require.config({"config": {
        "jsbuild":{"Threedadv_Customer/js/fabric.js":"/* build: `node build.js modules=ALL exclude=gestures,accessors,erasing requirejs minifier=uglifyjs` */\n/*! Fabric.js Copyright 2008-2015, Printio (Juriy Zaytsev, Maxim Chernyak) */\n\nvar fabric = fabric || { version: '5.2.1' };\nif (typeof exports !== 'undefined') {\n  exports.fabric = fabric;\n}\n/* _AMD_START_ */\nelse if (typeof define === 'function' && define.amd) {\n  define([], function() { return fabric; });\n}\n/* _AMD_END_ */\nif (typeof document !== 'undefined' && typeof window !== 'undefined') {\n  if (document instanceof (typeof HTMLDocument !== 'undefined' ? HTMLDocument : Document)) {\n    fabric.document = document;\n  }\n  else {\n    fabric.document = document.implementation.createHTMLDocument('');\n  }\n  fabric.window = window;\n}\nelse {\n  // assume we're running under node.js when document/window are not present\n  var jsdom = require('jsdom');\n  var virtualWindow = new jsdom.JSDOM(\n    decodeURIComponent('%3C!DOCTYPE%20html%3E%3Chtml%3E%3Chead%3E%3C%2Fhead%3E%3Cbody%3E%3C%2Fbody%3E%3C%2Fhtml%3E'),\n    {\n      features: {\n        FetchExternalResources: ['img']\n      },\n      resources: 'usable'\n    }).window;\n  fabric.document = virtualWindow.document;\n  fabric.jsdomImplForWrapper = require('jsdom/lib/jsdom/living/generated/utils').implForWrapper;\n  fabric.nodeCanvas = require('jsdom/lib/jsdom/utils').Canvas;\n  fabric.window = virtualWindow;\n  DOMParser = fabric.window.DOMParser;\n}\n\n/**\n * True when in environment that supports touch events\n * @type boolean\n */\nfabric.isTouchSupported = 'ontouchstart' in fabric.window || 'ontouchstart' in fabric.document ||\n  (fabric.window && fabric.window.navigator && fabric.window.navigator.maxTouchPoints > 0);\n\n/**\n * True when in environment that's probably Node.js\n * @type boolean\n */\nfabric.isLikelyNode = typeof Buffer !== 'undefined' &&\n                      typeof window === 'undefined';\n\n/* _FROM_SVG_START_ */\n/**\n * Attributes parsed from all SVG elements\n * @type array\n */\nfabric.SHARED_ATTRIBUTES = [\n  'display',\n  'transform',\n  'fill', 'fill-opacity', 'fill-rule',\n  'opacity',\n  'stroke', 'stroke-dasharray', 'stroke-linecap', 'stroke-dashoffset',\n  'stroke-linejoin', 'stroke-miterlimit',\n  'stroke-opacity', 'stroke-width',\n  'id', 'paint-order', 'vector-effect',\n  'instantiated_by_use', 'clip-path',\n];\n/* _FROM_SVG_END_ */\n\n/**\n * Pixel per Inch as a default value set to 96. Can be changed for more realistic conversion.\n */\nfabric.DPI = 96;\nfabric.reNum = '(?:[-+]?(?:\\\\d+|\\\\d*\\\\.\\\\d+)(?:[eE][-+]?\\\\d+)?)';\nfabric.commaWsp = '(?:\\\\s+,?\\\\s*|,\\\\s*)';\nfabric.rePathCommand = /([-+]?((\\d+\\.\\d+)|((\\d+)|(\\.\\d+)))(?:[eE][-+]?\\d+)?)/ig;\nfabric.reNonWord = /[ \\n\\.,;!\\?\\-]/;\nfabric.fontPaths = { };\nfabric.iMatrix = [1, 0, 0, 1, 0, 0];\nfabric.svgNS = 'http://www.w3.org/2000/svg';\n\n/**\n * Pixel limit for cache canvases. 1Mpx , 4Mpx should be fine.\n * @since 1.7.14\n * @type Number\n * @default\n */\nfabric.perfLimitSizeTotal = 2097152;\n\n/**\n * Pixel limit for cache canvases width or height. IE fixes the maximum at 5000\n * @since 1.7.14\n * @type Number\n * @default\n */\nfabric.maxCacheSideLimit = 4096;\n\n/**\n * Lowest pixel limit for cache canvases, set at 256PX\n * @since 1.7.14\n * @type Number\n * @default\n */\nfabric.minCacheSideLimit = 256;\n\n/**\n * Cache Object for widths of chars in text rendering.\n */\nfabric.charWidthsCache = { };\n\n/**\n * if webgl is enabled and available, textureSize will determine the size\n * of the canvas backend\n * @since 2.0.0\n * @type Number\n * @default\n */\nfabric.textureSize = 2048;\n\n/**\n * When 'true', style information is not retained when copy/pasting text, making\n * pasted text use destination style.\n * Defaults to 'false'.\n * @type Boolean\n * @default\n */\nfabric.disableStyleCopyPaste = false;\n\n/**\n * Enable webgl for filtering picture is available\n * A filtering backend will be initialized, this will both take memory and\n * time since a default 2048x2048 canvas will be created for the gl context\n * @since 2.0.0\n * @type Boolean\n * @default\n */\nfabric.enableGLFiltering = true;\n\n/**\n * Device Pixel Ratio\n * @see https://developer.apple.com/library/safari/documentation/AudioVideo/Conceptual/HTML-canvas-guide/SettingUptheCanvas/SettingUptheCanvas.html\n */\nfabric.devicePixelRatio = fabric.window.devicePixelRatio ||\n                          fabric.window.webkitDevicePixelRatio ||\n                          fabric.window.mozDevicePixelRatio ||\n                          1;\n/**\n * Browser-specific constant to adjust CanvasRenderingContext2D.shadowBlur value,\n * which is unitless and not rendered equally across browsers.\n *\n * Values that work quite well (as of October 2017) are:\n * - Chrome: 1.5\n * - Edge: 1.75\n * - Firefox: 0.9\n * - Safari: 0.95\n *\n * @since 2.0.0\n * @type Number\n * @default 1\n */\nfabric.browserShadowBlurConstant = 1;\n\n/**\n * This object contains the result of arc to bezier conversion for faster retrieving if the same arc needs to be converted again.\n * It was an internal variable, is accessible since version 2.3.4\n */\nfabric.arcToSegmentsCache = { };\n\n/**\n * This object keeps the results of the boundsOfCurve calculation mapped by the joined arguments necessary to calculate it.\n * It does speed up calculation, if you parse and add always the same paths, but in case of heavy usage of freedrawing\n * you do not get any speed benefit and you get a big object in memory.\n * The object was a private variable before, while now is appended to the lib so that you have access to it and you\n * can eventually clear it.\n * It was an internal variable, is accessible since version 2.3.4\n */\nfabric.boundsOfCurveCache = { };\n\n/**\n * If disabled boundsOfCurveCache is not used. For apps that make heavy usage of pencil drawing probably disabling it is better\n * @default true\n */\nfabric.cachesBoundsOfCurve = true;\n\n/**\n * Skip performance testing of setupGLContext and force the use of putImageData that seems to be the one that works best on\n * Chrome + old hardware. if your users are experiencing empty images after filtering you may try to force this to true\n * this has to be set before instantiating the filtering backend ( before filtering the first image )\n * @type Boolean\n * @default false\n */\nfabric.forceGLPutImageData = false;\n\nfabric.initFilterBackend = function() {\n  if (fabric.enableGLFiltering && fabric.isWebglSupported && fabric.isWebglSupported(fabric.textureSize)) {\n    console.log('max texture size: ' + fabric.maxTextureSize);\n    return (new fabric.WebglFilterBackend({ tileSize: fabric.textureSize }));\n  }\n  else if (fabric.Canvas2dFilterBackend) {\n    return (new fabric.Canvas2dFilterBackend());\n  }\n};\n\n\nif (typeof document !== 'undefined' && typeof window !== 'undefined') {\n  // ensure globality even if entire library were function wrapped (as in Meteor.js packaging system)\n  window.fabric = fabric;\n}\n\n\n(function() {\n\n  /**\n   * @private\n   * @param {String} eventName\n   * @param {Function} handler\n   */\n  function _removeEventListener(eventName, handler) {\n    if (!this.__eventListeners[eventName]) {\n      return;\n    }\n    var eventListener = this.__eventListeners[eventName];\n    if (handler) {\n      eventListener[eventListener.indexOf(handler)] = false;\n    }\n    else {\n      fabric.util.array.fill(eventListener, false);\n    }\n  }\n\n  /**\n   * Observes specified event\n   * @memberOf fabric.Observable\n   * @alias on\n   * @param {String|Object} eventName Event name (eg. 'after:render') or object with key/value pairs (eg. {'after:render': handler, 'selection:cleared': handler})\n   * @param {Function} handler Function that receives a notification when an event of the specified type occurs\n   * @return {Self} thisArg\n   * @chainable\n   */\n  function on(eventName, handler) {\n    if (!this.__eventListeners) {\n      this.__eventListeners = { };\n    }\n    // one object with key/value pairs was passed\n    if (arguments.length === 1) {\n      for (var prop in eventName) {\n        this.on(prop, eventName[prop]);\n      }\n    }\n    else {\n      if (!this.__eventListeners[eventName]) {\n        this.__eventListeners[eventName] = [];\n      }\n      this.__eventListeners[eventName].push(handler);\n    }\n    return this;\n  }\n\n  function _once(eventName, handler) {\n    var _handler = function () {\n      handler.apply(this, arguments);\n      this.off(eventName, _handler);\n    }.bind(this);\n    this.on(eventName, _handler);\n  }\n\n  function once(eventName, handler) {\n    // one object with key/value pairs was passed\n    if (arguments.length === 1) {\n      for (var prop in eventName) {\n        _once.call(this, prop, eventName[prop]);\n      }\n    }\n    else {\n      _once.call(this, eventName, handler);\n    }\n    return this;\n  }\n\n  /**\n   * Stops event observing for a particular event handler. Calling this method\n   * without arguments removes all handlers for all events\n   * @memberOf fabric.Observable\n   * @alias off\n   * @param {String|Object} eventName Event name (eg. 'after:render') or object with key/value pairs (eg. {'after:render': handler, 'selection:cleared': handler})\n   * @param {Function} handler Function to be deleted from EventListeners\n   * @return {Self} thisArg\n   * @chainable\n   */\n  function off(eventName, handler) {\n    if (!this.__eventListeners) {\n      return this;\n    }\n\n    // remove all key/value pairs (event name -> event handler)\n    if (arguments.length === 0) {\n      for (eventName in this.__eventListeners) {\n        _removeEventListener.call(this, eventName);\n      }\n    }\n    // one object with key/value pairs was passed\n    else if (arguments.length === 1 && typeof arguments[0] === 'object') {\n      for (var prop in eventName) {\n        _removeEventListener.call(this, prop, eventName[prop]);\n      }\n    }\n    else {\n      _removeEventListener.call(this, eventName, handler);\n    }\n    return this;\n  }\n\n  /**\n   * Fires event with an optional options object\n   * @memberOf fabric.Observable\n   * @param {String} eventName Event name to fire\n   * @param {Object} [options] Options object\n   * @return {Self} thisArg\n   * @chainable\n   */\n  function fire(eventName, options) {\n    if (!this.__eventListeners) {\n      return this;\n    }\n\n    var listenersForEvent = this.__eventListeners[eventName];\n    if (!listenersForEvent) {\n      return this;\n    }\n\n    for (var i = 0, len = listenersForEvent.length; i < len; i++) {\n      listenersForEvent[i] && listenersForEvent[i].call(this, options || { });\n    }\n    this.__eventListeners[eventName] = listenersForEvent.filter(function(value) {\n      return value !== false;\n    });\n    return this;\n  }\n\n  /**\n   * @namespace fabric.Observable\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-2#events}\n   * @see {@link http://fabricjs.com/events|Events demo}\n   */\n  fabric.Observable = {\n    fire: fire,\n    on: on,\n    once: once,\n    off: off,\n  };\n})();\n\n\n/**\n * @namespace fabric.Collection\n */\nfabric.Collection = {\n\n  _objects: [],\n\n  /**\n   * Adds objects to collection, Canvas or Group, then renders canvas\n   * (if `renderOnAddRemove` is not `false`).\n   * in case of Group no changes to bounding box are made.\n   * Objects should be instances of (or inherit from) fabric.Object\n   * Use of this function is highly discouraged for groups.\n   * you can add a bunch of objects with the add method but then you NEED\n   * to run a addWithUpdate call for the Group class or position/bbox will be wrong.\n   * @param {...fabric.Object} object Zero or more fabric instances\n   * @return {Self} thisArg\n   * @chainable\n   */\n  add: function () {\n    this._objects.push.apply(this._objects, arguments);\n    if (this._onObjectAdded) {\n      for (var i = 0, length = arguments.length; i < length; i++) {\n        this._onObjectAdded(arguments[i]);\n      }\n    }\n    this.renderOnAddRemove && this.requestRenderAll();\n    return this;\n  },\n\n  /**\n   * Inserts an object into collection at specified index, then renders canvas (if `renderOnAddRemove` is not `false`)\n   * An object should be an instance of (or inherit from) fabric.Object\n   * Use of this function is highly discouraged for groups.\n   * you can add a bunch of objects with the insertAt method but then you NEED\n   * to run a addWithUpdate call for the Group class or position/bbox will be wrong.\n   * @param {Object} object Object to insert\n   * @param {Number} index Index to insert object at\n   * @param {Boolean} nonSplicing When `true`, no splicing (shifting) of objects occurs\n   * @return {Self} thisArg\n   * @chainable\n   */\n  insertAt: function (object, index, nonSplicing) {\n    var objects = this._objects;\n    if (nonSplicing) {\n      objects[index] = object;\n    }\n    else {\n      objects.splice(index, 0, object);\n    }\n    this._onObjectAdded && this._onObjectAdded(object);\n    this.renderOnAddRemove && this.requestRenderAll();\n    return this;\n  },\n\n  /**\n   * Removes objects from a collection, then renders canvas (if `renderOnAddRemove` is not `false`)\n   * @param {...fabric.Object} object Zero or more fabric instances\n   * @return {Self} thisArg\n   * @chainable\n   */\n  remove: function() {\n    var objects = this._objects,\n        index, somethingRemoved = false;\n\n    for (var i = 0, length = arguments.length; i < length; i++) {\n      index = objects.indexOf(arguments[i]);\n\n      // only call onObjectRemoved if an object was actually removed\n      if (index !== -1) {\n        somethingRemoved = true;\n        objects.splice(index, 1);\n        this._onObjectRemoved && this._onObjectRemoved(arguments[i]);\n      }\n    }\n\n    this.renderOnAddRemove && somethingRemoved && this.requestRenderAll();\n    return this;\n  },\n\n  /**\n   * Executes given function for each object in this group\n   * @param {Function} callback\n   *                   Callback invoked with current object as first argument,\n   *                   index - as second and an array of all objects - as third.\n   *                   Callback is invoked in a context of Global Object (e.g. `window`)\n   *                   when no `context` argument is given\n   *\n   * @param {Object} context Context (aka thisObject)\n   * @return {Self} thisArg\n   * @chainable\n   */\n  forEachObject: function(callback, context) {\n    var objects = this.getObjects();\n    for (var i = 0, len = objects.length; i < len; i++) {\n      callback.call(context, objects[i], i, objects);\n    }\n    return this;\n  },\n\n  /**\n   * Returns an array of children objects of this instance\n   * Type parameter introduced in 1.3.10\n   * since 2.3.5 this method return always a COPY of the array;\n   * @param {String} [type] When specified, only objects of this type are returned\n   * @return {Array}\n   */\n  getObjects: function(type) {\n    if (typeof type === 'undefined') {\n      return this._objects.concat();\n    }\n    return this._objects.filter(function(o) {\n      return o.type === type;\n    });\n  },\n\n  /**\n   * Returns object at specified index\n   * @param {Number} index\n   * @return {Self} thisArg\n   */\n  item: function (index) {\n    return this._objects[index];\n  },\n\n  /**\n   * Returns true if collection contains no objects\n   * @return {Boolean} true if collection is empty\n   */\n  isEmpty: function () {\n    return this._objects.length === 0;\n  },\n\n  /**\n   * Returns a size of a collection (i.e: length of an array containing its objects)\n   * @return {Number} Collection size\n   */\n  size: function() {\n    return this._objects.length;\n  },\n\n  /**\n   * Returns true if collection contains an object\n   * @param {Object} object Object to check against\n   * @param {Boolean} [deep=false] `true` to check all descendants, `false` to check only `_objects`\n   * @return {Boolean} `true` if collection contains an object\n   */\n  contains: function (object, deep) {\n    if (this._objects.indexOf(object) > -1) {\n      return true;\n    }\n    else if (deep) {\n      return this._objects.some(function (obj) {\n        return typeof obj.contains === 'function' && obj.contains(object, true);\n      });\n    }\n    return false;\n  },\n\n  /**\n   * Returns number representation of a collection complexity\n   * @return {Number} complexity\n   */\n  complexity: function () {\n    return this._objects.reduce(function (memo, current) {\n      memo += current.complexity ? current.complexity() : 0;\n      return memo;\n    }, 0);\n  }\n};\n\n\n/**\n * @namespace fabric.CommonMethods\n */\nfabric.CommonMethods = {\n\n  /**\n   * Sets object's properties from options\n   * @param {Object} [options] Options object\n   */\n  _setOptions: function(options) {\n    for (var prop in options) {\n      this.set(prop, options[prop]);\n    }\n  },\n\n  /**\n   * @private\n   * @param {Object} [filler] Options object\n   * @param {String} [property] property to set the Gradient to\n   */\n  _initGradient: function(filler, property) {\n    if (filler && filler.colorStops && !(filler instanceof fabric.Gradient)) {\n      this.set(property, new fabric.Gradient(filler));\n    }\n  },\n\n  /**\n   * @private\n   * @param {Object} [filler] Options object\n   * @param {String} [property] property to set the Pattern to\n   * @param {Function} [callback] callback to invoke after pattern load\n   */\n  _initPattern: function(filler, property, callback) {\n    if (filler && filler.source && !(filler instanceof fabric.Pattern)) {\n      this.set(property, new fabric.Pattern(filler, callback));\n    }\n    else {\n      callback && callback();\n    }\n  },\n\n  /**\n   * @private\n   */\n  _setObject: function(obj) {\n    for (var prop in obj) {\n      this._set(prop, obj[prop]);\n    }\n  },\n\n  /**\n   * Sets property to a given value. When changing position/dimension -related properties (left, top, scale, angle, etc.) `set` does not update position of object's borders/controls. If you need to update those, call `setCoords()`.\n   * @param {String|Object} key Property name or object (if object, iterate over the object properties)\n   * @param {Object|Function} value Property value (if function, the value is passed into it and its return value is used as a new one)\n   * @return {fabric.Object} thisArg\n   * @chainable\n   */\n  set: function(key, value) {\n    if (typeof key === 'object') {\n      this._setObject(key);\n    }\n    else {\n      this._set(key, value);\n    }\n    return this;\n  },\n\n  _set: function(key, value) {\n    this[key] = value;\n  },\n\n  /**\n   * Toggles specified property from `true` to `false` or from `false` to `true`\n   * @param {String} property Property to toggle\n   * @return {fabric.Object} thisArg\n   * @chainable\n   */\n  toggle: function(property) {\n    var value = this.get(property);\n    if (typeof value === 'boolean') {\n      this.set(property, !value);\n    }\n    return this;\n  },\n\n  /**\n   * Basic getter\n   * @param {String} property Property name\n   * @return {*} value of a property\n   */\n  get: function(property) {\n    return this[property];\n  }\n};\n\n\n(function(global) {\n\n  var sqrt = Math.sqrt,\n      atan2 = Math.atan2,\n      pow = Math.pow,\n      PiBy180 = Math.PI / 180,\n      PiBy2 = Math.PI / 2;\n\n  /**\n   * @namespace fabric.util\n   */\n  fabric.util = {\n\n    /**\n     * Calculate the cos of an angle, avoiding returning floats for known results\n     * @static\n     * @memberOf fabric.util\n     * @param {Number} angle the angle in radians or in degree\n     * @return {Number}\n     */\n    cos: function(angle) {\n      if (angle === 0) { return 1; }\n      if (angle < 0) {\n        // cos(a) = cos(-a)\n        angle = -angle;\n      }\n      var angleSlice = angle / PiBy2;\n      switch (angleSlice) {\n        case 1: case 3: return 0;\n        case 2: return -1;\n      }\n      return Math.cos(angle);\n    },\n\n    /**\n     * Calculate the sin of an angle, avoiding returning floats for known results\n     * @static\n     * @memberOf fabric.util\n     * @param {Number} angle the angle in radians or in degree\n     * @return {Number}\n     */\n    sin: function(angle) {\n      if (angle === 0) { return 0; }\n      var angleSlice = angle / PiBy2, sign = 1;\n      if (angle < 0) {\n        // sin(-a) = -sin(a)\n        sign = -1;\n      }\n      switch (angleSlice) {\n        case 1: return sign;\n        case 2: return 0;\n        case 3: return -sign;\n      }\n      return Math.sin(angle);\n    },\n\n    /**\n     * Removes value from an array.\n     * Presence of value (and its position in an array) is determined via `Array.prototype.indexOf`\n     * @static\n     * @memberOf fabric.util\n     * @param {Array} array\n     * @param {*} value\n     * @return {Array} original array\n     */\n    removeFromArray: function(array, value) {\n      var idx = array.indexOf(value);\n      if (idx !== -1) {\n        array.splice(idx, 1);\n      }\n      return array;\n    },\n\n    /**\n     * Returns random number between 2 specified ones.\n     * @static\n     * @memberOf fabric.util\n     * @param {Number} min lower limit\n     * @param {Number} max upper limit\n     * @return {Number} random value (between min and max)\n     */\n    getRandomInt: function(min, max) {\n      return Math.floor(Math.random() * (max - min + 1)) + min;\n    },\n\n    /**\n     * Transforms degrees to radians.\n     * @static\n     * @memberOf fabric.util\n     * @param {Number} degrees value in degrees\n     * @return {Number} value in radians\n     */\n    degreesToRadians: function(degrees) {\n      return degrees * PiBy180;\n    },\n\n    /**\n     * Transforms radians to degrees.\n     * @static\n     * @memberOf fabric.util\n     * @param {Number} radians value in radians\n     * @return {Number} value in degrees\n     */\n    radiansToDegrees: function(radians) {\n      return radians / PiBy180;\n    },\n\n    /**\n     * Rotates `point` around `origin` with `radians`\n     * @static\n     * @memberOf fabric.util\n     * @param {fabric.Point} point The point to rotate\n     * @param {fabric.Point} origin The origin of the rotation\n     * @param {Number} radians The radians of the angle for the rotation\n     * @return {fabric.Point} The new rotated point\n     */\n    rotatePoint: function(point, origin, radians) {\n      var newPoint = new fabric.Point(point.x - origin.x, point.y - origin.y),\n          v = fabric.util.rotateVector(newPoint, radians);\n      return new fabric.Point(v.x, v.y).addEquals(origin);\n    },\n\n    /**\n     * Rotates `vector` with `radians`\n     * @static\n     * @memberOf fabric.util\n     * @param {Object} vector The vector to rotate (x and y)\n     * @param {Number} radians The radians of the angle for the rotation\n     * @return {Object} The new rotated point\n     */\n    rotateVector: function(vector, radians) {\n      var sin = fabric.util.sin(radians),\n          cos = fabric.util.cos(radians),\n          rx = vector.x * cos - vector.y * sin,\n          ry = vector.x * sin + vector.y * cos;\n      return {\n        x: rx,\n        y: ry\n      };\n    },\n\n    /**\n     * Creates a vetor from points represented as a point\n     * @static\n     * @memberOf fabric.util\n     *\n     * @typedef {Object} Point\n     * @property {number} x\n     * @property {number} y\n     *\n     * @param {Point} from\n     * @param {Point} to\n     * @returns {Point} vector\n     */\n    createVector: function (from, to) {\n      return new fabric.Point(to.x - from.x, to.y - from.y);\n    },\n\n    /**\n     * Calculates angle between 2 vectors using dot product\n     * @static\n     * @memberOf fabric.util\n     * @param {Point} a\n     * @param {Point} b\n     * @returns the angle in radian between the vectors\n     */\n    calcAngleBetweenVectors: function (a, b) {\n      return Math.acos((a.x * b.x + a.y * b.y) / (Math.hypot(a.x, a.y) * Math.hypot(b.x, b.y)));\n    },\n\n    /**\n     * @static\n     * @memberOf fabric.util\n     * @param {Point} v\n     * @returns {Point} vector representing the unit vector of pointing to the direction of `v`\n     */\n    getHatVector: function (v) {\n      return new fabric.Point(v.x, v.y).multiply(1 / Math.hypot(v.x, v.y));\n    },\n\n    /**\n     * @static\n     * @memberOf fabric.util\n     * @param {Point} A\n     * @param {Point} B\n     * @param {Point} C\n     * @returns {{ vector: Point, angle: number }} vector representing the bisector of A and A's angle\n     */\n    getBisector: function (A, B, C) {\n      var AB = fabric.util.createVector(A, B), AC = fabric.util.createVector(A, C);\n      var alpha = fabric.util.calcAngleBetweenVectors(AB, AC);\n      //  check if alpha is relative to AB->BC\n      var ro = fabric.util.calcAngleBetweenVectors(fabric.util.rotateVector(AB, alpha), AC);\n      var phi = alpha * (ro === 0 ? 1 : -1) / 2;\n      return {\n        vector: fabric.util.getHatVector(fabric.util.rotateVector(AB, phi)),\n        angle: alpha\n      };\n    },\n\n    /**\n     * Project stroke width on points returning 2 projections for each point as follows:\n     * - `miter`: 2 points corresponding to the outer boundary and the inner boundary of stroke.\n     * - `bevel`: 2 points corresponding to the bevel boundaries, tangent to the bisector.\n     * - `round`: same as `bevel`\n     * Used to calculate object's bounding box\n     * @static\n     * @memberOf fabric.util\n     * @param {Point[]} points\n     * @param {Object} options\n     * @param {number} options.strokeWidth\n     * @param {'miter'|'bevel'|'round'} options.strokeLineJoin\n     * @param {number} options.strokeMiterLimit https://developer.mozilla.org/en-US/docs/Web/SVG/Attribute/stroke-miterlimit\n     * @param {boolean} options.strokeUniform\n     * @param {number} options.scaleX\n     * @param {number} options.scaleY\n     * @param {boolean} [openPath] whether the shape is open or not, affects the calculations of the first and last points\n     * @returns {fabric.Point[]} array of size 2n/4n of all suspected points\n     */\n    projectStrokeOnPoints: function (points, options, openPath) {\n      var coords = [], s = options.strokeWidth / 2,\n          strokeUniformScalar = options.strokeUniform ?\n            new fabric.Point(1 / options.scaleX, 1 / options.scaleY) : new fabric.Point(1, 1),\n          getStrokeHatVector = function (v) {\n            var scalar = s / (Math.hypot(v.x, v.y));\n            return new fabric.Point(v.x * scalar * strokeUniformScalar.x, v.y * scalar * strokeUniformScalar.y);\n          };\n      if (points.length <= 1) {return coords;}\n      points.forEach(function (p, index) {\n        var A = new fabric.Point(p.x, p.y), B, C;\n        if (index === 0) {\n          C = points[index + 1];\n          B = openPath ? getStrokeHatVector(fabric.util.createVector(C, A)).addEquals(A) : points[points.length - 1];\n        }\n        else if (index === points.length - 1) {\n          B = points[index - 1];\n          C = openPath ? getStrokeHatVector(fabric.util.createVector(B, A)).addEquals(A) : points[0];\n        }\n        else {\n          B = points[index - 1];\n          C = points[index + 1];\n        }\n        var bisector = fabric.util.getBisector(A, B, C),\n            bisectorVector = bisector.vector,\n            alpha = bisector.angle,\n            scalar,\n            miterVector;\n        if (options.strokeLineJoin === 'miter') {\n          scalar = -s / Math.sin(alpha / 2);\n          miterVector = new fabric.Point(\n            bisectorVector.x * scalar * strokeUniformScalar.x,\n            bisectorVector.y * scalar * strokeUniformScalar.y\n          );\n          if (Math.hypot(miterVector.x, miterVector.y) / s <= options.strokeMiterLimit) {\n            coords.push(A.add(miterVector));\n            coords.push(A.subtract(miterVector));\n            return;\n          }\n        }\n        scalar = -s * Math.SQRT2;\n        miterVector = new fabric.Point(\n          bisectorVector.x * scalar * strokeUniformScalar.x,\n          bisectorVector.y * scalar * strokeUniformScalar.y\n        );\n        coords.push(A.add(miterVector));\n        coords.push(A.subtract(miterVector));\n      });\n      return coords;\n    },\n\n    /**\n     * Apply transform t to point p\n     * @static\n     * @memberOf fabric.util\n     * @param  {fabric.Point} p The point to transform\n     * @param  {Array} t The transform\n     * @param  {Boolean} [ignoreOffset] Indicates that the offset should not be applied\n     * @return {fabric.Point} The transformed point\n     */\n    transformPoint: function(p, t, ignoreOffset) {\n      if (ignoreOffset) {\n        return new fabric.Point(\n          t[0] * p.x + t[2] * p.y,\n          t[1] * p.x + t[3] * p.y\n        );\n      }\n      return new fabric.Point(\n        t[0] * p.x + t[2] * p.y + t[4],\n        t[1] * p.x + t[3] * p.y + t[5]\n      );\n    },\n\n    /**\n     * Returns coordinates of points's bounding rectangle (left, top, width, height)\n     * @param {Array} points 4 points array\n     * @param {Array} [transform] an array of 6 numbers representing a 2x3 transform matrix\n     * @return {Object} Object with left, top, width, height properties\n     */\n    makeBoundingBoxFromPoints: function(points, transform) {\n      if (transform) {\n        for (var i = 0; i < points.length; i++) {\n          points[i] = fabric.util.transformPoint(points[i], transform);\n        }\n      }\n      var xPoints = [points[0].x, points[1].x, points[2].x, points[3].x],\n          minX = fabric.util.array.min(xPoints),\n          maxX = fabric.util.array.max(xPoints),\n          width = maxX - minX,\n          yPoints = [points[0].y, points[1].y, points[2].y, points[3].y],\n          minY = fabric.util.array.min(yPoints),\n          maxY = fabric.util.array.max(yPoints),\n          height = maxY - minY;\n\n      return {\n        left: minX,\n        top: minY,\n        width: width,\n        height: height\n      };\n    },\n\n    /**\n     * Invert transformation t\n     * @static\n     * @memberOf fabric.util\n     * @param {Array} t The transform\n     * @return {Array} The inverted transform\n     */\n    invertTransform: function(t) {\n      var a = 1 / (t[0] * t[3] - t[1] * t[2]),\n          r = [a * t[3], -a * t[1], -a * t[2], a * t[0]],\n          o = fabric.util.transformPoint({ x: t[4], y: t[5] }, r, true);\n      r[4] = -o.x;\n      r[5] = -o.y;\n      return r;\n    },\n\n    /**\n     * A wrapper around Number#toFixed, which contrary to native method returns number, not string.\n     * @static\n     * @memberOf fabric.util\n     * @param {Number|String} number number to operate on\n     * @param {Number} fractionDigits number of fraction digits to \"leave\"\n     * @return {Number}\n     */\n    toFixed: function(number, fractionDigits) {\n      return parseFloat(Number(number).toFixed(fractionDigits));\n    },\n\n    /**\n     * Converts from attribute value to pixel value if applicable.\n     * Returns converted pixels or original value not converted.\n     * @param {Number|String} value number to operate on\n     * @param {Number} fontSize\n     * @return {Number|String}\n     */\n    parseUnit: function(value, fontSize) {\n      var unit = /\\D{0,2}$/.exec(value),\n          number = parseFloat(value);\n      if (!fontSize) {\n        fontSize = fabric.Text.DEFAULT_SVG_FONT_SIZE;\n      }\n      switch (unit[0]) {\n        case 'mm':\n          return number * fabric.DPI / 25.4;\n\n        case 'cm':\n          return number * fabric.DPI / 2.54;\n\n        case 'in':\n          return number * fabric.DPI;\n\n        case 'pt':\n          return number * fabric.DPI / 72; // or * 4 / 3\n\n        case 'pc':\n          return number * fabric.DPI / 72 * 12; // or * 16\n\n        case 'em':\n          return number * fontSize;\n\n        default:\n          return number;\n      }\n    },\n\n    /**\n     * Function which always returns `false`.\n     * @static\n     * @memberOf fabric.util\n     * @return {Boolean}\n     */\n    falseFunction: function() {\n      return false;\n    },\n\n    /**\n     * Returns klass \"Class\" object of given namespace\n     * @memberOf fabric.util\n     * @param {String} type Type of object (eg. 'circle')\n     * @param {String} namespace Namespace to get klass \"Class\" object from\n     * @return {Object} klass \"Class\"\n     */\n    getKlass: function(type, namespace) {\n      // capitalize first letter only\n      type = fabric.util.string.camelize(type.charAt(0).toUpperCase() + type.slice(1));\n      return fabric.util.resolveNamespace(namespace)[type];\n    },\n\n    /**\n     * Returns array of attributes for given svg that fabric parses\n     * @memberOf fabric.util\n     * @param {String} type Type of svg element (eg. 'circle')\n     * @return {Array} string names of supported attributes\n     */\n    getSvgAttributes: function(type) {\n      var attributes = [\n        'instantiated_by_use',\n        'style',\n        'id',\n        'class'\n      ];\n      switch (type) {\n        case 'linearGradient':\n          attributes = attributes.concat(['x1', 'y1', 'x2', 'y2', 'gradientUnits', 'gradientTransform']);\n          break;\n        case 'radialGradient':\n          attributes = attributes.concat(['gradientUnits', 'gradientTransform', 'cx', 'cy', 'r', 'fx', 'fy', 'fr']);\n          break;\n        case 'stop':\n          attributes = attributes.concat(['offset', 'stop-color', 'stop-opacity']);\n          break;\n      }\n      return attributes;\n    },\n\n    /**\n     * Returns object of given namespace\n     * @memberOf fabric.util\n     * @param {String} namespace Namespace string e.g. 'fabric.Image.filter' or 'fabric'\n     * @return {Object} Object for given namespace (default fabric)\n     */\n    resolveNamespace: function(namespace) {\n      if (!namespace) {\n        return fabric;\n      }\n\n      var parts = namespace.split('.'),\n          len = parts.length, i,\n          obj = global || fabric.window;\n\n      for (i = 0; i < len; ++i) {\n        obj = obj[parts[i]];\n      }\n\n      return obj;\n    },\n\n    /**\n     * Loads image element from given url and passes it to a callback\n     * @memberOf fabric.util\n     * @param {String} url URL representing an image\n     * @param {Function} callback Callback; invoked with loaded image\n     * @param {*} [context] Context to invoke callback in\n     * @param {Object} [crossOrigin] crossOrigin value to set image element to\n     */\n    loadImage: function(url, callback, context, crossOrigin) {\n      if (!url) {\n        callback && callback.call(context, url);\n        return;\n      }\n\n      var img = fabric.util.createImage();\n\n      /** @ignore */\n      var onLoadCallback = function () {\n        callback && callback.call(context, img, false);\n        img = img.onload = img.onerror = null;\n      };\n\n      img.onload = onLoadCallback;\n      /** @ignore */\n      img.onerror = function() {\n        fabric.log('Error loading ' + img.src);\n        callback && callback.call(context, null, true);\n        img = img.onload = img.onerror = null;\n      };\n\n      // data-urls appear to be buggy with crossOrigin\n      // https://github.com/kangax/fabric.js/commit/d0abb90f1cd5c5ef9d2a94d3fb21a22330da3e0a#commitcomment-4513767\n      // see https://code.google.com/p/chromium/issues/detail?id=315152\n      //     https://bugzilla.mozilla.org/show_bug.cgi?id=935069\n      // crossOrigin null is the same as not set.\n      if (url.indexOf('data') !== 0 &&\n        crossOrigin !== undefined &&\n        crossOrigin !== null) {\n        img.crossOrigin = crossOrigin;\n      }\n\n      // IE10 / IE11-Fix: SVG contents from data: URI\n      // will only be available if the IMG is present\n      // in the DOM (and visible)\n      if (url.substring(0,14) === 'data:image/svg') {\n        img.onload = null;\n        fabric.util.loadImageInDom(img, onLoadCallback);\n      }\n\n      img.src = url;\n    },\n\n    /**\n     * Attaches SVG image with data: URL to the dom\n     * @memberOf fabric.util\n     * @param {Object} img Image object with data:image/svg src\n     * @param {Function} callback Callback; invoked with loaded image\n     * @return {Object} DOM element (div containing the SVG image)\n     */\n    loadImageInDom: function(img, onLoadCallback) {\n      var div = fabric.document.createElement('div');\n      div.style.width = div.style.height = '1px';\n      div.style.left = div.style.top = '-100%';\n      div.style.position = 'absolute';\n      div.appendChild(img);\n      fabric.document.querySelector('body').appendChild(div);\n      /**\n       * Wrap in function to:\n       *   1. Call existing callback\n       *   2. Cleanup DOM\n       */\n      img.onload = function () {\n        onLoadCallback();\n        div.parentNode.removeChild(div);\n        div = null;\n      };\n    },\n\n    /**\n     * Creates corresponding fabric instances from their object representations\n     * @static\n     * @memberOf fabric.util\n     * @param {Array} objects Objects to enliven\n     * @param {Function} callback Callback to invoke when all objects are created\n     * @param {String} namespace Namespace to get klass \"Class\" object from\n     * @param {Function} reviver Method for further parsing of object elements,\n     * called after each fabric object created.\n     */\n    enlivenObjects: function(objects, callback, namespace, reviver) {\n      objects = objects || [];\n\n      var enlivenedObjects = [],\n          numLoadedObjects = 0,\n          numTotalObjects = objects.length;\n\n      function onLoaded() {\n        if (++numLoadedObjects === numTotalObjects) {\n          callback && callback(enlivenedObjects.filter(function(obj) {\n            // filter out undefined objects (objects that gave error)\n            return obj;\n          }));\n        }\n      }\n\n      if (!numTotalObjects) {\n        callback && callback(enlivenedObjects);\n        return;\n      }\n\n      objects.forEach(function (o, index) {\n        // if sparse array\n        if (!o || !o.type) {\n          onLoaded();\n          return;\n        }\n        var klass = fabric.util.getKlass(o.type, namespace);\n        klass.fromObject(o, function (obj, error) {\n          error || (enlivenedObjects[index] = obj);\n          reviver && reviver(o, obj, error);\n          onLoaded();\n        });\n      });\n    },\n\n    /**\n     * Creates corresponding fabric instances residing in an object, e.g. `clipPath`\n     * @see {@link fabric.Object.ENLIVEN_PROPS}\n     * @param {Object} object\n     * @param {Object} [context] assign enlived props to this object (pass null to skip this)\n     * @param {(objects:fabric.Object[]) => void} callback\n     */\n    enlivenObjectEnlivables: function (object, context, callback) {\n      var enlivenProps = fabric.Object.ENLIVEN_PROPS.filter(function (key) { return !!object[key]; });\n      fabric.util.enlivenObjects(enlivenProps.map(function (key) { return object[key]; }), function (enlivedProps) {\n        var objects = {};\n        enlivenProps.forEach(function (key, index) {\n          objects[key] = enlivedProps[index];\n          context && (context[key] = enlivedProps[index]);\n        });\n        callback && callback(objects);\n      });\n    },\n\n    /**\n     * Create and wait for loading of patterns\n     * @static\n     * @memberOf fabric.util\n     * @param {Array} patterns Objects to enliven\n     * @param {Function} callback Callback to invoke when all objects are created\n     * called after each fabric object created.\n     */\n    enlivenPatterns: function(patterns, callback) {\n      patterns = patterns || [];\n\n      function onLoaded() {\n        if (++numLoadedPatterns === numPatterns) {\n          callback && callback(enlivenedPatterns);\n        }\n      }\n\n      var enlivenedPatterns = [],\n          numLoadedPatterns = 0,\n          numPatterns = patterns.length;\n\n      if (!numPatterns) {\n        callback && callback(enlivenedPatterns);\n        return;\n      }\n\n      patterns.forEach(function (p, index) {\n        if (p && p.source) {\n          new fabric.Pattern(p, function(pattern) {\n            enlivenedPatterns[index] = pattern;\n            onLoaded();\n          });\n        }\n        else {\n          enlivenedPatterns[index] = p;\n          onLoaded();\n        }\n      });\n    },\n\n    /**\n     * Groups SVG elements (usually those retrieved from SVG document)\n     * @static\n     * @memberOf fabric.util\n     * @param {Array} elements SVG elements to group\n     * @param {Object} [options] Options object\n     * @param {String} path Value to set sourcePath to\n     * @return {fabric.Object|fabric.Group}\n     */\n    groupSVGElements: function(elements, options, path) {\n      var object;\n      if (elements && elements.length === 1) {\n        return elements[0];\n      }\n      if (options) {\n        if (options.width && options.height) {\n          options.centerPoint = {\n            x: options.width / 2,\n            y: options.height / 2\n          };\n        }\n        else {\n          delete options.width;\n          delete options.height;\n        }\n      }\n      object = new fabric.Group(elements, options);\n      if (typeof path !== 'undefined') {\n        object.sourcePath = path;\n      }\n      return object;\n    },\n\n    /**\n     * Populates an object with properties of another object\n     * @static\n     * @memberOf fabric.util\n     * @param {Object} source Source object\n     * @param {Object} destination Destination object\n     * @return {Array} properties Properties names to include\n     */\n    populateWithProperties: function(source, destination, properties) {\n      if (properties && Array.isArray(properties)) {\n        for (var i = 0, len = properties.length; i < len; i++) {\n          if (properties[i] in source) {\n            destination[properties[i]] = source[properties[i]];\n          }\n        }\n      }\n    },\n\n    /**\n     * Creates canvas element\n     * @static\n     * @memberOf fabric.util\n     * @return {CanvasElement} initialized canvas element\n     */\n    createCanvasElement: function() {\n      return fabric.document.createElement('canvas');\n    },\n\n    /**\n     * Creates a canvas element that is a copy of another and is also painted\n     * @param {CanvasElement} canvas to copy size and content of\n     * @static\n     * @memberOf fabric.util\n     * @return {CanvasElement} initialized canvas element\n     */\n    copyCanvasElement: function(canvas) {\n      var newCanvas = fabric.util.createCanvasElement();\n      newCanvas.width = canvas.width;\n      newCanvas.height = canvas.height;\n      newCanvas.getContext('2d').drawImage(canvas, 0, 0);\n      return newCanvas;\n    },\n\n    /**\n     * since 2.6.0 moved from canvas instance to utility.\n     * @param {CanvasElement} canvasEl to copy size and content of\n     * @param {String} format 'jpeg' or 'png', in some browsers 'webp' is ok too\n     * @param {Number} quality <= 1 and > 0\n     * @static\n     * @memberOf fabric.util\n     * @return {String} data url\n     */\n    toDataURL: function(canvasEl, format, quality) {\n      return canvasEl.toDataURL('image/' + format, quality);\n    },\n\n    /**\n     * Creates image element (works on client and node)\n     * @static\n     * @memberOf fabric.util\n     * @return {HTMLImageElement} HTML image element\n     */\n    createImage: function() {\n      return fabric.document.createElement('img');\n    },\n\n    /**\n     * Multiply matrix A by matrix B to nest transformations\n     * @static\n     * @memberOf fabric.util\n     * @param  {Array} a First transformMatrix\n     * @param  {Array} b Second transformMatrix\n     * @param  {Boolean} is2x2 flag to multiply matrices as 2x2 matrices\n     * @return {Array} The product of the two transform matrices\n     */\n    multiplyTransformMatrices: function(a, b, is2x2) {\n      // Matrix multiply a * b\n      return [\n        a[0] * b[0] + a[2] * b[1],\n        a[1] * b[0] + a[3] * b[1],\n        a[0] * b[2] + a[2] * b[3],\n        a[1] * b[2] + a[3] * b[3],\n        is2x2 ? 0 : a[0] * b[4] + a[2] * b[5] + a[4],\n        is2x2 ? 0 : a[1] * b[4] + a[3] * b[5] + a[5]\n      ];\n    },\n\n    /**\n     * Decomposes standard 2x3 matrix into transform components\n     * @static\n     * @memberOf fabric.util\n     * @param  {Array} a transformMatrix\n     * @return {Object} Components of transform\n     */\n    qrDecompose: function(a) {\n      var angle = atan2(a[1], a[0]),\n          denom = pow(a[0], 2) + pow(a[1], 2),\n          scaleX = sqrt(denom),\n          scaleY = (a[0] * a[3] - a[2] * a[1]) / scaleX,\n          skewX = atan2(a[0] * a[2] + a[1] * a [3], denom);\n      return {\n        angle: angle / PiBy180,\n        scaleX: scaleX,\n        scaleY: scaleY,\n        skewX: skewX / PiBy180,\n        skewY: 0,\n        translateX: a[4],\n        translateY: a[5]\n      };\n    },\n\n    /**\n     * Returns a transform matrix starting from an object of the same kind of\n     * the one returned from qrDecompose, useful also if you want to calculate some\n     * transformations from an object that is not enlived yet\n     * @static\n     * @memberOf fabric.util\n     * @param  {Object} options\n     * @param  {Number} [options.angle] angle in degrees\n     * @return {Number[]} transform matrix\n     */\n    calcRotateMatrix: function(options) {\n      if (!options.angle) {\n        return fabric.iMatrix.concat();\n      }\n      var theta = fabric.util.degreesToRadians(options.angle),\n          cos = fabric.util.cos(theta),\n          sin = fabric.util.sin(theta);\n      return [cos, sin, -sin, cos, 0, 0];\n    },\n\n    /**\n     * Returns a transform matrix starting from an object of the same kind of\n     * the one returned from qrDecompose, useful also if you want to calculate some\n     * transformations from an object that is not enlived yet.\n     * is called DimensionsTransformMatrix because those properties are the one that influence\n     * the size of the resulting box of the object.\n     * @static\n     * @memberOf fabric.util\n     * @param  {Object} options\n     * @param  {Number} [options.scaleX]\n     * @param  {Number} [options.scaleY]\n     * @param  {Boolean} [options.flipX]\n     * @param  {Boolean} [options.flipY]\n     * @param  {Number} [options.skewX]\n     * @param  {Number} [options.skewY]\n     * @return {Number[]} transform matrix\n     */\n    calcDimensionsMatrix: function(options) {\n      var scaleX = typeof options.scaleX === 'undefined' ? 1 : options.scaleX,\n          scaleY = typeof options.scaleY === 'undefined' ? 1 : options.scaleY,\n          scaleMatrix = [\n            options.flipX ? -scaleX : scaleX,\n            0,\n            0,\n            options.flipY ? -scaleY : scaleY,\n            0,\n            0],\n          multiply = fabric.util.multiplyTransformMatrices,\n          degreesToRadians = fabric.util.degreesToRadians;\n      if (options.skewX) {\n        scaleMatrix = multiply(\n          scaleMatrix,\n          [1, 0, Math.tan(degreesToRadians(options.skewX)), 1],\n          true);\n      }\n      if (options.skewY) {\n        scaleMatrix = multiply(\n          scaleMatrix,\n          [1, Math.tan(degreesToRadians(options.skewY)), 0, 1],\n          true);\n      }\n      return scaleMatrix;\n    },\n\n    /**\n     * Returns a transform matrix starting from an object of the same kind of\n     * the one returned from qrDecompose, useful also if you want to calculate some\n     * transformations from an object that is not enlived yet\n     * @static\n     * @memberOf fabric.util\n     * @param  {Object} options\n     * @param  {Number} [options.angle]\n     * @param  {Number} [options.scaleX]\n     * @param  {Number} [options.scaleY]\n     * @param  {Boolean} [options.flipX]\n     * @param  {Boolean} [options.flipY]\n     * @param  {Number} [options.skewX]\n     * @param  {Number} [options.skewX]\n     * @param  {Number} [options.translateX]\n     * @param  {Number} [options.translateY]\n     * @return {Number[]} transform matrix\n     */\n    composeMatrix: function(options) {\n      var matrix = [1, 0, 0, 1, options.translateX || 0, options.translateY || 0],\n          multiply = fabric.util.multiplyTransformMatrices;\n      if (options.angle) {\n        matrix = multiply(matrix, fabric.util.calcRotateMatrix(options));\n      }\n      if (options.scaleX !== 1 || options.scaleY !== 1 ||\n          options.skewX || options.skewY || options.flipX || options.flipY) {\n        matrix = multiply(matrix, fabric.util.calcDimensionsMatrix(options));\n      }\n      return matrix;\n    },\n\n    /**\n     * reset an object transform state to neutral. Top and left are not accounted for\n     * @static\n     * @memberOf fabric.util\n     * @param  {fabric.Object} target object to transform\n     */\n    resetObjectTransform: function (target) {\n      target.scaleX = 1;\n      target.scaleY = 1;\n      target.skewX = 0;\n      target.skewY = 0;\n      target.flipX = false;\n      target.flipY = false;\n      target.rotate(0);\n    },\n\n    /**\n     * Extract Object transform values\n     * @static\n     * @memberOf fabric.util\n     * @param  {fabric.Object} target object to read from\n     * @return {Object} Components of transform\n     */\n    saveObjectTransform: function (target) {\n      return {\n        scaleX: target.scaleX,\n        scaleY: target.scaleY,\n        skewX: target.skewX,\n        skewY: target.skewY,\n        angle: target.angle,\n        left: target.left,\n        flipX: target.flipX,\n        flipY: target.flipY,\n        top: target.top\n      };\n    },\n\n    /**\n     * Returns true if context has transparent pixel\n     * at specified location (taking tolerance into account)\n     * @param {CanvasRenderingContext2D} ctx context\n     * @param {Number} x x coordinate\n     * @param {Number} y y coordinate\n     * @param {Number} tolerance Tolerance\n     */\n    isTransparent: function(ctx, x, y, tolerance) {\n\n      // If tolerance is > 0 adjust start coords to take into account.\n      // If moves off Canvas fix to 0\n      if (tolerance > 0) {\n        if (x > tolerance) {\n          x -= tolerance;\n        }\n        else {\n          x = 0;\n        }\n        if (y > tolerance) {\n          y -= tolerance;\n        }\n        else {\n          y = 0;\n        }\n      }\n\n      var _isTransparent = true, i, temp,\n          imageData = ctx.getImageData(x, y, (tolerance * 2) || 1, (tolerance * 2) || 1),\n          l = imageData.data.length;\n\n      // Split image data - for tolerance > 1, pixelDataSize = 4;\n      for (i = 3; i < l; i += 4) {\n        temp = imageData.data[i];\n        _isTransparent = temp <= 0;\n        if (_isTransparent === false) {\n          break; // Stop if colour found\n        }\n      }\n\n      imageData = null;\n\n      return _isTransparent;\n    },\n\n    /**\n     * Parse preserveAspectRatio attribute from element\n     * @param {string} attribute to be parsed\n     * @return {Object} an object containing align and meetOrSlice attribute\n     */\n    parsePreserveAspectRatioAttribute: function(attribute) {\n      var meetOrSlice = 'meet', alignX = 'Mid', alignY = 'Mid',\n          aspectRatioAttrs = attribute.split(' '), align;\n\n      if (aspectRatioAttrs && aspectRatioAttrs.length) {\n        meetOrSlice = aspectRatioAttrs.pop();\n        if (meetOrSlice !== 'meet' && meetOrSlice !== 'slice') {\n          align = meetOrSlice;\n          meetOrSlice = 'meet';\n        }\n        else if (aspectRatioAttrs.length) {\n          align = aspectRatioAttrs.pop();\n        }\n      }\n      //divide align in alignX and alignY\n      alignX = align !== 'none' ? align.slice(1, 4) : 'none';\n      alignY = align !== 'none' ? align.slice(5, 8) : 'none';\n      return {\n        meetOrSlice: meetOrSlice,\n        alignX: alignX,\n        alignY: alignY\n      };\n    },\n\n    /**\n     * Clear char widths cache for the given font family or all the cache if no\n     * fontFamily is specified.\n     * Use it if you know you are loading fonts in a lazy way and you are not waiting\n     * for custom fonts to load properly when adding text objects to the canvas.\n     * If a text object is added when its own font is not loaded yet, you will get wrong\n     * measurement and so wrong bounding boxes.\n     * After the font cache is cleared, either change the textObject text content or call\n     * initDimensions() to trigger a recalculation\n     * @memberOf fabric.util\n     * @param {String} [fontFamily] font family to clear\n     */\n    clearFabricFontCache: function(fontFamily) {\n      fontFamily = (fontFamily || '').toLowerCase();\n      if (!fontFamily) {\n        fabric.charWidthsCache = { };\n      }\n      else if (fabric.charWidthsCache[fontFamily]) {\n        delete fabric.charWidthsCache[fontFamily];\n      }\n    },\n\n    /**\n     * Given current aspect ratio, determines the max width and height that can\n     * respect the total allowed area for the cache.\n     * @memberOf fabric.util\n     * @param {Number} ar aspect ratio\n     * @param {Number} maximumArea Maximum area you want to achieve\n     * @return {Object.x} Limited dimensions by X\n     * @return {Object.y} Limited dimensions by Y\n     */\n    limitDimsByArea: function(ar, maximumArea) {\n      var roughWidth = Math.sqrt(maximumArea * ar),\n          perfLimitSizeY = Math.floor(maximumArea / roughWidth);\n      return { x: Math.floor(roughWidth), y: perfLimitSizeY };\n    },\n\n    capValue: function(min, value, max) {\n      return Math.max(min, Math.min(value, max));\n    },\n\n    /**\n     * Finds the scale for the object source to fit inside the object destination,\n     * keeping aspect ratio intact.\n     * respect the total allowed area for the cache.\n     * @memberOf fabric.util\n     * @param {Object | fabric.Object} source\n     * @param {Number} source.height natural unscaled height of the object\n     * @param {Number} source.width natural unscaled width of the object\n     * @param {Object | fabric.Object} destination\n     * @param {Number} destination.height natural unscaled height of the object\n     * @param {Number} destination.width natural unscaled width of the object\n     * @return {Number} scale factor to apply to source to fit into destination\n     */\n    findScaleToFit: function(source, destination) {\n      return Math.min(destination.width / source.width, destination.height / source.height);\n    },\n\n    /**\n     * Finds the scale for the object source to cover entirely the object destination,\n     * keeping aspect ratio intact.\n     * respect the total allowed area for the cache.\n     * @memberOf fabric.util\n     * @param {Object | fabric.Object} source\n     * @param {Number} source.height natural unscaled height of the object\n     * @param {Number} source.width natural unscaled width of the object\n     * @param {Object | fabric.Object} destination\n     * @param {Number} destination.height natural unscaled height of the object\n     * @param {Number} destination.width natural unscaled width of the object\n     * @return {Number} scale factor to apply to source to cover destination\n     */\n    findScaleToCover: function(source, destination) {\n      return Math.max(destination.width / source.width, destination.height / source.height);\n    },\n\n    /**\n     * given an array of 6 number returns something like `\"matrix(...numbers)\"`\n     * @memberOf fabric.util\n     * @param {Array} transform an array with 6 numbers\n     * @return {String} transform matrix for svg\n     * @return {Object.y} Limited dimensions by Y\n     */\n    matrixToSVG: function(transform) {\n      return 'matrix(' + transform.map(function(value) {\n        return fabric.util.toFixed(value, fabric.Object.NUM_FRACTION_DIGITS);\n      }).join(' ') + ')';\n    },\n\n    /**\n     * given an object and a transform, apply the inverse transform to the object,\n     * this is equivalent to remove from that object that transformation, so that\n     * added in a space with the removed transform, the object will be the same as before.\n     * Removing from an object a transform that scale by 2 is like scaling it by 1/2.\n     * Removing from an object a transfrom that rotate by 30deg is like rotating by 30deg\n     * in the opposite direction.\n     * This util is used to add objects inside transformed groups or nested groups.\n     * @memberOf fabric.util\n     * @param {fabric.Object} object the object you want to transform\n     * @param {Array} transform the destination transform\n     */\n    removeTransformFromObject: function(object, transform) {\n      var inverted = fabric.util.invertTransform(transform),\n          finalTransform = fabric.util.multiplyTransformMatrices(inverted, object.calcOwnMatrix());\n      fabric.util.applyTransformToObject(object, finalTransform);\n    },\n\n    /**\n     * given an object and a transform, apply the transform to the object.\n     * this is equivalent to change the space where the object is drawn.\n     * Adding to an object a transform that scale by 2 is like scaling it by 2.\n     * This is used when removing an object from an active selection for example.\n     * @memberOf fabric.util\n     * @param {fabric.Object} object the object you want to transform\n     * @param {Array} transform the destination transform\n     */\n    addTransformToObject: function(object, transform) {\n      fabric.util.applyTransformToObject(\n        object,\n        fabric.util.multiplyTransformMatrices(transform, object.calcOwnMatrix())\n      );\n    },\n\n    /**\n     * discard an object transform state and apply the one from the matrix.\n     * @memberOf fabric.util\n     * @param {fabric.Object} object the object you want to transform\n     * @param {Array} transform the destination transform\n     */\n    applyTransformToObject: function(object, transform) {\n      var options = fabric.util.qrDecompose(transform),\n          center = new fabric.Point(options.translateX, options.translateY);\n      object.flipX = false;\n      object.flipY = false;\n      object.set('scaleX', options.scaleX);\n      object.set('scaleY', options.scaleY);\n      object.skewX = options.skewX;\n      object.skewY = options.skewY;\n      object.angle = options.angle;\n      object.setPositionByOrigin(center, 'center', 'center');\n    },\n\n    /**\n     * given a width and height, return the size of the bounding box\n     * that can contains the box with width/height with applied transform\n     * described in options.\n     * Use to calculate the boxes around objects for controls.\n     * @memberOf fabric.util\n     * @param {Number} width\n     * @param {Number} height\n     * @param {Object} options\n     * @param {Number} options.scaleX\n     * @param {Number} options.scaleY\n     * @param {Number} options.skewX\n     * @param {Number} options.skewY\n     * @return {Object.x} width of containing\n     * @return {Object.y} height of containing\n     */\n    sizeAfterTransform: function(width, height, options) {\n      var dimX = width / 2, dimY = height / 2,\n          points = [\n            {\n              x: -dimX,\n              y: -dimY\n            },\n            {\n              x: dimX,\n              y: -dimY\n            },\n            {\n              x: -dimX,\n              y: dimY\n            },\n            {\n              x: dimX,\n              y: dimY\n            }],\n          transformMatrix = fabric.util.calcDimensionsMatrix(options),\n          bbox = fabric.util.makeBoundingBoxFromPoints(points, transformMatrix);\n      return {\n        x: bbox.width,\n        y: bbox.height,\n      };\n    },\n\n    /**\n     * Merges 2 clip paths into one visually equal clip path\n     *\n     * **IMPORTANT**:\\\n     * Does **NOT** clone the arguments, clone them proir if necessary.\n     *\n     * Creates a wrapper (group) that contains one clip path and is clipped by the other so content is kept where both overlap.\n     * Use this method if both the clip paths may have nested clip paths of their own, so assigning one to the other's clip path property is not possible.\n     *\n     * In order to handle the `inverted` property we follow logic described in the following cases:\\\n     * **(1)** both clip paths are inverted - the clip paths pass the inverted prop to the wrapper and loose it themselves.\\\n     * **(2)** one is inverted and the other isn't - the wrapper shouldn't become inverted and the inverted clip path must clip the non inverted one to produce an identical visual effect.\\\n     * **(3)** both clip paths are not inverted - wrapper and clip paths remain unchanged.\n     *\n     * @memberOf fabric.util\n     * @param {fabric.Object} c1\n     * @param {fabric.Object} c2\n     * @returns {fabric.Object} merged clip path\n     */\n    mergeClipPaths: function (c1, c2) {\n      var a = c1, b = c2;\n      if (a.inverted && !b.inverted) {\n        //  case (2)\n        a = c2;\n        b = c1;\n      }\n      //  `b` becomes `a`'s clip path so we transform `b` to `a` coordinate plane\n      fabric.util.applyTransformToObject(\n        b,\n        fabric.util.multiplyTransformMatrices(\n          fabric.util.invertTransform(a.calcTransformMatrix()),\n          b.calcTransformMatrix()\n        )\n      );\n      //  assign the `inverted` prop to the wrapping group\n      var inverted = a.inverted && b.inverted;\n      if (inverted) {\n        //  case (1)\n        a.inverted = b.inverted = false;\n      }\n      return new fabric.Group([a], { clipPath: b, inverted: inverted });\n    },\n\n    /**\n     * @memberOf fabric.util\n     * @param {Object} prevStyle first style to compare\n     * @param {Object} thisStyle second style to compare\n     * @param {boolean} forTextSpans whether to check overline, underline, and line-through properties\n     * @return {boolean} true if the style changed\n     */\n    hasStyleChanged: function(prevStyle, thisStyle, forTextSpans) {\n      forTextSpans = forTextSpans || false;\n      return (prevStyle.fill !== thisStyle.fill ||\n              prevStyle.stroke !== thisStyle.stroke ||\n              prevStyle.strokeWidth !== thisStyle.strokeWidth ||\n              prevStyle.fontSize !== thisStyle.fontSize ||\n              prevStyle.fontFamily !== thisStyle.fontFamily ||\n              prevStyle.fontWeight !== thisStyle.fontWeight ||\n              prevStyle.fontStyle !== thisStyle.fontStyle ||\n              prevStyle.deltaY !== thisStyle.deltaY) ||\n              (forTextSpans &&\n                (prevStyle.overline !== thisStyle.overline ||\n                prevStyle.underline !== thisStyle.underline ||\n                prevStyle.linethrough !== thisStyle.linethrough));\n    },\n\n    /**\n     * Returns the array form of a text object's inline styles property with styles grouped in ranges\n     * rather than per character. This format is less verbose, and is better suited for storage\n     * so it is used in serialization (not during runtime).\n     * @memberOf fabric.util\n     * @param {object} styles per character styles for a text object\n     * @param {String} text the text string that the styles are applied to\n     * @return {{start: number, end: number, style: object}[]}\n     */\n    stylesToArray: function(styles, text) {\n      // clone style structure to prevent mutation\n      var styles = fabric.util.object.clone(styles, true),\n          textLines = text.split('\\n'),\n          charIndex = -1, prevStyle = {}, stylesArray = [];\n      //loop through each textLine\n      for (var i = 0; i < textLines.length; i++) {\n        if (!styles[i]) {\n          //no styles exist for this line, so add the line's length to the charIndex total\n          charIndex += textLines[i].length;\n          continue;\n        }\n        //loop through each character of the current line\n        for (var c = 0; c < textLines[i].length; c++) {\n          charIndex++;\n          var thisStyle = styles[i][c];\n          //check if style exists for this character\n          if (thisStyle) {\n            var styleChanged = fabric.util.hasStyleChanged(prevStyle, thisStyle, true);\n            if (styleChanged) {\n              stylesArray.push({\n                start: charIndex,\n                end: charIndex + 1,\n                style: thisStyle\n              });\n            }\n            else {\n              //if style is the same as previous character, increase end index\n              stylesArray[stylesArray.length - 1].end++;\n            }\n          }\n          prevStyle = thisStyle || {};\n        }\n      }\n      return stylesArray;\n    },\n\n    /**\n     * Returns the object form of the styles property with styles that are assigned per\n     * character rather than grouped by range. This format is more verbose, and is\n     * only used during runtime (not for serialization/storage)\n     * @memberOf fabric.util\n     * @param {Array} styles the serialized form of a text object's styles\n     * @param {String} text the text string that the styles are applied to\n     * @return {Object}\n     */\n    stylesFromArray: function(styles, text) {\n      if (!Array.isArray(styles)) {\n        return styles;\n      }\n      var textLines = text.split('\\n'),\n          charIndex = -1, styleIndex = 0, stylesObject = {};\n      //loop through each textLine\n      for (var i = 0; i < textLines.length; i++) {\n        //loop through each character of the current line\n        for (var c = 0; c < textLines[i].length; c++) {\n          charIndex++;\n          //check if there's a style collection that includes the current character\n          if (styles[styleIndex]\n            && styles[styleIndex].start <= charIndex\n            && charIndex < styles[styleIndex].end) {\n            //create object for line index if it doesn't exist\n            stylesObject[i] = stylesObject[i] || {};\n            //assign a style at this character's index\n            stylesObject[i][c] = Object.assign({}, styles[styleIndex].style);\n            //if character is at the end of the current style collection, move to the next\n            if (charIndex === styles[styleIndex].end - 1) {\n              styleIndex++;\n            }\n          }\n        }\n      }\n      return stylesObject;\n    }\n  };\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function() {\n  var _join = Array.prototype.join,\n      commandLengths = {\n        m: 2,\n        l: 2,\n        h: 1,\n        v: 1,\n        c: 6,\n        s: 4,\n        q: 4,\n        t: 2,\n        a: 7\n      },\n      repeatedCommands = {\n        m: 'l',\n        M: 'L'\n      };\n  function segmentToBezier(th2, th3, cosTh, sinTh, rx, ry, cx1, cy1, mT, fromX, fromY) {\n    var costh2 = fabric.util.cos(th2),\n        sinth2 = fabric.util.sin(th2),\n        costh3 = fabric.util.cos(th3),\n        sinth3 = fabric.util.sin(th3),\n        toX = cosTh * rx * costh3 - sinTh * ry * sinth3 + cx1,\n        toY = sinTh * rx * costh3 + cosTh * ry * sinth3 + cy1,\n        cp1X = fromX + mT * ( -cosTh * rx * sinth2 - sinTh * ry * costh2),\n        cp1Y = fromY + mT * ( -sinTh * rx * sinth2 + cosTh * ry * costh2),\n        cp2X = toX + mT * ( cosTh * rx * sinth3 + sinTh * ry * costh3),\n        cp2Y = toY + mT * ( sinTh * rx * sinth3 - cosTh * ry * costh3);\n\n    return ['C',\n      cp1X, cp1Y,\n      cp2X, cp2Y,\n      toX, toY\n    ];\n  }\n\n  /* Adapted from http://dxr.mozilla.org/mozilla-central/source/content/svg/content/src/nsSVGPathDataParser.cpp\n   * by Andrea Bogazzi code is under MPL. if you don't have a copy of the license you can take it here\n   * http://mozilla.org/MPL/2.0/\n   */\n  function arcToSegments(toX, toY, rx, ry, large, sweep, rotateX) {\n    var PI = Math.PI, th = rotateX * PI / 180,\n        sinTh = fabric.util.sin(th),\n        cosTh = fabric.util.cos(th),\n        fromX = 0, fromY = 0;\n\n    rx = Math.abs(rx);\n    ry = Math.abs(ry);\n\n    var px = -cosTh * toX * 0.5 - sinTh * toY * 0.5,\n        py = -cosTh * toY * 0.5 + sinTh * toX * 0.5,\n        rx2 = rx * rx, ry2 = ry * ry, py2 = py * py, px2 = px * px,\n        pl = rx2 * ry2 - rx2 * py2 - ry2 * px2,\n        root = 0;\n\n    if (pl < 0) {\n      var s = Math.sqrt(1 - pl / (rx2 * ry2));\n      rx *= s;\n      ry *= s;\n    }\n    else {\n      root = (large === sweep ? -1.0 : 1.0) *\n              Math.sqrt( pl / (rx2 * py2 + ry2 * px2));\n    }\n\n    var cx = root * rx * py / ry,\n        cy = -root * ry * px / rx,\n        cx1 = cosTh * cx - sinTh * cy + toX * 0.5,\n        cy1 = sinTh * cx + cosTh * cy + toY * 0.5,\n        mTheta = calcVectorAngle(1, 0, (px - cx) / rx, (py - cy) / ry),\n        dtheta = calcVectorAngle((px - cx) / rx, (py - cy) / ry, (-px - cx) / rx, (-py - cy) / ry);\n\n    if (sweep === 0 && dtheta > 0) {\n      dtheta -= 2 * PI;\n    }\n    else if (sweep === 1 && dtheta < 0) {\n      dtheta += 2 * PI;\n    }\n\n    // Convert into cubic bezier segments <= 90deg\n    var segments = Math.ceil(Math.abs(dtheta / PI * 2)),\n        result = [], mDelta = dtheta / segments,\n        mT = 8 / 3 * Math.sin(mDelta / 4) * Math.sin(mDelta / 4) / Math.sin(mDelta / 2),\n        th3 = mTheta + mDelta;\n\n    for (var i = 0; i < segments; i++) {\n      result[i] = segmentToBezier(mTheta, th3, cosTh, sinTh, rx, ry, cx1, cy1, mT, fromX, fromY);\n      fromX = result[i][5];\n      fromY = result[i][6];\n      mTheta = th3;\n      th3 += mDelta;\n    }\n    return result;\n  }\n\n  /*\n   * Private\n   */\n  function calcVectorAngle(ux, uy, vx, vy) {\n    var ta = Math.atan2(uy, ux),\n        tb = Math.atan2(vy, vx);\n    if (tb >= ta) {\n      return tb - ta;\n    }\n    else {\n      return 2 * Math.PI - (ta - tb);\n    }\n  }\n\n  /**\n   * Calculate bounding box of a beziercurve\n   * @param {Number} x0 starting point\n   * @param {Number} y0\n   * @param {Number} x1 first control point\n   * @param {Number} y1\n   * @param {Number} x2 secondo control point\n   * @param {Number} y2\n   * @param {Number} x3 end of bezier\n   * @param {Number} y3\n   */\n  // taken from http://jsbin.com/ivomiq/56/edit  no credits available for that.\n  // TODO: can we normalize this with the starting points set at 0 and then translated the bbox?\n  function getBoundsOfCurve(x0, y0, x1, y1, x2, y2, x3, y3) {\n    var argsString;\n    if (fabric.cachesBoundsOfCurve) {\n      argsString = _join.call(arguments);\n      if (fabric.boundsOfCurveCache[argsString]) {\n        return fabric.boundsOfCurveCache[argsString];\n      }\n    }\n\n    var sqrt = Math.sqrt,\n        min = Math.min, max = Math.max,\n        abs = Math.abs, tvalues = [],\n        bounds = [[], []],\n        a, b, c, t, t1, t2, b2ac, sqrtb2ac;\n\n    b = 6 * x0 - 12 * x1 + 6 * x2;\n    a = -3 * x0 + 9 * x1 - 9 * x2 + 3 * x3;\n    c = 3 * x1 - 3 * x0;\n\n    for (var i = 0; i < 2; ++i) {\n      if (i > 0) {\n        b = 6 * y0 - 12 * y1 + 6 * y2;\n        a = -3 * y0 + 9 * y1 - 9 * y2 + 3 * y3;\n        c = 3 * y1 - 3 * y0;\n      }\n\n      if (abs(a) < 1e-12) {\n        if (abs(b) < 1e-12) {\n          continue;\n        }\n        t = -c / b;\n        if (0 < t && t < 1) {\n          tvalues.push(t);\n        }\n        continue;\n      }\n      b2ac = b * b - 4 * c * a;\n      if (b2ac < 0) {\n        continue;\n      }\n      sqrtb2ac = sqrt(b2ac);\n      t1 = (-b + sqrtb2ac) / (2 * a);\n      if (0 < t1 && t1 < 1) {\n        tvalues.push(t1);\n      }\n      t2 = (-b - sqrtb2ac) / (2 * a);\n      if (0 < t2 && t2 < 1) {\n        tvalues.push(t2);\n      }\n    }\n\n    var x, y, j = tvalues.length, jlen = j, mt;\n    while (j--) {\n      t = tvalues[j];\n      mt = 1 - t;\n      x = (mt * mt * mt * x0) + (3 * mt * mt * t * x1) + (3 * mt * t * t * x2) + (t * t * t * x3);\n      bounds[0][j] = x;\n\n      y = (mt * mt * mt * y0) + (3 * mt * mt * t * y1) + (3 * mt * t * t * y2) + (t * t * t * y3);\n      bounds[1][j] = y;\n    }\n\n    bounds[0][jlen] = x0;\n    bounds[1][jlen] = y0;\n    bounds[0][jlen + 1] = x3;\n    bounds[1][jlen + 1] = y3;\n    var result = [\n      {\n        x: min.apply(null, bounds[0]),\n        y: min.apply(null, bounds[1])\n      },\n      {\n        x: max.apply(null, bounds[0]),\n        y: max.apply(null, bounds[1])\n      }\n    ];\n    if (fabric.cachesBoundsOfCurve) {\n      fabric.boundsOfCurveCache[argsString] = result;\n    }\n    return result;\n  }\n\n  /**\n   * Converts arc to a bunch of bezier curves\n   * @param {Number} fx starting point x\n   * @param {Number} fy starting point y\n   * @param {Array} coords Arc command\n   */\n  function fromArcToBeziers(fx, fy, coords) {\n    var rx = coords[1],\n        ry = coords[2],\n        rot = coords[3],\n        large = coords[4],\n        sweep = coords[5],\n        tx = coords[6],\n        ty = coords[7],\n        segsNorm = arcToSegments(tx - fx, ty - fy, rx, ry, large, sweep, rot);\n\n    for (var i = 0, len = segsNorm.length; i < len; i++) {\n      segsNorm[i][1] += fx;\n      segsNorm[i][2] += fy;\n      segsNorm[i][3] += fx;\n      segsNorm[i][4] += fy;\n      segsNorm[i][5] += fx;\n      segsNorm[i][6] += fy;\n    }\n    return segsNorm;\n  };\n\n  /**\n   * This function take a parsed SVG path and make it simpler for fabricJS logic.\n   * simplification consist of: only UPPERCASE absolute commands ( relative converted to absolute )\n   * S converted in C, T converted in Q, A converted in C.\n   * @param {Array} path the array of commands of a parsed svg path for fabric.Path\n   * @return {Array} the simplified array of commands of a parsed svg path for fabric.Path\n   */\n  function makePathSimpler(path) {\n    // x and y represent the last point of the path. the previous command point.\n    // we add them to each relative command to make it an absolute comment.\n    // we also swap the v V h H with L, because are easier to transform.\n    var x = 0, y = 0, len = path.length,\n        // x1 and y1 represent the last point of the subpath. the subpath is started with\n        // m or M command. When a z or Z command is drawn, x and y need to be resetted to\n        // the last x1 and y1.\n        x1 = 0, y1 = 0, current, i, converted,\n        // previous will host the letter of the previous command, to handle S and T.\n        // controlX and controlY will host the previous reflected control point\n        destinationPath = [], previous, controlX, controlY;\n    for (i = 0; i < len; ++i) {\n      converted = false;\n      current = path[i].slice(0);\n      switch (current[0]) { // first letter\n        case 'l': // lineto, relative\n          current[0] = 'L';\n          current[1] += x;\n          current[2] += y;\n          // falls through\n        case 'L':\n          x = current[1];\n          y = current[2];\n          break;\n        case 'h': // horizontal lineto, relative\n          current[1] += x;\n          // falls through\n        case 'H':\n          current[0] = 'L';\n          current[2] = y;\n          x = current[1];\n          break;\n        case 'v': // vertical lineto, relative\n          current[1] += y;\n          // falls through\n        case 'V':\n          current[0] = 'L';\n          y = current[1];\n          current[1] = x;\n          current[2] = y;\n          break;\n        case 'm': // moveTo, relative\n          current[0] = 'M';\n          current[1] += x;\n          current[2] += y;\n          // falls through\n        case 'M':\n          x = current[1];\n          y = current[2];\n          x1 = current[1];\n          y1 = current[2];\n          break;\n        case 'c': // bezierCurveTo, relative\n          current[0] = 'C';\n          current[1] += x;\n          current[2] += y;\n          current[3] += x;\n          current[4] += y;\n          current[5] += x;\n          current[6] += y;\n          // falls through\n        case 'C':\n          controlX = current[3];\n          controlY = current[4];\n          x = current[5];\n          y = current[6];\n          break;\n        case 's': // shorthand cubic bezierCurveTo, relative\n          current[0] = 'S';\n          current[1] += x;\n          current[2] += y;\n          current[3] += x;\n          current[4] += y;\n          // falls through\n        case 'S':\n          // would be sScC but since we are swapping sSc for C, we check just that.\n          if (previous === 'C') {\n            // calculate reflection of previous control points\n            controlX = 2 * x - controlX;\n            controlY = 2 * y - controlY;\n          }\n          else {\n            // If there is no previous command or if the previous command was not a C, c, S, or s,\n            // the control point is coincident with the current point\n            controlX = x;\n            controlY = y;\n          }\n          x = current[3];\n          y = current[4];\n          current[0] = 'C';\n          current[5] = current[3];\n          current[6] = current[4];\n          current[3] = current[1];\n          current[4] = current[2];\n          current[1] = controlX;\n          current[2] = controlY;\n          // current[3] and current[4] are NOW the second control point.\n          // we keep it for the next reflection.\n          controlX = current[3];\n          controlY = current[4];\n          break;\n        case 'q': // quadraticCurveTo, relative\n          current[0] = 'Q';\n          current[1] += x;\n          current[2] += y;\n          current[3] += x;\n          current[4] += y;\n          // falls through\n        case 'Q':\n          controlX = current[1];\n          controlY = current[2];\n          x = current[3];\n          y = current[4];\n          break;\n        case 't': // shorthand quadraticCurveTo, relative\n          current[0] = 'T';\n          current[1] += x;\n          current[2] += y;\n          // falls through\n        case 'T':\n          if (previous === 'Q') {\n            // calculate reflection of previous control point\n            controlX = 2 * x - controlX;\n            controlY = 2 * y - controlY;\n          }\n          else {\n            // If there is no previous command or if the previous command was not a Q, q, T or t,\n            // assume the control point is coincident with the current point\n            controlX = x;\n            controlY = y;\n          }\n          current[0] = 'Q';\n          x = current[1];\n          y = current[2];\n          current[1] = controlX;\n          current[2] = controlY;\n          current[3] = x;\n          current[4] = y;\n          break;\n        case 'a':\n          current[0] = 'A';\n          current[6] += x;\n          current[7] += y;\n          // falls through\n        case 'A':\n          converted = true;\n          destinationPath = destinationPath.concat(fromArcToBeziers(x, y, current));\n          x = current[6];\n          y = current[7];\n          break;\n        case 'z':\n        case 'Z':\n          x = x1;\n          y = y1;\n          break;\n        default:\n      }\n      if (!converted) {\n        destinationPath.push(current);\n      }\n      previous = current[0];\n    }\n    return destinationPath;\n  };\n\n  /**\n   * Calc length from point x1,y1 to x2,y2\n   * @param {Number} x1 starting point x\n   * @param {Number} y1 starting point y\n   * @param {Number} x2 starting point x\n   * @param {Number} y2 starting point y\n   * @return {Number} length of segment\n   */\n  function calcLineLength(x1, y1, x2, y2) {\n    return Math.sqrt((x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1));\n  }\n\n  // functions for the Cubic beizer\n  // taken from: https://github.com/konvajs/konva/blob/7.0.5/src/shapes/Path.ts#L350\n  function CB1(t) {\n    return t * t * t;\n  }\n  function CB2(t) {\n    return 3 * t * t * (1 - t);\n  }\n  function CB3(t) {\n    return 3 * t * (1 - t) * (1 - t);\n  }\n  function CB4(t) {\n    return (1 - t) * (1 - t) * (1 - t);\n  }\n\n  function getPointOnCubicBezierIterator(p1x, p1y, p2x, p2y, p3x, p3y, p4x, p4y) {\n    return function(pct) {\n      var c1 = CB1(pct), c2 = CB2(pct), c3 = CB3(pct), c4 = CB4(pct);\n      return {\n        x: p4x * c1 + p3x * c2 + p2x * c3 + p1x * c4,\n        y: p4y * c1 + p3y * c2 + p2y * c3 + p1y * c4\n      };\n    };\n  }\n\n  function getTangentCubicIterator(p1x, p1y, p2x, p2y, p3x, p3y, p4x, p4y) {\n    return function (pct) {\n      var invT = 1 - pct,\n          tangentX = (3 * invT * invT * (p2x - p1x)) + (6 * invT * pct * (p3x - p2x)) +\n          (3 * pct * pct * (p4x - p3x)),\n          tangentY = (3 * invT * invT * (p2y - p1y)) + (6 * invT * pct * (p3y - p2y)) +\n          (3 * pct * pct * (p4y - p3y));\n      return Math.atan2(tangentY, tangentX);\n    };\n  }\n\n  function QB1(t) {\n    return t * t;\n  }\n\n  function QB2(t) {\n    return 2 * t * (1 - t);\n  }\n\n  function QB3(t) {\n    return (1 - t) * (1 - t);\n  }\n\n  function getPointOnQuadraticBezierIterator(p1x, p1y, p2x, p2y, p3x, p3y) {\n    return function(pct) {\n      var c1 = QB1(pct), c2 = QB2(pct), c3 = QB3(pct);\n      return {\n        x: p3x * c1 + p2x * c2 + p1x * c3,\n        y: p3y * c1 + p2y * c2 + p1y * c3\n      };\n    };\n  }\n\n  function getTangentQuadraticIterator(p1x, p1y, p2x, p2y, p3x, p3y) {\n    return function (pct) {\n      var invT = 1 - pct,\n          tangentX = (2 * invT * (p2x - p1x)) + (2 * pct * (p3x - p2x)),\n          tangentY = (2 * invT * (p2y - p1y)) + (2 * pct * (p3y - p2y));\n      return Math.atan2(tangentY, tangentX);\n    };\n  }\n\n\n  // this will run over a path segment ( a cubic or quadratic segment) and approximate it\n  // with 100 segemnts. This will good enough to calculate the length of the curve\n  function pathIterator(iterator, x1, y1) {\n    var tempP = { x: x1, y: y1 }, p, tmpLen = 0, perc;\n    for (perc = 1; perc <= 100; perc += 1) {\n      p = iterator(perc / 100);\n      tmpLen += calcLineLength(tempP.x, tempP.y, p.x, p.y);\n      tempP = p;\n    }\n    return tmpLen;\n  }\n\n  /**\n   * Given a pathInfo, and a distance in pixels, find the percentage from 0 to 1\n   * that correspond to that pixels run over the path.\n   * The percentage will be then used to find the correct point on the canvas for the path.\n   * @param {Array} segInfo fabricJS collection of information on a parsed path\n   * @param {Number} distance from starting point, in pixels.\n   * @return {Object} info object with x and y ( the point on canvas ) and angle, the tangent on that point;\n   */\n  function findPercentageForDistance(segInfo, distance) {\n    var perc = 0, tmpLen = 0, iterator = segInfo.iterator, tempP = { x: segInfo.x, y: segInfo.y },\n        p, nextLen, nextStep = 0.01, angleFinder = segInfo.angleFinder, lastPerc;\n    // nextStep > 0.0001 covers 0.00015625 that 1/64th of 1/100\n    // the path\n    while (tmpLen < distance && nextStep > 0.0001) {\n      p = iterator(perc);\n      lastPerc = perc;\n      nextLen = calcLineLength(tempP.x, tempP.y, p.x, p.y);\n      // compare tmpLen each cycle with distance, decide next perc to test.\n      if ((nextLen + tmpLen) > distance) {\n        // we discard this step and we make smaller steps.\n        perc -= nextStep;\n        nextStep /= 2;\n      }\n      else {\n        tempP = p;\n        perc += nextStep;\n        tmpLen += nextLen;\n      }\n    }\n    p.angle = angleFinder(lastPerc);\n    return p;\n  }\n\n  /**\n   * Run over a parsed and simplifed path and extrac some informations.\n   * informations are length of each command and starting point\n   * @param {Array} path fabricJS parsed path commands\n   * @return {Array} path commands informations\n   */\n  function getPathSegmentsInfo(path) {\n    var totalLength = 0, len = path.length, current,\n        //x2 and y2 are the coords of segment start\n        //x1 and y1 are the coords of the current point\n        x1 = 0, y1 = 0, x2 = 0, y2 = 0, info = [], iterator, tempInfo, angleFinder;\n    for (var i = 0; i < len; i++) {\n      current = path[i];\n      tempInfo = {\n        x: x1,\n        y: y1,\n        command: current[0],\n      };\n      switch (current[0]) { //first letter\n        case 'M':\n          tempInfo.length = 0;\n          x2 = x1 = current[1];\n          y2 = y1 = current[2];\n          break;\n        case 'L':\n          tempInfo.length = calcLineLength(x1, y1, current[1], current[2]);\n          x1 = current[1];\n          y1 = current[2];\n          break;\n        case 'C':\n          iterator = getPointOnCubicBezierIterator(\n            x1,\n            y1,\n            current[1],\n            current[2],\n            current[3],\n            current[4],\n            current[5],\n            current[6]\n          );\n          angleFinder = getTangentCubicIterator(\n            x1,\n            y1,\n            current[1],\n            current[2],\n            current[3],\n            current[4],\n            current[5],\n            current[6]\n          );\n          tempInfo.iterator = iterator;\n          tempInfo.angleFinder = angleFinder;\n          tempInfo.length = pathIterator(iterator, x1, y1);\n          x1 = current[5];\n          y1 = current[6];\n          break;\n        case 'Q':\n          iterator = getPointOnQuadraticBezierIterator(\n            x1,\n            y1,\n            current[1],\n            current[2],\n            current[3],\n            current[4]\n          );\n          angleFinder = getTangentQuadraticIterator(\n            x1,\n            y1,\n            current[1],\n            current[2],\n            current[3],\n            current[4]\n          );\n          tempInfo.iterator = iterator;\n          tempInfo.angleFinder = angleFinder;\n          tempInfo.length = pathIterator(iterator, x1, y1);\n          x1 = current[3];\n          y1 = current[4];\n          break;\n        case 'Z':\n        case 'z':\n          // we add those in order to ease calculations later\n          tempInfo.destX = x2;\n          tempInfo.destY = y2;\n          tempInfo.length = calcLineLength(x1, y1, x2, y2);\n          x1 = x2;\n          y1 = y2;\n          break;\n      }\n      totalLength += tempInfo.length;\n      info.push(tempInfo);\n    }\n    info.push({ length: totalLength, x: x1, y: y1 });\n    return info;\n  }\n\n  function getPointOnPath(path, distance, infos) {\n    if (!infos) {\n      infos = getPathSegmentsInfo(path);\n    }\n    var i = 0;\n    while ((distance - infos[i].length > 0) && i < (infos.length - 2)) {\n      distance -= infos[i].length;\n      i++;\n    }\n    // var distance = infos[infos.length - 1] * perc;\n    var segInfo = infos[i], segPercent = distance / segInfo.length,\n        command = segInfo.command, segment = path[i], info;\n\n    switch (command) {\n      case 'M':\n        return { x: segInfo.x, y: segInfo.y, angle: 0 };\n      case 'Z':\n      case 'z':\n        info = new fabric.Point(segInfo.x, segInfo.y).lerp(\n          new fabric.Point(segInfo.destX, segInfo.destY),\n          segPercent\n        );\n        info.angle = Math.atan2(segInfo.destY - segInfo.y, segInfo.destX - segInfo.x);\n        return info;\n      case 'L':\n        info = new fabric.Point(segInfo.x, segInfo.y).lerp(\n          new fabric.Point(segment[1], segment[2]),\n          segPercent\n        );\n        info.angle = Math.atan2(segment[2] - segInfo.y, segment[1] - segInfo.x);\n        return info;\n      case 'C':\n        return findPercentageForDistance(segInfo, distance);\n      case 'Q':\n        return findPercentageForDistance(segInfo, distance);\n    }\n  }\n\n  /**\n   *\n   * @param {string} pathString\n   * @return {(string|number)[][]} An array of SVG path commands\n   * @example <caption>Usage</caption>\n   * parsePath('M 3 4 Q 3 5 2 1 4 0 Q 9 12 2 1 4 0') === [\n   *   ['M', 3, 4],\n   *   ['Q', 3, 5, 2, 1, 4, 0],\n   *   ['Q', 9, 12, 2, 1, 4, 0],\n   * ];\n   *\n   */\n  function parsePath(pathString) {\n    var result = [],\n        coords = [],\n        currentPath,\n        parsed,\n        re = fabric.rePathCommand,\n        rNumber = '[-+]?(?:\\\\d*\\\\.\\\\d+|\\\\d+\\\\.?)(?:[eE][-+]?\\\\d+)?\\\\s*',\n        rNumberCommaWsp = '(' + rNumber + ')' + fabric.commaWsp,\n        rFlagCommaWsp = '([01])' + fabric.commaWsp + '?',\n        rArcSeq = rNumberCommaWsp + '?' + rNumberCommaWsp + '?' + rNumberCommaWsp + rFlagCommaWsp + rFlagCommaWsp +\n          rNumberCommaWsp + '?(' + rNumber + ')',\n        regArcArgumentSequence = new RegExp(rArcSeq, 'g'),\n        match,\n        coordsStr,\n        // one of commands (m,M,l,L,q,Q,c,C,etc.) followed by non-command characters (i.e. command values)\n        path;\n    if (!pathString || !pathString.match) {\n      return result;\n    }\n    path = pathString.match(/[mzlhvcsqta][^mzlhvcsqta]*/gi);\n\n    for (var i = 0, coordsParsed, len = path.length; i < len; i++) {\n      currentPath = path[i];\n\n      coordsStr = currentPath.slice(1).trim();\n      coords.length = 0;\n\n      var command = currentPath.charAt(0);\n      coordsParsed = [command];\n\n      if (command.toLowerCase() === 'a') {\n        // arcs have special flags that apparently don't require spaces so handle special\n        for (var args; (args = regArcArgumentSequence.exec(coordsStr));) {\n          for (var j = 1; j < args.length; j++) {\n            coords.push(args[j]);\n          }\n        }\n      }\n      else {\n        while ((match = re.exec(coordsStr))) {\n          coords.push(match[0]);\n        }\n      }\n\n      for (var j = 0, jlen = coords.length; j < jlen; j++) {\n        parsed = parseFloat(coords[j]);\n        if (!isNaN(parsed)) {\n          coordsParsed.push(parsed);\n        }\n      }\n\n      var commandLength = commandLengths[command.toLowerCase()],\n          repeatedCommand = repeatedCommands[command] || command;\n\n      if (coordsParsed.length - 1 > commandLength) {\n        for (var k = 1, klen = coordsParsed.length; k < klen; k += commandLength) {\n          result.push([command].concat(coordsParsed.slice(k, k + commandLength)));\n          command = repeatedCommand;\n        }\n      }\n      else {\n        result.push(coordsParsed);\n      }\n    }\n\n    return result;\n  };\n\n  /**\n   *\n   * Converts points to a smooth SVG path\n   * @param {{ x: number,y: number }[]} points Array of points\n   * @param {number} [correction] Apply a correction to the path (usually we use `width / 1000`). If value is undefined 0 is used as the correction value.\n   * @return {(string|number)[][]} An array of SVG path commands\n   */\n  function getSmoothPathFromPoints(points, correction) {\n    var path = [], i,\n        p1 = new fabric.Point(points[0].x, points[0].y),\n        p2 = new fabric.Point(points[1].x, points[1].y),\n        len = points.length, multSignX = 1, multSignY = 0, manyPoints = len > 2;\n    correction = correction || 0;\n\n    if (manyPoints) {\n      multSignX = points[2].x < p2.x ? -1 : points[2].x === p2.x ? 0 : 1;\n      multSignY = points[2].y < p2.y ? -1 : points[2].y === p2.y ? 0 : 1;\n    }\n    path.push(['M', p1.x - multSignX * correction, p1.y - multSignY * correction]);\n    for (i = 1; i < len; i++) {\n      if (!p1.eq(p2)) {\n        var midPoint = p1.midPointFrom(p2);\n        // p1 is our bezier control point\n        // midpoint is our endpoint\n        // start point is p(i-1) value.\n        path.push(['Q', p1.x, p1.y, midPoint.x, midPoint.y]);\n      }\n      p1 = points[i];\n      if ((i + 1) < points.length) {\n        p2 = points[i + 1];\n      }\n    }\n    if (manyPoints) {\n      multSignX = p1.x > points[i - 2].x ? 1 : p1.x === points[i - 2].x ? 0 : -1;\n      multSignY = p1.y > points[i - 2].y ? 1 : p1.y === points[i - 2].y ? 0 : -1;\n    }\n    path.push(['L', p1.x + multSignX * correction, p1.y + multSignY * correction]);\n    return path;\n  }\n  /**\n   * Transform a path by transforming each segment.\n   * it has to be a simplified path or it won't work.\n   * WARNING: this depends from pathOffset for correct operation\n   * @param {Array} path fabricJS parsed and simplified path commands\n   * @param {Array} transform matrix that represent the transformation\n   * @param {Object} [pathOffset] the fabric.Path pathOffset\n   * @param {Number} pathOffset.x\n   * @param {Number} pathOffset.y\n   * @returns {Array} the transformed path\n   */\n  function transformPath(path, transform, pathOffset) {\n    if (pathOffset) {\n      transform = fabric.util.multiplyTransformMatrices(\n        transform,\n        [1, 0, 0, 1, -pathOffset.x, -pathOffset.y]\n      );\n    }\n    return path.map(function(pathSegment) {\n      var newSegment = pathSegment.slice(0), point = {};\n      for (var i = 1; i < pathSegment.length - 1; i += 2) {\n        point.x = pathSegment[i];\n        point.y = pathSegment[i + 1];\n        point = fabric.util.transformPoint(point, transform);\n        newSegment[i] = point.x;\n        newSegment[i + 1] = point.y;\n      }\n      return newSegment;\n    });\n  }\n\n  /**\n   * Join path commands to go back to svg format\n   * @param {Array} pathData fabricJS parsed path commands\n   * @return {String} joined path 'M 0 0 L 20 30'\n   */\n  fabric.util.joinPath = function(pathData) {\n    return pathData.map(function (segment) { return segment.join(' '); }).join(' ');\n  };\n  fabric.util.parsePath = parsePath;\n  fabric.util.makePathSimpler = makePathSimpler;\n  fabric.util.getSmoothPathFromPoints = getSmoothPathFromPoints;\n  fabric.util.getPathSegmentsInfo = getPathSegmentsInfo;\n  fabric.util.getBoundsOfCurve = getBoundsOfCurve;\n  fabric.util.getPointOnPath = getPointOnPath;\n  fabric.util.transformPath = transformPath;\n})();\n\n\n(function() {\n\n  var slice = Array.prototype.slice;\n\n  /**\n   * Invokes method on all items in a given array\n   * @memberOf fabric.util.array\n   * @param {Array} array Array to iterate over\n   * @param {String} method Name of a method to invoke\n   * @return {Array}\n   */\n  function invoke(array, method) {\n    var args = slice.call(arguments, 2), result = [];\n    for (var i = 0, len = array.length; i < len; i++) {\n      result[i] = args.length ? array[i][method].apply(array[i], args) : array[i][method].call(array[i]);\n    }\n    return result;\n  }\n\n  /**\n   * Finds maximum value in array (not necessarily \"first\" one)\n   * @memberOf fabric.util.array\n   * @param {Array} array Array to iterate over\n   * @param {String} byProperty\n   * @return {*}\n   */\n  function max(array, byProperty) {\n    return find(array, byProperty, function(value1, value2) {\n      return value1 >= value2;\n    });\n  }\n\n  /**\n   * Finds minimum value in array (not necessarily \"first\" one)\n   * @memberOf fabric.util.array\n   * @param {Array} array Array to iterate over\n   * @param {String} byProperty\n   * @return {*}\n   */\n  function min(array, byProperty) {\n    return find(array, byProperty, function(value1, value2) {\n      return value1 < value2;\n    });\n  }\n\n  /**\n   * @private\n   */\n  function fill(array, value) {\n    var k = array.length;\n    while (k--) {\n      array[k] = value;\n    }\n    return array;\n  }\n\n  /**\n   * @private\n   */\n  function find(array, byProperty, condition) {\n    if (!array || array.length === 0) {\n      return;\n    }\n\n    var i = array.length - 1,\n        result = byProperty ? array[i][byProperty] : array[i];\n    if (byProperty) {\n      while (i--) {\n        if (condition(array[i][byProperty], result)) {\n          result = array[i][byProperty];\n        }\n      }\n    }\n    else {\n      while (i--) {\n        if (condition(array[i], result)) {\n          result = array[i];\n        }\n      }\n    }\n    return result;\n  }\n\n  /**\n   * @namespace fabric.util.array\n   */\n  fabric.util.array = {\n    fill: fill,\n    invoke: invoke,\n    min: min,\n    max: max\n  };\n\n})();\n\n\n(function() {\n  /**\n   * Copies all enumerable properties of one js object to another\n   * this does not and cannot compete with generic utils.\n   * Does not clone or extend fabric.Object subclasses.\n   * This is mostly for internal use and has extra handling for fabricJS objects\n   * it skips the canvas and group properties in deep cloning.\n   * @memberOf fabric.util.object\n   * @param {Object} destination Where to copy to\n   * @param {Object} source Where to copy from\n   * @param {Boolean} [deep] Whether to extend nested objects\n   * @return {Object}\n   */\n\n  function extend(destination, source, deep) {\n    // JScript DontEnum bug is not taken care of\n    // the deep clone is for internal use, is not meant to avoid\n    // javascript traps or cloning html element or self referenced objects.\n    if (deep) {\n      if (!fabric.isLikelyNode && source instanceof Element) {\n        // avoid cloning deep images, canvases,\n        destination = source;\n      }\n      else if (source instanceof Array) {\n        destination = [];\n        for (var i = 0, len = source.length; i < len; i++) {\n          destination[i] = extend({ }, source[i], deep);\n        }\n      }\n      else if (source && typeof source === 'object') {\n        for (var property in source) {\n          if (property === 'canvas' || property === 'group') {\n            // we do not want to clone this props at all.\n            // we want to keep the keys in the copy\n            destination[property] = null;\n          }\n          else if (source.hasOwnProperty(property)) {\n            destination[property] = extend({ }, source[property], deep);\n          }\n        }\n      }\n      else {\n        // this sounds odd for an extend but is ok for recursive use\n        destination = source;\n      }\n    }\n    else {\n      for (var property in source) {\n        destination[property] = source[property];\n      }\n    }\n    return destination;\n  }\n\n  /**\n   * Creates an empty object and copies all enumerable properties of another object to it\n   * This method is mostly for internal use, and not intended for duplicating shapes in canvas.\n   * @memberOf fabric.util.object\n   * @param {Object} object Object to clone\n   * @param {Boolean} [deep] Whether to clone nested objects\n   * @return {Object}\n   */\n\n  //TODO: this function return an empty object if you try to clone null\n  function clone(object, deep) {\n    return extend({ }, object, deep);\n  }\n\n  /** @namespace fabric.util.object */\n  fabric.util.object = {\n    extend: extend,\n    clone: clone\n  };\n  fabric.util.object.extend(fabric.util, fabric.Observable);\n})();\n\n\n(function() {\n\n  /**\n   * Camelizes a string\n   * @memberOf fabric.util.string\n   * @param {String} string String to camelize\n   * @return {String} Camelized version of a string\n   */\n  function camelize(string) {\n    return string.replace(/-+(.)?/g, function(match, character) {\n      return character ? character.toUpperCase() : '';\n    });\n  }\n\n  /**\n   * Capitalizes a string\n   * @memberOf fabric.util.string\n   * @param {String} string String to capitalize\n   * @param {Boolean} [firstLetterOnly] If true only first letter is capitalized\n   * and other letters stay untouched, if false first letter is capitalized\n   * and other letters are converted to lowercase.\n   * @return {String} Capitalized version of a string\n   */\n  function capitalize(string, firstLetterOnly) {\n    return string.charAt(0).toUpperCase() +\n      (firstLetterOnly ? string.slice(1) : string.slice(1).toLowerCase());\n  }\n\n  /**\n   * Escapes XML in a string\n   * @memberOf fabric.util.string\n   * @param {String} string String to escape\n   * @return {String} Escaped version of a string\n   */\n  function escapeXml(string) {\n    return string.replace(/&/g, '&amp;')\n      .replace(/\"/g, '&quot;')\n      .replace(/'/g, '&apos;')\n      .replace(/</g, '&lt;')\n      .replace(/>/g, '&gt;');\n  }\n\n  /**\n   * Divide a string in the user perceived single units\n   * @memberOf fabric.util.string\n   * @param {String} textstring String to escape\n   * @return {Array} array containing the graphemes\n   */\n  function graphemeSplit(textstring) {\n    var i = 0, chr, graphemes = [];\n    for (i = 0, chr; i < textstring.length; i++) {\n      if ((chr = getWholeChar(textstring, i)) === false) {\n        continue;\n      }\n      graphemes.push(chr);\n    }\n    return graphemes;\n  }\n\n  // taken from mdn in the charAt doc page.\n  function getWholeChar(str, i) {\n    var code = str.charCodeAt(i);\n\n    if (isNaN(code)) {\n      return ''; // Position not found\n    }\n    if (code < 0xD800 || code > 0xDFFF) {\n      return str.charAt(i);\n    }\n\n    // High surrogate (could change last hex to 0xDB7F to treat high private\n    // surrogates as single characters)\n    if (0xD800 <= code && code <= 0xDBFF) {\n      if (str.length <= (i + 1)) {\n        throw 'High surrogate without following low surrogate';\n      }\n      var next = str.charCodeAt(i + 1);\n      if (0xDC00 > next || next > 0xDFFF) {\n        throw 'High surrogate without following low surrogate';\n      }\n      return str.charAt(i) + str.charAt(i + 1);\n    }\n    // Low surrogate (0xDC00 <= code && code <= 0xDFFF)\n    if (i === 0) {\n      throw 'Low surrogate without preceding high surrogate';\n    }\n    var prev = str.charCodeAt(i - 1);\n\n    // (could change last hex to 0xDB7F to treat high private\n    // surrogates as single characters)\n    if (0xD800 > prev || prev > 0xDBFF) {\n      throw 'Low surrogate without preceding high surrogate';\n    }\n    // We can pass over low surrogates now as the second component\n    // in a pair which we have already processed\n    return false;\n  }\n\n\n  /**\n   * String utilities\n   * @namespace fabric.util.string\n   */\n  fabric.util.string = {\n    camelize: camelize,\n    capitalize: capitalize,\n    escapeXml: escapeXml,\n    graphemeSplit: graphemeSplit\n  };\n})();\n\n\n(function() {\n\n  var slice = Array.prototype.slice, emptyFunction = function() { },\n\n      IS_DONTENUM_BUGGY = (function() {\n        for (var p in { toString: 1 }) {\n          if (p === 'toString') {\n            return false;\n          }\n        }\n        return true;\n      })(),\n\n      /** @ignore */\n      addMethods = function(klass, source, parent) {\n        for (var property in source) {\n\n          if (property in klass.prototype &&\n              typeof klass.prototype[property] === 'function' &&\n              (source[property] + '').indexOf('callSuper') > -1) {\n\n            klass.prototype[property] = (function(property) {\n              return function() {\n\n                var superclass = this.constructor.superclass;\n                this.constructor.superclass = parent;\n                var returnValue = source[property].apply(this, arguments);\n                this.constructor.superclass = superclass;\n\n                if (property !== 'initialize') {\n                  return returnValue;\n                }\n              };\n            })(property);\n          }\n          else {\n            klass.prototype[property] = source[property];\n          }\n\n          if (IS_DONTENUM_BUGGY) {\n            if (source.toString !== Object.prototype.toString) {\n              klass.prototype.toString = source.toString;\n            }\n            if (source.valueOf !== Object.prototype.valueOf) {\n              klass.prototype.valueOf = source.valueOf;\n            }\n          }\n        }\n      };\n\n  function Subclass() { }\n\n  function callSuper(methodName) {\n    var parentMethod = null,\n        _this = this;\n\n    // climb prototype chain to find method not equal to callee's method\n    while (_this.constructor.superclass) {\n      var superClassMethod = _this.constructor.superclass.prototype[methodName];\n      if (_this[methodName] !== superClassMethod) {\n        parentMethod = superClassMethod;\n        break;\n      }\n      // eslint-disable-next-line\n      _this = _this.constructor.superclass.prototype;\n    }\n\n    if (!parentMethod) {\n      return console.log('tried to callSuper ' + methodName + ', method not found in prototype chain', this);\n    }\n\n    return (arguments.length > 1)\n      ? parentMethod.apply(this, slice.call(arguments, 1))\n      : parentMethod.call(this);\n  }\n\n  /**\n   * Helper for creation of \"classes\".\n   * @memberOf fabric.util\n   * @param {Function} [parent] optional \"Class\" to inherit from\n   * @param {Object} [properties] Properties shared by all instances of this class\n   *                  (be careful modifying objects defined here as this would affect all instances)\n   */\n  function createClass() {\n    var parent = null,\n        properties = slice.call(arguments, 0);\n\n    if (typeof properties[0] === 'function') {\n      parent = properties.shift();\n    }\n    function klass() {\n      this.initialize.apply(this, arguments);\n    }\n\n    klass.superclass = parent;\n    klass.subclasses = [];\n\n    if (parent) {\n      Subclass.prototype = parent.prototype;\n      klass.prototype = new Subclass();\n      parent.subclasses.push(klass);\n    }\n    for (var i = 0, length = properties.length; i < length; i++) {\n      addMethods(klass, properties[i], parent);\n    }\n    if (!klass.prototype.initialize) {\n      klass.prototype.initialize = emptyFunction;\n    }\n    klass.prototype.constructor = klass;\n    klass.prototype.callSuper = callSuper;\n    return klass;\n  }\n\n  fabric.util.createClass = createClass;\n})();\n\n\n(function () {\n  // since ie11 can use addEventListener but they do not support options, i need to check\n  var couldUseAttachEvent = !!fabric.document.createElement('div').attachEvent,\n      touchEvents = ['touchstart', 'touchmove', 'touchend'];\n  /**\n   * Adds an event listener to an element\n   * @function\n   * @memberOf fabric.util\n   * @param {HTMLElement} element\n   * @param {String} eventName\n   * @param {Function} handler\n   */\n  fabric.util.addListener = function(element, eventName, handler, options) {\n    element && element.addEventListener(eventName, handler, couldUseAttachEvent ? false : options);\n  };\n\n  /**\n   * Removes an event listener from an element\n   * @function\n   * @memberOf fabric.util\n   * @param {HTMLElement} element\n   * @param {String} eventName\n   * @param {Function} handler\n   */\n  fabric.util.removeListener = function(element, eventName, handler, options) {\n    element && element.removeEventListener(eventName, handler, couldUseAttachEvent ? false : options);\n  };\n\n  function getTouchInfo(event) {\n    var touchProp = event.changedTouches;\n    if (touchProp && touchProp[0]) {\n      return touchProp[0];\n    }\n    return event;\n  }\n\n  fabric.util.getPointer = function(event) {\n    var element = event.target,\n        scroll = fabric.util.getScrollLeftTop(element),\n        _evt = getTouchInfo(event);\n    return {\n      x: _evt.clientX + scroll.left,\n      y: _evt.clientY + scroll.top\n    };\n  };\n\n  fabric.util.isTouchEvent = function(event) {\n    return touchEvents.indexOf(event.type) > -1 || event.pointerType === 'touch';\n  };\n})();\n\n\n(function () {\n\n  /**\n   * Cross-browser wrapper for setting element's style\n   * @memberOf fabric.util\n   * @param {HTMLElement} element\n   * @param {Object} styles\n   * @return {HTMLElement} Element that was passed as a first argument\n   */\n  function setStyle(element, styles) {\n    var elementStyle = element.style;\n    if (!elementStyle) {\n      return element;\n    }\n    if (typeof styles === 'string') {\n      element.style.cssText += ';' + styles;\n      return styles.indexOf('opacity') > -1\n        ? setOpacity(element, styles.match(/opacity:\\s*(\\d?\\.?\\d*)/)[1])\n        : element;\n    }\n    for (var property in styles) {\n      if (property === 'opacity') {\n        setOpacity(element, styles[property]);\n      }\n      else {\n        var normalizedProperty = (property === 'float' || property === 'cssFloat')\n          ? (typeof elementStyle.styleFloat === 'undefined' ? 'cssFloat' : 'styleFloat')\n          : property;\n        elementStyle.setProperty(normalizedProperty, styles[property]);\n      }\n    }\n    return element;\n  }\n\n  var parseEl = fabric.document.createElement('div'),\n      supportsOpacity = typeof parseEl.style.opacity === 'string',\n      supportsFilters = typeof parseEl.style.filter === 'string',\n      reOpacity = /alpha\\s*\\(\\s*opacity\\s*=\\s*([^\\)]+)\\)/,\n\n      /** @ignore */\n      setOpacity = function (element) { return element; };\n\n  if (supportsOpacity) {\n    /** @ignore */\n    setOpacity = function(element, value) {\n      element.style.opacity = value;\n      return element;\n    };\n  }\n  else if (supportsFilters) {\n    /** @ignore */\n    setOpacity = function(element, value) {\n      var es = element.style;\n      if (element.currentStyle && !element.currentStyle.hasLayout) {\n        es.zoom = 1;\n      }\n      if (reOpacity.test(es.filter)) {\n        value = value >= 0.9999 ? '' : ('alpha(opacity=' + (value * 100) + ')');\n        es.filter = es.filter.replace(reOpacity, value);\n      }\n      else {\n        es.filter += ' alpha(opacity=' + (value * 100) + ')';\n      }\n      return element;\n    };\n  }\n\n  fabric.util.setStyle = setStyle;\n\n})();\n\n\n(function() {\n\n  var _slice = Array.prototype.slice;\n\n  /**\n   * Takes id and returns an element with that id (if one exists in a document)\n   * @memberOf fabric.util\n   * @param {String|HTMLElement} id\n   * @return {HTMLElement|null}\n   */\n  function getById(id) {\n    return typeof id === 'string' ? fabric.document.getElementById(id) : id;\n  }\n\n  var sliceCanConvertNodelists,\n      /**\n       * Converts an array-like object (e.g. arguments or NodeList) to an array\n       * @memberOf fabric.util\n       * @param {Object} arrayLike\n       * @return {Array}\n       */\n      toArray = function(arrayLike) {\n        return _slice.call(arrayLike, 0);\n      };\n\n  try {\n    sliceCanConvertNodelists = toArray(fabric.document.childNodes) instanceof Array;\n  }\n  catch (err) { }\n\n  if (!sliceCanConvertNodelists) {\n    toArray = function(arrayLike) {\n      var arr = new Array(arrayLike.length), i = arrayLike.length;\n      while (i--) {\n        arr[i] = arrayLike[i];\n      }\n      return arr;\n    };\n  }\n\n  /**\n   * Creates specified element with specified attributes\n   * @memberOf fabric.util\n   * @param {String} tagName Type of an element to create\n   * @param {Object} [attributes] Attributes to set on an element\n   * @return {HTMLElement} Newly created element\n   */\n  function makeElement(tagName, attributes) {\n    var el = fabric.document.createElement(tagName);\n    for (var prop in attributes) {\n      if (prop === 'class') {\n        el.className = attributes[prop];\n      }\n      else if (prop === 'for') {\n        el.htmlFor = attributes[prop];\n      }\n      else {\n        el.setAttribute(prop, attributes[prop]);\n      }\n    }\n    return el;\n  }\n\n  /**\n   * Adds class to an element\n   * @memberOf fabric.util\n   * @param {HTMLElement} element Element to add class to\n   * @param {String} className Class to add to an element\n   */\n  function addClass(element, className) {\n    if (element && (' ' + element.className + ' ').indexOf(' ' + className + ' ') === -1) {\n      element.className += (element.className ? ' ' : '') + className;\n    }\n  }\n\n  /**\n   * Wraps element with another element\n   * @memberOf fabric.util\n   * @param {HTMLElement} element Element to wrap\n   * @param {HTMLElement|String} wrapper Element to wrap with\n   * @param {Object} [attributes] Attributes to set on a wrapper\n   * @return {HTMLElement} wrapper\n   */\n  function wrapElement(element, wrapper, attributes) {\n    if (typeof wrapper === 'string') {\n      wrapper = makeElement(wrapper, attributes);\n    }\n    if (element.parentNode) {\n      element.parentNode.replaceChild(wrapper, element);\n    }\n    wrapper.appendChild(element);\n    return wrapper;\n  }\n\n  /**\n   * Returns element scroll offsets\n   * @memberOf fabric.util\n   * @param {HTMLElement} element Element to operate on\n   * @return {Object} Object with left/top values\n   */\n  function getScrollLeftTop(element) {\n\n    var left = 0,\n        top = 0,\n        docElement = fabric.document.documentElement,\n        body = fabric.document.body || {\n          scrollLeft: 0, scrollTop: 0\n        };\n\n    // While loop checks (and then sets element to) .parentNode OR .host\n    //  to account for ShadowDOM. We still want to traverse up out of ShadowDOM,\n    //  but the .parentNode of a root ShadowDOM node will always be null, instead\n    //  it should be accessed through .host. See http://stackoverflow.com/a/24765528/4383938\n    while (element && (element.parentNode || element.host)) {\n\n      // Set element to element parent, or 'host' in case of ShadowDOM\n      element = element.parentNode || element.host;\n\n      if (element === fabric.document) {\n        left = body.scrollLeft || docElement.scrollLeft || 0;\n        top = body.scrollTop ||  docElement.scrollTop || 0;\n      }\n      else {\n        left += element.scrollLeft || 0;\n        top += element.scrollTop || 0;\n      }\n\n      if (element.nodeType === 1 && element.style.position === 'fixed') {\n        break;\n      }\n    }\n\n    return { left: left, top: top };\n  }\n\n  /**\n   * Returns offset for a given element\n   * @function\n   * @memberOf fabric.util\n   * @param {HTMLElement} element Element to get offset for\n   * @return {Object} Object with \"left\" and \"top\" properties\n   */\n  function getElementOffset(element) {\n    var docElem,\n        doc = element && element.ownerDocument,\n        box = { left: 0, top: 0 },\n        offset = { left: 0, top: 0 },\n        scrollLeftTop,\n        offsetAttributes = {\n          borderLeftWidth: 'left',\n          borderTopWidth:  'top',\n          paddingLeft:     'left',\n          paddingTop:      'top'\n        };\n\n    if (!doc) {\n      return offset;\n    }\n\n    for (var attr in offsetAttributes) {\n      offset[offsetAttributes[attr]] += parseInt(getElementStyle(element, attr), 10) || 0;\n    }\n\n    docElem = doc.documentElement;\n    if ( typeof element.getBoundingClientRect !== 'undefined' ) {\n      box = element.getBoundingClientRect();\n    }\n\n    scrollLeftTop = getScrollLeftTop(element);\n\n    return {\n      left: box.left + scrollLeftTop.left - (docElem.clientLeft || 0) + offset.left,\n      top: box.top + scrollLeftTop.top - (docElem.clientTop || 0)  + offset.top\n    };\n  }\n\n  /**\n   * Returns style attribute value of a given element\n   * @memberOf fabric.util\n   * @param {HTMLElement} element Element to get style attribute for\n   * @param {String} attr Style attribute to get for element\n   * @return {String} Style attribute value of the given element.\n   */\n  var getElementStyle;\n  if (fabric.document.defaultView && fabric.document.defaultView.getComputedStyle) {\n    getElementStyle = function(element, attr) {\n      var style = fabric.document.defaultView.getComputedStyle(element, null);\n      return style ? style[attr] : undefined;\n    };\n  }\n  else {\n    getElementStyle = function(element, attr) {\n      var value = element.style[attr];\n      if (!value && element.currentStyle) {\n        value = element.currentStyle[attr];\n      }\n      return value;\n    };\n  }\n\n  (function () {\n    var style = fabric.document.documentElement.style,\n        selectProp = 'userSelect' in style\n          ? 'userSelect'\n          : 'MozUserSelect' in style\n            ? 'MozUserSelect'\n            : 'WebkitUserSelect' in style\n              ? 'WebkitUserSelect'\n              : 'KhtmlUserSelect' in style\n                ? 'KhtmlUserSelect'\n                : '';\n\n    /**\n     * Makes element unselectable\n     * @memberOf fabric.util\n     * @param {HTMLElement} element Element to make unselectable\n     * @return {HTMLElement} Element that was passed in\n     */\n    function makeElementUnselectable(element) {\n      if (typeof element.onselectstart !== 'undefined') {\n        element.onselectstart = fabric.util.falseFunction;\n      }\n      if (selectProp) {\n        element.style[selectProp] = 'none';\n      }\n      else if (typeof element.unselectable === 'string') {\n        element.unselectable = 'on';\n      }\n      return element;\n    }\n\n    /**\n     * Makes element selectable\n     * @memberOf fabric.util\n     * @param {HTMLElement} element Element to make selectable\n     * @return {HTMLElement} Element that was passed in\n     */\n    function makeElementSelectable(element) {\n      if (typeof element.onselectstart !== 'undefined') {\n        element.onselectstart = null;\n      }\n      if (selectProp) {\n        element.style[selectProp] = '';\n      }\n      else if (typeof element.unselectable === 'string') {\n        element.unselectable = '';\n      }\n      return element;\n    }\n\n    fabric.util.makeElementUnselectable = makeElementUnselectable;\n    fabric.util.makeElementSelectable = makeElementSelectable;\n  })();\n\n  function getNodeCanvas(element) {\n    var impl = fabric.jsdomImplForWrapper(element);\n    return impl._canvas || impl._image;\n  };\n\n  function cleanUpJsdomNode(element) {\n    if (!fabric.isLikelyNode) {\n      return;\n    }\n    var impl = fabric.jsdomImplForWrapper(element);\n    if (impl) {\n      impl._image = null;\n      impl._canvas = null;\n      // unsure if necessary\n      impl._currentSrc = null;\n      impl._attributes = null;\n      impl._classList = null;\n    }\n  }\n\n  function setImageSmoothing(ctx, value) {\n    ctx.imageSmoothingEnabled = ctx.imageSmoothingEnabled || ctx.webkitImageSmoothingEnabled\n      || ctx.mozImageSmoothingEnabled || ctx.msImageSmoothingEnabled || ctx.oImageSmoothingEnabled;\n    ctx.imageSmoothingEnabled = value;\n  }\n\n  /**\n   * setImageSmoothing sets the context imageSmoothingEnabled property.\n   * Used by canvas and by ImageObject.\n   * @memberOf fabric.util\n   * @since 4.0.0\n   * @param {HTMLRenderingContext2D} ctx to set on\n   * @param {Boolean} value true or false\n   */\n  fabric.util.setImageSmoothing = setImageSmoothing;\n  fabric.util.getById = getById;\n  fabric.util.toArray = toArray;\n  fabric.util.addClass = addClass;\n  fabric.util.makeElement = makeElement;\n  fabric.util.wrapElement = wrapElement;\n  fabric.util.getScrollLeftTop = getScrollLeftTop;\n  fabric.util.getElementOffset = getElementOffset;\n  fabric.util.getNodeCanvas = getNodeCanvas;\n  fabric.util.cleanUpJsdomNode = cleanUpJsdomNode;\n\n})();\n\n\n(function() {\n\n  function addParamToUrl(url, param) {\n    return url + (/\\?/.test(url) ? '&' : '?') + param;\n  }\n\n  function emptyFn() { }\n\n  /**\n   * Cross-browser abstraction for sending XMLHttpRequest\n   * @memberOf fabric.util\n   * @param {String} url URL to send XMLHttpRequest to\n   * @param {Object} [options] Options object\n   * @param {String} [options.method=\"GET\"]\n   * @param {String} [options.parameters] parameters to append to url in GET or in body\n   * @param {String} [options.body] body to send with POST or PUT request\n   * @param {Function} options.onComplete Callback to invoke when request is completed\n   * @return {XMLHttpRequest} request\n   */\n  function request(url, options) {\n    options || (options = { });\n\n    var method = options.method ? options.method.toUpperCase() : 'GET',\n        onComplete = options.onComplete || function() { },\n        xhr = new fabric.window.XMLHttpRequest(),\n        body = options.body || options.parameters;\n\n    /** @ignore */\n    xhr.onreadystatechange = function() {\n      if (xhr.readyState === 4) {\n        onComplete(xhr);\n        xhr.onreadystatechange = emptyFn;\n      }\n    };\n\n    if (method === 'GET') {\n      body = null;\n      if (typeof options.parameters === 'string') {\n        url = addParamToUrl(url, options.parameters);\n      }\n    }\n\n    xhr.open(method, url, true);\n\n    if (method === 'POST' || method === 'PUT') {\n      xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');\n    }\n\n    xhr.send(body);\n    return xhr;\n  }\n\n  fabric.util.request = request;\n})();\n\n\n/**\n * Wrapper around `console.log` (when available)\n * @param {*} [values] Values to log\n */\nfabric.log = console.log;\n\n/**\n * Wrapper around `console.warn` (when available)\n * @param {*} [values] Values to log as a warning\n */\nfabric.warn = console.warn;\n\n\n(function () {\n\n  var extend = fabric.util.object.extend,\n      clone = fabric.util.object.clone;\n\n  /**\n   * @typedef {Object} AnimationOptions\n   * Animation of a value or list of values.\n   * When using lists, think of something like this:\n   * fabric.util.animate({\n   *   startValue: [1, 2, 3],\n   *   endValue: [2, 4, 6],\n   *   onChange: function([a, b, c]) {\n   *     canvas.zoomToPoint({x: b, y: c}, a)\n   *     canvas.renderAll()\n   *   }\n   * });\n   * @example\n   * @property {Function} [onChange] Callback; invoked on every value change\n   * @property {Function} [onComplete] Callback; invoked when value change is completed\n   * @example\n   * // Note: startValue, endValue, and byValue must match the type\n   * var animationOptions = { startValue: 0, endValue: 1, byValue: 0.25 }\n   * var animationOptions = { startValue: [0, 1], endValue: [1, 2], byValue: [0.25, 0.25] }\n   * @property {number | number[]} [startValue=0] Starting value\n   * @property {number | number[]} [endValue=100] Ending value\n   * @property {number | number[]} [byValue=100] Value to modify the property by\n   * @property {Function} [easing] Easing function\n   * @property {Number} [duration=500] Duration of change (in ms)\n   * @property {Function} [abort] Additional function with logic. If returns true, animation aborts.\n   *\n   * @typedef {() => void} CancelFunction\n   *\n   * @typedef {Object} AnimationCurrentState\n   * @property {number | number[]} currentValue value in range [`startValue`, `endValue`]\n   * @property {number} completionRate value in range [0, 1]\n   * @property {number} durationRate value in range [0, 1]\n   *\n   * @typedef {(AnimationOptions & AnimationCurrentState & { cancel: CancelFunction }} AnimationContext\n   */\n\n  /**\n   * Array holding all running animations\n   * @memberof fabric\n   * @type {AnimationContext[]}\n   */\n  var RUNNING_ANIMATIONS = [];\n  fabric.util.object.extend(RUNNING_ANIMATIONS, {\n\n    /**\n     * cancel all running animations at the next requestAnimFrame\n     * @returns {AnimationContext[]}\n     */\n    cancelAll: function () {\n      var animations = this.splice(0);\n      animations.forEach(function (animation) {\n        animation.cancel();\n      });\n      return animations;\n    },\n\n    /**\n     * cancel all running animations attached to canvas at the next requestAnimFrame\n     * @param {fabric.Canvas} canvas\n     * @returns {AnimationContext[]}\n     */\n    cancelByCanvas: function (canvas) {\n      if (!canvas) {\n        return [];\n      }\n      var cancelled = this.filter(function (animation) {\n        return typeof animation.target === 'object' && animation.target.canvas === canvas;\n      });\n      cancelled.forEach(function (animation) {\n        animation.cancel();\n      });\n      return cancelled;\n    },\n\n    /**\n     * cancel all running animations for target at the next requestAnimFrame\n     * @param {*} target\n     * @returns {AnimationContext[]}\n     */\n    cancelByTarget: function (target) {\n      var cancelled = this.findAnimationsByTarget(target);\n      cancelled.forEach(function (animation) {\n        animation.cancel();\n      });\n      return cancelled;\n    },\n\n    /**\n     *\n     * @param {CancelFunction} cancelFunc the function returned by animate\n     * @returns {number}\n     */\n    findAnimationIndex: function (cancelFunc) {\n      return this.indexOf(this.findAnimation(cancelFunc));\n    },\n\n    /**\n     *\n     * @param {CancelFunction} cancelFunc the function returned by animate\n     * @returns {AnimationContext | undefined} animation's options object\n     */\n    findAnimation: function (cancelFunc) {\n      return this.find(function (animation) {\n        return animation.cancel === cancelFunc;\n      });\n    },\n\n    /**\n     *\n     * @param {*} target the object that is assigned to the target property of the animation context\n     * @returns {AnimationContext[]} array of animation options object associated with target\n     */\n    findAnimationsByTarget: function (target) {\n      if (!target) {\n        return [];\n      }\n      return this.filter(function (animation) {\n        return animation.target === target;\n      });\n    }\n  });\n\n  function noop() {\n    return false;\n  }\n\n  function defaultEasing(t, b, c, d) {\n    return -c * Math.cos(t / d * (Math.PI / 2)) + c + b;\n  }\n\n  /**\n   * Changes value from one to another within certain period of time, invoking callbacks as value is being changed.\n   * @memberOf fabric.util\n   * @param {AnimationOptions} [options] Animation options\n   * @example\n   * // Note: startValue, endValue, and byValue must match the type\n   * fabric.util.animate({ startValue: 0, endValue: 1, byValue: 0.25 })\n   * fabric.util.animate({ startValue: [0, 1], endValue: [1, 2], byValue: [0.25, 0.25] })\n   * @returns {CancelFunction} cancel function\n   */\n  function animate(options) {\n    options || (options = {});\n    var cancel = false,\n        context,\n        removeFromRegistry = function () {\n          var index = fabric.runningAnimations.indexOf(context);\n          return index > -1 && fabric.runningAnimations.splice(index, 1)[0];\n        };\n\n    context = extend(clone(options), {\n      cancel: function () {\n        cancel = true;\n        return removeFromRegistry();\n      },\n      currentValue: 'startValue' in options ? options.startValue : 0,\n      completionRate: 0,\n      durationRate: 0\n    });\n    fabric.runningAnimations.push(context);\n\n    requestAnimFrame(function(timestamp) {\n      var start = timestamp || +new Date(),\n          duration = options.duration || 500,\n          finish = start + duration, time,\n          onChange = options.onChange || noop,\n          abort = options.abort || noop,\n          onComplete = options.onComplete || noop,\n          easing = options.easing || defaultEasing,\n          isMany = 'startValue' in options ? options.startValue.length > 0 : false,\n          startValue = 'startValue' in options ? options.startValue : 0,\n          endValue = 'endValue' in options ? options.endValue : 100,\n          byValue = options.byValue || (isMany ? startValue.map(function(value, i) {\n            return endValue[i] - startValue[i];\n          }) : endValue - startValue);\n\n      options.onStart && options.onStart();\n\n      (function tick(ticktime) {\n        time = ticktime || +new Date();\n        var currentTime = time > finish ? duration : (time - start),\n            timePerc = currentTime / duration,\n            current = isMany ? startValue.map(function(_value, i) {\n              return easing(currentTime, startValue[i], byValue[i], duration);\n            }) : easing(currentTime, startValue, byValue, duration),\n            valuePerc = isMany ? Math.abs((current[0] - startValue[0]) / byValue[0])\n              : Math.abs((current - startValue) / byValue);\n        //  update context\n        context.currentValue = isMany ? current.slice() : current;\n        context.completionRate = valuePerc;\n        context.durationRate = timePerc;\n        if (cancel) {\n          return;\n        }\n        if (abort(current, valuePerc, timePerc)) {\n          removeFromRegistry();\n          return;\n        }\n        if (time > finish) {\n          //  update context\n          context.currentValue = isMany ? endValue.slice() : endValue;\n          context.completionRate = 1;\n          context.durationRate = 1;\n          //  execute callbacks\n          onChange(isMany ? endValue.slice() : endValue, 1, 1);\n          onComplete(endValue, 1, 1);\n          removeFromRegistry();\n          return;\n        }\n        else {\n          onChange(current, valuePerc, timePerc);\n          requestAnimFrame(tick);\n        }\n      })(start);\n    });\n\n    return context.cancel;\n  }\n\n  var _requestAnimFrame = fabric.window.requestAnimationFrame       ||\n                          fabric.window.webkitRequestAnimationFrame ||\n                          fabric.window.mozRequestAnimationFrame    ||\n                          fabric.window.oRequestAnimationFrame      ||\n                          fabric.window.msRequestAnimationFrame     ||\n                          function(callback) {\n                            return fabric.window.setTimeout(callback, 1000 / 60);\n                          };\n\n  var _cancelAnimFrame = fabric.window.cancelAnimationFrame || fabric.window.clearTimeout;\n\n  /**\n   * requestAnimationFrame polyfill based on http://paulirish.com/2011/requestanimationframe-for-smart-animating/\n   * In order to get a precise start time, `requestAnimFrame` should be called as an entry into the method\n   * @memberOf fabric.util\n   * @param {Function} callback Callback to invoke\n   * @param {DOMElement} element optional Element to associate with animation\n   */\n  function requestAnimFrame() {\n    return _requestAnimFrame.apply(fabric.window, arguments);\n  }\n\n  function cancelAnimFrame() {\n    return _cancelAnimFrame.apply(fabric.window, arguments);\n  }\n\n  fabric.util.animate = animate;\n  fabric.util.requestAnimFrame = requestAnimFrame;\n  fabric.util.cancelAnimFrame = cancelAnimFrame;\n  fabric.runningAnimations = RUNNING_ANIMATIONS;\n})();\n\n\n(function() {\n  // Calculate an in-between color. Returns a \"rgba()\" string.\n  // Credit: Edwin Martin <edwin@bitstorm.org>\n  //         http://www.bitstorm.org/jquery/color-animation/jquery.animate-colors.js\n  function calculateColor(begin, end, pos) {\n    var color = 'rgba('\n        + parseInt((begin[0] + pos * (end[0] - begin[0])), 10) + ','\n        + parseInt((begin[1] + pos * (end[1] - begin[1])), 10) + ','\n        + parseInt((begin[2] + pos * (end[2] - begin[2])), 10);\n\n    color += ',' + (begin && end ? parseFloat(begin[3] + pos * (end[3] - begin[3])) : 1);\n    color += ')';\n    return color;\n  }\n\n  /**\n   * Changes the color from one to another within certain period of time, invoking callbacks as value is being changed.\n   * @memberOf fabric.util\n   * @param {String} fromColor The starting color in hex or rgb(a) format.\n   * @param {String} toColor The starting color in hex or rgb(a) format.\n   * @param {Number} [duration] Duration of change (in ms).\n   * @param {Object} [options] Animation options\n   * @param {Function} [options.onChange] Callback; invoked on every value change\n   * @param {Function} [options.onComplete] Callback; invoked when value change is completed\n   * @param {Function} [options.colorEasing] Easing function. Note that this function only take two arguments (currentTime, duration). Thus the regular animation easing functions cannot be used.\n   * @param {Function} [options.abort] Additional function with logic. If returns true, onComplete is called.\n   * @returns {Function} abort function\n   */\n  function animateColor(fromColor, toColor, duration, options) {\n    var startColor = new fabric.Color(fromColor).getSource(),\n        endColor = new fabric.Color(toColor).getSource(),\n        originalOnComplete = options.onComplete,\n        originalOnChange = options.onChange;\n    options = options || {};\n\n    return fabric.util.animate(fabric.util.object.extend(options, {\n      duration: duration || 500,\n      startValue: startColor,\n      endValue: endColor,\n      byValue: endColor,\n      easing: function (currentTime, startValue, byValue, duration) {\n        var posValue = options.colorEasing\n          ? options.colorEasing(currentTime, duration)\n          : 1 - Math.cos(currentTime / duration * (Math.PI / 2));\n        return calculateColor(startValue, byValue, posValue);\n      },\n      // has to take in account for color restoring;\n      onComplete: function(current, valuePerc, timePerc) {\n        if (originalOnComplete) {\n          return originalOnComplete(\n            calculateColor(endColor, endColor, 0),\n            valuePerc,\n            timePerc\n          );\n        }\n      },\n      onChange: function(current, valuePerc, timePerc) {\n        if (originalOnChange) {\n          if (Array.isArray(current)) {\n            return originalOnChange(\n              calculateColor(current, current, 0),\n              valuePerc,\n              timePerc\n            );\n          }\n          originalOnChange(current, valuePerc, timePerc);\n        }\n      }\n    }));\n  }\n\n  fabric.util.animateColor = animateColor;\n\n})();\n\n\n(function() {\n\n  function normalize(a, c, p, s) {\n    if (a < Math.abs(c)) {\n      a = c;\n      s = p / 4;\n    }\n    else {\n      //handle the 0/0 case:\n      if (c === 0 && a === 0) {\n        s = p / (2 * Math.PI) * Math.asin(1);\n      }\n      else {\n        s = p / (2 * Math.PI) * Math.asin(c / a);\n      }\n    }\n    return { a: a, c: c, p: p, s: s };\n  }\n\n  function elastic(opts, t, d) {\n    return opts.a *\n      Math.pow(2, 10 * (t -= 1)) *\n      Math.sin( (t * d - opts.s) * (2 * Math.PI) / opts.p );\n  }\n\n  /**\n   * Cubic easing out\n   * @memberOf fabric.util.ease\n   */\n  function easeOutCubic(t, b, c, d) {\n    return c * ((t = t / d - 1) * t * t + 1) + b;\n  }\n\n  /**\n   * Cubic easing in and out\n   * @memberOf fabric.util.ease\n   */\n  function easeInOutCubic(t, b, c, d) {\n    t /= d / 2;\n    if (t < 1) {\n      return c / 2 * t * t * t + b;\n    }\n    return c / 2 * ((t -= 2) * t * t + 2) + b;\n  }\n\n  /**\n   * Quartic easing in\n   * @memberOf fabric.util.ease\n   */\n  function easeInQuart(t, b, c, d) {\n    return c * (t /= d) * t * t * t + b;\n  }\n\n  /**\n   * Quartic easing out\n   * @memberOf fabric.util.ease\n   */\n  function easeOutQuart(t, b, c, d) {\n    return -c * ((t = t / d - 1) * t * t * t - 1) + b;\n  }\n\n  /**\n   * Quartic easing in and out\n   * @memberOf fabric.util.ease\n   */\n  function easeInOutQuart(t, b, c, d) {\n    t /= d / 2;\n    if (t < 1) {\n      return c / 2 * t * t * t * t + b;\n    }\n    return -c / 2 * ((t -= 2) * t * t * t - 2) + b;\n  }\n\n  /**\n   * Quintic easing in\n   * @memberOf fabric.util.ease\n   */\n  function easeInQuint(t, b, c, d) {\n    return c * (t /= d) * t * t * t * t + b;\n  }\n\n  /**\n   * Quintic easing out\n   * @memberOf fabric.util.ease\n   */\n  function easeOutQuint(t, b, c, d) {\n    return c * ((t = t / d - 1) * t * t * t * t + 1) + b;\n  }\n\n  /**\n   * Quintic easing in and out\n   * @memberOf fabric.util.ease\n   */\n  function easeInOutQuint(t, b, c, d) {\n    t /= d / 2;\n    if (t < 1) {\n      return c / 2 * t * t * t * t * t + b;\n    }\n    return c / 2 * ((t -= 2) * t * t * t * t + 2) + b;\n  }\n\n  /**\n   * Sinusoidal easing in\n   * @memberOf fabric.util.ease\n   */\n  function easeInSine(t, b, c, d) {\n    return -c * Math.cos(t / d * (Math.PI / 2)) + c + b;\n  }\n\n  /**\n   * Sinusoidal easing out\n   * @memberOf fabric.util.ease\n   */\n  function easeOutSine(t, b, c, d) {\n    return c * Math.sin(t / d * (Math.PI / 2)) + b;\n  }\n\n  /**\n   * Sinusoidal easing in and out\n   * @memberOf fabric.util.ease\n   */\n  function easeInOutSine(t, b, c, d) {\n    return -c / 2 * (Math.cos(Math.PI * t / d) - 1) + b;\n  }\n\n  /**\n   * Exponential easing in\n   * @memberOf fabric.util.ease\n   */\n  function easeInExpo(t, b, c, d) {\n    return (t === 0) ? b : c * Math.pow(2, 10 * (t / d - 1)) + b;\n  }\n\n  /**\n   * Exponential easing out\n   * @memberOf fabric.util.ease\n   */\n  function easeOutExpo(t, b, c, d) {\n    return (t === d) ? b + c : c * (-Math.pow(2, -10 * t / d) + 1) + b;\n  }\n\n  /**\n   * Exponential easing in and out\n   * @memberOf fabric.util.ease\n   */\n  function easeInOutExpo(t, b, c, d) {\n    if (t === 0) {\n      return b;\n    }\n    if (t === d) {\n      return b + c;\n    }\n    t /= d / 2;\n    if (t < 1) {\n      return c / 2 * Math.pow(2, 10 * (t - 1)) + b;\n    }\n    return c / 2 * (-Math.pow(2, -10 * --t) + 2) + b;\n  }\n\n  /**\n   * Circular easing in\n   * @memberOf fabric.util.ease\n   */\n  function easeInCirc(t, b, c, d) {\n    return -c * (Math.sqrt(1 - (t /= d) * t) - 1) + b;\n  }\n\n  /**\n   * Circular easing out\n   * @memberOf fabric.util.ease\n   */\n  function easeOutCirc(t, b, c, d) {\n    return c * Math.sqrt(1 - (t = t / d - 1) * t) + b;\n  }\n\n  /**\n   * Circular easing in and out\n   * @memberOf fabric.util.ease\n   */\n  function easeInOutCirc(t, b, c, d) {\n    t /= d / 2;\n    if (t < 1) {\n      return -c / 2 * (Math.sqrt(1 - t * t) - 1) + b;\n    }\n    return c / 2 * (Math.sqrt(1 - (t -= 2) * t) + 1) + b;\n  }\n\n  /**\n   * Elastic easing in\n   * @memberOf fabric.util.ease\n   */\n  function easeInElastic(t, b, c, d) {\n    var s = 1.70158, p = 0, a = c;\n    if (t === 0) {\n      return b;\n    }\n    t /= d;\n    if (t === 1) {\n      return b + c;\n    }\n    if (!p) {\n      p = d * 0.3;\n    }\n    var opts = normalize(a, c, p, s);\n    return -elastic(opts, t, d) + b;\n  }\n\n  /**\n   * Elastic easing out\n   * @memberOf fabric.util.ease\n   */\n  function easeOutElastic(t, b, c, d) {\n    var s = 1.70158, p = 0, a = c;\n    if (t === 0) {\n      return b;\n    }\n    t /= d;\n    if (t === 1) {\n      return b + c;\n    }\n    if (!p) {\n      p = d * 0.3;\n    }\n    var opts = normalize(a, c, p, s);\n    return opts.a * Math.pow(2, -10 * t) * Math.sin((t * d - opts.s) * (2 * Math.PI) / opts.p ) + opts.c + b;\n  }\n\n  /**\n   * Elastic easing in and out\n   * @memberOf fabric.util.ease\n   */\n  function easeInOutElastic(t, b, c, d) {\n    var s = 1.70158, p = 0, a = c;\n    if (t === 0) {\n      return b;\n    }\n    t /= d / 2;\n    if (t === 2) {\n      return b + c;\n    }\n    if (!p) {\n      p = d * (0.3 * 1.5);\n    }\n    var opts = normalize(a, c, p, s);\n    if (t < 1) {\n      return -0.5 * elastic(opts, t, d) + b;\n    }\n    return opts.a * Math.pow(2, -10 * (t -= 1)) *\n      Math.sin((t * d - opts.s) * (2 * Math.PI) / opts.p ) * 0.5 + opts.c + b;\n  }\n\n  /**\n   * Backwards easing in\n   * @memberOf fabric.util.ease\n   */\n  function easeInBack(t, b, c, d, s) {\n    if (s === undefined) {\n      s = 1.70158;\n    }\n    return c * (t /= d) * t * ((s + 1) * t - s) + b;\n  }\n\n  /**\n   * Backwards easing out\n   * @memberOf fabric.util.ease\n   */\n  function easeOutBack(t, b, c, d, s) {\n    if (s === undefined) {\n      s = 1.70158;\n    }\n    return c * ((t = t / d - 1) * t * ((s + 1) * t + s) + 1) + b;\n  }\n\n  /**\n   * Backwards easing in and out\n   * @memberOf fabric.util.ease\n   */\n  function easeInOutBack(t, b, c, d, s) {\n    if (s === undefined) {\n      s = 1.70158;\n    }\n    t /= d / 2;\n    if (t < 1) {\n      return c / 2 * (t * t * (((s *= (1.525)) + 1) * t - s)) + b;\n    }\n    return c / 2 * ((t -= 2) * t * (((s *= (1.525)) + 1) * t + s) + 2) + b;\n  }\n\n  /**\n   * Bouncing easing in\n   * @memberOf fabric.util.ease\n   */\n  function easeInBounce(t, b, c, d) {\n    return c - easeOutBounce (d - t, 0, c, d) + b;\n  }\n\n  /**\n   * Bouncing easing out\n   * @memberOf fabric.util.ease\n   */\n  function easeOutBounce(t, b, c, d) {\n    if ((t /= d) < (1 / 2.75)) {\n      return c * (7.5625 * t * t) + b;\n    }\n    else if (t < (2 / 2.75)) {\n      return c * (7.5625 * (t -= (1.5 / 2.75)) * t + 0.75) + b;\n    }\n    else if (t < (2.5 / 2.75)) {\n      return c * (7.5625 * (t -= (2.25 / 2.75)) * t + 0.9375) + b;\n    }\n    else {\n      return c * (7.5625 * (t -= (2.625 / 2.75)) * t + 0.984375) + b;\n    }\n  }\n\n  /**\n   * Bouncing easing in and out\n   * @memberOf fabric.util.ease\n   */\n  function easeInOutBounce(t, b, c, d) {\n    if (t < d / 2) {\n      return easeInBounce (t * 2, 0, c, d) * 0.5 + b;\n    }\n    return easeOutBounce(t * 2 - d, 0, c, d) * 0.5 + c * 0.5 + b;\n  }\n\n  /**\n   * Easing functions\n   * See <a href=\"http://gizma.com/easing/\">Easing Equations by Robert Penner</a>\n   * @namespace fabric.util.ease\n   */\n  fabric.util.ease = {\n\n    /**\n     * Quadratic easing in\n     * @memberOf fabric.util.ease\n     */\n    easeInQuad: function(t, b, c, d) {\n      return c * (t /= d) * t + b;\n    },\n\n    /**\n     * Quadratic easing out\n     * @memberOf fabric.util.ease\n     */\n    easeOutQuad: function(t, b, c, d) {\n      return -c * (t /= d) * (t - 2) + b;\n    },\n\n    /**\n     * Quadratic easing in and out\n     * @memberOf fabric.util.ease\n     */\n    easeInOutQuad: function(t, b, c, d) {\n      t /= (d / 2);\n      if (t < 1) {\n        return c / 2 * t * t + b;\n      }\n      return -c / 2 * ((--t) * (t - 2) - 1) + b;\n    },\n\n    /**\n     * Cubic easing in\n     * @memberOf fabric.util.ease\n     */\n    easeInCubic: function(t, b, c, d) {\n      return c * (t /= d) * t * t + b;\n    },\n\n    easeOutCubic: easeOutCubic,\n    easeInOutCubic: easeInOutCubic,\n    easeInQuart: easeInQuart,\n    easeOutQuart: easeOutQuart,\n    easeInOutQuart: easeInOutQuart,\n    easeInQuint: easeInQuint,\n    easeOutQuint: easeOutQuint,\n    easeInOutQuint: easeInOutQuint,\n    easeInSine: easeInSine,\n    easeOutSine: easeOutSine,\n    easeInOutSine: easeInOutSine,\n    easeInExpo: easeInExpo,\n    easeOutExpo: easeOutExpo,\n    easeInOutExpo: easeInOutExpo,\n    easeInCirc: easeInCirc,\n    easeOutCirc: easeOutCirc,\n    easeInOutCirc: easeInOutCirc,\n    easeInElastic: easeInElastic,\n    easeOutElastic: easeOutElastic,\n    easeInOutElastic: easeInOutElastic,\n    easeInBack: easeInBack,\n    easeOutBack: easeOutBack,\n    easeInOutBack: easeInOutBack,\n    easeInBounce: easeInBounce,\n    easeOutBounce: easeOutBounce,\n    easeInOutBounce: easeInOutBounce\n  };\n\n})();\n\n\n(function(global) {\n\n  'use strict';\n\n  /**\n   * @name fabric\n   * @namespace\n   */\n\n  var fabric = global.fabric || (global.fabric = { }),\n      extend = fabric.util.object.extend,\n      clone = fabric.util.object.clone,\n      toFixed = fabric.util.toFixed,\n      parseUnit = fabric.util.parseUnit,\n      multiplyTransformMatrices = fabric.util.multiplyTransformMatrices,\n\n      svgValidTagNames = ['path', 'circle', 'polygon', 'polyline', 'ellipse', 'rect', 'line',\n        'image', 'text'],\n      svgViewBoxElements = ['symbol', 'image', 'marker', 'pattern', 'view', 'svg'],\n      svgInvalidAncestors = ['pattern', 'defs', 'symbol', 'metadata', 'clipPath', 'mask', 'desc'],\n      svgValidParents = ['symbol', 'g', 'a', 'svg', 'clipPath', 'defs'],\n\n      attributesMap = {\n        cx:                   'left',\n        x:                    'left',\n        r:                    'radius',\n        cy:                   'top',\n        y:                    'top',\n        display:              'visible',\n        visibility:           'visible',\n        transform:            'transformMatrix',\n        'fill-opacity':       'fillOpacity',\n        'fill-rule':          'fillRule',\n        'font-family':        'fontFamily',\n        'font-size':          'fontSize',\n        'font-style':         'fontStyle',\n        'font-weight':        'fontWeight',\n        'letter-spacing':     'charSpacing',\n        'paint-order':        'paintFirst',\n        'stroke-dasharray':   'strokeDashArray',\n        'stroke-dashoffset':  'strokeDashOffset',\n        'stroke-linecap':     'strokeLineCap',\n        'stroke-linejoin':    'strokeLineJoin',\n        'stroke-miterlimit':  'strokeMiterLimit',\n        'stroke-opacity':     'strokeOpacity',\n        'stroke-width':       'strokeWidth',\n        'text-decoration':    'textDecoration',\n        'text-anchor':        'textAnchor',\n        opacity:              'opacity',\n        'clip-path':          'clipPath',\n        'clip-rule':          'clipRule',\n        'vector-effect':      'strokeUniform',\n        'image-rendering':    'imageSmoothing',\n      },\n\n      colorAttributes = {\n        stroke: 'strokeOpacity',\n        fill:   'fillOpacity'\n      },\n\n      fSize = 'font-size', cPath = 'clip-path';\n\n  fabric.svgValidTagNamesRegEx = getSvgRegex(svgValidTagNames);\n  fabric.svgViewBoxElementsRegEx = getSvgRegex(svgViewBoxElements);\n  fabric.svgInvalidAncestorsRegEx = getSvgRegex(svgInvalidAncestors);\n  fabric.svgValidParentsRegEx = getSvgRegex(svgValidParents);\n\n  fabric.cssRules = { };\n  fabric.gradientDefs = { };\n  fabric.clipPaths = { };\n\n  function normalizeAttr(attr) {\n    // transform attribute names\n    if (attr in attributesMap) {\n      return attributesMap[attr];\n    }\n    return attr;\n  }\n\n  function normalizeValue(attr, value, parentAttributes, fontSize) {\n    var isArray = Array.isArray(value), parsed;\n\n    if ((attr === 'fill' || attr === 'stroke') && value === 'none') {\n      value = '';\n    }\n    else if (attr === 'strokeUniform') {\n      return (value === 'non-scaling-stroke');\n    }\n    else if (attr === 'strokeDashArray') {\n      if (value === 'none') {\n        value = null;\n      }\n      else {\n        value = value.replace(/,/g, ' ').split(/\\s+/).map(parseFloat);\n      }\n    }\n    else if (attr === 'transformMatrix') {\n      if (parentAttributes && parentAttributes.transformMatrix) {\n        value = multiplyTransformMatrices(\n          parentAttributes.transformMatrix, fabric.parseTransformAttribute(value));\n      }\n      else {\n        value = fabric.parseTransformAttribute(value);\n      }\n    }\n    else if (attr === 'visible') {\n      value = value !== 'none' && value !== 'hidden';\n      // display=none on parent element always takes precedence over child element\n      if (parentAttributes && parentAttributes.visible === false) {\n        value = false;\n      }\n    }\n    else if (attr === 'opacity') {\n      value = parseFloat(value);\n      if (parentAttributes && typeof parentAttributes.opacity !== 'undefined') {\n        value *= parentAttributes.opacity;\n      }\n    }\n    else if (attr === 'textAnchor' /* text-anchor */) {\n      value = value === 'start' ? 'left' : value === 'end' ? 'right' : 'center';\n    }\n    else if (attr === 'charSpacing') {\n      // parseUnit returns px and we convert it to em\n      parsed = parseUnit(value, fontSize) / fontSize * 1000;\n    }\n    else if (attr === 'paintFirst') {\n      var fillIndex = value.indexOf('fill');\n      var strokeIndex = value.indexOf('stroke');\n      var value = 'fill';\n      if (fillIndex > -1 && strokeIndex > -1 && strokeIndex < fillIndex) {\n        value = 'stroke';\n      }\n      else if (fillIndex === -1 && strokeIndex > -1) {\n        value = 'stroke';\n      }\n    }\n    else if (attr === 'href' || attr === 'xlink:href' || attr === 'font') {\n      return value;\n    }\n    else if (attr === 'imageSmoothing') {\n      return (value === 'optimizeQuality');\n    }\n    else {\n      parsed = isArray ? value.map(parseUnit) : parseUnit(value, fontSize);\n    }\n\n    return (!isArray && isNaN(parsed) ? value : parsed);\n  }\n\n  /**\n    * @private\n    */\n  function getSvgRegex(arr) {\n    return new RegExp('^(' + arr.join('|') + ')\\\\b', 'i');\n  }\n\n  /**\n   * @private\n   * @param {Object} attributes Array of attributes to parse\n   */\n  function _setStrokeFillOpacity(attributes) {\n    for (var attr in colorAttributes) {\n\n      if (typeof attributes[colorAttributes[attr]] === 'undefined' || attributes[attr] === '') {\n        continue;\n      }\n\n      if (typeof attributes[attr] === 'undefined') {\n        if (!fabric.Object.prototype[attr]) {\n          continue;\n        }\n        attributes[attr] = fabric.Object.prototype[attr];\n      }\n\n      if (attributes[attr].indexOf('url(') === 0) {\n        continue;\n      }\n\n      var color = new fabric.Color(attributes[attr]);\n      attributes[attr] = color.setAlpha(toFixed(color.getAlpha() * attributes[colorAttributes[attr]], 2)).toRgba();\n    }\n    return attributes;\n  }\n\n  /**\n   * @private\n   */\n  function _getMultipleNodes(doc, nodeNames) {\n    var nodeName, nodeArray = [], nodeList, i, len;\n    for (i = 0, len = nodeNames.length; i < len; i++) {\n      nodeName = nodeNames[i];\n      nodeList = doc.getElementsByTagName(nodeName);\n      nodeArray = nodeArray.concat(Array.prototype.slice.call(nodeList));\n    }\n    return nodeArray;\n  }\n\n  /**\n   * Parses \"transform\" attribute, returning an array of values\n   * @static\n   * @function\n   * @memberOf fabric\n   * @param {String} attributeValue String containing attribute value\n   * @return {Array} Array of 6 elements representing transformation matrix\n   */\n  fabric.parseTransformAttribute = (function() {\n    function rotateMatrix(matrix, args) {\n      var cos = fabric.util.cos(args[0]), sin = fabric.util.sin(args[0]),\n          x = 0, y = 0;\n      if (args.length === 3) {\n        x = args[1];\n        y = args[2];\n      }\n\n      matrix[0] = cos;\n      matrix[1] = sin;\n      matrix[2] = -sin;\n      matrix[3] = cos;\n      matrix[4] = x - (cos * x - sin * y);\n      matrix[5] = y - (sin * x + cos * y);\n    }\n\n    function scaleMatrix(matrix, args) {\n      var multiplierX = args[0],\n          multiplierY = (args.length === 2) ? args[1] : args[0];\n\n      matrix[0] = multiplierX;\n      matrix[3] = multiplierY;\n    }\n\n    function skewMatrix(matrix, args, pos) {\n      matrix[pos] = Math.tan(fabric.util.degreesToRadians(args[0]));\n    }\n\n    function translateMatrix(matrix, args) {\n      matrix[4] = args[0];\n      if (args.length === 2) {\n        matrix[5] = args[1];\n      }\n    }\n\n    // identity matrix\n    var iMatrix = fabric.iMatrix,\n\n        // == begin transform regexp\n        number = fabric.reNum,\n\n        commaWsp = fabric.commaWsp,\n\n        skewX = '(?:(skewX)\\\\s*\\\\(\\\\s*(' + number + ')\\\\s*\\\\))',\n\n        skewY = '(?:(skewY)\\\\s*\\\\(\\\\s*(' + number + ')\\\\s*\\\\))',\n\n        rotate = '(?:(rotate)\\\\s*\\\\(\\\\s*(' + number + ')(?:' +\n                    commaWsp + '(' + number + ')' +\n                    commaWsp + '(' + number + '))?\\\\s*\\\\))',\n\n        scale = '(?:(scale)\\\\s*\\\\(\\\\s*(' + number + ')(?:' +\n                    commaWsp + '(' + number + '))?\\\\s*\\\\))',\n\n        translate = '(?:(translate)\\\\s*\\\\(\\\\s*(' + number + ')(?:' +\n                    commaWsp + '(' + number + '))?\\\\s*\\\\))',\n\n        matrix = '(?:(matrix)\\\\s*\\\\(\\\\s*' +\n                  '(' + number + ')' + commaWsp +\n                  '(' + number + ')' + commaWsp +\n                  '(' + number + ')' + commaWsp +\n                  '(' + number + ')' + commaWsp +\n                  '(' + number + ')' + commaWsp +\n                  '(' + number + ')' +\n                  '\\\\s*\\\\))',\n\n        transform = '(?:' +\n                    matrix + '|' +\n                    translate + '|' +\n                    scale + '|' +\n                    rotate + '|' +\n                    skewX + '|' +\n                    skewY +\n                    ')',\n\n        transforms = '(?:' + transform + '(?:' + commaWsp + '*' + transform + ')*' + ')',\n\n        transformList = '^\\\\s*(?:' + transforms + '?)\\\\s*$',\n\n        // http://www.w3.org/TR/SVG/coords.html#TransformAttribute\n        reTransformList = new RegExp(transformList),\n        // == end transform regexp\n\n        reTransform = new RegExp(transform, 'g');\n\n    return function(attributeValue) {\n\n      // start with identity matrix\n      var matrix = iMatrix.concat(),\n          matrices = [];\n\n      // return if no argument was given or\n      // an argument does not match transform attribute regexp\n      if (!attributeValue || (attributeValue && !reTransformList.test(attributeValue))) {\n        return matrix;\n      }\n\n      attributeValue.replace(reTransform, function(match) {\n\n        var m = new RegExp(transform).exec(match).filter(function (match) {\n              // match !== '' && match != null\n              return (!!match);\n            }),\n            operation = m[1],\n            args = m.slice(2).map(parseFloat);\n\n        switch (operation) {\n          case 'translate':\n            translateMatrix(matrix, args);\n            break;\n          case 'rotate':\n            args[0] = fabric.util.degreesToRadians(args[0]);\n            rotateMatrix(matrix, args);\n            break;\n          case 'scale':\n            scaleMatrix(matrix, args);\n            break;\n          case 'skewX':\n            skewMatrix(matrix, args, 2);\n            break;\n          case 'skewY':\n            skewMatrix(matrix, args, 1);\n            break;\n          case 'matrix':\n            matrix = args;\n            break;\n        }\n\n        // snapshot current matrix into matrices array\n        matrices.push(matrix.concat());\n        // reset\n        matrix = iMatrix.concat();\n      });\n\n      var combinedMatrix = matrices[0];\n      while (matrices.length > 1) {\n        matrices.shift();\n        combinedMatrix = fabric.util.multiplyTransformMatrices(combinedMatrix, matrices[0]);\n      }\n      return combinedMatrix;\n    };\n  })();\n\n  /**\n   * @private\n   */\n  function parseStyleString(style, oStyle) {\n    var attr, value;\n    style.replace(/;\\s*$/, '').split(';').forEach(function (chunk) {\n      var pair = chunk.split(':');\n\n      attr = pair[0].trim().toLowerCase();\n      value =  pair[1].trim();\n\n      oStyle[attr] = value;\n    });\n  }\n\n  /**\n   * @private\n   */\n  function parseStyleObject(style, oStyle) {\n    var attr, value;\n    for (var prop in style) {\n      if (typeof style[prop] === 'undefined') {\n        continue;\n      }\n\n      attr = prop.toLowerCase();\n      value = style[prop];\n\n      oStyle[attr] = value;\n    }\n  }\n\n  /**\n   * @private\n   */\n  function getGlobalStylesForElement(element, svgUid) {\n    var styles = { };\n    for (var rule in fabric.cssRules[svgUid]) {\n      if (elementMatchesRule(element, rule.split(' '))) {\n        for (var property in fabric.cssRules[svgUid][rule]) {\n          styles[property] = fabric.cssRules[svgUid][rule][property];\n        }\n      }\n    }\n    return styles;\n  }\n\n  /**\n   * @private\n   */\n  function elementMatchesRule(element, selectors) {\n    var firstMatching, parentMatching = true;\n    //start from rightmost selector.\n    firstMatching = selectorMatches(element, selectors.pop());\n    if (firstMatching && selectors.length) {\n      parentMatching = doesSomeParentMatch(element, selectors);\n    }\n    return firstMatching && parentMatching && (selectors.length === 0);\n  }\n\n  function doesSomeParentMatch(element, selectors) {\n    var selector, parentMatching = true;\n    while (element.parentNode && element.parentNode.nodeType === 1 && selectors.length) {\n      if (parentMatching) {\n        selector = selectors.pop();\n      }\n      element = element.parentNode;\n      parentMatching = selectorMatches(element, selector);\n    }\n    return selectors.length === 0;\n  }\n\n  /**\n   * @private\n   */\n  function selectorMatches(element, selector) {\n    var nodeName = element.nodeName,\n        classNames = element.getAttribute('class'),\n        id = element.getAttribute('id'), matcher, i;\n    // i check if a selector matches slicing away part from it.\n    // if i get empty string i should match\n    matcher = new RegExp('^' + nodeName, 'i');\n    selector = selector.replace(matcher, '');\n    if (id && selector.length) {\n      matcher = new RegExp('#' + id + '(?![a-zA-Z\\\\-]+)', 'i');\n      selector = selector.replace(matcher, '');\n    }\n    if (classNames && selector.length) {\n      classNames = classNames.split(' ');\n      for (i = classNames.length; i--;) {\n        matcher = new RegExp('\\\\.' + classNames[i] + '(?![a-zA-Z\\\\-]+)', 'i');\n        selector = selector.replace(matcher, '');\n      }\n    }\n    return selector.length === 0;\n  }\n\n  /**\n   * @private\n   * to support IE8 missing getElementById on SVGdocument and on node xmlDOM\n   */\n  function elementById(doc, id) {\n    var el;\n    doc.getElementById && (el = doc.getElementById(id));\n    if (el) {\n      return el;\n    }\n    var node, i, len, nodelist = doc.getElementsByTagName('*');\n    for (i = 0, len = nodelist.length; i < len; i++) {\n      node = nodelist[i];\n      if (id === node.getAttribute('id')) {\n        return node;\n      }\n    }\n  }\n\n  /**\n   * @private\n   */\n  function parseUseDirectives(doc) {\n    var nodelist = _getMultipleNodes(doc, ['use', 'svg:use']), i = 0;\n    while (nodelist.length && i < nodelist.length) {\n      var el = nodelist[i],\n          xlinkAttribute = el.getAttribute('xlink:href') || el.getAttribute('href');\n\n      if (xlinkAttribute === null) {\n        return;\n      }\n\n      var xlink = xlinkAttribute.slice(1),\n          x = el.getAttribute('x') || 0,\n          y = el.getAttribute('y') || 0,\n          el2 = elementById(doc, xlink).cloneNode(true),\n          currentTrans = (el2.getAttribute('transform') || '') + ' translate(' + x + ', ' + y + ')',\n          parentNode,\n          oldLength = nodelist.length, attr,\n          j,\n          attrs,\n          len,\n          namespace = fabric.svgNS;\n\n      applyViewboxTransform(el2);\n      if (/^svg$/i.test(el2.nodeName)) {\n        var el3 = el2.ownerDocument.createElementNS(namespace, 'g');\n        for (j = 0, attrs = el2.attributes, len = attrs.length; j < len; j++) {\n          attr = attrs.item(j);\n          el3.setAttributeNS(namespace, attr.nodeName, attr.nodeValue);\n        }\n        // el2.firstChild != null\n        while (el2.firstChild) {\n          el3.appendChild(el2.firstChild);\n        }\n        el2 = el3;\n      }\n\n      for (j = 0, attrs = el.attributes, len = attrs.length; j < len; j++) {\n        attr = attrs.item(j);\n        if (attr.nodeName === 'x' || attr.nodeName === 'y' ||\n          attr.nodeName === 'xlink:href' || attr.nodeName === 'href') {\n          continue;\n        }\n\n        if (attr.nodeName === 'transform') {\n          currentTrans = attr.nodeValue + ' ' + currentTrans;\n        }\n        else {\n          el2.setAttribute(attr.nodeName, attr.nodeValue);\n        }\n      }\n\n      el2.setAttribute('transform', currentTrans);\n      el2.setAttribute('instantiated_by_use', '1');\n      el2.removeAttribute('id');\n      parentNode = el.parentNode;\n      parentNode.replaceChild(el2, el);\n      // some browsers do not shorten nodelist after replaceChild (IE8)\n      if (nodelist.length === oldLength) {\n        i++;\n      }\n    }\n  }\n\n  // http://www.w3.org/TR/SVG/coords.html#ViewBoxAttribute\n  // matches, e.g.: +14.56e-12, etc.\n  var reViewBoxAttrValue = new RegExp(\n    '^' +\n    '\\\\s*(' + fabric.reNum + '+)\\\\s*,?' +\n    '\\\\s*(' + fabric.reNum + '+)\\\\s*,?' +\n    '\\\\s*(' + fabric.reNum + '+)\\\\s*,?' +\n    '\\\\s*(' + fabric.reNum + '+)\\\\s*' +\n    '$'\n  );\n\n  /**\n   * Add a <g> element that envelop all child elements and makes the viewbox transformMatrix descend on all elements\n   */\n  function applyViewboxTransform(element) {\n    if (!fabric.svgViewBoxElementsRegEx.test(element.nodeName)) {\n      return {};\n    }\n    var viewBoxAttr = element.getAttribute('viewBox'),\n        scaleX = 1,\n        scaleY = 1,\n        minX = 0,\n        minY = 0,\n        viewBoxWidth, viewBoxHeight, matrix, el,\n        widthAttr = element.getAttribute('width'),\n        heightAttr = element.getAttribute('height'),\n        x = element.getAttribute('x') || 0,\n        y = element.getAttribute('y') || 0,\n        preserveAspectRatio = element.getAttribute('preserveAspectRatio') || '',\n        missingViewBox = (!viewBoxAttr || !(viewBoxAttr = viewBoxAttr.match(reViewBoxAttrValue))),\n        missingDimAttr = (!widthAttr || !heightAttr || widthAttr === '100%' || heightAttr === '100%'),\n        toBeParsed = missingViewBox && missingDimAttr,\n        parsedDim = { }, translateMatrix = '', widthDiff = 0, heightDiff = 0;\n\n    parsedDim.width = 0;\n    parsedDim.height = 0;\n    parsedDim.toBeParsed = toBeParsed;\n\n    if (missingViewBox) {\n      if (((x || y) && element.parentNode && element.parentNode.nodeName !== '#document')) {\n        translateMatrix = ' translate(' + parseUnit(x) + ' ' + parseUnit(y) + ') ';\n        matrix = (element.getAttribute('transform') || '') + translateMatrix;\n        element.setAttribute('transform', matrix);\n        element.removeAttribute('x');\n        element.removeAttribute('y');\n      }\n    }\n\n    if (toBeParsed) {\n      return parsedDim;\n    }\n\n    if (missingViewBox) {\n      parsedDim.width = parseUnit(widthAttr);\n      parsedDim.height = parseUnit(heightAttr);\n      // set a transform for elements that have x y and are inner(only) SVGs\n      return parsedDim;\n    }\n    minX = -parseFloat(viewBoxAttr[1]);\n    minY = -parseFloat(viewBoxAttr[2]);\n    viewBoxWidth = parseFloat(viewBoxAttr[3]);\n    viewBoxHeight = parseFloat(viewBoxAttr[4]);\n    parsedDim.minX = minX;\n    parsedDim.minY = minY;\n    parsedDim.viewBoxWidth = viewBoxWidth;\n    parsedDim.viewBoxHeight = viewBoxHeight;\n    if (!missingDimAttr) {\n      parsedDim.width = parseUnit(widthAttr);\n      parsedDim.height = parseUnit(heightAttr);\n      scaleX = parsedDim.width / viewBoxWidth;\n      scaleY = parsedDim.height / viewBoxHeight;\n    }\n    else {\n      parsedDim.width = viewBoxWidth;\n      parsedDim.height = viewBoxHeight;\n    }\n\n    // default is to preserve aspect ratio\n    preserveAspectRatio = fabric.util.parsePreserveAspectRatioAttribute(preserveAspectRatio);\n    if (preserveAspectRatio.alignX !== 'none') {\n      //translate all container for the effect of Mid, Min, Max\n      if (preserveAspectRatio.meetOrSlice === 'meet') {\n        scaleY = scaleX = (scaleX > scaleY ? scaleY : scaleX);\n        // calculate additional translation to move the viewbox\n      }\n      if (preserveAspectRatio.meetOrSlice === 'slice') {\n        scaleY = scaleX = (scaleX > scaleY ? scaleX : scaleY);\n        // calculate additional translation to move the viewbox\n      }\n      widthDiff = parsedDim.width - viewBoxWidth * scaleX;\n      heightDiff = parsedDim.height - viewBoxHeight * scaleX;\n      if (preserveAspectRatio.alignX === 'Mid') {\n        widthDiff /= 2;\n      }\n      if (preserveAspectRatio.alignY === 'Mid') {\n        heightDiff /= 2;\n      }\n      if (preserveAspectRatio.alignX === 'Min') {\n        widthDiff = 0;\n      }\n      if (preserveAspectRatio.alignY === 'Min') {\n        heightDiff = 0;\n      }\n    }\n\n    if (scaleX === 1 && scaleY === 1 && minX === 0 && minY === 0 && x === 0 && y === 0) {\n      return parsedDim;\n    }\n    if ((x || y) && element.parentNode.nodeName !== '#document') {\n      translateMatrix = ' translate(' + parseUnit(x) + ' ' + parseUnit(y) + ') ';\n    }\n\n    matrix = translateMatrix + ' matrix(' + scaleX +\n                  ' 0' +\n                  ' 0 ' +\n                  scaleY + ' ' +\n                  (minX * scaleX + widthDiff) + ' ' +\n                  (minY * scaleY + heightDiff) + ') ';\n    // seems unused.\n    // parsedDim.viewboxTransform = fabric.parseTransformAttribute(matrix);\n    if (element.nodeName === 'svg') {\n      el = element.ownerDocument.createElementNS(fabric.svgNS, 'g');\n      // element.firstChild != null\n      while (element.firstChild) {\n        el.appendChild(element.firstChild);\n      }\n      element.appendChild(el);\n    }\n    else {\n      el = element;\n      el.removeAttribute('x');\n      el.removeAttribute('y');\n      matrix = el.getAttribute('transform') + matrix;\n    }\n    el.setAttribute('transform', matrix);\n    return parsedDim;\n  }\n\n  function hasAncestorWithNodeName(element, nodeName) {\n    while (element && (element = element.parentNode)) {\n      if (element.nodeName && nodeName.test(element.nodeName.replace('svg:', ''))\n        && !element.getAttribute('instantiated_by_use')) {\n        return true;\n      }\n    }\n    return false;\n  }\n\n  /**\n   * Parses an SVG document, converts it to an array of corresponding fabric.* instances and passes them to a callback\n   * @static\n   * @function\n   * @memberOf fabric\n   * @param {SVGDocument} doc SVG document to parse\n   * @param {Function} callback Callback to call when parsing is finished;\n   * It's being passed an array of elements (parsed from a document).\n   * @param {Function} [reviver] Method for further parsing of SVG elements, called after each fabric object created.\n   * @param {Object} [parsingOptions] options for parsing document\n   * @param {String} [parsingOptions.crossOrigin] crossOrigin settings\n   */\n  fabric.parseSVGDocument = function(doc, callback, reviver, parsingOptions) {\n    if (!doc) {\n      return;\n    }\n\n    parseUseDirectives(doc);\n\n    var svgUid =  fabric.Object.__uid++, i, len,\n        options = applyViewboxTransform(doc),\n        descendants = fabric.util.toArray(doc.getElementsByTagName('*'));\n    options.crossOrigin = parsingOptions && parsingOptions.crossOrigin;\n    options.svgUid = svgUid;\n\n    if (descendants.length === 0 && fabric.isLikelyNode) {\n      // we're likely in node, where \"o3-xml\" library fails to gEBTN(\"*\")\n      // https://github.com/ajaxorg/node-o3-xml/issues/21\n      descendants = doc.selectNodes('//*[name(.)!=\"svg\"]');\n      var arr = [];\n      for (i = 0, len = descendants.length; i < len; i++) {\n        arr[i] = descendants[i];\n      }\n      descendants = arr;\n    }\n\n    var elements = descendants.filter(function(el) {\n      applyViewboxTransform(el);\n      return fabric.svgValidTagNamesRegEx.test(el.nodeName.replace('svg:', '')) &&\n            !hasAncestorWithNodeName(el, fabric.svgInvalidAncestorsRegEx); // http://www.w3.org/TR/SVG/struct.html#DefsElement\n    });\n    if (!elements || (elements && !elements.length)) {\n      callback && callback([], {});\n      return;\n    }\n    var clipPaths = { };\n    descendants.filter(function(el) {\n      return el.nodeName.replace('svg:', '') === 'clipPath';\n    }).forEach(function(el) {\n      var id = el.getAttribute('id');\n      clipPaths[id] = fabric.util.toArray(el.getElementsByTagName('*')).filter(function(el) {\n        return fabric.svgValidTagNamesRegEx.test(el.nodeName.replace('svg:', ''));\n      });\n    });\n    fabric.gradientDefs[svgUid] = fabric.getGradientDefs(doc);\n    fabric.cssRules[svgUid] = fabric.getCSSRules(doc);\n    fabric.clipPaths[svgUid] = clipPaths;\n    // Precedence of rules:   style > class > attribute\n    fabric.parseElements(elements, function(instances, elements) {\n      if (callback) {\n        callback(instances, options, elements, descendants);\n        delete fabric.gradientDefs[svgUid];\n        delete fabric.cssRules[svgUid];\n        delete fabric.clipPaths[svgUid];\n      }\n    }, clone(options), reviver, parsingOptions);\n  };\n\n  function recursivelyParseGradientsXlink(doc, gradient) {\n    var gradientsAttrs = ['gradientTransform', 'x1', 'x2', 'y1', 'y2', 'gradientUnits', 'cx', 'cy', 'r', 'fx', 'fy'],\n        xlinkAttr = 'xlink:href',\n        xLink = gradient.getAttribute(xlinkAttr).slice(1),\n        referencedGradient = elementById(doc, xLink);\n    if (referencedGradient && referencedGradient.getAttribute(xlinkAttr)) {\n      recursivelyParseGradientsXlink(doc, referencedGradient);\n    }\n    gradientsAttrs.forEach(function(attr) {\n      if (referencedGradient && !gradient.hasAttribute(attr) && referencedGradient.hasAttribute(attr)) {\n        gradient.setAttribute(attr, referencedGradient.getAttribute(attr));\n      }\n    });\n    if (!gradient.children.length) {\n      var referenceClone = referencedGradient.cloneNode(true);\n      while (referenceClone.firstChild) {\n        gradient.appendChild(referenceClone.firstChild);\n      }\n    }\n    gradient.removeAttribute(xlinkAttr);\n  }\n\n  var reFontDeclaration = new RegExp(\n    '(normal|italic)?\\\\s*(normal|small-caps)?\\\\s*' +\n    '(normal|bold|bolder|lighter|100|200|300|400|500|600|700|800|900)?\\\\s*(' +\n      fabric.reNum +\n    '(?:px|cm|mm|em|pt|pc|in)*)(?:\\\\/(normal|' + fabric.reNum + '))?\\\\s+(.*)');\n\n  extend(fabric, {\n    /**\n     * Parses a short font declaration, building adding its properties to a style object\n     * @static\n     * @function\n     * @memberOf fabric\n     * @param {String} value font declaration\n     * @param {Object} oStyle definition\n     */\n    parseFontDeclaration: function(value, oStyle) {\n      var match = value.match(reFontDeclaration);\n\n      if (!match) {\n        return;\n      }\n      var fontStyle = match[1],\n          // font variant is not used\n          // fontVariant = match[2],\n          fontWeight = match[3],\n          fontSize = match[4],\n          lineHeight = match[5],\n          fontFamily = match[6];\n\n      if (fontStyle) {\n        oStyle.fontStyle = fontStyle;\n      }\n      if (fontWeight) {\n        oStyle.fontWeight = isNaN(parseFloat(fontWeight)) ? fontWeight : parseFloat(fontWeight);\n      }\n      if (fontSize) {\n        oStyle.fontSize = parseUnit(fontSize);\n      }\n      if (fontFamily) {\n        oStyle.fontFamily = fontFamily;\n      }\n      if (lineHeight) {\n        oStyle.lineHeight = lineHeight === 'normal' ? 1 : lineHeight;\n      }\n    },\n\n    /**\n     * Parses an SVG document, returning all of the gradient declarations found in it\n     * @static\n     * @function\n     * @memberOf fabric\n     * @param {SVGDocument} doc SVG document to parse\n     * @return {Object} Gradient definitions; key corresponds to element id, value -- to gradient definition element\n     */\n    getGradientDefs: function(doc) {\n      var tagArray = [\n            'linearGradient',\n            'radialGradient',\n            'svg:linearGradient',\n            'svg:radialGradient'],\n          elList = _getMultipleNodes(doc, tagArray),\n          el, j = 0, gradientDefs = { };\n      j = elList.length;\n      while (j--) {\n        el = elList[j];\n        if (el.getAttribute('xlink:href')) {\n          recursivelyParseGradientsXlink(doc, el);\n        }\n        gradientDefs[el.getAttribute('id')] = el;\n      }\n      return gradientDefs;\n    },\n\n    /**\n     * Returns an object of attributes' name/value, given element and an array of attribute names;\n     * Parses parent \"g\" nodes recursively upwards.\n     * @static\n     * @memberOf fabric\n     * @param {DOMElement} element Element to parse\n     * @param {Array} attributes Array of attributes to parse\n     * @return {Object} object containing parsed attributes' names/values\n     */\n    parseAttributes: function(element, attributes, svgUid) {\n\n      if (!element) {\n        return;\n      }\n\n      var value,\n          parentAttributes = { },\n          fontSize, parentFontSize;\n\n      if (typeof svgUid === 'undefined') {\n        svgUid = element.getAttribute('svgUid');\n      }\n      // if there's a parent container (`g` or `a` or `symbol` node), parse its attributes recursively upwards\n      if (element.parentNode && fabric.svgValidParentsRegEx.test(element.parentNode.nodeName)) {\n        parentAttributes = fabric.parseAttributes(element.parentNode, attributes, svgUid);\n      }\n\n      var ownAttributes = attributes.reduce(function(memo, attr) {\n        value = element.getAttribute(attr);\n        if (value) { // eslint-disable-line\n          memo[attr] = value;\n        }\n        return memo;\n      }, { });\n      // add values parsed from style, which take precedence over attributes\n      // (see: http://www.w3.org/TR/SVG/styling.html#UsingPresentationAttributes)\n      var cssAttrs = extend(\n        getGlobalStylesForElement(element, svgUid),\n        fabric.parseStyleAttribute(element)\n      );\n      ownAttributes = extend(\n        ownAttributes,\n        cssAttrs\n      );\n      if (cssAttrs[cPath]) {\n        element.setAttribute(cPath, cssAttrs[cPath]);\n      }\n      fontSize = parentFontSize = parentAttributes.fontSize || fabric.Text.DEFAULT_SVG_FONT_SIZE;\n      if (ownAttributes[fSize]) {\n        // looks like the minimum should be 9px when dealing with ems. this is what looks like in browsers.\n        ownAttributes[fSize] = fontSize = parseUnit(ownAttributes[fSize], parentFontSize);\n      }\n\n      var normalizedAttr, normalizedValue, normalizedStyle = {};\n      for (var attr in ownAttributes) {\n        normalizedAttr = normalizeAttr(attr);\n        normalizedValue = normalizeValue(normalizedAttr, ownAttributes[attr], parentAttributes, fontSize);\n        normalizedStyle[normalizedAttr] = normalizedValue;\n      }\n      if (normalizedStyle && normalizedStyle.font) {\n        fabric.parseFontDeclaration(normalizedStyle.font, normalizedStyle);\n      }\n      var mergedAttrs = extend(parentAttributes, normalizedStyle);\n      return fabric.svgValidParentsRegEx.test(element.nodeName) ? mergedAttrs : _setStrokeFillOpacity(mergedAttrs);\n    },\n\n    /**\n     * Transforms an array of svg elements to corresponding fabric.* instances\n     * @static\n     * @memberOf fabric\n     * @param {Array} elements Array of elements to parse\n     * @param {Function} callback Being passed an array of fabric instances (transformed from SVG elements)\n     * @param {Object} [options] Options object\n     * @param {Function} [reviver] Method for further parsing of SVG elements, called after each fabric object created.\n     */\n    parseElements: function(elements, callback, options, reviver, parsingOptions) {\n      new fabric.ElementsParser(elements, callback, options, reviver, parsingOptions).parse();\n    },\n\n    /**\n     * Parses \"style\" attribute, retuning an object with values\n     * @static\n     * @memberOf fabric\n     * @param {SVGElement} element Element to parse\n     * @return {Object} Objects with values parsed from style attribute of an element\n     */\n    parseStyleAttribute: function(element) {\n      var oStyle = { },\n          style = element.getAttribute('style');\n\n      if (!style) {\n        return oStyle;\n      }\n\n      if (typeof style === 'string') {\n        parseStyleString(style, oStyle);\n      }\n      else {\n        parseStyleObject(style, oStyle);\n      }\n\n      return oStyle;\n    },\n\n    /**\n     * Parses \"points\" attribute, returning an array of values\n     * @static\n     * @memberOf fabric\n     * @param {String} points points attribute string\n     * @return {Array} array of points\n     */\n    parsePointsAttribute: function(points) {\n\n      // points attribute is required and must not be empty\n      if (!points) {\n        return null;\n      }\n\n      // replace commas with whitespace and remove bookending whitespace\n      points = points.replace(/,/g, ' ').trim();\n\n      points = points.split(/\\s+/);\n      var parsedPoints = [], i, len;\n\n      for (i = 0, len = points.length; i < len; i += 2) {\n        parsedPoints.push({\n          x: parseFloat(points[i]),\n          y: parseFloat(points[i + 1])\n        });\n      }\n\n      // odd number of points is an error\n      // if (parsedPoints.length % 2 !== 0) {\n      //   return null;\n      // }\n\n      return parsedPoints;\n    },\n\n    /**\n     * Returns CSS rules for a given SVG document\n     * @static\n     * @function\n     * @memberOf fabric\n     * @param {SVGDocument} doc SVG document to parse\n     * @return {Object} CSS rules of this document\n     */\n    getCSSRules: function(doc) {\n      var styles = doc.getElementsByTagName('style'), i, len,\n          allRules = { }, rules;\n\n      // very crude parsing of style contents\n      for (i = 0, len = styles.length; i < len; i++) {\n        var styleContents = styles[i].textContent;\n\n        // remove comments\n        styleContents = styleContents.replace(/\\/\\*[\\s\\S]*?\\*\\//g, '');\n        if (styleContents.trim() === '') {\n          continue;\n        }\n        // recovers all the rule in this form `body { style code... }`\n        // rules = styleContents.match(/[^{]*\\{[\\s\\S]*?\\}/g);\n        rules = styleContents.split('}');\n        // remove empty rules.\n        rules = rules.filter(function(rule) { return rule.trim(); });\n        // at this point we have hopefully an array of rules `body { style code... `\n        // eslint-disable-next-line no-loop-func\n        rules.forEach(function(rule) {\n\n          var match = rule.split('{'),\n              ruleObj = { }, declaration = match[1].trim(),\n              propertyValuePairs = declaration.split(';').filter(function(pair) { return pair.trim(); });\n\n          for (i = 0, len = propertyValuePairs.length; i < len; i++) {\n            var pair = propertyValuePairs[i].split(':'),\n                property = pair[0].trim(),\n                value = pair[1].trim();\n            ruleObj[property] = value;\n          }\n          rule = match[0].trim();\n          rule.split(',').forEach(function(_rule) {\n            _rule = _rule.replace(/^svg/i, '').trim();\n            if (_rule === '') {\n              return;\n            }\n            if (allRules[_rule]) {\n              fabric.util.object.extend(allRules[_rule], ruleObj);\n            }\n            else {\n              allRules[_rule] = fabric.util.object.clone(ruleObj);\n            }\n          });\n        });\n      }\n      return allRules;\n    },\n\n    /**\n     * Takes url corresponding to an SVG document, and parses it into a set of fabric objects.\n     * Note that SVG is fetched via XMLHttpRequest, so it needs to conform to SOP (Same Origin Policy)\n     * @memberOf fabric\n     * @param {String} url\n     * @param {Function} callback\n     * @param {Function} [reviver] Method for further parsing of SVG elements, called after each fabric object created.\n     * @param {Object} [options] Object containing options for parsing\n     * @param {String} [options.crossOrigin] crossOrigin crossOrigin setting to use for external resources\n     */\n    loadSVGFromURL: function(url, callback, reviver, options) {\n\n      url = url.replace(/^\\n\\s*/, '').trim();\n      new fabric.util.request(url, {\n        method: 'get',\n        onComplete: onComplete\n      });\n\n      function onComplete(r) {\n\n        var xml = r.responseXML;\n        if (!xml || !xml.documentElement) {\n          callback && callback(null);\n          return false;\n        }\n\n        fabric.parseSVGDocument(xml.documentElement, function (results, _options, elements, allElements) {\n          callback && callback(results, _options, elements, allElements);\n        }, reviver, options);\n      }\n    },\n\n    /**\n     * Takes string corresponding to an SVG document, and parses it into a set of fabric objects\n     * @memberOf fabric\n     * @param {String} string\n     * @param {Function} callback\n     * @param {Function} [reviver] Method for further parsing of SVG elements, called after each fabric object created.\n     * @param {Object} [options] Object containing options for parsing\n     * @param {String} [options.crossOrigin] crossOrigin crossOrigin setting to use for external resources\n     */\n    loadSVGFromString: function(string, callback, reviver, options) {\n      var parser = new fabric.window.DOMParser(),\n          doc = parser.parseFromString(string.trim(), 'text/xml');\n      fabric.parseSVGDocument(doc.documentElement, function (results, _options, elements, allElements) {\n        callback(results, _options, elements, allElements);\n      }, reviver, options);\n    }\n  });\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\nfabric.ElementsParser = function(elements, callback, options, reviver, parsingOptions, doc) {\n  this.elements = elements;\n  this.callback = callback;\n  this.options = options;\n  this.reviver = reviver;\n  this.svgUid = (options && options.svgUid) || 0;\n  this.parsingOptions = parsingOptions;\n  this.regexUrl = /^url\\(['\"]?#([^'\"]+)['\"]?\\)/g;\n  this.doc = doc;\n};\n\n(function(proto) {\n  proto.parse = function() {\n    this.instances = new Array(this.elements.length);\n    this.numElements = this.elements.length;\n    this.createObjects();\n  };\n\n  proto.createObjects = function() {\n    var _this = this;\n    this.elements.forEach(function(element, i) {\n      element.setAttribute('svgUid', _this.svgUid);\n      _this.createObject(element, i);\n    });\n  };\n\n  proto.findTag = function(el) {\n    return fabric[fabric.util.string.capitalize(el.tagName.replace('svg:', ''))];\n  };\n\n  proto.createObject = function(el, index) {\n    var klass = this.findTag(el);\n    if (klass && klass.fromElement) {\n      try {\n        klass.fromElement(el, this.createCallback(index, el), this.options);\n      }\n      catch (err) {\n        fabric.log(err);\n      }\n    }\n    else {\n      this.checkIfDone();\n    }\n  };\n\n  proto.createCallback = function(index, el) {\n    var _this = this;\n    return function(obj) {\n      var _options;\n      _this.resolveGradient(obj, el, 'fill');\n      _this.resolveGradient(obj, el, 'stroke');\n      if (obj instanceof fabric.Image && obj._originalElement) {\n        _options = obj.parsePreserveAspectRatioAttribute(el);\n      }\n      obj._removeTransformMatrix(_options);\n      _this.resolveClipPath(obj, el);\n      _this.reviver && _this.reviver(el, obj);\n      _this.instances[index] = obj;\n      _this.checkIfDone();\n    };\n  };\n\n  proto.extractPropertyDefinition = function(obj, property, storage) {\n    var value = obj[property], regex = this.regexUrl;\n    if (!regex.test(value)) {\n      return;\n    }\n    regex.lastIndex = 0;\n    var id = regex.exec(value)[1];\n    regex.lastIndex = 0;\n    return fabric[storage][this.svgUid][id];\n  };\n\n  proto.resolveGradient = function(obj, el, property) {\n    var gradientDef = this.extractPropertyDefinition(obj, property, 'gradientDefs');\n    if (gradientDef) {\n      var opacityAttr = el.getAttribute(property + '-opacity');\n      var gradient = fabric.Gradient.fromElement(gradientDef, obj, opacityAttr, this.options);\n      obj.set(property, gradient);\n    }\n  };\n\n  proto.createClipPathCallback = function(obj, container) {\n    return function(_newObj) {\n      _newObj._removeTransformMatrix();\n      _newObj.fillRule = _newObj.clipRule;\n      container.push(_newObj);\n    };\n  };\n\n  proto.resolveClipPath = function(obj, usingElement) {\n    var clipPath = this.extractPropertyDefinition(obj, 'clipPath', 'clipPaths'),\n        element, klass, objTransformInv, container, gTransform, options;\n    if (clipPath) {\n      container = [];\n      objTransformInv = fabric.util.invertTransform(obj.calcTransformMatrix());\n      // move the clipPath tag as sibling to the real element that is using it\n      var clipPathTag = clipPath[0].parentNode;\n      var clipPathOwner = usingElement;\n      while (clipPathOwner.parentNode && clipPathOwner.getAttribute('clip-path') !== obj.clipPath) {\n        clipPathOwner = clipPathOwner.parentNode;\n      }\n      clipPathOwner.parentNode.appendChild(clipPathTag);\n      for (var i = 0; i < clipPath.length; i++) {\n        element = clipPath[i];\n        klass = this.findTag(element);\n        klass.fromElement(\n          element,\n          this.createClipPathCallback(obj, container),\n          this.options\n        );\n      }\n      if (container.length === 1) {\n        clipPath = container[0];\n      }\n      else {\n        clipPath = new fabric.Group(container);\n      }\n      gTransform = fabric.util.multiplyTransformMatrices(\n        objTransformInv,\n        clipPath.calcTransformMatrix()\n      );\n      if (clipPath.clipPath) {\n        this.resolveClipPath(clipPath, clipPathOwner);\n      }\n      var options = fabric.util.qrDecompose(gTransform);\n      clipPath.flipX = false;\n      clipPath.flipY = false;\n      clipPath.set('scaleX', options.scaleX);\n      clipPath.set('scaleY', options.scaleY);\n      clipPath.angle = options.angle;\n      clipPath.skewX = options.skewX;\n      clipPath.skewY = 0;\n      clipPath.setPositionByOrigin({ x: options.translateX, y: options.translateY }, 'center', 'center');\n      obj.clipPath = clipPath;\n    }\n    else {\n      // if clip-path does not resolve to any element, delete the property.\n      delete obj.clipPath;\n    }\n  };\n\n  proto.checkIfDone = function() {\n    if (--this.numElements === 0) {\n      this.instances = this.instances.filter(function(el) {\n        // eslint-disable-next-line no-eq-null, eqeqeq\n        return el != null;\n      });\n      this.callback(this.instances, this.elements);\n    }\n  };\n})(fabric.ElementsParser.prototype);\n\n\n(function(global) {\n\n  'use strict';\n\n  /* Adaptation of work of Kevin Lindsey (kevin@kevlindev.com) */\n\n  var fabric = global.fabric || (global.fabric = { });\n\n  if (fabric.Point) {\n    fabric.warn('fabric.Point is already defined');\n    return;\n  }\n\n  fabric.Point = Point;\n\n  /**\n   * Point class\n   * @class fabric.Point\n   * @memberOf fabric\n   * @constructor\n   * @param {Number} x\n   * @param {Number} y\n   * @return {fabric.Point} thisArg\n   */\n  function Point(x, y) {\n    this.x = x;\n    this.y = y;\n  }\n\n  Point.prototype = /** @lends fabric.Point.prototype */ {\n\n    type: 'point',\n\n    constructor: Point,\n\n    /**\n     * Adds another point to this one and returns another one\n     * @param {fabric.Point} that\n     * @return {fabric.Point} new Point instance with added values\n     */\n    add: function (that) {\n      return new Point(this.x + that.x, this.y + that.y);\n    },\n\n    /**\n     * Adds another point to this one\n     * @param {fabric.Point} that\n     * @return {fabric.Point} thisArg\n     * @chainable\n     */\n    addEquals: function (that) {\n      this.x += that.x;\n      this.y += that.y;\n      return this;\n    },\n\n    /**\n     * Adds value to this point and returns a new one\n     * @param {Number} scalar\n     * @return {fabric.Point} new Point with added value\n     */\n    scalarAdd: function (scalar) {\n      return new Point(this.x + scalar, this.y + scalar);\n    },\n\n    /**\n     * Adds value to this point\n     * @param {Number} scalar\n     * @return {fabric.Point} thisArg\n     * @chainable\n     */\n    scalarAddEquals: function (scalar) {\n      this.x += scalar;\n      this.y += scalar;\n      return this;\n    },\n\n    /**\n     * Subtracts another point from this point and returns a new one\n     * @param {fabric.Point} that\n     * @return {fabric.Point} new Point object with subtracted values\n     */\n    subtract: function (that) {\n      return new Point(this.x - that.x, this.y - that.y);\n    },\n\n    /**\n     * Subtracts another point from this point\n     * @param {fabric.Point} that\n     * @return {fabric.Point} thisArg\n     * @chainable\n     */\n    subtractEquals: function (that) {\n      this.x -= that.x;\n      this.y -= that.y;\n      return this;\n    },\n\n    /**\n     * Subtracts value from this point and returns a new one\n     * @param {Number} scalar\n     * @return {fabric.Point}\n     */\n    scalarSubtract: function (scalar) {\n      return new Point(this.x - scalar, this.y - scalar);\n    },\n\n    /**\n     * Subtracts value from this point\n     * @param {Number} scalar\n     * @return {fabric.Point} thisArg\n     * @chainable\n     */\n    scalarSubtractEquals: function (scalar) {\n      this.x -= scalar;\n      this.y -= scalar;\n      return this;\n    },\n\n    /**\n     * Multiplies this point by a value and returns a new one\n     * TODO: rename in scalarMultiply in 2.0\n     * @param {Number} scalar\n     * @return {fabric.Point}\n     */\n    multiply: function (scalar) {\n      return new Point(this.x * scalar, this.y * scalar);\n    },\n\n    /**\n     * Multiplies this point by a value\n     * TODO: rename in scalarMultiplyEquals in 2.0\n     * @param {Number} scalar\n     * @return {fabric.Point} thisArg\n     * @chainable\n     */\n    multiplyEquals: function (scalar) {\n      this.x *= scalar;\n      this.y *= scalar;\n      return this;\n    },\n\n    /**\n     * Divides this point by a value and returns a new one\n     * TODO: rename in scalarDivide in 2.0\n     * @param {Number} scalar\n     * @return {fabric.Point}\n     */\n    divide: function (scalar) {\n      return new Point(this.x / scalar, this.y / scalar);\n    },\n\n    /**\n     * Divides this point by a value\n     * TODO: rename in scalarDivideEquals in 2.0\n     * @param {Number} scalar\n     * @return {fabric.Point} thisArg\n     * @chainable\n     */\n    divideEquals: function (scalar) {\n      this.x /= scalar;\n      this.y /= scalar;\n      return this;\n    },\n\n    /**\n     * Returns true if this point is equal to another one\n     * @param {fabric.Point} that\n     * @return {Boolean}\n     */\n    eq: function (that) {\n      return (this.x === that.x && this.y === that.y);\n    },\n\n    /**\n     * Returns true if this point is less than another one\n     * @param {fabric.Point} that\n     * @return {Boolean}\n     */\n    lt: function (that) {\n      return (this.x < that.x && this.y < that.y);\n    },\n\n    /**\n     * Returns true if this point is less than or equal to another one\n     * @param {fabric.Point} that\n     * @return {Boolean}\n     */\n    lte: function (that) {\n      return (this.x <= that.x && this.y <= that.y);\n    },\n\n    /**\n\n     * Returns true if this point is greater another one\n     * @param {fabric.Point} that\n     * @return {Boolean}\n     */\n    gt: function (that) {\n      return (this.x > that.x && this.y > that.y);\n    },\n\n    /**\n     * Returns true if this point is greater than or equal to another one\n     * @param {fabric.Point} that\n     * @return {Boolean}\n     */\n    gte: function (that) {\n      return (this.x >= that.x && this.y >= that.y);\n    },\n\n    /**\n     * Returns new point which is the result of linear interpolation with this one and another one\n     * @param {fabric.Point} that\n     * @param {Number} t , position of interpolation, between 0 and 1 default 0.5\n     * @return {fabric.Point}\n     */\n    lerp: function (that, t) {\n      if (typeof t === 'undefined') {\n        t = 0.5;\n      }\n      t = Math.max(Math.min(1, t), 0);\n      return new Point(this.x + (that.x - this.x) * t, this.y + (that.y - this.y) * t);\n    },\n\n    /**\n     * Returns distance from this point and another one\n     * @param {fabric.Point} that\n     * @return {Number}\n     */\n    distanceFrom: function (that) {\n      var dx = this.x - that.x,\n          dy = this.y - that.y;\n      return Math.sqrt(dx * dx + dy * dy);\n    },\n\n    /**\n     * Returns the point between this point and another one\n     * @param {fabric.Point} that\n     * @return {fabric.Point}\n     */\n    midPointFrom: function (that) {\n      return this.lerp(that);\n    },\n\n    /**\n     * Returns a new point which is the min of this and another one\n     * @param {fabric.Point} that\n     * @return {fabric.Point}\n     */\n    min: function (that) {\n      return new Point(Math.min(this.x, that.x), Math.min(this.y, that.y));\n    },\n\n    /**\n     * Returns a new point which is the max of this and another one\n     * @param {fabric.Point} that\n     * @return {fabric.Point}\n     */\n    max: function (that) {\n      return new Point(Math.max(this.x, that.x), Math.max(this.y, that.y));\n    },\n\n    /**\n     * Returns string representation of this point\n     * @return {String}\n     */\n    toString: function () {\n      return this.x + ',' + this.y;\n    },\n\n    /**\n     * Sets x/y of this point\n     * @param {Number} x\n     * @param {Number} y\n     * @chainable\n     */\n    setXY: function (x, y) {\n      this.x = x;\n      this.y = y;\n      return this;\n    },\n\n    /**\n     * Sets x of this point\n     * @param {Number} x\n     * @chainable\n     */\n    setX: function (x) {\n      this.x = x;\n      return this;\n    },\n\n    /**\n     * Sets y of this point\n     * @param {Number} y\n     * @chainable\n     */\n    setY: function (y) {\n      this.y = y;\n      return this;\n    },\n\n    /**\n     * Sets x/y of this point from another point\n     * @param {fabric.Point} that\n     * @chainable\n     */\n    setFromPoint: function (that) {\n      this.x = that.x;\n      this.y = that.y;\n      return this;\n    },\n\n    /**\n     * Swaps x/y of this point and another point\n     * @param {fabric.Point} that\n     */\n    swap: function (that) {\n      var x = this.x,\n          y = this.y;\n      this.x = that.x;\n      this.y = that.y;\n      that.x = x;\n      that.y = y;\n    },\n\n    /**\n     * return a cloned instance of the point\n     * @return {fabric.Point}\n     */\n    clone: function () {\n      return new Point(this.x, this.y);\n    }\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  /* Adaptation of work of Kevin Lindsey (kevin@kevlindev.com) */\n  var fabric = global.fabric || (global.fabric = { });\n\n  if (fabric.Intersection) {\n    fabric.warn('fabric.Intersection is already defined');\n    return;\n  }\n\n  /**\n   * Intersection class\n   * @class fabric.Intersection\n   * @memberOf fabric\n   * @constructor\n   */\n  function Intersection(status) {\n    this.status = status;\n    this.points = [];\n  }\n\n  fabric.Intersection = Intersection;\n\n  fabric.Intersection.prototype = /** @lends fabric.Intersection.prototype */ {\n\n    constructor: Intersection,\n\n    /**\n     * Appends a point to intersection\n     * @param {fabric.Point} point\n     * @return {fabric.Intersection} thisArg\n     * @chainable\n     */\n    appendPoint: function (point) {\n      this.points.push(point);\n      return this;\n    },\n\n    /**\n     * Appends points to intersection\n     * @param {Array} points\n     * @return {fabric.Intersection} thisArg\n     * @chainable\n     */\n    appendPoints: function (points) {\n      this.points = this.points.concat(points);\n      return this;\n    }\n  };\n\n  /**\n   * Checks if one line intersects another\n   * TODO: rename in intersectSegmentSegment\n   * @static\n   * @param {fabric.Point} a1\n   * @param {fabric.Point} a2\n   * @param {fabric.Point} b1\n   * @param {fabric.Point} b2\n   * @return {fabric.Intersection}\n   */\n  fabric.Intersection.intersectLineLine = function (a1, a2, b1, b2) {\n    var result,\n        uaT = (b2.x - b1.x) * (a1.y - b1.y) - (b2.y - b1.y) * (a1.x - b1.x),\n        ubT = (a2.x - a1.x) * (a1.y - b1.y) - (a2.y - a1.y) * (a1.x - b1.x),\n        uB = (b2.y - b1.y) * (a2.x - a1.x) - (b2.x - b1.x) * (a2.y - a1.y);\n    if (uB !== 0) {\n      var ua = uaT / uB,\n          ub = ubT / uB;\n      if (0 <= ua && ua <= 1 && 0 <= ub && ub <= 1) {\n        result = new Intersection('Intersection');\n        result.appendPoint(new fabric.Point(a1.x + ua * (a2.x - a1.x), a1.y + ua * (a2.y - a1.y)));\n      }\n      else {\n        result = new Intersection();\n      }\n    }\n    else {\n      if (uaT === 0 || ubT === 0) {\n        result = new Intersection('Coincident');\n      }\n      else {\n        result = new Intersection('Parallel');\n      }\n    }\n    return result;\n  };\n\n  /**\n   * Checks if line intersects polygon\n   * TODO: rename in intersectSegmentPolygon\n   * fix detection of coincident\n   * @static\n   * @param {fabric.Point} a1\n   * @param {fabric.Point} a2\n   * @param {Array} points\n   * @return {fabric.Intersection}\n   */\n  fabric.Intersection.intersectLinePolygon = function(a1, a2, points) {\n    var result = new Intersection(),\n        length = points.length,\n        b1, b2, inter, i;\n\n    for (i = 0; i < length; i++) {\n      b1 = points[i];\n      b2 = points[(i + 1) % length];\n      inter = Intersection.intersectLineLine(a1, a2, b1, b2);\n\n      result.appendPoints(inter.points);\n    }\n    if (result.points.length > 0) {\n      result.status = 'Intersection';\n    }\n    return result;\n  };\n\n  /**\n   * Checks if polygon intersects another polygon\n   * @static\n   * @param {Array} points1\n   * @param {Array} points2\n   * @return {fabric.Intersection}\n   */\n  fabric.Intersection.intersectPolygonPolygon = function (points1, points2) {\n    var result = new Intersection(),\n        length = points1.length, i;\n\n    for (i = 0; i < length; i++) {\n      var a1 = points1[i],\n          a2 = points1[(i + 1) % length],\n          inter = Intersection.intersectLinePolygon(a1, a2, points2);\n\n      result.appendPoints(inter.points);\n    }\n    if (result.points.length > 0) {\n      result.status = 'Intersection';\n    }\n    return result;\n  };\n\n  /**\n   * Checks if polygon intersects rectangle\n   * @static\n   * @param {Array} points\n   * @param {fabric.Point} r1\n   * @param {fabric.Point} r2\n   * @return {fabric.Intersection}\n   */\n  fabric.Intersection.intersectPolygonRectangle = function (points, r1, r2) {\n    var min = r1.min(r2),\n        max = r1.max(r2),\n        topRight = new fabric.Point(max.x, min.y),\n        bottomLeft = new fabric.Point(min.x, max.y),\n        inter1 = Intersection.intersectLinePolygon(min, topRight, points),\n        inter2 = Intersection.intersectLinePolygon(topRight, max, points),\n        inter3 = Intersection.intersectLinePolygon(max, bottomLeft, points),\n        inter4 = Intersection.intersectLinePolygon(bottomLeft, min, points),\n        result = new Intersection();\n\n    result.appendPoints(inter1.points);\n    result.appendPoints(inter2.points);\n    result.appendPoints(inter3.points);\n    result.appendPoints(inter4.points);\n\n    if (result.points.length > 0) {\n      result.status = 'Intersection';\n    }\n    return result;\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { });\n\n  if (fabric.Color) {\n    fabric.warn('fabric.Color is already defined.');\n    return;\n  }\n\n  /**\n   * Color class\n   * The purpose of {@link fabric.Color} is to abstract and encapsulate common color operations;\n   * {@link fabric.Color} is a constructor and creates instances of {@link fabric.Color} objects.\n   *\n   * @class fabric.Color\n   * @param {String} color optional in hex or rgb(a) or hsl format or from known color list\n   * @return {fabric.Color} thisArg\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-2/#colors}\n   */\n  function Color(color) {\n    if (!color) {\n      this.setSource([0, 0, 0, 1]);\n    }\n    else {\n      this._tryParsingColor(color);\n    }\n  }\n\n  fabric.Color = Color;\n\n  fabric.Color.prototype = /** @lends fabric.Color.prototype */ {\n\n    /**\n     * @private\n     * @param {String|Array} color Color value to parse\n     */\n    _tryParsingColor: function(color) {\n      var source;\n\n      if (color in Color.colorNameMap) {\n        color = Color.colorNameMap[color];\n      }\n\n      if (color === 'transparent') {\n        source = [255, 255, 255, 0];\n      }\n\n      if (!source) {\n        source = Color.sourceFromHex(color);\n      }\n      if (!source) {\n        source = Color.sourceFromRgb(color);\n      }\n      if (!source) {\n        source = Color.sourceFromHsl(color);\n      }\n      if (!source) {\n        //if color is not recognize let's make black as canvas does\n        source = [0, 0, 0, 1];\n      }\n      if (source) {\n        this.setSource(source);\n      }\n    },\n\n    /**\n     * Adapted from <a href=\"https://rawgithub.com/mjijackson/mjijackson.github.com/master/2008/02/rgb-to-hsl-and-rgb-to-hsv-color-model-conversion-algorithms-in-javascript.html\">https://github.com/mjijackson</a>\n     * @private\n     * @param {Number} r Red color value\n     * @param {Number} g Green color value\n     * @param {Number} b Blue color value\n     * @return {Array} Hsl color\n     */\n    _rgbToHsl: function(r, g, b) {\n      r /= 255; g /= 255; b /= 255;\n\n      var h, s, l,\n          max = fabric.util.array.max([r, g, b]),\n          min = fabric.util.array.min([r, g, b]);\n\n      l = (max + min) / 2;\n\n      if (max === min) {\n        h = s = 0; // achromatic\n      }\n      else {\n        var d = max - min;\n        s = l > 0.5 ? d / (2 - max - min) : d / (max + min);\n        switch (max) {\n          case r:\n            h = (g - b) / d + (g < b ? 6 : 0);\n            break;\n          case g:\n            h = (b - r) / d + 2;\n            break;\n          case b:\n            h = (r - g) / d + 4;\n            break;\n        }\n        h /= 6;\n      }\n\n      return [\n        Math.round(h * 360),\n        Math.round(s * 100),\n        Math.round(l * 100)\n      ];\n    },\n\n    /**\n     * Returns source of this color (where source is an array representation; ex: [200, 200, 100, 1])\n     * @return {Array}\n     */\n    getSource: function() {\n      return this._source;\n    },\n\n    /**\n     * Sets source of this color (where source is an array representation; ex: [200, 200, 100, 1])\n     * @param {Array} source\n     */\n    setSource: function(source) {\n      this._source = source;\n    },\n\n    /**\n     * Returns color representation in RGB format\n     * @return {String} ex: rgb(0-255,0-255,0-255)\n     */\n    toRgb: function() {\n      var source = this.getSource();\n      return 'rgb(' + source[0] + ',' + source[1] + ',' + source[2] + ')';\n    },\n\n    /**\n     * Returns color representation in RGBA format\n     * @return {String} ex: rgba(0-255,0-255,0-255,0-1)\n     */\n    toRgba: function() {\n      var source = this.getSource();\n      return 'rgba(' + source[0] + ',' + source[1] + ',' + source[2] + ',' + source[3] + ')';\n    },\n\n    /**\n     * Returns color representation in HSL format\n     * @return {String} ex: hsl(0-360,0%-100%,0%-100%)\n     */\n    toHsl: function() {\n      var source = this.getSource(),\n          hsl = this._rgbToHsl(source[0], source[1], source[2]);\n\n      return 'hsl(' + hsl[0] + ',' + hsl[1] + '%,' + hsl[2] + '%)';\n    },\n\n    /**\n     * Returns color representation in HSLA format\n     * @return {String} ex: hsla(0-360,0%-100%,0%-100%,0-1)\n     */\n    toHsla: function() {\n      var source = this.getSource(),\n          hsl = this._rgbToHsl(source[0], source[1], source[2]);\n\n      return 'hsla(' + hsl[0] + ',' + hsl[1] + '%,' + hsl[2] + '%,' + source[3] + ')';\n    },\n\n    /**\n     * Returns color representation in HEX format\n     * @return {String} ex: FF5555\n     */\n    toHex: function() {\n      var source = this.getSource(), r, g, b;\n\n      r = source[0].toString(16);\n      r = (r.length === 1) ? ('0' + r) : r;\n\n      g = source[1].toString(16);\n      g = (g.length === 1) ? ('0' + g) : g;\n\n      b = source[2].toString(16);\n      b = (b.length === 1) ? ('0' + b) : b;\n\n      return r.toUpperCase() + g.toUpperCase() + b.toUpperCase();\n    },\n\n    /**\n     * Returns color representation in HEXA format\n     * @return {String} ex: FF5555CC\n     */\n    toHexa: function() {\n      var source = this.getSource(), a;\n\n      a = Math.round(source[3] * 255);\n      a = a.toString(16);\n      a = (a.length === 1) ? ('0' + a) : a;\n\n      return this.toHex() + a.toUpperCase();\n    },\n\n    /**\n     * Gets value of alpha channel for this color\n     * @return {Number} 0-1\n     */\n    getAlpha: function() {\n      return this.getSource()[3];\n    },\n\n    /**\n     * Sets value of alpha channel for this color\n     * @param {Number} alpha Alpha value 0-1\n     * @return {fabric.Color} thisArg\n     */\n    setAlpha: function(alpha) {\n      var source = this.getSource();\n      source[3] = alpha;\n      this.setSource(source);\n      return this;\n    },\n\n    /**\n     * Transforms color to its grayscale representation\n     * @return {fabric.Color} thisArg\n     */\n    toGrayscale: function() {\n      var source = this.getSource(),\n          average = parseInt((source[0] * 0.3 + source[1] * 0.59 + source[2] * 0.11).toFixed(0), 10),\n          currentAlpha = source[3];\n      this.setSource([average, average, average, currentAlpha]);\n      return this;\n    },\n\n    /**\n     * Transforms color to its black and white representation\n     * @param {Number} threshold\n     * @return {fabric.Color} thisArg\n     */\n    toBlackWhite: function(threshold) {\n      var source = this.getSource(),\n          average = (source[0] * 0.3 + source[1] * 0.59 + source[2] * 0.11).toFixed(0),\n          currentAlpha = source[3];\n\n      threshold = threshold || 127;\n\n      average = (Number(average) < Number(threshold)) ? 0 : 255;\n      this.setSource([average, average, average, currentAlpha]);\n      return this;\n    },\n\n    /**\n     * Overlays color with another color\n     * @param {String|fabric.Color} otherColor\n     * @return {fabric.Color} thisArg\n     */\n    overlayWith: function(otherColor) {\n      if (!(otherColor instanceof Color)) {\n        otherColor = new Color(otherColor);\n      }\n\n      var result = [],\n          alpha = this.getAlpha(),\n          otherAlpha = 0.5,\n          source = this.getSource(),\n          otherSource = otherColor.getSource(), i;\n\n      for (i = 0; i < 3; i++) {\n        result.push(Math.round((source[i] * (1 - otherAlpha)) + (otherSource[i] * otherAlpha)));\n      }\n\n      result[3] = alpha;\n      this.setSource(result);\n      return this;\n    }\n  };\n\n  /**\n   * Regex matching color in RGB or RGBA formats (ex: rgb(0, 0, 0), rgba(255, 100, 10, 0.5), rgba( 255 , 100 , 10 , 0.5 ), rgb(1,1,1), rgba(100%, 60%, 10%, 0.5))\n   * @static\n   * @field\n   * @memberOf fabric.Color\n   */\n  // eslint-disable-next-line max-len\n  fabric.Color.reRGBa = /^rgba?\\(\\s*(\\d{1,3}(?:\\.\\d+)?\\%?)\\s*,\\s*(\\d{1,3}(?:\\.\\d+)?\\%?)\\s*,\\s*(\\d{1,3}(?:\\.\\d+)?\\%?)\\s*(?:\\s*,\\s*((?:\\d*\\.?\\d+)?)\\s*)?\\)$/i;\n\n  /**\n   * Regex matching color in HSL or HSLA formats (ex: hsl(200, 80%, 10%), hsla(300, 50%, 80%, 0.5), hsla( 300 , 50% , 80% , 0.5 ))\n   * @static\n   * @field\n   * @memberOf fabric.Color\n   */\n  fabric.Color.reHSLa = /^hsla?\\(\\s*(\\d{1,3})\\s*,\\s*(\\d{1,3}\\%)\\s*,\\s*(\\d{1,3}\\%)\\s*(?:\\s*,\\s*(\\d+(?:\\.\\d+)?)\\s*)?\\)$/i;\n\n  /**\n   * Regex matching color in HEX format (ex: #FF5544CC, #FF5555, 010155, aff)\n   * @static\n   * @field\n   * @memberOf fabric.Color\n   */\n  fabric.Color.reHex = /^#?([0-9a-f]{8}|[0-9a-f]{6}|[0-9a-f]{4}|[0-9a-f]{3})$/i;\n\n  /**\n   * Map of the 148 color names with HEX code\n   * @static\n   * @field\n   * @memberOf fabric.Color\n   * @see: https://www.w3.org/TR/css3-color/#svg-color\n   */\n  fabric.Color.colorNameMap = {\n    aliceblue:            '#F0F8FF',\n    antiquewhite:         '#FAEBD7',\n    aqua:                 '#00FFFF',\n    aquamarine:           '#7FFFD4',\n    azure:                '#F0FFFF',\n    beige:                '#F5F5DC',\n    bisque:               '#FFE4C4',\n    black:                '#000000',\n    blanchedalmond:       '#FFEBCD',\n    blue:                 '#0000FF',\n    blueviolet:           '#8A2BE2',\n    brown:                '#A52A2A',\n    burlywood:            '#DEB887',\n    cadetblue:            '#5F9EA0',\n    chartreuse:           '#7FFF00',\n    chocolate:            '#D2691E',\n    coral:                '#FF7F50',\n    cornflowerblue:       '#6495ED',\n    cornsilk:             '#FFF8DC',\n    crimson:              '#DC143C',\n    cyan:                 '#00FFFF',\n    darkblue:             '#00008B',\n    darkcyan:             '#008B8B',\n    darkgoldenrod:        '#B8860B',\n    darkgray:             '#A9A9A9',\n    darkgrey:             '#A9A9A9',\n    darkgreen:            '#006400',\n    darkkhaki:            '#BDB76B',\n    darkmagenta:          '#8B008B',\n    darkolivegreen:       '#556B2F',\n    darkorange:           '#FF8C00',\n    darkorchid:           '#9932CC',\n    darkred:              '#8B0000',\n    darksalmon:           '#E9967A',\n    darkseagreen:         '#8FBC8F',\n    darkslateblue:        '#483D8B',\n    darkslategray:        '#2F4F4F',\n    darkslategrey:        '#2F4F4F',\n    darkturquoise:        '#00CED1',\n    darkviolet:           '#9400D3',\n    deeppink:             '#FF1493',\n    deepskyblue:          '#00BFFF',\n    dimgray:              '#696969',\n    dimgrey:              '#696969',\n    dodgerblue:           '#1E90FF',\n    firebrick:            '#B22222',\n    floralwhite:          '#FFFAF0',\n    forestgreen:          '#228B22',\n    fuchsia:              '#FF00FF',\n    gainsboro:            '#DCDCDC',\n    ghostwhite:           '#F8F8FF',\n    gold:                 '#FFD700',\n    goldenrod:            '#DAA520',\n    gray:                 '#808080',\n    grey:                 '#808080',\n    green:                '#008000',\n    greenyellow:          '#ADFF2F',\n    honeydew:             '#F0FFF0',\n    hotpink:              '#FF69B4',\n    indianred:            '#CD5C5C',\n    indigo:               '#4B0082',\n    ivory:                '#FFFFF0',\n    khaki:                '#F0E68C',\n    lavender:             '#E6E6FA',\n    lavenderblush:        '#FFF0F5',\n    lawngreen:            '#7CFC00',\n    lemonchiffon:         '#FFFACD',\n    lightblue:            '#ADD8E6',\n    lightcoral:           '#F08080',\n    lightcyan:            '#E0FFFF',\n    lightgoldenrodyellow: '#FAFAD2',\n    lightgray:            '#D3D3D3',\n    lightgrey:            '#D3D3D3',\n    lightgreen:           '#90EE90',\n    lightpink:            '#FFB6C1',\n    lightsalmon:          '#FFA07A',\n    lightseagreen:        '#20B2AA',\n    lightskyblue:         '#87CEFA',\n    lightslategray:       '#778899',\n    lightslategrey:       '#778899',\n    lightsteelblue:       '#B0C4DE',\n    lightyellow:          '#FFFFE0',\n    lime:                 '#00FF00',\n    limegreen:            '#32CD32',\n    linen:                '#FAF0E6',\n    magenta:              '#FF00FF',\n    maroon:               '#800000',\n    mediumaquamarine:     '#66CDAA',\n    mediumblue:           '#0000CD',\n    mediumorchid:         '#BA55D3',\n    mediumpurple:         '#9370DB',\n    mediumseagreen:       '#3CB371',\n    mediumslateblue:      '#7B68EE',\n    mediumspringgreen:    '#00FA9A',\n    mediumturquoise:      '#48D1CC',\n    mediumvioletred:      '#C71585',\n    midnightblue:         '#191970',\n    mintcream:            '#F5FFFA',\n    mistyrose:            '#FFE4E1',\n    moccasin:             '#FFE4B5',\n    navajowhite:          '#FFDEAD',\n    navy:                 '#000080',\n    oldlace:              '#FDF5E6',\n    olive:                '#808000',\n    olivedrab:            '#6B8E23',\n    orange:               '#FFA500',\n    orangered:            '#FF4500',\n    orchid:               '#DA70D6',\n    palegoldenrod:        '#EEE8AA',\n    palegreen:            '#98FB98',\n    paleturquoise:        '#AFEEEE',\n    palevioletred:        '#DB7093',\n    papayawhip:           '#FFEFD5',\n    peachpuff:            '#FFDAB9',\n    peru:                 '#CD853F',\n    pink:                 '#FFC0CB',\n    plum:                 '#DDA0DD',\n    powderblue:           '#B0E0E6',\n    purple:               '#800080',\n    rebeccapurple:        '#663399',\n    red:                  '#FF0000',\n    rosybrown:            '#BC8F8F',\n    royalblue:            '#4169E1',\n    saddlebrown:          '#8B4513',\n    salmon:               '#FA8072',\n    sandybrown:           '#F4A460',\n    seagreen:             '#2E8B57',\n    seashell:             '#FFF5EE',\n    sienna:               '#A0522D',\n    silver:               '#C0C0C0',\n    skyblue:              '#87CEEB',\n    slateblue:            '#6A5ACD',\n    slategray:            '#708090',\n    slategrey:            '#708090',\n    snow:                 '#FFFAFA',\n    springgreen:          '#00FF7F',\n    steelblue:            '#4682B4',\n    tan:                  '#D2B48C',\n    teal:                 '#008080',\n    thistle:              '#D8BFD8',\n    tomato:               '#FF6347',\n    turquoise:            '#40E0D0',\n    violet:               '#EE82EE',\n    wheat:                '#F5DEB3',\n    white:                '#FFFFFF',\n    whitesmoke:           '#F5F5F5',\n    yellow:               '#FFFF00',\n    yellowgreen:          '#9ACD32'\n  };\n\n  /**\n   * @private\n   * @param {Number} p\n   * @param {Number} q\n   * @param {Number} t\n   * @return {Number}\n   */\n  function hue2rgb(p, q, t) {\n    if (t < 0) {\n      t += 1;\n    }\n    if (t > 1) {\n      t -= 1;\n    }\n    if (t < 1 / 6) {\n      return p + (q - p) * 6 * t;\n    }\n    if (t < 1 / 2) {\n      return q;\n    }\n    if (t < 2 / 3) {\n      return p + (q - p) * (2 / 3 - t) * 6;\n    }\n    return p;\n  }\n\n  /**\n   * Returns new color object, when given a color in RGB format\n   * @memberOf fabric.Color\n   * @param {String} color Color value ex: rgb(0-255,0-255,0-255)\n   * @return {fabric.Color}\n   */\n  fabric.Color.fromRgb = function(color) {\n    return Color.fromSource(Color.sourceFromRgb(color));\n  };\n\n  /**\n   * Returns array representation (ex: [100, 100, 200, 1]) of a color that's in RGB or RGBA format\n   * @memberOf fabric.Color\n   * @param {String} color Color value ex: rgb(0-255,0-255,0-255), rgb(0%-100%,0%-100%,0%-100%)\n   * @return {Array} source\n   */\n  fabric.Color.sourceFromRgb = function(color) {\n    var match = color.match(Color.reRGBa);\n    if (match) {\n      var r = parseInt(match[1], 10) / (/%$/.test(match[1]) ? 100 : 1) * (/%$/.test(match[1]) ? 255 : 1),\n          g = parseInt(match[2], 10) / (/%$/.test(match[2]) ? 100 : 1) * (/%$/.test(match[2]) ? 255 : 1),\n          b = parseInt(match[3], 10) / (/%$/.test(match[3]) ? 100 : 1) * (/%$/.test(match[3]) ? 255 : 1);\n\n      return [\n        parseInt(r, 10),\n        parseInt(g, 10),\n        parseInt(b, 10),\n        match[4] ? parseFloat(match[4]) : 1\n      ];\n    }\n  };\n\n  /**\n   * Returns new color object, when given a color in RGBA format\n   * @static\n   * @function\n   * @memberOf fabric.Color\n   * @param {String} color\n   * @return {fabric.Color}\n   */\n  fabric.Color.fromRgba = Color.fromRgb;\n\n  /**\n   * Returns new color object, when given a color in HSL format\n   * @param {String} color Color value ex: hsl(0-260,0%-100%,0%-100%)\n   * @memberOf fabric.Color\n   * @return {fabric.Color}\n   */\n  fabric.Color.fromHsl = function(color) {\n    return Color.fromSource(Color.sourceFromHsl(color));\n  };\n\n  /**\n   * Returns array representation (ex: [100, 100, 200, 1]) of a color that's in HSL or HSLA format.\n   * Adapted from <a href=\"https://rawgithub.com/mjijackson/mjijackson.github.com/master/2008/02/rgb-to-hsl-and-rgb-to-hsv-color-model-conversion-algorithms-in-javascript.html\">https://github.com/mjijackson</a>\n   * @memberOf fabric.Color\n   * @param {String} color Color value ex: hsl(0-360,0%-100%,0%-100%) or hsla(0-360,0%-100%,0%-100%, 0-1)\n   * @return {Array} source\n   * @see http://http://www.w3.org/TR/css3-color/#hsl-color\n   */\n  fabric.Color.sourceFromHsl = function(color) {\n    var match = color.match(Color.reHSLa);\n    if (!match) {\n      return;\n    }\n\n    var h = (((parseFloat(match[1]) % 360) + 360) % 360) / 360,\n        s = parseFloat(match[2]) / (/%$/.test(match[2]) ? 100 : 1),\n        l = parseFloat(match[3]) / (/%$/.test(match[3]) ? 100 : 1),\n        r, g, b;\n\n    if (s === 0) {\n      r = g = b = l;\n    }\n    else {\n      var q = l <= 0.5 ? l * (s + 1) : l + s - l * s,\n          p = l * 2 - q;\n\n      r = hue2rgb(p, q, h + 1 / 3);\n      g = hue2rgb(p, q, h);\n      b = hue2rgb(p, q, h - 1 / 3);\n    }\n\n    return [\n      Math.round(r * 255),\n      Math.round(g * 255),\n      Math.round(b * 255),\n      match[4] ? parseFloat(match[4]) : 1\n    ];\n  };\n\n  /**\n   * Returns new color object, when given a color in HSLA format\n   * @static\n   * @function\n   * @memberOf fabric.Color\n   * @param {String} color\n   * @return {fabric.Color}\n   */\n  fabric.Color.fromHsla = Color.fromHsl;\n\n  /**\n   * Returns new color object, when given a color in HEX format\n   * @static\n   * @memberOf fabric.Color\n   * @param {String} color Color value ex: FF5555\n   * @return {fabric.Color}\n   */\n  fabric.Color.fromHex = function(color) {\n    return Color.fromSource(Color.sourceFromHex(color));\n  };\n\n  /**\n   * Returns array representation (ex: [100, 100, 200, 1]) of a color that's in HEX format\n   * @static\n   * @memberOf fabric.Color\n   * @param {String} color ex: FF5555 or FF5544CC (RGBa)\n   * @return {Array} source\n   */\n  fabric.Color.sourceFromHex = function(color) {\n    if (color.match(Color.reHex)) {\n      var value = color.slice(color.indexOf('#') + 1),\n          isShortNotation = (value.length === 3 || value.length === 4),\n          isRGBa = (value.length === 8 || value.length === 4),\n          r = isShortNotation ? (value.charAt(0) + value.charAt(0)) : value.substring(0, 2),\n          g = isShortNotation ? (value.charAt(1) + value.charAt(1)) : value.substring(2, 4),\n          b = isShortNotation ? (value.charAt(2) + value.charAt(2)) : value.substring(4, 6),\n          a = isRGBa ? (isShortNotation ? (value.charAt(3) + value.charAt(3)) : value.substring(6, 8)) : 'FF';\n\n      return [\n        parseInt(r, 16),\n        parseInt(g, 16),\n        parseInt(b, 16),\n        parseFloat((parseInt(a, 16) / 255).toFixed(2))\n      ];\n    }\n  };\n\n  /**\n   * Returns new color object, when given color in array representation (ex: [200, 100, 100, 0.5])\n   * @static\n   * @memberOf fabric.Color\n   * @param {Array} source\n   * @return {fabric.Color}\n   */\n  fabric.Color.fromSource = function(source) {\n    var oColor = new Color();\n    oColor.setSource(source);\n    return oColor;\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      scaleMap = ['e', 'se', 's', 'sw', 'w', 'nw', 'n', 'ne', 'e'],\n      skewMap = ['ns', 'nesw', 'ew', 'nwse'],\n      controls = {},\n      LEFT = 'left', TOP = 'top', RIGHT = 'right', BOTTOM = 'bottom', CENTER = 'center',\n      opposite = {\n        top: BOTTOM,\n        bottom: TOP,\n        left: RIGHT,\n        right: LEFT,\n        center: CENTER,\n      }, radiansToDegrees = fabric.util.radiansToDegrees,\n      sign = (Math.sign || function(x) { return ((x > 0) - (x < 0)) || +x; });\n\n  /**\n   * Combine control position and object angle to find the control direction compared\n   * to the object center.\n   * @param {fabric.Object} fabricObject the fabric object for which we are rendering controls\n   * @param {fabric.Control} control the control class\n   * @return {Number} 0 - 7 a quadrant number\n   */\n  function findCornerQuadrant(fabricObject, control) {\n    var cornerAngle = fabricObject.angle + radiansToDegrees(Math.atan2(control.y, control.x)) + 360;\n    return Math.round((cornerAngle % 360) / 45);\n  }\n\n  function fireEvent(eventName, options) {\n    var target = options.transform.target,\n        canvas = target.canvas,\n        canvasOptions = fabric.util.object.clone(options);\n    canvasOptions.target = target;\n    canvas && canvas.fire('object:' + eventName, canvasOptions);\n    target.fire(eventName, options);\n  }\n\n  /**\n   * Inspect event and fabricObject properties to understand if the scaling action\n   * @param {Event} eventData from the user action\n   * @param {fabric.Object} fabricObject the fabric object about to scale\n   * @return {Boolean} true if scale is proportional\n   */\n  function scaleIsProportional(eventData, fabricObject) {\n    var canvas = fabricObject.canvas, uniScaleKey = canvas.uniScaleKey,\n        uniformIsToggled = eventData[uniScaleKey];\n    return (canvas.uniformScaling && !uniformIsToggled) ||\n    (!canvas.uniformScaling && uniformIsToggled);\n  }\n\n  /**\n   * Checks if transform is centered\n   * @param {Object} transform transform data\n   * @return {Boolean} true if transform is centered\n   */\n  function isTransformCentered(transform) {\n    return transform.originX === CENTER && transform.originY === CENTER;\n  }\n\n  /**\n   * Inspect fabricObject to understand if the current scaling action is allowed\n   * @param {fabric.Object} fabricObject the fabric object about to scale\n   * @param {String} by 'x' or 'y' or ''\n   * @param {Boolean} scaleProportionally true if we are trying to scale proportionally\n   * @return {Boolean} true if scaling is not allowed at current conditions\n   */\n  function scalingIsForbidden(fabricObject, by, scaleProportionally) {\n    var lockX = fabricObject.lockScalingX, lockY = fabricObject.lockScalingY;\n    if (lockX && lockY) {\n      return true;\n    }\n    if (!by && (lockX || lockY) && scaleProportionally) {\n      return true;\n    }\n    if (lockX && by === 'x') {\n      return true;\n    }\n    if (lockY && by === 'y') {\n      return true;\n    }\n    return false;\n  }\n\n  /**\n   * return the correct cursor style for the scale action\n   * @param {Event} eventData the javascript event that is causing the scale\n   * @param {fabric.Control} control the control that is interested in the action\n   * @param {fabric.Object} fabricObject the fabric object that is interested in the action\n   * @return {String} a valid css string for the cursor\n   */\n  function scaleCursorStyleHandler(eventData, control, fabricObject) {\n    var notAllowed = 'not-allowed',\n        scaleProportionally = scaleIsProportional(eventData, fabricObject),\n        by = '';\n    if (control.x !== 0 && control.y === 0) {\n      by = 'x';\n    }\n    else if (control.x === 0 && control.y !== 0) {\n      by = 'y';\n    }\n    if (scalingIsForbidden(fabricObject, by, scaleProportionally)) {\n      return notAllowed;\n    }\n    var n = findCornerQuadrant(fabricObject, control);\n    return scaleMap[n] + '-resize';\n  }\n\n  /**\n   * return the correct cursor style for the skew action\n   * @param {Event} eventData the javascript event that is causing the scale\n   * @param {fabric.Control} control the control that is interested in the action\n   * @param {fabric.Object} fabricObject the fabric object that is interested in the action\n   * @return {String} a valid css string for the cursor\n   */\n  function skewCursorStyleHandler(eventData, control, fabricObject) {\n    var notAllowed = 'not-allowed';\n    if (control.x !== 0 && fabricObject.lockSkewingY) {\n      return notAllowed;\n    }\n    if (control.y !== 0 && fabricObject.lockSkewingX) {\n      return notAllowed;\n    }\n    var n = findCornerQuadrant(fabricObject, control) % 4;\n    return skewMap[n] + '-resize';\n  }\n\n  /**\n   * Combine skew and scale style handlers to cover fabric standard use case\n   * @param {Event} eventData the javascript event that is causing the scale\n   * @param {fabric.Control} control the control that is interested in the action\n   * @param {fabric.Object} fabricObject the fabric object that is interested in the action\n   * @return {String} a valid css string for the cursor\n   */\n  function scaleSkewCursorStyleHandler(eventData, control, fabricObject) {\n    if (eventData[fabricObject.canvas.altActionKey]) {\n      return controls.skewCursorStyleHandler(eventData, control, fabricObject);\n    }\n    return controls.scaleCursorStyleHandler(eventData, control, fabricObject);\n  }\n\n  /**\n   * Inspect event, control and fabricObject to return the correct action name\n   * @param {Event} eventData the javascript event that is causing the scale\n   * @param {fabric.Control} control the control that is interested in the action\n   * @param {fabric.Object} fabricObject the fabric object that is interested in the action\n   * @return {String} an action name\n   */\n  function scaleOrSkewActionName(eventData, control, fabricObject) {\n    var isAlternative = eventData[fabricObject.canvas.altActionKey];\n    if (control.x === 0) {\n      // then is scaleY or skewX\n      return isAlternative ? 'skewX' : 'scaleY';\n    }\n    if (control.y === 0) {\n      // then is scaleY or skewX\n      return isAlternative ? 'skewY' : 'scaleX';\n    }\n  }\n\n  /**\n   * Find the correct style for the control that is used for rotation.\n   * this function is very simple and it just take care of not-allowed or standard cursor\n   * @param {Event} eventData the javascript event that is causing the scale\n   * @param {fabric.Control} control the control that is interested in the action\n   * @param {fabric.Object} fabricObject the fabric object that is interested in the action\n   * @return {String} a valid css string for the cursor\n   */\n  function rotationStyleHandler(eventData, control, fabricObject) {\n    if (fabricObject.lockRotation) {\n      return 'not-allowed';\n    }\n    return control.cursorStyle;\n  }\n\n  function commonEventInfo(eventData, transform, x, y) {\n    return {\n      e: eventData,\n      transform: transform,\n      pointer: {\n        x: x,\n        y: y,\n      }\n    };\n  }\n\n  /**\n   * Wrap an action handler with saving/restoring object position on the transform.\n   * this is the code that permits to objects to keep their position while transforming.\n   * @param {Function} actionHandler the function to wrap\n   * @return {Function} a function with an action handler signature\n   */\n  function wrapWithFixedAnchor(actionHandler) {\n    return function(eventData, transform, x, y) {\n      var target = transform.target, centerPoint = target.getCenterPoint(),\n          constraint = target.translateToOriginPoint(centerPoint, transform.originX, transform.originY),\n          actionPerformed = actionHandler(eventData, transform, x, y);\n      target.setPositionByOrigin(constraint, transform.originX, transform.originY);\n      return actionPerformed;\n    };\n  }\n\n  /**\n   * Wrap an action handler with firing an event if the action is performed\n   * @param {Function} actionHandler the function to wrap\n   * @return {Function} a function with an action handler signature\n   */\n  function wrapWithFireEvent(eventName, actionHandler) {\n    return function(eventData, transform, x, y) {\n      var actionPerformed = actionHandler(eventData, transform, x, y);\n      if (actionPerformed) {\n        fireEvent(eventName, commonEventInfo(eventData, transform, x, y));\n      }\n      return actionPerformed;\n    };\n  }\n\n  /**\n   * Transforms a point described by x and y in a distance from the top left corner of the object\n   * bounding box.\n   * @param {Object} transform\n   * @param {String} originX\n   * @param {String} originY\n   * @param {number} x\n   * @param {number} y\n   * @return {Fabric.Point} the normalized point\n   */\n  function getLocalPoint(transform, originX, originY, x, y) {\n    var target = transform.target,\n        control = target.controls[transform.corner],\n        zoom = target.canvas.getZoom(),\n        padding = target.padding / zoom,\n        localPoint = target.toLocalPoint(new fabric.Point(x, y), originX, originY);\n    if (localPoint.x >= padding) {\n      localPoint.x -= padding;\n    }\n    if (localPoint.x <= -padding) {\n      localPoint.x += padding;\n    }\n    if (localPoint.y >= padding) {\n      localPoint.y -= padding;\n    }\n    if (localPoint.y <= padding) {\n      localPoint.y += padding;\n    }\n    localPoint.x -= control.offsetX;\n    localPoint.y -= control.offsetY;\n    return localPoint;\n  }\n\n  /**\n   * Detect if the fabric object is flipped on one side.\n   * @param {fabric.Object} target\n   * @return {Boolean} true if one flip, but not two.\n   */\n  function targetHasOneFlip(target) {\n    return target.flipX !== target.flipY;\n  }\n\n  /**\n   * Utility function to compensate the scale factor when skew is applied on both axes\n   * @private\n   */\n  function compensateScaleForSkew(target, oppositeSkew, scaleToCompensate, axis, reference) {\n    if (target[oppositeSkew] !== 0) {\n      var newDim = target._getTransformedDimensions()[axis];\n      var newValue = reference / newDim * target[scaleToCompensate];\n      target.set(scaleToCompensate, newValue);\n    }\n  }\n\n  /**\n   * Action handler for skewing on the X axis\n   * @private\n   */\n  function skewObjectX(eventData, transform, x, y) {\n    var target = transform.target,\n        // find how big the object would be, if there was no skewX. takes in account scaling\n        dimNoSkew = target._getTransformedDimensions(0, target.skewY),\n        localPoint = getLocalPoint(transform, transform.originX, transform.originY, x, y),\n        // the mouse is in the center of the object, and we want it to stay there.\n        // so the object will grow twice as much as the mouse.\n        // this makes the skew growth to localPoint * 2 - dimNoSkew.\n        totalSkewSize = Math.abs(localPoint.x * 2) - dimNoSkew.x,\n        currentSkew = target.skewX, newSkew;\n    if (totalSkewSize < 2) {\n      // let's make it easy to go back to position 0.\n      newSkew = 0;\n    }\n    else {\n      newSkew = radiansToDegrees(\n        Math.atan2((totalSkewSize / target.scaleX), (dimNoSkew.y / target.scaleY))\n      );\n      // now we have to find the sign of the skew.\n      // it mostly depend on the origin of transformation.\n      if (transform.originX === LEFT && transform.originY === BOTTOM) {\n        newSkew = -newSkew;\n      }\n      if (transform.originX === RIGHT && transform.originY === TOP) {\n        newSkew = -newSkew;\n      }\n      if (targetHasOneFlip(target)) {\n        newSkew = -newSkew;\n      }\n    }\n    var hasSkewed = currentSkew !== newSkew;\n    if (hasSkewed) {\n      var dimBeforeSkewing = target._getTransformedDimensions().y;\n      target.set('skewX', newSkew);\n      compensateScaleForSkew(target, 'skewY', 'scaleY', 'y', dimBeforeSkewing);\n    }\n    return hasSkewed;\n  }\n\n  /**\n   * Action handler for skewing on the Y axis\n   * @private\n   */\n  function skewObjectY(eventData, transform, x, y) {\n    var target = transform.target,\n        // find how big the object would be, if there was no skewX. takes in account scaling\n        dimNoSkew = target._getTransformedDimensions(target.skewX, 0),\n        localPoint = getLocalPoint(transform, transform.originX, transform.originY, x, y),\n        // the mouse is in the center of the object, and we want it to stay there.\n        // so the object will grow twice as much as the mouse.\n        // this makes the skew growth to localPoint * 2 - dimNoSkew.\n        totalSkewSize = Math.abs(localPoint.y * 2) - dimNoSkew.y,\n        currentSkew = target.skewY, newSkew;\n    if (totalSkewSize < 2) {\n      // let's make it easy to go back to position 0.\n      newSkew = 0;\n    }\n    else {\n      newSkew = radiansToDegrees(\n        Math.atan2((totalSkewSize / target.scaleY), (dimNoSkew.x / target.scaleX))\n      );\n      // now we have to find the sign of the skew.\n      // it mostly depend on the origin of transformation.\n      if (transform.originX === LEFT && transform.originY === BOTTOM) {\n        newSkew = -newSkew;\n      }\n      if (transform.originX === RIGHT && transform.originY === TOP) {\n        newSkew = -newSkew;\n      }\n      if (targetHasOneFlip(target)) {\n        newSkew = -newSkew;\n      }\n    }\n    var hasSkewed = currentSkew !== newSkew;\n    if (hasSkewed) {\n      var dimBeforeSkewing = target._getTransformedDimensions().x;\n      target.set('skewY', newSkew);\n      compensateScaleForSkew(target, 'skewX', 'scaleX', 'x', dimBeforeSkewing);\n    }\n    return hasSkewed;\n  }\n\n  /**\n   * Wrapped Action handler for skewing on the Y axis, takes care of the\n   * skew direction and determine the correct transform origin for the anchor point\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if some change happened\n   */\n  function skewHandlerX(eventData, transform, x, y) {\n    // step1 figure out and change transform origin.\n    // if skewX > 0 and originY bottom we anchor on right\n    // if skewX > 0 and originY top we anchor on left\n    // if skewX < 0 and originY bottom we anchor on left\n    // if skewX < 0 and originY top we anchor on right\n    // if skewX is 0, we look for mouse position to understand where are we going.\n    var target = transform.target, currentSkew = target.skewX, originX, originY = transform.originY;\n    if (target.lockSkewingX) {\n      return false;\n    }\n    if (currentSkew === 0) {\n      var localPointFromCenter = getLocalPoint(transform, CENTER, CENTER, x, y);\n      if (localPointFromCenter.x > 0) {\n        // we are pulling right, anchor left;\n        originX = LEFT;\n      }\n      else {\n        // we are pulling right, anchor right\n        originX = RIGHT;\n      }\n    }\n    else {\n      if (currentSkew > 0) {\n        originX = originY === TOP ? LEFT : RIGHT;\n      }\n      if (currentSkew < 0) {\n        originX = originY === TOP ? RIGHT : LEFT;\n      }\n      // is the object flipped on one side only? swap the origin.\n      if (targetHasOneFlip(target)) {\n        originX = originX === LEFT ? RIGHT : LEFT;\n      }\n    }\n\n    // once we have the origin, we find the anchor point\n    transform.originX = originX;\n    var finalHandler = wrapWithFireEvent('skewing', wrapWithFixedAnchor(skewObjectX));\n    return finalHandler(eventData, transform, x, y);\n  }\n\n  /**\n   * Wrapped Action handler for skewing on the Y axis, takes care of the\n   * skew direction and determine the correct transform origin for the anchor point\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if some change happened\n   */\n  function skewHandlerY(eventData, transform, x, y) {\n    // step1 figure out and change transform origin.\n    // if skewY > 0 and originX left we anchor on top\n    // if skewY > 0 and originX right we anchor on bottom\n    // if skewY < 0 and originX left we anchor on bottom\n    // if skewY < 0 and originX right we anchor on top\n    // if skewY is 0, we look for mouse position to understand where are we going.\n    var target = transform.target, currentSkew = target.skewY, originY, originX = transform.originX;\n    if (target.lockSkewingY) {\n      return false;\n    }\n    if (currentSkew === 0) {\n      var localPointFromCenter = getLocalPoint(transform, CENTER, CENTER, x, y);\n      if (localPointFromCenter.y > 0) {\n        // we are pulling down, anchor up;\n        originY = TOP;\n      }\n      else {\n        // we are pulling up, anchor down\n        originY = BOTTOM;\n      }\n    }\n    else {\n      if (currentSkew > 0) {\n        originY = originX === LEFT ? TOP : BOTTOM;\n      }\n      if (currentSkew < 0) {\n        originY = originX === LEFT ? BOTTOM : TOP;\n      }\n      // is the object flipped on one side only? swap the origin.\n      if (targetHasOneFlip(target)) {\n        originY = originY === TOP ? BOTTOM : TOP;\n      }\n    }\n\n    // once we have the origin, we find the anchor point\n    transform.originY = originY;\n    var finalHandler = wrapWithFireEvent('skewing', wrapWithFixedAnchor(skewObjectY));\n    return finalHandler(eventData, transform, x, y);\n  }\n\n  /**\n   * Action handler for rotation and snapping, without anchor point.\n   * Needs to be wrapped with `wrapWithFixedAnchor` to be effective\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if some change happened\n   * @private\n   */\n  function rotationWithSnapping(eventData, transform, x, y) {\n    var t = transform,\n        target = t.target,\n        pivotPoint = target.translateToOriginPoint(target.getCenterPoint(), t.originX, t.originY);\n\n    if (target.lockRotation) {\n      return false;\n    }\n\n    var lastAngle = Math.atan2(t.ey - pivotPoint.y, t.ex - pivotPoint.x),\n        curAngle = Math.atan2(y - pivotPoint.y, x - pivotPoint.x),\n        angle = radiansToDegrees(curAngle - lastAngle + t.theta),\n        hasRotated = true;\n\n    if (target.snapAngle > 0) {\n      var snapAngle  = target.snapAngle,\n          snapThreshold  = target.snapThreshold || snapAngle,\n          rightAngleLocked = Math.ceil(angle / snapAngle) * snapAngle,\n          leftAngleLocked = Math.floor(angle / snapAngle) * snapAngle;\n\n      if (Math.abs(angle - leftAngleLocked) < snapThreshold) {\n        angle = leftAngleLocked;\n      }\n      else if (Math.abs(angle - rightAngleLocked) < snapThreshold) {\n        angle = rightAngleLocked;\n      }\n    }\n\n    // normalize angle to positive value\n    if (angle < 0) {\n      angle = 360 + angle;\n    }\n    angle %= 360;\n\n    hasRotated = target.angle !== angle;\n    target.angle = angle;\n    return hasRotated;\n  }\n\n  /**\n   * Basic scaling logic, reused with different constrain for scaling X,Y, freely or equally.\n   * Needs to be wrapped with `wrapWithFixedAnchor` to be effective\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @param {Object} options additional information for scaling\n   * @param {String} options.by 'x', 'y', 'equally' or '' to indicate type of scaling\n   * @return {Boolean} true if some change happened\n   * @private\n   */\n  function scaleObject(eventData, transform, x, y, options) {\n    options = options || {};\n    var target = transform.target,\n        lockScalingX = target.lockScalingX, lockScalingY = target.lockScalingY,\n        by = options.by, newPoint, scaleX, scaleY, dim,\n        scaleProportionally = scaleIsProportional(eventData, target),\n        forbidScaling = scalingIsForbidden(target, by, scaleProportionally),\n        signX, signY, gestureScale = transform.gestureScale;\n\n    if (forbidScaling) {\n      return false;\n    }\n    if (gestureScale) {\n      scaleX = transform.scaleX * gestureScale;\n      scaleY = transform.scaleY * gestureScale;\n    }\n    else {\n      newPoint = getLocalPoint(transform, transform.originX, transform.originY, x, y);\n      // use of sign: We use sign to detect change of direction of an action. sign usually change when\n      // we cross the origin point with the mouse. So a scale flip for example. There is an issue when scaling\n      // by center and scaling using one middle control ( default: mr, mt, ml, mb), the mouse movement can easily\n      // cross many time the origin point and flip the object. so we need a way to filter out the noise.\n      // This ternary here should be ok to filter out X scaling when we want Y only and vice versa.\n      signX = by !== 'y' ? sign(newPoint.x) : 1;\n      signY = by !== 'x' ? sign(newPoint.y) : 1;\n      if (!transform.signX) {\n        transform.signX = signX;\n      }\n      if (!transform.signY) {\n        transform.signY = signY;\n      }\n\n      if (target.lockScalingFlip &&\n        (transform.signX !== signX || transform.signY !== signY)\n      ) {\n        return false;\n      }\n\n      dim = target._getTransformedDimensions();\n      // missing detection of flip and logic to switch the origin\n      if (scaleProportionally && !by) {\n        // uniform scaling\n        var distance = Math.abs(newPoint.x) + Math.abs(newPoint.y),\n            original = transform.original,\n            originalDistance = Math.abs(dim.x * original.scaleX / target.scaleX) +\n              Math.abs(dim.y * original.scaleY / target.scaleY),\n            scale = distance / originalDistance;\n        scaleX = original.scaleX * scale;\n        scaleY = original.scaleY * scale;\n      }\n      else {\n        scaleX = Math.abs(newPoint.x * target.scaleX / dim.x);\n        scaleY = Math.abs(newPoint.y * target.scaleY / dim.y);\n      }\n      // if we are scaling by center, we need to double the scale\n      if (isTransformCentered(transform)) {\n        scaleX *= 2;\n        scaleY *= 2;\n      }\n      if (transform.signX !== signX && by !== 'y') {\n        transform.originX = opposite[transform.originX];\n        scaleX *= -1;\n        transform.signX = signX;\n      }\n      if (transform.signY !== signY && by !== 'x') {\n        transform.originY = opposite[transform.originY];\n        scaleY *= -1;\n        transform.signY = signY;\n      }\n    }\n    // minScale is taken are in the setter.\n    var oldScaleX = target.scaleX, oldScaleY = target.scaleY;\n    if (!by) {\n      !lockScalingX && target.set('scaleX', scaleX);\n      !lockScalingY && target.set('scaleY', scaleY);\n    }\n    else {\n      // forbidden cases already handled on top here.\n      by === 'x' && target.set('scaleX', scaleX);\n      by === 'y' && target.set('scaleY', scaleY);\n    }\n    return oldScaleX !== target.scaleX || oldScaleY !== target.scaleY;\n  }\n\n  /**\n   * Generic scaling logic, to scale from corners either equally or freely.\n   * Needs to be wrapped with `wrapWithFixedAnchor` to be effective\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if some change happened\n   */\n  function scaleObjectFromCorner(eventData, transform, x, y) {\n    return scaleObject(eventData, transform, x, y);\n  }\n\n  /**\n   * Scaling logic for the X axis.\n   * Needs to be wrapped with `wrapWithFixedAnchor` to be effective\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if some change happened\n   */\n  function scaleObjectX(eventData, transform, x, y) {\n    return scaleObject(eventData, transform, x, y , { by: 'x' });\n  }\n\n  /**\n   * Scaling logic for the Y axis.\n   * Needs to be wrapped with `wrapWithFixedAnchor` to be effective\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if some change happened\n   */\n  function scaleObjectY(eventData, transform, x, y) {\n    return scaleObject(eventData, transform, x, y , { by: 'y' });\n  }\n\n  /**\n   * Composed action handler to either scale Y or skew X\n   * Needs to be wrapped with `wrapWithFixedAnchor` to be effective\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if some change happened\n   */\n  function scalingYOrSkewingX(eventData, transform, x, y) {\n    // ok some safety needed here.\n    if (eventData[transform.target.canvas.altActionKey]) {\n      return controls.skewHandlerX(eventData, transform, x, y);\n    }\n    return controls.scalingY(eventData, transform, x, y);\n  }\n\n  /**\n   * Composed action handler to either scale X or skew Y\n   * Needs to be wrapped with `wrapWithFixedAnchor` to be effective\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if some change happened\n   */\n  function scalingXOrSkewingY(eventData, transform, x, y) {\n    // ok some safety needed here.\n    if (eventData[transform.target.canvas.altActionKey]) {\n      return controls.skewHandlerY(eventData, transform, x, y);\n    }\n    return controls.scalingX(eventData, transform, x, y);\n  }\n\n  /**\n   * Action handler to change textbox width\n   * Needs to be wrapped with `wrapWithFixedAnchor` to be effective\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if some change happened\n   */\n  function changeWidth(eventData, transform, x, y) {\n    var target = transform.target, localPoint = getLocalPoint(transform, transform.originX, transform.originY, x, y),\n        strokePadding = target.strokeWidth / (target.strokeUniform ? target.scaleX : 1),\n        multiplier = isTransformCentered(transform) ? 2 : 1,\n        oldWidth = target.width,\n        newWidth = Math.abs(localPoint.x * multiplier / target.scaleX) - strokePadding;\n    target.set('width', Math.max(newWidth, 0));\n    return oldWidth !== newWidth;\n  }\n\n  /**\n   * Action handler\n   * @private\n   * @param {Event} eventData javascript event that is doing the transform\n   * @param {Object} transform javascript object containing a series of information around the current transform\n   * @param {number} x current mouse x position, canvas normalized\n   * @param {number} y current mouse y position, canvas normalized\n   * @return {Boolean} true if the translation occurred\n   */\n  function dragHandler(eventData, transform, x, y) {\n    var target = transform.target,\n        newLeft = x - transform.offsetX,\n        newTop = y - transform.offsetY,\n        moveX = !target.get('lockMovementX') && target.left !== newLeft,\n        moveY = !target.get('lockMovementY') && target.top !== newTop;\n    moveX && target.set('left', newLeft);\n    moveY && target.set('top', newTop);\n    if (moveX || moveY) {\n      fireEvent('moving', commonEventInfo(eventData, transform, x, y));\n    }\n    return moveX || moveY;\n  }\n\n  controls.scaleCursorStyleHandler = scaleCursorStyleHandler;\n  controls.skewCursorStyleHandler = skewCursorStyleHandler;\n  controls.scaleSkewCursorStyleHandler = scaleSkewCursorStyleHandler;\n  controls.rotationWithSnapping = wrapWithFireEvent('rotating', wrapWithFixedAnchor(rotationWithSnapping));\n  controls.scalingEqually = wrapWithFireEvent('scaling', wrapWithFixedAnchor( scaleObjectFromCorner));\n  controls.scalingX = wrapWithFireEvent('scaling', wrapWithFixedAnchor(scaleObjectX));\n  controls.scalingY = wrapWithFireEvent('scaling', wrapWithFixedAnchor(scaleObjectY));\n  controls.scalingYOrSkewingX = scalingYOrSkewingX;\n  controls.scalingXOrSkewingY = scalingXOrSkewingY;\n  controls.changeWidth = wrapWithFireEvent('resizing', wrapWithFixedAnchor(changeWidth));\n  controls.skewHandlerX = skewHandlerX;\n  controls.skewHandlerY = skewHandlerY;\n  controls.dragHandler = dragHandler;\n  controls.scaleOrSkewActionName = scaleOrSkewActionName;\n  controls.rotationStyleHandler = rotationStyleHandler;\n  controls.fireEvent = fireEvent;\n  controls.wrapWithFixedAnchor = wrapWithFixedAnchor;\n  controls.wrapWithFireEvent = wrapWithFireEvent;\n  controls.getLocalPoint = getLocalPoint;\n  fabric.controlsUtils = controls;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      degreesToRadians = fabric.util.degreesToRadians,\n      controls = fabric.controlsUtils;\n\n  /**\n   * Render a round control, as per fabric features.\n   * This function is written to respect object properties like transparentCorners, cornerSize\n   * cornerColor, cornerStrokeColor\n   * plus the addition of offsetY and offsetX.\n   * @param {CanvasRenderingContext2D} ctx context to render on\n   * @param {Number} left x coordinate where the control center should be\n   * @param {Number} top y coordinate where the control center should be\n   * @param {Object} styleOverride override for fabric.Object controls style\n   * @param {fabric.Object} fabricObject the fabric object for which we are rendering controls\n   */\n  function renderCircleControl (ctx, left, top, styleOverride, fabricObject) {\n    styleOverride = styleOverride || {};\n    var xSize = this.sizeX || styleOverride.cornerSize || fabricObject.cornerSize,\n        ySize = this.sizeY || styleOverride.cornerSize || fabricObject.cornerSize,\n        transparentCorners = typeof styleOverride.transparentCorners !== 'undefined' ?\n          styleOverride.transparentCorners : fabricObject.transparentCorners,\n        methodName = transparentCorners ? 'stroke' : 'fill',\n        stroke = !transparentCorners && (styleOverride.cornerStrokeColor || fabricObject.cornerStrokeColor),\n        myLeft = left,\n        myTop = top, size;\n    ctx.save();\n    ctx.fillStyle = styleOverride.cornerColor || fabricObject.cornerColor;\n    ctx.strokeStyle = styleOverride.cornerStrokeColor || fabricObject.cornerStrokeColor;\n    // as soon as fabric react v5, remove ie11, use proper ellipse code.\n    if (xSize > ySize) {\n      size = xSize;\n      ctx.scale(1.0, ySize / xSize);\n      myTop = top * xSize / ySize;\n    }\n    else if (ySize > xSize) {\n      size = ySize;\n      ctx.scale(xSize / ySize, 1.0);\n      myLeft = left * ySize / xSize;\n    }\n    else {\n      size = xSize;\n    }\n    // this is still wrong\n    ctx.lineWidth = 1;\n    ctx.beginPath();\n    ctx.arc(myLeft, myTop, size / 2, 0, 2 * Math.PI, false);\n    ctx[methodName]();\n    if (stroke) {\n      ctx.stroke();\n    }\n    ctx.restore();\n  }\n\n  /**\n   * Render a square control, as per fabric features.\n   * This function is written to respect object properties like transparentCorners, cornerSize\n   * cornerColor, cornerStrokeColor\n   * plus the addition of offsetY and offsetX.\n   * @param {CanvasRenderingContext2D} ctx context to render on\n   * @param {Number} left x coordinate where the control center should be\n   * @param {Number} top y coordinate where the control center should be\n   * @param {Object} styleOverride override for fabric.Object controls style\n   * @param {fabric.Object} fabricObject the fabric object for which we are rendering controls\n   */\n  function renderSquareControl(ctx, left, top, styleOverride, fabricObject) {\n    styleOverride = styleOverride || {};\n    var xSize = this.sizeX || styleOverride.cornerSize || fabricObject.cornerSize,\n        ySize = this.sizeY || styleOverride.cornerSize || fabricObject.cornerSize,\n        transparentCorners = typeof styleOverride.transparentCorners !== 'undefined' ?\n          styleOverride.transparentCorners : fabricObject.transparentCorners,\n        methodName = transparentCorners ? 'stroke' : 'fill',\n        stroke = !transparentCorners && (\n          styleOverride.cornerStrokeColor || fabricObject.cornerStrokeColor\n        ), xSizeBy2 = xSize / 2, ySizeBy2 = ySize / 2;\n    ctx.save();\n    ctx.fillStyle = styleOverride.cornerColor || fabricObject.cornerColor;\n    ctx.strokeStyle = styleOverride.cornerStrokeColor || fabricObject.cornerStrokeColor;\n    // this is still wrong\n    ctx.lineWidth = 1;\n    ctx.translate(left, top);\n    ctx.rotate(degreesToRadians(fabricObject.angle));\n    // this does not work, and fixed with ( && ) does not make sense.\n    // to have real transparent corners we need the controls on upperCanvas\n    // transparentCorners || ctx.clearRect(-xSizeBy2, -ySizeBy2, xSize, ySize);\n    ctx[methodName + 'Rect'](-xSizeBy2, -ySizeBy2, xSize, ySize);\n    if (stroke) {\n      ctx.strokeRect(-xSizeBy2, -ySizeBy2, xSize, ySize);\n    }\n    ctx.restore();\n  }\n\n  controls.renderCircleControl = renderCircleControl;\n  controls.renderSquareControl = renderSquareControl;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { });\n\n  function Control(options) {\n    for (var i in options) {\n      this[i] = options[i];\n    }\n  }\n\n  fabric.Control = Control;\n\n  fabric.Control.prototype = /** @lends fabric.Control.prototype */ {\n\n    /**\n     * keep track of control visibility.\n     * mainly for backward compatibility.\n     * if you do not want to see a control, you can remove it\n     * from the controlset.\n     * @type {Boolean}\n     * @default true\n     */\n    visible: true,\n\n    /**\n     * Name of the action that the control will likely execute.\n     * This is optional. FabricJS uses to identify what the user is doing for some\n     * extra optimizations. If you are writing a custom control and you want to know\n     * somewhere else in the code what is going on, you can use this string here.\n     * you can also provide a custom getActionName if your control run multiple actions\n     * depending on some external state.\n     * default to scale since is the most common, used on 4 corners by default\n     * @type {String}\n     * @default 'scale'\n     */\n    actionName: 'scale',\n\n    /**\n     * Drawing angle of the control.\n     * NOT used for now, but name marked as needed for internal logic\n     * example: to reuse the same drawing function for different rotated controls\n     * @type {Number}\n     * @default 0\n     */\n    angle: 0,\n\n    /**\n     * Relative position of the control. X\n     * 0,0 is the center of the Object, while -0.5 (left) or 0.5 (right) are the extremities\n     * of the bounding box.\n     * @type {Number}\n     * @default 0\n     */\n    x: 0,\n\n    /**\n     * Relative position of the control. Y\n     * 0,0 is the center of the Object, while -0.5 (top) or 0.5 (bottom) are the extremities\n     * of the bounding box.\n     * @type {Number}\n     * @default 0\n     */\n    y: 0,\n\n    /**\n     * Horizontal offset of the control from the defined position. In pixels\n     * Positive offset moves the control to the right, negative to the left.\n     * It used when you want to have position of control that does not scale with\n     * the bounding box. Example: rotation control is placed at x:0, y: 0.5 on\n     * the boundindbox, with an offset of 30 pixels vertically. Those 30 pixels will\n     * stay 30 pixels no matter how the object is big. Another example is having 2\n     * controls in the corner, that stay in the same position when the object scale.\n     * of the bounding box.\n     * @type {Number}\n     * @default 0\n     */\n    offsetX: 0,\n\n    /**\n     * Vertical offset of the control from the defined position. In pixels\n     * Positive offset moves the control to the bottom, negative to the top.\n     * @type {Number}\n     * @default 0\n     */\n    offsetY: 0,\n\n    /**\n     * Sets the length of the control. If null, defaults to object's cornerSize.\n     * Expects both sizeX and sizeY to be set when set.\n     * @type {?Number}\n     * @default null\n     */\n    sizeX: null,\n\n    /**\n     * Sets the height of the control. If null, defaults to object's cornerSize.\n     * Expects both sizeX and sizeY to be set when set.\n     * @type {?Number}\n     * @default null\n     */\n    sizeY: null,\n\n    /**\n     * Sets the length of the touch area of the control. If null, defaults to object's touchCornerSize.\n     * Expects both touchSizeX and touchSizeY to be set when set.\n     * @type {?Number}\n     * @default null\n     */\n    touchSizeX: null,\n\n    /**\n     * Sets the height of the touch area of the control. If null, defaults to object's touchCornerSize.\n     * Expects both touchSizeX and touchSizeY to be set when set.\n     * @type {?Number}\n     * @default null\n     */\n    touchSizeY: null,\n\n    /**\n     * Css cursor style to display when the control is hovered.\n     * if the method `cursorStyleHandler` is provided, this property is ignored.\n     * @type {String}\n     * @default 'crosshair'\n     */\n    cursorStyle: 'crosshair',\n\n    /**\n     * If controls has an offsetY or offsetX, draw a line that connects\n     * the control to the bounding box\n     * @type {Boolean}\n     * @default false\n     */\n    withConnection: false,\n\n    /**\n     * The control actionHandler, provide one to handle action ( control being moved )\n     * @param {Event} eventData the native mouse event\n     * @param {Object} transformData properties of the current transform\n     * @param {Number} x x position of the cursor\n     * @param {Number} y y position of the cursor\n     * @return {Boolean} true if the action/event modified the object\n     */\n    actionHandler: function(/* eventData, transformData, x, y */) { },\n\n    /**\n     * The control handler for mouse down, provide one to handle mouse down on control\n     * @param {Event} eventData the native mouse event\n     * @param {Object} transformData properties of the current transform\n     * @param {Number} x x position of the cursor\n     * @param {Number} y y position of the cursor\n     * @return {Boolean} true if the action/event modified the object\n     */\n    mouseDownHandler: function(/* eventData, transformData, x, y */) { },\n\n    /**\n     * The control mouseUpHandler, provide one to handle an effect on mouse up.\n     * @param {Event} eventData the native mouse event\n     * @param {Object} transformData properties of the current transform\n     * @param {Number} x x position of the cursor\n     * @param {Number} y y position of the cursor\n     * @return {Boolean} true if the action/event modified the object\n     */\n    mouseUpHandler: function(/* eventData, transformData, x, y */) { },\n\n    /**\n     * Returns control actionHandler\n     * @param {Event} eventData the native mouse event\n     * @param {fabric.Object} fabricObject on which the control is displayed\n     * @param {fabric.Control} control control for which the action handler is being asked\n     * @return {Function} the action handler\n     */\n    getActionHandler: function(/* eventData, fabricObject, control */) {\n      return this.actionHandler;\n    },\n\n    /**\n     * Returns control mouseDown handler\n     * @param {Event} eventData the native mouse event\n     * @param {fabric.Object} fabricObject on which the control is displayed\n     * @param {fabric.Control} control control for which the action handler is being asked\n     * @return {Function} the action handler\n     */\n    getMouseDownHandler: function(/* eventData, fabricObject, control */) {\n      return this.mouseDownHandler;\n    },\n\n    /**\n     * Returns control mouseUp handler\n     * @param {Event} eventData the native mouse event\n     * @param {fabric.Object} fabricObject on which the control is displayed\n     * @param {fabric.Control} control control for which the action handler is being asked\n     * @return {Function} the action handler\n     */\n    getMouseUpHandler: function(/* eventData, fabricObject, control */) {\n      return this.mouseUpHandler;\n    },\n\n    /**\n     * Returns control cursorStyle for css using cursorStyle. If you need a more elaborate\n     * function you can pass one in the constructor\n     * the cursorStyle property\n     * @param {Event} eventData the native mouse event\n     * @param {fabric.Control} control the current control ( likely this)\n     * @param {fabric.Object} object on which the control is displayed\n     * @return {String}\n     */\n    cursorStyleHandler: function(eventData, control /* fabricObject */) {\n      return control.cursorStyle;\n    },\n\n    /**\n     * Returns the action name. The basic implementation just return the actionName property.\n     * @param {Event} eventData the native mouse event\n     * @param {fabric.Control} control the current control ( likely this)\n     * @param {fabric.Object} object on which the control is displayed\n     * @return {String}\n     */\n    getActionName: function(eventData, control /* fabricObject */) {\n      return control.actionName;\n    },\n\n    /**\n     * Returns controls visibility\n     * @param {fabric.Object} object on which the control is displayed\n     * @param {String} controlKey key where the control is memorized on the\n     * @return {Boolean}\n     */\n    getVisibility: function(fabricObject, controlKey) {\n      var objectVisibility = fabricObject._controlsVisibility;\n      if (objectVisibility && typeof objectVisibility[controlKey] !== 'undefined') {\n        return objectVisibility[controlKey];\n      }\n      return this.visible;\n    },\n\n    /**\n     * Sets controls visibility\n     * @param {Boolean} visibility for the object\n     * @return {Void}\n     */\n    setVisibility: function(visibility /* name, fabricObject */) {\n      this.visible = visibility;\n    },\n\n\n    positionHandler: function(dim, finalMatrix /*, fabricObject, currentControl */) {\n      var point = fabric.util.transformPoint({\n        x: this.x * dim.x + this.offsetX,\n        y: this.y * dim.y + this.offsetY }, finalMatrix);\n      return point;\n    },\n\n    /**\n     * Returns the coords for this control based on object values.\n     * @param {Number} objectAngle angle from the fabric object holding the control\n     * @param {Number} objectCornerSize cornerSize from the fabric object holding the control (or touchCornerSize if\n     *   isTouch is true)\n     * @param {Number} centerX x coordinate where the control center should be\n     * @param {Number} centerY y coordinate where the control center should be\n     * @param {boolean} isTouch true if touch corner, false if normal corner\n     */\n    calcCornerCoords: function(objectAngle, objectCornerSize, centerX, centerY, isTouch) {\n      var cosHalfOffset,\n          sinHalfOffset,\n          cosHalfOffsetComp,\n          sinHalfOffsetComp,\n          xSize = (isTouch) ? this.touchSizeX : this.sizeX,\n          ySize = (isTouch) ? this.touchSizeY : this.sizeY;\n      if (xSize && ySize && xSize !== ySize) {\n        // handle rectangular corners\n        var controlTriangleAngle = Math.atan2(ySize, xSize);\n        var cornerHypotenuse = Math.sqrt(xSize * xSize + ySize * ySize) / 2;\n        var newTheta = controlTriangleAngle - fabric.util.degreesToRadians(objectAngle);\n        var newThetaComp = Math.PI / 2 - controlTriangleAngle - fabric.util.degreesToRadians(objectAngle);\n        cosHalfOffset = cornerHypotenuse * fabric.util.cos(newTheta);\n        sinHalfOffset = cornerHypotenuse * fabric.util.sin(newTheta);\n        // use complementary angle for two corners\n        cosHalfOffsetComp = cornerHypotenuse * fabric.util.cos(newThetaComp);\n        sinHalfOffsetComp = cornerHypotenuse * fabric.util.sin(newThetaComp);\n      }\n      else {\n        // handle square corners\n        // use default object corner size unless size is defined\n        var cornerSize = (xSize && ySize) ? xSize : objectCornerSize;\n        /* 0.7071067812 stands for sqrt(2)/2 */\n        cornerHypotenuse = cornerSize * 0.7071067812;\n        // complementary angles are equal since they're both 45 degrees\n        var newTheta = fabric.util.degreesToRadians(45 - objectAngle);\n        cosHalfOffset = cosHalfOffsetComp = cornerHypotenuse * fabric.util.cos(newTheta);\n        sinHalfOffset = sinHalfOffsetComp = cornerHypotenuse * fabric.util.sin(newTheta);\n      }\n\n      return {\n        tl: {\n          x: centerX - sinHalfOffsetComp,\n          y: centerY - cosHalfOffsetComp,\n        },\n        tr: {\n          x: centerX + cosHalfOffset,\n          y: centerY - sinHalfOffset,\n        },\n        bl: {\n          x: centerX - cosHalfOffset,\n          y: centerY + sinHalfOffset,\n        },\n        br: {\n          x: centerX + sinHalfOffsetComp,\n          y: centerY + cosHalfOffsetComp,\n        },\n      };\n    },\n\n    /**\n    * Render function for the control.\n    * When this function runs the context is unscaled. unrotate. Just retina scaled.\n    * all the functions will have to translate to the point left,top before starting Drawing\n    * if they want to draw a control where the position is detected.\n    * left and top are the result of the positionHandler function\n    * @param {RenderingContext2D} ctx the context where the control will be drawn\n    * @param {Number} left position of the canvas where we are about to render the control.\n    * @param {Number} top position of the canvas where we are about to render the control.\n    * @param {Object} styleOverride\n    * @param {fabric.Object} fabricObject the object where the control is about to be rendered\n    */\n    render: function(ctx, left, top, styleOverride, fabricObject) {\n      styleOverride = styleOverride || {};\n      switch (styleOverride.cornerStyle || fabricObject.cornerStyle) {\n        case 'circle':\n          fabric.controlsUtils.renderCircleControl.call(this, ctx, left, top, styleOverride, fabricObject);\n          break;\n        default:\n          fabric.controlsUtils.renderSquareControl.call(this, ctx, left, top, styleOverride, fabricObject);\n      }\n    },\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function() {\n\n  /* _FROM_SVG_START_ */\n  function getColorStop(el, multiplier) {\n    var style = el.getAttribute('style'),\n        offset = el.getAttribute('offset') || 0,\n        color, colorAlpha, opacity, i;\n\n    // convert percents to absolute values\n    offset = parseFloat(offset) / (/%$/.test(offset) ? 100 : 1);\n    offset = offset < 0 ? 0 : offset > 1 ? 1 : offset;\n    if (style) {\n      var keyValuePairs = style.split(/\\s*;\\s*/);\n\n      if (keyValuePairs[keyValuePairs.length - 1] === '') {\n        keyValuePairs.pop();\n      }\n\n      for (i = keyValuePairs.length; i--; ) {\n\n        var split = keyValuePairs[i].split(/\\s*:\\s*/),\n            key = split[0].trim(),\n            value = split[1].trim();\n\n        if (key === 'stop-color') {\n          color = value;\n        }\n        else if (key === 'stop-opacity') {\n          opacity = value;\n        }\n      }\n    }\n\n    if (!color) {\n      color = el.getAttribute('stop-color') || 'rgb(0,0,0)';\n    }\n    if (!opacity) {\n      opacity = el.getAttribute('stop-opacity');\n    }\n\n    color = new fabric.Color(color);\n    colorAlpha = color.getAlpha();\n    opacity = isNaN(parseFloat(opacity)) ? 1 : parseFloat(opacity);\n    opacity *= colorAlpha * multiplier;\n\n    return {\n      offset: offset,\n      color: color.toRgb(),\n      opacity: opacity\n    };\n  }\n\n  function getLinearCoords(el) {\n    return {\n      x1: el.getAttribute('x1') || 0,\n      y1: el.getAttribute('y1') || 0,\n      x2: el.getAttribute('x2') || '100%',\n      y2: el.getAttribute('y2') || 0\n    };\n  }\n\n  function getRadialCoords(el) {\n    return {\n      x1: el.getAttribute('fx') || el.getAttribute('cx') || '50%',\n      y1: el.getAttribute('fy') || el.getAttribute('cy') || '50%',\n      r1: 0,\n      x2: el.getAttribute('cx') || '50%',\n      y2: el.getAttribute('cy') || '50%',\n      r2: el.getAttribute('r') || '50%'\n    };\n  }\n  /* _FROM_SVG_END_ */\n\n  var clone = fabric.util.object.clone;\n\n  /**\n   * Gradient class\n   * @class fabric.Gradient\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-2#gradients}\n   * @see {@link fabric.Gradient#initialize} for constructor definition\n   */\n  fabric.Gradient = fabric.util.createClass(/** @lends fabric.Gradient.prototype */ {\n\n    /**\n     * Horizontal offset for aligning gradients coming from SVG when outside pathgroups\n     * @type Number\n     * @default 0\n     */\n    offsetX: 0,\n\n    /**\n     * Vertical offset for aligning gradients coming from SVG when outside pathgroups\n     * @type Number\n     * @default 0\n     */\n    offsetY: 0,\n\n    /**\n     * A transform matrix to apply to the gradient before painting.\n     * Imported from svg gradients, is not applied with the current transform in the center.\n     * Before this transform is applied, the origin point is at the top left corner of the object\n     * plus the addition of offsetY and offsetX.\n     * @type Number[]\n     * @default null\n     */\n    gradientTransform: null,\n\n    /**\n     * coordinates units for coords.\n     * If `pixels`, the number of coords are in the same unit of width / height.\n     * If set as `percentage` the coords are still a number, but 1 means 100% of width\n     * for the X and 100% of the height for the y. It can be bigger than 1 and negative.\n     * allowed values pixels or percentage.\n     * @type String\n     * @default 'pixels'\n     */\n    gradientUnits: 'pixels',\n\n    /**\n     * Gradient type linear or radial\n     * @type String\n     * @default 'pixels'\n     */\n    type: 'linear',\n\n    /**\n     * Constructor\n     * @param {Object} options Options object with type, coords, gradientUnits and colorStops\n     * @param {Object} [options.type] gradient type linear or radial\n     * @param {Object} [options.gradientUnits] gradient units\n     * @param {Object} [options.offsetX] SVG import compatibility\n     * @param {Object} [options.offsetY] SVG import compatibility\n     * @param {Object[]} options.colorStops contains the colorstops.\n     * @param {Object} options.coords contains the coords of the gradient\n     * @param {Number} [options.coords.x1] X coordiante of the first point for linear or of the focal point for radial\n     * @param {Number} [options.coords.y1] Y coordiante of the first point for linear or of the focal point for radial\n     * @param {Number} [options.coords.x2] X coordiante of the second point for linear or of the center point for radial\n     * @param {Number} [options.coords.y2] Y coordiante of the second point for linear or of the center point for radial\n     * @param {Number} [options.coords.r1] only for radial gradient, radius of the inner circle\n     * @param {Number} [options.coords.r2] only for radial gradient, radius of the external circle\n     * @return {fabric.Gradient} thisArg\n     */\n    initialize: function(options) {\n      options || (options = { });\n      options.coords || (options.coords = { });\n\n      var coords, _this = this;\n\n      // sets everything, then coords and colorstops get sets again\n      Object.keys(options).forEach(function(option) {\n        _this[option] = options[option];\n      });\n\n      if (this.id) {\n        this.id += '_' + fabric.Object.__uid++;\n      }\n      else {\n        this.id = fabric.Object.__uid++;\n      }\n\n      coords = {\n        x1: options.coords.x1 || 0,\n        y1: options.coords.y1 || 0,\n        x2: options.coords.x2 || 0,\n        y2: options.coords.y2 || 0\n      };\n\n      if (this.type === 'radial') {\n        coords.r1 = options.coords.r1 || 0;\n        coords.r2 = options.coords.r2 || 0;\n      }\n\n      this.coords = coords;\n      this.colorStops = options.colorStops.slice();\n    },\n\n    /**\n     * Adds another colorStop\n     * @param {Object} colorStop Object with offset and color\n     * @return {fabric.Gradient} thisArg\n     */\n    addColorStop: function(colorStops) {\n      for (var position in colorStops) {\n        var color = new fabric.Color(colorStops[position]);\n        this.colorStops.push({\n          offset: parseFloat(position),\n          color: color.toRgb(),\n          opacity: color.getAlpha()\n        });\n      }\n      return this;\n    },\n\n    /**\n     * Returns object representation of a gradient\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object}\n     */\n    toObject: function(propertiesToInclude) {\n      var object = {\n        type: this.type,\n        coords: this.coords,\n        colorStops: this.colorStops,\n        offsetX: this.offsetX,\n        offsetY: this.offsetY,\n        gradientUnits: this.gradientUnits,\n        gradientTransform: this.gradientTransform ? this.gradientTransform.concat() : this.gradientTransform\n      };\n      fabric.util.populateWithProperties(this, object, propertiesToInclude);\n\n      return object;\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns SVG representation of an gradient\n     * @param {Object} object Object to create a gradient for\n     * @return {String} SVG representation of an gradient (linear/radial)\n     */\n    toSVG: function(object, options) {\n      var coords = clone(this.coords, true), i, len, options = options || {},\n          markup, commonAttributes, colorStops = clone(this.colorStops, true),\n          needsSwap = coords.r1 > coords.r2,\n          transform = this.gradientTransform ? this.gradientTransform.concat() : fabric.iMatrix.concat(),\n          offsetX = -this.offsetX, offsetY = -this.offsetY,\n          withViewport = !!options.additionalTransform,\n          gradientUnits = this.gradientUnits === 'pixels' ? 'userSpaceOnUse' : 'objectBoundingBox';\n      // colorStops must be sorted ascending\n      colorStops.sort(function(a, b) {\n        return a.offset - b.offset;\n      });\n\n      if (gradientUnits === 'objectBoundingBox') {\n        offsetX /= object.width;\n        offsetY /= object.height;\n      }\n      else {\n        offsetX += object.width / 2;\n        offsetY += object.height / 2;\n      }\n      if (object.type === 'path' && this.gradientUnits !== 'percentage') {\n        offsetX -= object.pathOffset.x;\n        offsetY -= object.pathOffset.y;\n      }\n\n\n      transform[4] -= offsetX;\n      transform[5] -= offsetY;\n\n      commonAttributes = 'id=\"SVGID_' + this.id +\n                     '\" gradientUnits=\"' + gradientUnits + '\"';\n      commonAttributes += ' gradientTransform=\"' + (withViewport ?\n        options.additionalTransform + ' ' : '') + fabric.util.matrixToSVG(transform) + '\" ';\n\n      if (this.type === 'linear') {\n        markup = [\n          '<linearGradient ',\n          commonAttributes,\n          ' x1=\"', coords.x1,\n          '\" y1=\"', coords.y1,\n          '\" x2=\"', coords.x2,\n          '\" y2=\"', coords.y2,\n          '\">\\n'\n        ];\n      }\n      else if (this.type === 'radial') {\n        // svg radial gradient has just 1 radius. the biggest.\n        markup = [\n          '<radialGradient ',\n          commonAttributes,\n          ' cx=\"', needsSwap ? coords.x1 : coords.x2,\n          '\" cy=\"', needsSwap ? coords.y1 : coords.y2,\n          '\" r=\"', needsSwap ? coords.r1 : coords.r2,\n          '\" fx=\"', needsSwap ? coords.x2 : coords.x1,\n          '\" fy=\"', needsSwap ? coords.y2 : coords.y1,\n          '\">\\n'\n        ];\n      }\n\n      if (this.type === 'radial') {\n        if (needsSwap) {\n          // svg goes from internal to external radius. if radius are inverted, swap color stops.\n          colorStops = colorStops.concat();\n          colorStops.reverse();\n          for (i = 0, len = colorStops.length; i < len; i++) {\n            colorStops[i].offset = 1 - colorStops[i].offset;\n          }\n        }\n        var minRadius = Math.min(coords.r1, coords.r2);\n        if (minRadius > 0) {\n          // i have to shift all colorStops and add new one in 0.\n          var maxRadius = Math.max(coords.r1, coords.r2),\n              percentageShift = minRadius / maxRadius;\n          for (i = 0, len = colorStops.length; i < len; i++) {\n            colorStops[i].offset += percentageShift * (1 - colorStops[i].offset);\n          }\n        }\n      }\n\n      for (i = 0, len = colorStops.length; i < len; i++) {\n        var colorStop = colorStops[i];\n        markup.push(\n          '<stop ',\n          'offset=\"', (colorStop.offset * 100) + '%',\n          '\" style=\"stop-color:', colorStop.color,\n          (typeof colorStop.opacity !== 'undefined' ? ';stop-opacity: ' + colorStop.opacity : ';'),\n          '\"/>\\n'\n        );\n      }\n\n      markup.push((this.type === 'linear' ? '</linearGradient>\\n' : '</radialGradient>\\n'));\n\n      return markup.join('');\n    },\n    /* _TO_SVG_END_ */\n\n    /**\n     * Returns an instance of CanvasGradient\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @return {CanvasGradient}\n     */\n    toLive: function(ctx) {\n      var gradient, coords = fabric.util.object.clone(this.coords), i, len;\n\n      if (!this.type) {\n        return;\n      }\n\n      if (this.type === 'linear') {\n        gradient = ctx.createLinearGradient(\n          coords.x1, coords.y1, coords.x2, coords.y2);\n      }\n      else if (this.type === 'radial') {\n        gradient = ctx.createRadialGradient(\n          coords.x1, coords.y1, coords.r1, coords.x2, coords.y2, coords.r2);\n      }\n\n      for (i = 0, len = this.colorStops.length; i < len; i++) {\n        var color = this.colorStops[i].color,\n            opacity = this.colorStops[i].opacity,\n            offset = this.colorStops[i].offset;\n\n        if (typeof opacity !== 'undefined') {\n          color = new fabric.Color(color).setAlpha(opacity).toRgba();\n        }\n        gradient.addColorStop(offset, color);\n      }\n\n      return gradient;\n    }\n  });\n\n  fabric.util.object.extend(fabric.Gradient, {\n\n    /* _FROM_SVG_START_ */\n    /**\n     * Returns {@link fabric.Gradient} instance from an SVG element\n     * @static\n     * @memberOf fabric.Gradient\n     * @param {SVGGradientElement} el SVG gradient element\n     * @param {fabric.Object} instance\n     * @param {String} opacityAttr A fill-opacity or stroke-opacity attribute to multiply to each stop's opacity.\n     * @param {Object} svgOptions an object containing the size of the SVG in order to parse correctly gradients\n     * that uses gradientUnits as 'userSpaceOnUse' and percentages.\n     * @param {Object.number} viewBoxWidth width part of the viewBox attribute on svg\n     * @param {Object.number} viewBoxHeight height part of the viewBox attribute on svg\n     * @param {Object.number} width width part of the svg tag if viewBox is not specified\n     * @param {Object.number} height height part of the svg tag if viewBox is not specified\n     * @return {fabric.Gradient} Gradient instance\n     * @see http://www.w3.org/TR/SVG/pservers.html#LinearGradientElement\n     * @see http://www.w3.org/TR/SVG/pservers.html#RadialGradientElement\n     */\n    fromElement: function(el, instance, opacityAttr, svgOptions) {\n      /**\n       *  @example:\n       *\n       *  <linearGradient id=\"linearGrad1\">\n       *    <stop offset=\"0%\" stop-color=\"white\"/>\n       *    <stop offset=\"100%\" stop-color=\"black\"/>\n       *  </linearGradient>\n       *\n       *  OR\n       *\n       *  <linearGradient id=\"linearGrad2\">\n       *    <stop offset=\"0\" style=\"stop-color:rgb(255,255,255)\"/>\n       *    <stop offset=\"1\" style=\"stop-color:rgb(0,0,0)\"/>\n       *  </linearGradient>\n       *\n       *  OR\n       *\n       *  <radialGradient id=\"radialGrad1\">\n       *    <stop offset=\"0%\" stop-color=\"white\" stop-opacity=\"1\" />\n       *    <stop offset=\"50%\" stop-color=\"black\" stop-opacity=\"0.5\" />\n       *    <stop offset=\"100%\" stop-color=\"white\" stop-opacity=\"1\" />\n       *  </radialGradient>\n       *\n       *  OR\n       *\n       *  <radialGradient id=\"radialGrad2\">\n       *    <stop offset=\"0\" stop-color=\"rgb(255,255,255)\" />\n       *    <stop offset=\"0.5\" stop-color=\"rgb(0,0,0)\" />\n       *    <stop offset=\"1\" stop-color=\"rgb(255,255,255)\" />\n       *  </radialGradient>\n       *\n       */\n\n      var multiplier = parseFloat(opacityAttr) / (/%$/.test(opacityAttr) ? 100 : 1);\n      multiplier = multiplier < 0 ? 0 : multiplier > 1 ? 1 : multiplier;\n      if (isNaN(multiplier)) {\n        multiplier = 1;\n      }\n\n      var colorStopEls = el.getElementsByTagName('stop'),\n          type,\n          gradientUnits = el.getAttribute('gradientUnits') === 'userSpaceOnUse' ?\n            'pixels' : 'percentage',\n          gradientTransform = el.getAttribute('gradientTransform') || '',\n          colorStops = [],\n          coords, i, offsetX = 0, offsetY = 0,\n          transformMatrix;\n      if (el.nodeName === 'linearGradient' || el.nodeName === 'LINEARGRADIENT') {\n        type = 'linear';\n        coords = getLinearCoords(el);\n      }\n      else {\n        type = 'radial';\n        coords = getRadialCoords(el);\n      }\n\n      for (i = colorStopEls.length; i--; ) {\n        colorStops.push(getColorStop(colorStopEls[i], multiplier));\n      }\n\n      transformMatrix = fabric.parseTransformAttribute(gradientTransform);\n\n      __convertPercentUnitsToValues(instance, coords, svgOptions, gradientUnits);\n\n      if (gradientUnits === 'pixels') {\n        offsetX = -instance.left;\n        offsetY = -instance.top;\n      }\n\n      var gradient = new fabric.Gradient({\n        id: el.getAttribute('id'),\n        type: type,\n        coords: coords,\n        colorStops: colorStops,\n        gradientUnits: gradientUnits,\n        gradientTransform: transformMatrix,\n        offsetX: offsetX,\n        offsetY: offsetY,\n      });\n\n      return gradient;\n    }\n    /* _FROM_SVG_END_ */\n  });\n\n  /**\n   * @private\n   */\n  function __convertPercentUnitsToValues(instance, options, svgOptions, gradientUnits) {\n    var propValue, finalValue;\n    Object.keys(options).forEach(function(prop) {\n      propValue = options[prop];\n      if (propValue === 'Infinity') {\n        finalValue = 1;\n      }\n      else if (propValue === '-Infinity') {\n        finalValue = 0;\n      }\n      else {\n        finalValue = parseFloat(options[prop], 10);\n        if (typeof propValue === 'string' && /^(\\d+\\.\\d+)%|(\\d+)%$/.test(propValue)) {\n          finalValue *= 0.01;\n          if (gradientUnits === 'pixels') {\n            // then we need to fix those percentages here in svg parsing\n            if (prop === 'x1' || prop === 'x2' || prop === 'r2') {\n              finalValue *= svgOptions.viewBoxWidth || svgOptions.width;\n            }\n            if (prop === 'y1' || prop === 'y2') {\n              finalValue *= svgOptions.viewBoxHeight || svgOptions.height;\n            }\n          }\n        }\n      }\n      options[prop] = finalValue;\n    });\n  }\n})();\n\n\n(function() {\n\n  'use strict';\n\n  var toFixed = fabric.util.toFixed;\n\n  /**\n   * Pattern class\n   * @class fabric.Pattern\n   * @see {@link http://fabricjs.com/patterns|Pattern demo}\n   * @see {@link http://fabricjs.com/dynamic-patterns|DynamicPattern demo}\n   * @see {@link fabric.Pattern#initialize} for constructor definition\n   */\n\n\n  fabric.Pattern = fabric.util.createClass(/** @lends fabric.Pattern.prototype */ {\n\n    /**\n     * Repeat property of a pattern (one of repeat, repeat-x, repeat-y or no-repeat)\n     * @type String\n     * @default\n     */\n    repeat: 'repeat',\n\n    /**\n     * Pattern horizontal offset from object's left/top corner\n     * @type Number\n     * @default\n     */\n    offsetX: 0,\n\n    /**\n     * Pattern vertical offset from object's left/top corner\n     * @type Number\n     * @default\n     */\n    offsetY: 0,\n\n    /**\n     * crossOrigin value (one of \"\", \"anonymous\", \"use-credentials\")\n     * @see https://developer.mozilla.org/en-US/docs/HTML/CORS_settings_attributes\n     * @type String\n     * @default\n     */\n    crossOrigin: '',\n\n    /**\n     * transform matrix to change the pattern, imported from svgs.\n     * @type Array\n     * @default\n     */\n    patternTransform: null,\n\n    /**\n     * Constructor\n     * @param {Object} [options] Options object\n     * @param {Function} [callback] function to invoke after callback init.\n     * @return {fabric.Pattern} thisArg\n     */\n    initialize: function(options, callback) {\n      options || (options = { });\n\n      this.id = fabric.Object.__uid++;\n      this.setOptions(options);\n      if (!options.source || (options.source && typeof options.source !== 'string')) {\n        callback && callback(this);\n        return;\n      }\n      else {\n        // img src string\n        var _this = this;\n        this.source = fabric.util.createImage();\n        fabric.util.loadImage(options.source, function(img, isError) {\n          _this.source = img;\n          callback && callback(_this, isError);\n        }, null, this.crossOrigin);\n      }\n    },\n\n    /**\n     * Returns object representation of a pattern\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} Object representation of a pattern instance\n     */\n    toObject: function(propertiesToInclude) {\n      var NUM_FRACTION_DIGITS = fabric.Object.NUM_FRACTION_DIGITS,\n          source, object;\n\n      // <img> element\n      if (typeof this.source.src === 'string') {\n        source = this.source.src;\n      }\n      // <canvas> element\n      else if (typeof this.source === 'object' && this.source.toDataURL) {\n        source = this.source.toDataURL();\n      }\n\n      object = {\n        type: 'pattern',\n        source: source,\n        repeat: this.repeat,\n        crossOrigin: this.crossOrigin,\n        offsetX: toFixed(this.offsetX, NUM_FRACTION_DIGITS),\n        offsetY: toFixed(this.offsetY, NUM_FRACTION_DIGITS),\n        patternTransform: this.patternTransform ? this.patternTransform.concat() : null\n      };\n      fabric.util.populateWithProperties(this, object, propertiesToInclude);\n\n      return object;\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns SVG representation of a pattern\n     * @param {fabric.Object} object\n     * @return {String} SVG representation of a pattern\n     */\n    toSVG: function(object) {\n      var patternSource = typeof this.source === 'function' ? this.source() : this.source,\n          patternWidth = patternSource.width / object.width,\n          patternHeight = patternSource.height / object.height,\n          patternOffsetX = this.offsetX / object.width,\n          patternOffsetY = this.offsetY / object.height,\n          patternImgSrc = '';\n      if (this.repeat === 'repeat-x' || this.repeat === 'no-repeat') {\n        patternHeight = 1;\n        if (patternOffsetY) {\n          patternHeight += Math.abs(patternOffsetY);\n        }\n      }\n      if (this.repeat === 'repeat-y' || this.repeat === 'no-repeat') {\n        patternWidth = 1;\n        if (patternOffsetX) {\n          patternWidth += Math.abs(patternOffsetX);\n        }\n\n      }\n      if (patternSource.src) {\n        patternImgSrc = patternSource.src;\n      }\n      else if (patternSource.toDataURL) {\n        patternImgSrc = patternSource.toDataURL();\n      }\n\n      return '<pattern id=\"SVGID_' + this.id +\n                    '\" x=\"' + patternOffsetX +\n                    '\" y=\"' + patternOffsetY +\n                    '\" width=\"' + patternWidth +\n                    '\" height=\"' + patternHeight + '\">\\n' +\n               '<image x=\"0\" y=\"0\"' +\n                      ' width=\"' + patternSource.width +\n                      '\" height=\"' + patternSource.height +\n                      '\" xlink:href=\"' + patternImgSrc +\n               '\"></image>\\n' +\n             '</pattern>\\n';\n    },\n    /* _TO_SVG_END_ */\n\n    setOptions: function(options) {\n      for (var prop in options) {\n        this[prop] = options[prop];\n      }\n    },\n\n    /**\n     * Returns an instance of CanvasPattern\n     * @param {CanvasRenderingContext2D} ctx Context to create pattern\n     * @return {CanvasPattern}\n     */\n    toLive: function(ctx) {\n      var source = this.source;\n      // if the image failed to load, return, and allow rest to continue loading\n      if (!source) {\n        return '';\n      }\n\n      // if an image\n      if (typeof source.src !== 'undefined') {\n        if (!source.complete) {\n          return '';\n        }\n        if (source.naturalWidth === 0 || source.naturalHeight === 0) {\n          return '';\n        }\n      }\n      return ctx.createPattern(source, this.repeat);\n    }\n  });\n})();\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      toFixed = fabric.util.toFixed;\n\n  if (fabric.Shadow) {\n    fabric.warn('fabric.Shadow is already defined.');\n    return;\n  }\n\n  /**\n   * Shadow class\n   * @class fabric.Shadow\n   * @see {@link http://fabricjs.com/shadows|Shadow demo}\n   * @see {@link fabric.Shadow#initialize} for constructor definition\n   */\n  fabric.Shadow = fabric.util.createClass(/** @lends fabric.Shadow.prototype */ {\n\n    /**\n     * Shadow color\n     * @type String\n     * @default\n     */\n    color: 'rgb(0,0,0)',\n\n    /**\n     * Shadow blur\n     * @type Number\n     */\n    blur: 0,\n\n    /**\n     * Shadow horizontal offset\n     * @type Number\n     * @default\n     */\n    offsetX: 0,\n\n    /**\n     * Shadow vertical offset\n     * @type Number\n     * @default\n     */\n    offsetY: 0,\n\n    /**\n     * Whether the shadow should affect stroke operations\n     * @type Boolean\n     * @default\n     */\n    affectStroke: false,\n\n    /**\n     * Indicates whether toObject should include default values\n     * @type Boolean\n     * @default\n     */\n    includeDefaultValues: true,\n\n    /**\n     * When `false`, the shadow will scale with the object.\n     * When `true`, the shadow's offsetX, offsetY, and blur will not be affected by the object's scale.\n     * default to false\n     * @type Boolean\n     * @default\n     */\n    nonScaling: false,\n\n    /**\n     * Constructor\n     * @param {Object|String} [options] Options object with any of color, blur, offsetX, offsetY properties or string (e.g. \"rgba(0,0,0,0.2) 2px 2px 10px\")\n     * @return {fabric.Shadow} thisArg\n     */\n    initialize: function(options) {\n\n      if (typeof options === 'string') {\n        options = this._parseShadow(options);\n      }\n\n      for (var prop in options) {\n        this[prop] = options[prop];\n      }\n\n      this.id = fabric.Object.__uid++;\n    },\n\n    /**\n     * @private\n     * @param {String} shadow Shadow value to parse\n     * @return {Object} Shadow object with color, offsetX, offsetY and blur\n     */\n    _parseShadow: function(shadow) {\n      var shadowStr = shadow.trim(),\n          offsetsAndBlur = fabric.Shadow.reOffsetsAndBlur.exec(shadowStr) || [],\n          color = shadowStr.replace(fabric.Shadow.reOffsetsAndBlur, '') || 'rgb(0,0,0)';\n\n      return {\n        color: color.trim(),\n        offsetX: parseFloat(offsetsAndBlur[1], 10) || 0,\n        offsetY: parseFloat(offsetsAndBlur[2], 10) || 0,\n        blur: parseFloat(offsetsAndBlur[3], 10) || 0\n      };\n    },\n\n    /**\n     * Returns a string representation of an instance\n     * @see http://www.w3.org/TR/css-text-decor-3/#text-shadow\n     * @return {String} Returns CSS3 text-shadow declaration\n     */\n    toString: function() {\n      return [this.offsetX, this.offsetY, this.blur, this.color].join('px ');\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns SVG representation of a shadow\n     * @param {fabric.Object} object\n     * @return {String} SVG representation of a shadow\n     */\n    toSVG: function(object) {\n      var fBoxX = 40, fBoxY = 40, NUM_FRACTION_DIGITS = fabric.Object.NUM_FRACTION_DIGITS,\n          offset = fabric.util.rotateVector(\n            { x: this.offsetX, y: this.offsetY },\n            fabric.util.degreesToRadians(-object.angle)),\n          BLUR_BOX = 20, color = new fabric.Color(this.color);\n\n      if (object.width && object.height) {\n        //http://www.w3.org/TR/SVG/filters.html#FilterEffectsRegion\n        // we add some extra space to filter box to contain the blur ( 20 )\n        fBoxX = toFixed((Math.abs(offset.x) + this.blur) / object.width, NUM_FRACTION_DIGITS) * 100 + BLUR_BOX;\n        fBoxY = toFixed((Math.abs(offset.y) + this.blur) / object.height, NUM_FRACTION_DIGITS) * 100 + BLUR_BOX;\n      }\n      if (object.flipX) {\n        offset.x *= -1;\n      }\n      if (object.flipY) {\n        offset.y *= -1;\n      }\n\n      return (\n        '<filter id=\"SVGID_' + this.id + '\" y=\"-' + fBoxY + '%\" height=\"' + (100 + 2 * fBoxY) + '%\" ' +\n          'x=\"-' + fBoxX + '%\" width=\"' + (100 + 2 * fBoxX) + '%\" ' + '>\\n' +\n          '\\t<feGaussianBlur in=\"SourceAlpha\" stdDeviation=\"' +\n            toFixed(this.blur ? this.blur / 2 : 0, NUM_FRACTION_DIGITS) + '\"></feGaussianBlur>\\n' +\n          '\\t<feOffset dx=\"' + toFixed(offset.x, NUM_FRACTION_DIGITS) +\n          '\" dy=\"' + toFixed(offset.y, NUM_FRACTION_DIGITS) + '\" result=\"oBlur\" ></feOffset>\\n' +\n          '\\t<feFlood flood-color=\"' + color.toRgb() + '\" flood-opacity=\"' + color.getAlpha() + '\"/>\\n' +\n          '\\t<feComposite in2=\"oBlur\" operator=\"in\" />\\n' +\n          '\\t<feMerge>\\n' +\n            '\\t\\t<feMergeNode></feMergeNode>\\n' +\n            '\\t\\t<feMergeNode in=\"SourceGraphic\"></feMergeNode>\\n' +\n          '\\t</feMerge>\\n' +\n        '</filter>\\n');\n    },\n    /* _TO_SVG_END_ */\n\n    /**\n     * Returns object representation of a shadow\n     * @return {Object} Object representation of a shadow instance\n     */\n    toObject: function() {\n      if (this.includeDefaultValues) {\n        return {\n          color: this.color,\n          blur: this.blur,\n          offsetX: this.offsetX,\n          offsetY: this.offsetY,\n          affectStroke: this.affectStroke,\n          nonScaling: this.nonScaling\n        };\n      }\n      var obj = { }, proto = fabric.Shadow.prototype;\n\n      ['color', 'blur', 'offsetX', 'offsetY', 'affectStroke', 'nonScaling'].forEach(function(prop) {\n        if (this[prop] !== proto[prop]) {\n          obj[prop] = this[prop];\n        }\n      }, this);\n\n      return obj;\n    }\n  });\n\n  /**\n   * Regex matching shadow offsetX, offsetY and blur (ex: \"2px 2px 10px rgba(0,0,0,0.2)\", \"rgb(0,255,0) 2px 2px\")\n   * @static\n   * @field\n   * @memberOf fabric.Shadow\n   */\n  // eslint-disable-next-line max-len\n  fabric.Shadow.reOffsetsAndBlur = /(?:\\s|^)(-?\\d+(?:\\.\\d*)?(?:px)?(?:\\s?|$))?(-?\\d+(?:\\.\\d*)?(?:px)?(?:\\s?|$))?(\\d+(?:\\.\\d*)?(?:px)?)?(?:\\s?|$)(?:$|\\s)/;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function () {\n\n  'use strict';\n\n  if (fabric.StaticCanvas) {\n    fabric.warn('fabric.StaticCanvas is already defined.');\n    return;\n  }\n\n  // aliases for faster resolution\n  var extend = fabric.util.object.extend,\n      getElementOffset = fabric.util.getElementOffset,\n      removeFromArray = fabric.util.removeFromArray,\n      toFixed = fabric.util.toFixed,\n      transformPoint = fabric.util.transformPoint,\n      invertTransform = fabric.util.invertTransform,\n      getNodeCanvas = fabric.util.getNodeCanvas,\n      createCanvasElement = fabric.util.createCanvasElement,\n\n      CANVAS_INIT_ERROR = new Error('Could not initialize `canvas` element');\n\n  /**\n   * Static canvas class\n   * @class fabric.StaticCanvas\n   * @mixes fabric.Collection\n   * @mixes fabric.Observable\n   * @see {@link http://fabricjs.com/static_canvas|StaticCanvas demo}\n   * @see {@link fabric.StaticCanvas#initialize} for constructor definition\n   * @fires before:render\n   * @fires after:render\n   * @fires canvas:cleared\n   * @fires object:added\n   * @fires object:removed\n   */\n  fabric.StaticCanvas = fabric.util.createClass(fabric.CommonMethods, /** @lends fabric.StaticCanvas.prototype */ {\n\n    /**\n     * Constructor\n     * @param {HTMLElement | String} el &lt;canvas> element to initialize instance on\n     * @param {Object} [options] Options object\n     * @return {Object} thisArg\n     */\n    initialize: function(el, options) {\n      options || (options = { });\n      this.renderAndResetBound = this.renderAndReset.bind(this);\n      this.requestRenderAllBound = this.requestRenderAll.bind(this);\n      this._initStatic(el, options);\n    },\n\n    /**\n     * Background color of canvas instance.\n     * Should be set via {@link fabric.StaticCanvas#setBackgroundColor}.\n     * @type {(String|fabric.Pattern)}\n     * @default\n     */\n    backgroundColor: '',\n\n    /**\n     * Background image of canvas instance.\n     * since 2.4.0 image caching is active, please when putting an image as background, add to the\n     * canvas property a reference to the canvas it is on. Otherwise the image cannot detect the zoom\n     * vale. As an alternative you can disable image objectCaching\n     * @type fabric.Image\n     * @default\n     */\n    backgroundImage: null,\n\n    /**\n     * Overlay color of canvas instance.\n     * Should be set via {@link fabric.StaticCanvas#setOverlayColor}\n     * @since 1.3.9\n     * @type {(String|fabric.Pattern)}\n     * @default\n     */\n    overlayColor: '',\n\n    /**\n     * Overlay image of canvas instance.\n     * since 2.4.0 image caching is active, please when putting an image as overlay, add to the\n     * canvas property a reference to the canvas it is on. Otherwise the image cannot detect the zoom\n     * vale. As an alternative you can disable image objectCaching\n     * @type fabric.Image\n     * @default\n     */\n    overlayImage: null,\n\n    /**\n     * Indicates whether toObject/toDatalessObject should include default values\n     * if set to false, takes precedence over the object value.\n     * @type Boolean\n     * @default\n     */\n    includeDefaultValues: true,\n\n    /**\n     * Indicates whether objects' state should be saved\n     * @type Boolean\n     * @default\n     */\n    stateful: false,\n\n    /**\n     * Indicates whether {@link fabric.Collection.add}, {@link fabric.Collection.insertAt} and {@link fabric.Collection.remove},\n     * {@link fabric.StaticCanvas.moveTo}, {@link fabric.StaticCanvas.clear} and many more, should also re-render canvas.\n     * Disabling this option will not give a performance boost when adding/removing a lot of objects to/from canvas at once\n     * since the renders are quequed and executed one per frame.\n     * Disabling is suggested anyway and managing the renders of the app manually is not a big effort ( canvas.requestRenderAll() )\n     * Left default to true to do not break documentation and old app, fiddles.\n     * @type Boolean\n     * @default\n     */\n    renderOnAddRemove: true,\n\n    /**\n     * Indicates whether object controls (borders/controls) are rendered above overlay image\n     * @type Boolean\n     * @default\n     */\n    controlsAboveOverlay: false,\n\n    /**\n     * Indicates whether the browser can be scrolled when using a touchscreen and dragging on the canvas\n     * @type Boolean\n     * @default\n     */\n    allowTouchScrolling: false,\n\n    /**\n     * Indicates whether this canvas will use image smoothing, this is on by default in browsers\n     * @type Boolean\n     * @default\n     */\n    imageSmoothingEnabled: true,\n\n    /**\n     * The transformation (a Canvas 2D API transform matrix) which focuses the viewport\n     * @type Array\n     * @example <caption>Default transform</caption>\n     * canvas.viewportTransform = [1, 0, 0, 1, 0, 0];\n     * @example <caption>Scale by 70% and translate toward bottom-right by 50, without skewing</caption>\n     * canvas.viewportTransform = [0.7, 0, 0, 0.7, 50, 50];\n     * @default\n     */\n    viewportTransform: fabric.iMatrix.concat(),\n\n    /**\n     * if set to false background image is not affected by viewport transform\n     * @since 1.6.3\n     * @type Boolean\n     * @default\n     */\n    backgroundVpt: true,\n\n    /**\n     * if set to false overlya image is not affected by viewport transform\n     * @since 1.6.3\n     * @type Boolean\n     * @default\n     */\n    overlayVpt: true,\n\n    /**\n     * When true, canvas is scaled by devicePixelRatio for better rendering on retina screens\n     * @type Boolean\n     * @default\n     */\n    enableRetinaScaling: true,\n\n    /**\n     * Describe canvas element extension over design\n     * properties are tl,tr,bl,br.\n     * if canvas is not zoomed/panned those points are the four corner of canvas\n     * if canvas is viewportTransformed you those points indicate the extension\n     * of canvas element in plain untrasformed coordinates\n     * The coordinates get updated with @method calcViewportBoundaries.\n     * @memberOf fabric.StaticCanvas.prototype\n     */\n    vptCoords: { },\n\n    /**\n     * Based on vptCoords and object.aCoords, skip rendering of objects that\n     * are not included in current viewport.\n     * May greatly help in applications with crowded canvas and use of zoom/pan\n     * If One of the corner of the bounding box of the object is on the canvas\n     * the objects get rendered.\n     * @memberOf fabric.StaticCanvas.prototype\n     * @type Boolean\n     * @default\n     */\n    skipOffscreen: true,\n\n    /**\n     * a fabricObject that, without stroke define a clipping area with their shape. filled in black\n     * the clipPath object gets used when the canvas has rendered, and the context is placed in the\n     * top left corner of the canvas.\n     * clipPath will clip away controls, if you do not want this to happen use controlsAboveOverlay = true\n     * @type fabric.Object\n     */\n    clipPath: undefined,\n\n    /**\n     * @private\n     * @param {HTMLElement | String} el &lt;canvas> element to initialize instance on\n     * @param {Object} [options] Options object\n     */\n    _initStatic: function(el, options) {\n      var cb = this.requestRenderAllBound;\n      this._objects = [];\n      this._createLowerCanvas(el);\n      this._initOptions(options);\n      // only initialize retina scaling once\n      if (!this.interactive) {\n        this._initRetinaScaling();\n      }\n\n      if (options.overlayImage) {\n        this.setOverlayImage(options.overlayImage, cb);\n      }\n      if (options.backgroundImage) {\n        this.setBackgroundImage(options.backgroundImage, cb);\n      }\n      if (options.backgroundColor) {\n        this.setBackgroundColor(options.backgroundColor, cb);\n      }\n      if (options.overlayColor) {\n        this.setOverlayColor(options.overlayColor, cb);\n      }\n      this.calcOffset();\n    },\n\n    /**\n     * @private\n     */\n    _isRetinaScaling: function() {\n      return (fabric.devicePixelRatio > 1 && this.enableRetinaScaling);\n    },\n\n    /**\n     * @private\n     * @return {Number} retinaScaling if applied, otherwise 1;\n     */\n    getRetinaScaling: function() {\n      return this._isRetinaScaling() ? Math.max(1, fabric.devicePixelRatio) : 1;\n    },\n\n    /**\n     * @private\n     */\n    _initRetinaScaling: function() {\n      if (!this._isRetinaScaling()) {\n        return;\n      }\n      var scaleRatio = fabric.devicePixelRatio;\n      this.__initRetinaScaling(scaleRatio, this.lowerCanvasEl, this.contextContainer);\n      if (this.upperCanvasEl) {\n        this.__initRetinaScaling(scaleRatio, this.upperCanvasEl, this.contextTop);\n      }\n    },\n\n    __initRetinaScaling: function(scaleRatio, canvas, context) {\n      canvas.setAttribute('width', this.width * scaleRatio);\n      canvas.setAttribute('height', this.height * scaleRatio);\n      context.scale(scaleRatio, scaleRatio);\n    },\n\n\n    /**\n     * Calculates canvas element offset relative to the document\n     * This method is also attached as \"resize\" event handler of window\n     * @return {fabric.Canvas} instance\n     * @chainable\n     */\n    calcOffset: function () {\n      this._offset = getElementOffset(this.lowerCanvasEl);\n      return this;\n    },\n\n    /**\n     * Sets {@link fabric.StaticCanvas#overlayImage|overlay image} for this canvas\n     * @param {(fabric.Image|String)} image fabric.Image instance or URL of an image to set overlay to\n     * @param {Function} callback callback to invoke when image is loaded and set as an overlay\n     * @param {Object} [options] Optional options to set for the {@link fabric.Image|overlay image}.\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     * @see {@link http://jsfiddle.net/fabricjs/MnzHT/|jsFiddle demo}\n     * @example <caption>Normal overlayImage with left/top = 0</caption>\n     * canvas.setOverlayImage('http://fabricjs.com/assets/jail_cell_bars.png', canvas.renderAll.bind(canvas), {\n     *   // Needed to position overlayImage at 0/0\n     *   originX: 'left',\n     *   originY: 'top'\n     * });\n     * @example <caption>overlayImage with different properties</caption>\n     * canvas.setOverlayImage('http://fabricjs.com/assets/jail_cell_bars.png', canvas.renderAll.bind(canvas), {\n     *   opacity: 0.5,\n     *   angle: 45,\n     *   left: 400,\n     *   top: 400,\n     *   originX: 'left',\n     *   originY: 'top'\n     * });\n     * @example <caption>Stretched overlayImage #1 - width/height correspond to canvas width/height</caption>\n     * fabric.Image.fromURL('http://fabricjs.com/assets/jail_cell_bars.png', function(img, isError) {\n     *    img.set({width: canvas.width, height: canvas.height, originX: 'left', originY: 'top'});\n     *    canvas.setOverlayImage(img, canvas.renderAll.bind(canvas));\n     * });\n     * @example <caption>Stretched overlayImage #2 - width/height correspond to canvas width/height</caption>\n     * canvas.setOverlayImage('http://fabricjs.com/assets/jail_cell_bars.png', canvas.renderAll.bind(canvas), {\n     *   width: canvas.width,\n     *   height: canvas.height,\n     *   // Needed to position overlayImage at 0/0\n     *   originX: 'left',\n     *   originY: 'top'\n     * });\n     * @example <caption>overlayImage loaded from cross-origin</caption>\n     * canvas.setOverlayImage('http://fabricjs.com/assets/jail_cell_bars.png', canvas.renderAll.bind(canvas), {\n     *   opacity: 0.5,\n     *   angle: 45,\n     *   left: 400,\n     *   top: 400,\n     *   originX: 'left',\n     *   originY: 'top',\n     *   crossOrigin: 'anonymous'\n     * });\n     */\n    setOverlayImage: function (image, callback, options) {\n      return this.__setBgOverlayImage('overlayImage', image, callback, options);\n    },\n\n    /**\n     * Sets {@link fabric.StaticCanvas#backgroundImage|background image} for this canvas\n     * @param {(fabric.Image|String)} image fabric.Image instance or URL of an image to set background to\n     * @param {Function} callback Callback to invoke when image is loaded and set as background\n     * @param {Object} [options] Optional options to set for the {@link fabric.Image|background image}.\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     * @see {@link http://jsfiddle.net/djnr8o7a/28/|jsFiddle demo}\n     * @example <caption>Normal backgroundImage with left/top = 0</caption>\n     * canvas.setBackgroundImage('http://fabricjs.com/assets/honey_im_subtle.png', canvas.renderAll.bind(canvas), {\n     *   // Needed to position backgroundImage at 0/0\n     *   originX: 'left',\n     *   originY: 'top'\n     * });\n     * @example <caption>backgroundImage with different properties</caption>\n     * canvas.setBackgroundImage('http://fabricjs.com/assets/honey_im_subtle.png', canvas.renderAll.bind(canvas), {\n     *   opacity: 0.5,\n     *   angle: 45,\n     *   left: 400,\n     *   top: 400,\n     *   originX: 'left',\n     *   originY: 'top'\n     * });\n     * @example <caption>Stretched backgroundImage #1 - width/height correspond to canvas width/height</caption>\n     * fabric.Image.fromURL('http://fabricjs.com/assets/honey_im_subtle.png', function(img, isError) {\n     *    img.set({width: canvas.width, height: canvas.height, originX: 'left', originY: 'top'});\n     *    canvas.setBackgroundImage(img, canvas.renderAll.bind(canvas));\n     * });\n     * @example <caption>Stretched backgroundImage #2 - width/height correspond to canvas width/height</caption>\n     * canvas.setBackgroundImage('http://fabricjs.com/assets/honey_im_subtle.png', canvas.renderAll.bind(canvas), {\n     *   width: canvas.width,\n     *   height: canvas.height,\n     *   // Needed to position backgroundImage at 0/0\n     *   originX: 'left',\n     *   originY: 'top'\n     * });\n     * @example <caption>backgroundImage loaded from cross-origin</caption>\n     * canvas.setBackgroundImage('http://fabricjs.com/assets/honey_im_subtle.png', canvas.renderAll.bind(canvas), {\n     *   opacity: 0.5,\n     *   angle: 45,\n     *   left: 400,\n     *   top: 400,\n     *   originX: 'left',\n     *   originY: 'top',\n     *   crossOrigin: 'anonymous'\n     * });\n     */\n    // TODO: fix stretched examples\n    setBackgroundImage: function (image, callback, options) {\n      return this.__setBgOverlayImage('backgroundImage', image, callback, options);\n    },\n\n    /**\n     * Sets {@link fabric.StaticCanvas#overlayColor|foreground color} for this canvas\n     * @param {(String|fabric.Pattern)} overlayColor Color or pattern to set foreground color to\n     * @param {Function} callback Callback to invoke when foreground color is set\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     * @see {@link http://jsfiddle.net/fabricjs/pB55h/|jsFiddle demo}\n     * @example <caption>Normal overlayColor - color value</caption>\n     * canvas.setOverlayColor('rgba(255, 73, 64, 0.6)', canvas.renderAll.bind(canvas));\n     * @example <caption>fabric.Pattern used as overlayColor</caption>\n     * canvas.setOverlayColor({\n     *   source: 'http://fabricjs.com/assets/escheresque_ste.png'\n     * }, canvas.renderAll.bind(canvas));\n     * @example <caption>fabric.Pattern used as overlayColor with repeat and offset</caption>\n     * canvas.setOverlayColor({\n     *   source: 'http://fabricjs.com/assets/escheresque_ste.png',\n     *   repeat: 'repeat',\n     *   offsetX: 200,\n     *   offsetY: 100\n     * }, canvas.renderAll.bind(canvas));\n     */\n    setOverlayColor: function(overlayColor, callback) {\n      return this.__setBgOverlayColor('overlayColor', overlayColor, callback);\n    },\n\n    /**\n     * Sets {@link fabric.StaticCanvas#backgroundColor|background color} for this canvas\n     * @param {(String|fabric.Pattern)} backgroundColor Color or pattern to set background color to\n     * @param {Function} callback Callback to invoke when background color is set\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     * @see {@link http://jsfiddle.net/fabricjs/hXzvk/|jsFiddle demo}\n     * @example <caption>Normal backgroundColor - color value</caption>\n     * canvas.setBackgroundColor('rgba(255, 73, 64, 0.6)', canvas.renderAll.bind(canvas));\n     * @example <caption>fabric.Pattern used as backgroundColor</caption>\n     * canvas.setBackgroundColor({\n     *   source: 'http://fabricjs.com/assets/escheresque_ste.png'\n     * }, canvas.renderAll.bind(canvas));\n     * @example <caption>fabric.Pattern used as backgroundColor with repeat and offset</caption>\n     * canvas.setBackgroundColor({\n     *   source: 'http://fabricjs.com/assets/escheresque_ste.png',\n     *   repeat: 'repeat',\n     *   offsetX: 200,\n     *   offsetY: 100\n     * }, canvas.renderAll.bind(canvas));\n     */\n    setBackgroundColor: function(backgroundColor, callback) {\n      return this.__setBgOverlayColor('backgroundColor', backgroundColor, callback);\n    },\n\n    /**\n     * @private\n     * @param {String} property Property to set ({@link fabric.StaticCanvas#backgroundImage|backgroundImage}\n     * or {@link fabric.StaticCanvas#overlayImage|overlayImage})\n     * @param {(fabric.Image|String|null)} image fabric.Image instance, URL of an image or null to set background or overlay to\n     * @param {Function} callback Callback to invoke when image is loaded and set as background or overlay. The first argument is the created image, the second argument is a flag indicating whether an error occurred or not.\n     * @param {Object} [options] Optional options to set for the {@link fabric.Image|image}.\n     */\n    __setBgOverlayImage: function(property, image, callback, options) {\n      if (typeof image === 'string') {\n        fabric.util.loadImage(image, function(img, isError) {\n          if (img) {\n            var instance = new fabric.Image(img, options);\n            this[property] = instance;\n            instance.canvas = this;\n          }\n          callback && callback(img, isError);\n        }, this, options && options.crossOrigin);\n      }\n      else {\n        options && image.setOptions(options);\n        this[property] = image;\n        image && (image.canvas = this);\n        callback && callback(image, false);\n      }\n\n      return this;\n    },\n\n    /**\n     * @private\n     * @param {String} property Property to set ({@link fabric.StaticCanvas#backgroundColor|backgroundColor}\n     * or {@link fabric.StaticCanvas#overlayColor|overlayColor})\n     * @param {(Object|String|null)} color Object with pattern information, color value or null\n     * @param {Function} [callback] Callback is invoked when color is set\n     */\n    __setBgOverlayColor: function(property, color, callback) {\n      this[property] = color;\n      this._initGradient(color, property);\n      this._initPattern(color, property, callback);\n      return this;\n    },\n\n    /**\n     * @private\n     */\n    _createCanvasElement: function() {\n      var element = createCanvasElement();\n      if (!element) {\n        throw CANVAS_INIT_ERROR;\n      }\n      if (!element.style) {\n        element.style = { };\n      }\n      if (typeof element.getContext === 'undefined') {\n        throw CANVAS_INIT_ERROR;\n      }\n      return element;\n    },\n\n    /**\n     * @private\n     * @param {Object} [options] Options object\n     */\n    _initOptions: function (options) {\n      var lowerCanvasEl = this.lowerCanvasEl;\n      this._setOptions(options);\n\n      this.width = this.width || parseInt(lowerCanvasEl.width, 10) || 0;\n      this.height = this.height || parseInt(lowerCanvasEl.height, 10) || 0;\n\n      if (!this.lowerCanvasEl.style) {\n        return;\n      }\n\n      lowerCanvasEl.width = this.width;\n      lowerCanvasEl.height = this.height;\n\n      lowerCanvasEl.style.width = this.width + 'px';\n      lowerCanvasEl.style.height = this.height + 'px';\n\n      this.viewportTransform = this.viewportTransform.slice();\n    },\n\n    /**\n     * Creates a bottom canvas\n     * @private\n     * @param {HTMLElement} [canvasEl]\n     */\n    _createLowerCanvas: function (canvasEl) {\n      // canvasEl === 'HTMLCanvasElement' does not work on jsdom/node\n      if (canvasEl && canvasEl.getContext) {\n        this.lowerCanvasEl = canvasEl;\n      }\n      else {\n        this.lowerCanvasEl = fabric.util.getById(canvasEl) || this._createCanvasElement();\n      }\n\n      fabric.util.addClass(this.lowerCanvasEl, 'lower-canvas');\n      this._originalCanvasStyle = this.lowerCanvasEl.style;\n      if (this.interactive) {\n        this._applyCanvasStyle(this.lowerCanvasEl);\n      }\n\n      this.contextContainer = this.lowerCanvasEl.getContext('2d');\n    },\n\n    /**\n     * Returns canvas width (in px)\n     * @return {Number}\n     */\n    getWidth: function () {\n      return this.width;\n    },\n\n    /**\n     * Returns canvas height (in px)\n     * @return {Number}\n     */\n    getHeight: function () {\n      return this.height;\n    },\n\n    /**\n     * Sets width of this canvas instance\n     * @param {Number|String} value                         Value to set width to\n     * @param {Object}        [options]                     Options object\n     * @param {Boolean}       [options.backstoreOnly=false] Set the given dimensions only as canvas backstore dimensions\n     * @param {Boolean}       [options.cssOnly=false]       Set the given dimensions only as css dimensions\n     * @return {fabric.Canvas} instance\n     * @chainable true\n     */\n    setWidth: function (value, options) {\n      return this.setDimensions({ width: value }, options);\n    },\n\n    /**\n     * Sets height of this canvas instance\n     * @param {Number|String} value                         Value to set height to\n     * @param {Object}        [options]                     Options object\n     * @param {Boolean}       [options.backstoreOnly=false] Set the given dimensions only as canvas backstore dimensions\n     * @param {Boolean}       [options.cssOnly=false]       Set the given dimensions only as css dimensions\n     * @return {fabric.Canvas} instance\n     * @chainable true\n     */\n    setHeight: function (value, options) {\n      return this.setDimensions({ height: value }, options);\n    },\n\n    /**\n     * Sets dimensions (width, height) of this canvas instance. when options.cssOnly flag active you should also supply the unit of measure (px/%/em)\n     * @param {Object}        dimensions                    Object with width/height properties\n     * @param {Number|String} [dimensions.width]            Width of canvas element\n     * @param {Number|String} [dimensions.height]           Height of canvas element\n     * @param {Object}        [options]                     Options object\n     * @param {Boolean}       [options.backstoreOnly=false] Set the given dimensions only as canvas backstore dimensions\n     * @param {Boolean}       [options.cssOnly=false]       Set the given dimensions only as css dimensions\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    setDimensions: function (dimensions, options) {\n      var cssValue;\n\n      options = options || {};\n\n      for (var prop in dimensions) {\n        cssValue = dimensions[prop];\n\n        if (!options.cssOnly) {\n          this._setBackstoreDimension(prop, dimensions[prop]);\n          cssValue += 'px';\n          this.hasLostContext = true;\n        }\n\n        if (!options.backstoreOnly) {\n          this._setCssDimension(prop, cssValue);\n        }\n      }\n      if (this._isCurrentlyDrawing) {\n        this.freeDrawingBrush && this.freeDrawingBrush._setBrushStyles(this.contextTop);\n      }\n      this._initRetinaScaling();\n      this.calcOffset();\n\n      if (!options.cssOnly) {\n        this.requestRenderAll();\n      }\n\n      return this;\n    },\n\n    /**\n     * Helper for setting width/height\n     * @private\n     * @param {String} prop property (width|height)\n     * @param {Number} value value to set property to\n     * @return {fabric.Canvas} instance\n     * @chainable true\n     */\n    _setBackstoreDimension: function (prop, value) {\n      this.lowerCanvasEl[prop] = value;\n\n      if (this.upperCanvasEl) {\n        this.upperCanvasEl[prop] = value;\n      }\n\n      if (this.cacheCanvasEl) {\n        this.cacheCanvasEl[prop] = value;\n      }\n\n      this[prop] = value;\n\n      return this;\n    },\n\n    /**\n     * Helper for setting css width/height\n     * @private\n     * @param {String} prop property (width|height)\n     * @param {String} value value to set property to\n     * @return {fabric.Canvas} instance\n     * @chainable true\n     */\n    _setCssDimension: function (prop, value) {\n      this.lowerCanvasEl.style[prop] = value;\n\n      if (this.upperCanvasEl) {\n        this.upperCanvasEl.style[prop] = value;\n      }\n\n      if (this.wrapperEl) {\n        this.wrapperEl.style[prop] = value;\n      }\n\n      return this;\n    },\n\n    /**\n     * Returns canvas zoom level\n     * @return {Number}\n     */\n    getZoom: function () {\n      return this.viewportTransform[0];\n    },\n\n    /**\n     * Sets viewport transformation of this canvas instance\n     * @param {Array} vpt a Canvas 2D API transform matrix\n     * @return {fabric.Canvas} instance\n     * @chainable true\n     */\n    setViewportTransform: function (vpt) {\n      var activeObject = this._activeObject,\n          backgroundObject = this.backgroundImage,\n          overlayObject = this.overlayImage,\n          object, i, len;\n      this.viewportTransform = vpt;\n      for (i = 0, len = this._objects.length; i < len; i++) {\n        object = this._objects[i];\n        object.group || object.setCoords(true);\n      }\n      if (activeObject) {\n        activeObject.setCoords();\n      }\n      if (backgroundObject) {\n        backgroundObject.setCoords(true);\n      }\n      if (overlayObject) {\n        overlayObject.setCoords(true);\n      }\n      this.calcViewportBoundaries();\n      this.renderOnAddRemove && this.requestRenderAll();\n      return this;\n    },\n\n    /**\n     * Sets zoom level of this canvas instance, the zoom centered around point\n     * meaning that following zoom to point with the same point will have the visual\n     * effect of the zoom originating from that point. The point won't move.\n     * It has nothing to do with canvas center or visual center of the viewport.\n     * @param {fabric.Point} point to zoom with respect to\n     * @param {Number} value to set zoom to, less than 1 zooms out\n     * @return {fabric.Canvas} instance\n     * @chainable true\n     */\n    zoomToPoint: function (point, value) {\n      // TODO: just change the scale, preserve other transformations\n      var before = point, vpt = this.viewportTransform.slice(0);\n      point = transformPoint(point, invertTransform(this.viewportTransform));\n      vpt[0] = value;\n      vpt[3] = value;\n      var after = transformPoint(point, vpt);\n      vpt[4] += before.x - after.x;\n      vpt[5] += before.y - after.y;\n      return this.setViewportTransform(vpt);\n    },\n\n    /**\n     * Sets zoom level of this canvas instance\n     * @param {Number} value to set zoom to, less than 1 zooms out\n     * @return {fabric.Canvas} instance\n     * @chainable true\n     */\n    setZoom: function (value) {\n      this.zoomToPoint(new fabric.Point(0, 0), value);\n      return this;\n    },\n\n    /**\n     * Pan viewport so as to place point at top left corner of canvas\n     * @param {fabric.Point} point to move to\n     * @return {fabric.Canvas} instance\n     * @chainable true\n     */\n    absolutePan: function (point) {\n      var vpt = this.viewportTransform.slice(0);\n      vpt[4] = -point.x;\n      vpt[5] = -point.y;\n      return this.setViewportTransform(vpt);\n    },\n\n    /**\n     * Pans viewpoint relatively\n     * @param {fabric.Point} point (position vector) to move by\n     * @return {fabric.Canvas} instance\n     * @chainable true\n     */\n    relativePan: function (point) {\n      return this.absolutePan(new fabric.Point(\n        -point.x - this.viewportTransform[4],\n        -point.y - this.viewportTransform[5]\n      ));\n    },\n\n    /**\n     * Returns &lt;canvas> element corresponding to this instance\n     * @return {HTMLCanvasElement}\n     */\n    getElement: function () {\n      return this.lowerCanvasEl;\n    },\n\n    /**\n     * @private\n     * @param {fabric.Object} obj Object that was added\n     */\n    _onObjectAdded: function(obj) {\n      this.stateful && obj.setupState();\n      obj._set('canvas', this);\n      obj.setCoords();\n      this.fire('object:added', { target: obj });\n      obj.fire('added');\n    },\n\n    /**\n     * @private\n     * @param {fabric.Object} obj Object that was removed\n     */\n    _onObjectRemoved: function(obj) {\n      this.fire('object:removed', { target: obj });\n      obj.fire('removed');\n      delete obj.canvas;\n    },\n\n    /**\n     * Clears specified context of canvas element\n     * @param {CanvasRenderingContext2D} ctx Context to clear\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    clearContext: function(ctx) {\n      ctx.clearRect(0, 0, this.width, this.height);\n      return this;\n    },\n\n    /**\n     * Returns context of canvas where objects are drawn\n     * @return {CanvasRenderingContext2D}\n     */\n    getContext: function () {\n      return this.contextContainer;\n    },\n\n    /**\n     * Clears all contexts (background, main, top) of an instance\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    clear: function () {\n      this.remove.apply(this, this.getObjects());\n      this.backgroundImage = null;\n      this.overlayImage = null;\n      this.backgroundColor = '';\n      this.overlayColor = '';\n      if (this._hasITextHandlers) {\n        this.off('mouse:up', this._mouseUpITextHandler);\n        this._iTextInstances = null;\n        this._hasITextHandlers = false;\n      }\n      this.clearContext(this.contextContainer);\n      this.fire('canvas:cleared');\n      this.renderOnAddRemove && this.requestRenderAll();\n      return this;\n    },\n\n    /**\n     * Renders the canvas\n     * @return {fabric.Canvas} instance\n     * @chainable\n     */\n    renderAll: function () {\n      var canvasToDrawOn = this.contextContainer;\n      this.renderCanvas(canvasToDrawOn, this._objects);\n      return this;\n    },\n\n    /**\n     * Function created to be instance bound at initialization\n     * used in requestAnimationFrame rendering\n     * Let the fabricJS call it. If you call it manually you could have more\n     * animationFrame stacking on to of each other\n     * for an imperative rendering, use canvas.renderAll\n     * @private\n     * @return {fabric.Canvas} instance\n     * @chainable\n     */\n    renderAndReset: function() {\n      this.isRendering = 0;\n      this.renderAll();\n    },\n\n    /**\n     * Append a renderAll request to next animation frame.\n     * unless one is already in progress, in that case nothing is done\n     * a boolean flag will avoid appending more.\n     * @return {fabric.Canvas} instance\n     * @chainable\n     */\n    requestRenderAll: function () {\n      if (!this.isRendering) {\n        this.isRendering = fabric.util.requestAnimFrame(this.renderAndResetBound);\n      }\n      return this;\n    },\n\n    /**\n     * Calculate the position of the 4 corner of canvas with current viewportTransform.\n     * helps to determinate when an object is in the current rendering viewport using\n     * object absolute coordinates ( aCoords )\n     * @return {Object} points.tl\n     * @chainable\n     */\n    calcViewportBoundaries: function() {\n      var points = { }, width = this.width, height = this.height,\n          iVpt = invertTransform(this.viewportTransform);\n      points.tl = transformPoint({ x: 0, y: 0 }, iVpt);\n      points.br = transformPoint({ x: width, y: height }, iVpt);\n      points.tr = new fabric.Point(points.br.x, points.tl.y);\n      points.bl = new fabric.Point(points.tl.x, points.br.y);\n      this.vptCoords = points;\n      return points;\n    },\n\n    cancelRequestedRender: function() {\n      if (this.isRendering) {\n        fabric.util.cancelAnimFrame(this.isRendering);\n        this.isRendering = 0;\n      }\n    },\n\n    /**\n     * Renders background, objects, overlay and controls.\n     * @param {CanvasRenderingContext2D} ctx\n     * @param {Array} objects to render\n     * @return {fabric.Canvas} instance\n     * @chainable\n     */\n    renderCanvas: function(ctx, objects) {\n      var v = this.viewportTransform, path = this.clipPath;\n      this.cancelRequestedRender();\n      this.calcViewportBoundaries();\n      this.clearContext(ctx);\n      fabric.util.setImageSmoothing(ctx, this.imageSmoothingEnabled);\n      this.fire('before:render', { ctx: ctx, });\n      this._renderBackground(ctx);\n\n      ctx.save();\n      //apply viewport transform once for all rendering process\n      ctx.transform(v[0], v[1], v[2], v[3], v[4], v[5]);\n      this._renderObjects(ctx, objects);\n      ctx.restore();\n      if (!this.controlsAboveOverlay && this.interactive) {\n        this.drawControls(ctx);\n      }\n      if (path) {\n        path.canvas = this;\n        // needed to setup a couple of variables\n        path.shouldCache();\n        path._transformDone = true;\n        path.renderCache({ forClipping: true });\n        this.drawClipPathOnCanvas(ctx);\n      }\n      this._renderOverlay(ctx);\n      if (this.controlsAboveOverlay && this.interactive) {\n        this.drawControls(ctx);\n      }\n      this.fire('after:render', { ctx: ctx, });\n    },\n\n    /**\n     * Paint the cached clipPath on the lowerCanvasEl\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    drawClipPathOnCanvas: function(ctx) {\n      var v = this.viewportTransform, path = this.clipPath;\n      ctx.save();\n      ctx.transform(v[0], v[1], v[2], v[3], v[4], v[5]);\n      // DEBUG: uncomment this line, comment the following\n      // ctx.globalAlpha = 0.4;\n      ctx.globalCompositeOperation = 'destination-in';\n      path.transform(ctx);\n      ctx.scale(1 / path.zoomX, 1 / path.zoomY);\n      ctx.drawImage(path._cacheCanvas, -path.cacheTranslationX, -path.cacheTranslationY);\n      ctx.restore();\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {Array} objects to render\n     */\n    _renderObjects: function(ctx, objects) {\n      var i, len;\n      for (i = 0, len = objects.length; i < len; ++i) {\n        objects[i] && objects[i].render(ctx);\n      }\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {string} property 'background' or 'overlay'\n     */\n    _renderBackgroundOrOverlay: function(ctx, property) {\n      var fill = this[property + 'Color'], object = this[property + 'Image'],\n          v = this.viewportTransform, needsVpt = this[property + 'Vpt'];\n      if (!fill && !object) {\n        return;\n      }\n      if (fill) {\n        ctx.save();\n        ctx.beginPath();\n        ctx.moveTo(0, 0);\n        ctx.lineTo(this.width, 0);\n        ctx.lineTo(this.width, this.height);\n        ctx.lineTo(0, this.height);\n        ctx.closePath();\n        ctx.fillStyle = fill.toLive\n          ? fill.toLive(ctx, this)\n          : fill;\n        if (needsVpt) {\n          ctx.transform(v[0], v[1], v[2], v[3], v[4], v[5]);\n        }\n        ctx.transform(1, 0, 0, 1, fill.offsetX || 0, fill.offsetY || 0);\n        var m = fill.gradientTransform || fill.patternTransform;\n        m && ctx.transform(m[0], m[1], m[2], m[3], m[4], m[5]);\n        ctx.fill();\n        ctx.restore();\n      }\n      if (object) {\n        ctx.save();\n        if (needsVpt) {\n          ctx.transform(v[0], v[1], v[2], v[3], v[4], v[5]);\n        }\n        object.render(ctx);\n        ctx.restore();\n      }\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderBackground: function(ctx) {\n      this._renderBackgroundOrOverlay(ctx, 'background');\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderOverlay: function(ctx) {\n      this._renderBackgroundOrOverlay(ctx, 'overlay');\n    },\n\n    /**\n     * Returns coordinates of a center of canvas.\n     * Returned value is an object with top and left properties\n     * @return {Object} object with \"top\" and \"left\" number values\n     * @deprecated migrate to `getCenterPoint`\n     */\n    getCenter: function () {\n      return {\n        top: this.height / 2,\n        left: this.width / 2\n      };\n    },\n\n    /**\n     * Returns coordinates of a center of canvas.\n     * @return {fabric.Point}\n     */\n    getCenterPoint: function () {\n      return new fabric.Point(this.width / 2, this.height / 2);\n    },\n\n    /**\n     * Centers object horizontally in the canvas\n     * @param {fabric.Object} object Object to center horizontally\n     * @return {fabric.Canvas} thisArg\n     */\n    centerObjectH: function (object) {\n      return this._centerObject(object, new fabric.Point(this.getCenterPoint().x, object.getCenterPoint().y));\n    },\n\n    /**\n     * Centers object vertically in the canvas\n     * @param {fabric.Object} object Object to center vertically\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    centerObjectV: function (object) {\n      return this._centerObject(object, new fabric.Point(object.getCenterPoint().x, this.getCenterPoint().y));\n    },\n\n    /**\n     * Centers object vertically and horizontally in the canvas\n     * @param {fabric.Object} object Object to center vertically and horizontally\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    centerObject: function(object) {\n      var center = this.getCenterPoint();\n      return this._centerObject(object, center);\n    },\n\n    /**\n     * Centers object vertically and horizontally in the viewport\n     * @param {fabric.Object} object Object to center vertically and horizontally\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    viewportCenterObject: function(object) {\n      var vpCenter = this.getVpCenter();\n      return this._centerObject(object, vpCenter);\n    },\n\n    /**\n     * Centers object horizontally in the viewport, object.top is unchanged\n     * @param {fabric.Object} object Object to center vertically and horizontally\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    viewportCenterObjectH: function(object) {\n      var vpCenter = this.getVpCenter();\n      this._centerObject(object, new fabric.Point(vpCenter.x, object.getCenterPoint().y));\n      return this;\n    },\n\n    /**\n     * Centers object Vertically in the viewport, object.top is unchanged\n     * @param {fabric.Object} object Object to center vertically and horizontally\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    viewportCenterObjectV: function(object) {\n      var vpCenter = this.getVpCenter();\n\n      return this._centerObject(object, new fabric.Point(object.getCenterPoint().x, vpCenter.y));\n    },\n\n    /**\n     * Calculate the point in canvas that correspond to the center of actual viewport.\n     * @return {fabric.Point} vpCenter, viewport center\n     * @chainable\n     */\n    getVpCenter: function() {\n      var center = this.getCenterPoint(),\n          iVpt = invertTransform(this.viewportTransform);\n      return transformPoint(center, iVpt);\n    },\n\n    /**\n     * @private\n     * @param {fabric.Object} object Object to center\n     * @param {fabric.Point} center Center point\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    _centerObject: function(object, center) {\n      object.setPositionByOrigin(center, 'center', 'center');\n      object.setCoords();\n      this.renderOnAddRemove && this.requestRenderAll();\n      return this;\n    },\n\n    /**\n     * Returns dataless JSON representation of canvas\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {String} json string\n     */\n    toDatalessJSON: function (propertiesToInclude) {\n      return this.toDatalessObject(propertiesToInclude);\n    },\n\n    /**\n     * Returns object representation of canvas\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toObject: function (propertiesToInclude) {\n      return this._toObjectMethod('toObject', propertiesToInclude);\n    },\n\n    /**\n     * Returns dataless object representation of canvas\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toDatalessObject: function (propertiesToInclude) {\n      return this._toObjectMethod('toDatalessObject', propertiesToInclude);\n    },\n\n    /**\n     * @private\n     */\n    _toObjectMethod: function (methodName, propertiesToInclude) {\n\n      var clipPath = this.clipPath, data = {\n        version: fabric.version,\n        objects: this._toObjects(methodName, propertiesToInclude),\n      };\n      if (clipPath && !clipPath.excludeFromExport) {\n        data.clipPath = this._toObject(this.clipPath, methodName, propertiesToInclude);\n      }\n      extend(data, this.__serializeBgOverlay(methodName, propertiesToInclude));\n\n      fabric.util.populateWithProperties(this, data, propertiesToInclude);\n\n      return data;\n    },\n\n    /**\n     * @private\n     */\n    _toObjects: function(methodName, propertiesToInclude) {\n      return this._objects.filter(function(object) {\n        return !object.excludeFromExport;\n      }).map(function(instance) {\n        return this._toObject(instance, methodName, propertiesToInclude);\n      }, this);\n    },\n\n    /**\n     * @private\n     */\n    _toObject: function(instance, methodName, propertiesToInclude) {\n      var originalValue;\n\n      if (!this.includeDefaultValues) {\n        originalValue = instance.includeDefaultValues;\n        instance.includeDefaultValues = false;\n      }\n\n      var object = instance[methodName](propertiesToInclude);\n      if (!this.includeDefaultValues) {\n        instance.includeDefaultValues = originalValue;\n      }\n      return object;\n    },\n\n    /**\n     * @private\n     */\n    __serializeBgOverlay: function(methodName, propertiesToInclude) {\n      var data = {}, bgImage = this.backgroundImage, overlayImage = this.overlayImage,\n          bgColor = this.backgroundColor, overlayColor = this.overlayColor;\n\n      if (bgColor && bgColor.toObject) {\n        if (!bgColor.excludeFromExport) {\n          data.background = bgColor.toObject(propertiesToInclude);\n        }\n      }\n      else if (bgColor) {\n        data.background = bgColor;\n      }\n\n      if (overlayColor && overlayColor.toObject) {\n        if (!overlayColor.excludeFromExport) {\n          data.overlay = overlayColor.toObject(propertiesToInclude);\n        }\n      }\n      else if (overlayColor) {\n        data.overlay = overlayColor;\n      }\n\n      if (bgImage && !bgImage.excludeFromExport) {\n        data.backgroundImage = this._toObject(bgImage, methodName, propertiesToInclude);\n      }\n      if (overlayImage && !overlayImage.excludeFromExport) {\n        data.overlayImage = this._toObject(overlayImage, methodName, propertiesToInclude);\n      }\n\n      return data;\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * When true, getSvgTransform() will apply the StaticCanvas.viewportTransform to the SVG transformation. When true,\n     * a zoomed canvas will then produce zoomed SVG output.\n     * @type Boolean\n     * @default\n     */\n    svgViewportTransformation: true,\n\n    /**\n     * Returns SVG representation of canvas\n     * @function\n     * @param {Object} [options] Options object for SVG output\n     * @param {Boolean} [options.suppressPreamble=false] If true xml tag is not included\n     * @param {Object} [options.viewBox] SVG viewbox object\n     * @param {Number} [options.viewBox.x] x-coordinate of viewbox\n     * @param {Number} [options.viewBox.y] y-coordinate of viewbox\n     * @param {Number} [options.viewBox.width] Width of viewbox\n     * @param {Number} [options.viewBox.height] Height of viewbox\n     * @param {String} [options.encoding=UTF-8] Encoding of SVG output\n     * @param {String} [options.width] desired width of svg with or without units\n     * @param {String} [options.height] desired height of svg with or without units\n     * @param {Function} [reviver] Method for further parsing of svg elements, called after each fabric object converted into svg representation.\n     * @return {String} SVG string\n     * @tutorial {@link http://fabricjs.com/fabric-intro-part-3#serialization}\n     * @see {@link http://jsfiddle.net/fabricjs/jQ3ZZ/|jsFiddle demo}\n     * @example <caption>Normal SVG output</caption>\n     * var svg = canvas.toSVG();\n     * @example <caption>SVG output without preamble (without &lt;?xml ../>)</caption>\n     * var svg = canvas.toSVG({suppressPreamble: true});\n     * @example <caption>SVG output with viewBox attribute</caption>\n     * var svg = canvas.toSVG({\n     *   viewBox: {\n     *     x: 100,\n     *     y: 100,\n     *     width: 200,\n     *     height: 300\n     *   }\n     * });\n     * @example <caption>SVG output with different encoding (default: UTF-8)</caption>\n     * var svg = canvas.toSVG({encoding: 'ISO-8859-1'});\n     * @example <caption>Modify SVG output with reviver function</caption>\n     * var svg = canvas.toSVG(null, function(svg) {\n     *   return svg.replace('stroke-dasharray: ; stroke-linecap: butt; stroke-linejoin: miter; stroke-miterlimit: 10; ', '');\n     * });\n     */\n    toSVG: function(options, reviver) {\n      options || (options = { });\n      options.reviver = reviver;\n      var markup = [];\n\n      this._setSVGPreamble(markup, options);\n      this._setSVGHeader(markup, options);\n      if (this.clipPath) {\n        markup.push('<g clip-path=\"url(#' + this.clipPath.clipPathId + ')\" >\\n');\n      }\n      this._setSVGBgOverlayColor(markup, 'background');\n      this._setSVGBgOverlayImage(markup, 'backgroundImage', reviver);\n      this._setSVGObjects(markup, reviver);\n      if (this.clipPath) {\n        markup.push('</g>\\n');\n      }\n      this._setSVGBgOverlayColor(markup, 'overlay');\n      this._setSVGBgOverlayImage(markup, 'overlayImage', reviver);\n\n      markup.push('</svg>');\n\n      return markup.join('');\n    },\n\n    /**\n     * @private\n     */\n    _setSVGPreamble: function(markup, options) {\n      if (options.suppressPreamble) {\n        return;\n      }\n      markup.push(\n        '<?xml version=\"1.0\" encoding=\"', (options.encoding || 'UTF-8'), '\" standalone=\"no\" ?>\\n',\n        '<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\" ',\n        '\"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">\\n'\n      );\n    },\n\n    /**\n     * @private\n     */\n    _setSVGHeader: function(markup, options) {\n      var width = options.width || this.width,\n          height = options.height || this.height,\n          vpt, viewBox = 'viewBox=\"0 0 ' + this.width + ' ' + this.height + '\" ',\n          NUM_FRACTION_DIGITS = fabric.Object.NUM_FRACTION_DIGITS;\n\n      if (options.viewBox) {\n        viewBox = 'viewBox=\"' +\n                options.viewBox.x + ' ' +\n                options.viewBox.y + ' ' +\n                options.viewBox.width + ' ' +\n                options.viewBox.height + '\" ';\n      }\n      else {\n        if (this.svgViewportTransformation) {\n          vpt = this.viewportTransform;\n          viewBox = 'viewBox=\"' +\n                  toFixed(-vpt[4] / vpt[0], NUM_FRACTION_DIGITS) + ' ' +\n                  toFixed(-vpt[5] / vpt[3], NUM_FRACTION_DIGITS) + ' ' +\n                  toFixed(this.width / vpt[0], NUM_FRACTION_DIGITS) + ' ' +\n                  toFixed(this.height / vpt[3], NUM_FRACTION_DIGITS) + '\" ';\n        }\n      }\n\n      markup.push(\n        '<svg ',\n        'xmlns=\"http://www.w3.org/2000/svg\" ',\n        'xmlns:xlink=\"http://www.w3.org/1999/xlink\" ',\n        'version=\"1.1\" ',\n        'width=\"', width, '\" ',\n        'height=\"', height, '\" ',\n        viewBox,\n        'xml:space=\"preserve\">\\n',\n        '<desc>Created with Fabric.js ', fabric.version, '</desc>\\n',\n        '<defs>\\n',\n        this.createSVGFontFacesMarkup(),\n        this.createSVGRefElementsMarkup(),\n        this.createSVGClipPathMarkup(options),\n        '</defs>\\n'\n      );\n    },\n\n    createSVGClipPathMarkup: function(options) {\n      var clipPath = this.clipPath;\n      if (clipPath) {\n        clipPath.clipPathId = 'CLIPPATH_' + fabric.Object.__uid++;\n        return  '<clipPath id=\"' + clipPath.clipPathId + '\" >\\n' +\n          this.clipPath.toClipPathSVG(options.reviver) +\n          '</clipPath>\\n';\n      }\n      return '';\n    },\n\n    /**\n     * Creates markup containing SVG referenced elements like patterns, gradients etc.\n     * @return {String}\n     */\n    createSVGRefElementsMarkup: function() {\n      var _this = this,\n          markup = ['background', 'overlay'].map(function(prop) {\n            var fill = _this[prop + 'Color'];\n            if (fill && fill.toLive) {\n              var shouldTransform = _this[prop + 'Vpt'], vpt = _this.viewportTransform,\n                  object = {\n                    width: _this.width / (shouldTransform ? vpt[0] : 1),\n                    height: _this.height / (shouldTransform ? vpt[3] : 1)\n                  };\n              return fill.toSVG(\n                object,\n                { additionalTransform: shouldTransform ? fabric.util.matrixToSVG(vpt) : '' }\n              );\n            }\n          });\n      return markup.join('');\n    },\n\n    /**\n     * Creates markup containing SVG font faces,\n     * font URLs for font faces must be collected by developers\n     * and are not extracted from the DOM by fabricjs\n     * @param {Array} objects Array of fabric objects\n     * @return {String}\n     */\n    createSVGFontFacesMarkup: function() {\n      var markup = '', fontList = { }, obj, fontFamily,\n          style, row, rowIndex, _char, charIndex, i, len,\n          fontPaths = fabric.fontPaths, objects = [];\n\n      this._objects.forEach(function add(object) {\n        objects.push(object);\n        if (object._objects) {\n          object._objects.forEach(add);\n        }\n      });\n\n      for (i = 0, len = objects.length; i < len; i++) {\n        obj = objects[i];\n        fontFamily = obj.fontFamily;\n        if (obj.type.indexOf('text') === -1 || fontList[fontFamily] || !fontPaths[fontFamily]) {\n          continue;\n        }\n        fontList[fontFamily] = true;\n        if (!obj.styles) {\n          continue;\n        }\n        style = obj.styles;\n        for (rowIndex in style) {\n          row = style[rowIndex];\n          for (charIndex in row) {\n            _char = row[charIndex];\n            fontFamily = _char.fontFamily;\n            if (!fontList[fontFamily] && fontPaths[fontFamily]) {\n              fontList[fontFamily] = true;\n            }\n          }\n        }\n      }\n\n      for (var j in fontList) {\n        markup += [\n          '\\t\\t@font-face {\\n',\n          '\\t\\t\\tfont-family: \\'', j, '\\';\\n',\n          '\\t\\t\\tsrc: url(\\'', fontPaths[j], '\\');\\n',\n          '\\t\\t}\\n'\n        ].join('');\n      }\n\n      if (markup) {\n        markup = [\n          '\\t<style type=\"text/css\">',\n          '<![CDATA[\\n',\n          markup,\n          ']]>',\n          '</style>\\n'\n        ].join('');\n      }\n\n      return markup;\n    },\n\n    /**\n     * @private\n     */\n    _setSVGObjects: function(markup, reviver) {\n      var instance, i, len, objects = this._objects;\n      for (i = 0, len = objects.length; i < len; i++) {\n        instance = objects[i];\n        if (instance.excludeFromExport) {\n          continue;\n        }\n        this._setSVGObject(markup, instance, reviver);\n      }\n    },\n\n    /**\n     * @private\n     */\n    _setSVGObject: function(markup, instance, reviver) {\n      markup.push(instance.toSVG(reviver));\n    },\n\n    /**\n     * @private\n     */\n    _setSVGBgOverlayImage: function(markup, property, reviver) {\n      if (this[property] && !this[property].excludeFromExport && this[property].toSVG) {\n        markup.push(this[property].toSVG(reviver));\n      }\n    },\n\n    /**\n     * @private\n     */\n    _setSVGBgOverlayColor: function(markup, property) {\n      var filler = this[property + 'Color'], vpt = this.viewportTransform, finalWidth = this.width,\n          finalHeight = this.height;\n      if (!filler) {\n        return;\n      }\n      if (filler.toLive) {\n        var repeat = filler.repeat, iVpt = fabric.util.invertTransform(vpt), shouldInvert = this[property + 'Vpt'],\n            additionalTransform = shouldInvert ? fabric.util.matrixToSVG(iVpt) : '';\n        markup.push(\n          '<rect transform=\"' + additionalTransform + ' translate(', finalWidth / 2, ',', finalHeight / 2, ')\"',\n          ' x=\"', filler.offsetX - finalWidth / 2,\n          '\" y=\"', filler.offsetY - finalHeight / 2, '\" ',\n          'width=\"',\n          (repeat === 'repeat-y' || repeat === 'no-repeat'\n            ? filler.source.width\n            : finalWidth ),\n          '\" height=\"',\n          (repeat === 'repeat-x' || repeat === 'no-repeat'\n            ? filler.source.height\n            : finalHeight),\n          '\" fill=\"url(#SVGID_' + filler.id + ')\"',\n          '></rect>\\n'\n        );\n      }\n      else {\n        markup.push(\n          '<rect x=\"0\" y=\"0\" width=\"100%\" height=\"100%\" ',\n          'fill=\"', filler, '\"',\n          '></rect>\\n'\n        );\n      }\n    },\n    /* _TO_SVG_END_ */\n\n    /**\n     * Moves an object or the objects of a multiple selection\n     * to the bottom of the stack of drawn objects\n     * @param {fabric.Object} object Object to send to back\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    sendToBack: function (object) {\n      if (!object) {\n        return this;\n      }\n      var activeSelection = this._activeObject,\n          i, obj, objs;\n      if (object === activeSelection && object.type === 'activeSelection') {\n        objs = activeSelection._objects;\n        for (i = objs.length; i--;) {\n          obj = objs[i];\n          removeFromArray(this._objects, obj);\n          this._objects.unshift(obj);\n        }\n      }\n      else {\n        removeFromArray(this._objects, object);\n        this._objects.unshift(object);\n      }\n      this.renderOnAddRemove && this.requestRenderAll();\n      return this;\n    },\n\n    /**\n     * Moves an object or the objects of a multiple selection\n     * to the top of the stack of drawn objects\n     * @param {fabric.Object} object Object to send\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    bringToFront: function (object) {\n      if (!object) {\n        return this;\n      }\n      var activeSelection = this._activeObject,\n          i, obj, objs;\n      if (object === activeSelection && object.type === 'activeSelection') {\n        objs = activeSelection._objects;\n        for (i = 0; i < objs.length; i++) {\n          obj = objs[i];\n          removeFromArray(this._objects, obj);\n          this._objects.push(obj);\n        }\n      }\n      else {\n        removeFromArray(this._objects, object);\n        this._objects.push(object);\n      }\n      this.renderOnAddRemove && this.requestRenderAll();\n      return this;\n    },\n\n    /**\n     * Moves an object or a selection down in stack of drawn objects\n     * An optional parameter, intersecting allows to move the object in behind\n     * the first intersecting object. Where intersection is calculated with\n     * bounding box. If no intersection is found, there will not be change in the\n     * stack.\n     * @param {fabric.Object} object Object to send\n     * @param {Boolean} [intersecting] If `true`, send object behind next lower intersecting object\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    sendBackwards: function (object, intersecting) {\n      if (!object) {\n        return this;\n      }\n      var activeSelection = this._activeObject,\n          i, obj, idx, newIdx, objs, objsMoved = 0;\n\n      if (object === activeSelection && object.type === 'activeSelection') {\n        objs = activeSelection._objects;\n        for (i = 0; i < objs.length; i++) {\n          obj = objs[i];\n          idx = this._objects.indexOf(obj);\n          if (idx > 0 + objsMoved) {\n            newIdx = idx - 1;\n            removeFromArray(this._objects, obj);\n            this._objects.splice(newIdx, 0, obj);\n          }\n          objsMoved++;\n        }\n      }\n      else {\n        idx = this._objects.indexOf(object);\n        if (idx !== 0) {\n          // if object is not on the bottom of stack\n          newIdx = this._findNewLowerIndex(object, idx, intersecting);\n          removeFromArray(this._objects, object);\n          this._objects.splice(newIdx, 0, object);\n        }\n      }\n      this.renderOnAddRemove && this.requestRenderAll();\n      return this;\n    },\n\n    /**\n     * @private\n     */\n    _findNewLowerIndex: function(object, idx, intersecting) {\n      var newIdx, i;\n\n      if (intersecting) {\n        newIdx = idx;\n\n        // traverse down the stack looking for the nearest intersecting object\n        for (i = idx - 1; i >= 0; --i) {\n\n          var isIntersecting = object.intersectsWithObject(this._objects[i]) ||\n                               object.isContainedWithinObject(this._objects[i]) ||\n                               this._objects[i].isContainedWithinObject(object);\n\n          if (isIntersecting) {\n            newIdx = i;\n            break;\n          }\n        }\n      }\n      else {\n        newIdx = idx - 1;\n      }\n\n      return newIdx;\n    },\n\n    /**\n     * Moves an object or a selection up in stack of drawn objects\n     * An optional parameter, intersecting allows to move the object in front\n     * of the first intersecting object. Where intersection is calculated with\n     * bounding box. If no intersection is found, there will not be change in the\n     * stack.\n     * @param {fabric.Object} object Object to send\n     * @param {Boolean} [intersecting] If `true`, send object in front of next upper intersecting object\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    bringForward: function (object, intersecting) {\n      if (!object) {\n        return this;\n      }\n      var activeSelection = this._activeObject,\n          i, obj, idx, newIdx, objs, objsMoved = 0;\n\n      if (object === activeSelection && object.type === 'activeSelection') {\n        objs = activeSelection._objects;\n        for (i = objs.length; i--;) {\n          obj = objs[i];\n          idx = this._objects.indexOf(obj);\n          if (idx < this._objects.length - 1 - objsMoved) {\n            newIdx = idx + 1;\n            removeFromArray(this._objects, obj);\n            this._objects.splice(newIdx, 0, obj);\n          }\n          objsMoved++;\n        }\n      }\n      else {\n        idx = this._objects.indexOf(object);\n        if (idx !== this._objects.length - 1) {\n          // if object is not on top of stack (last item in an array)\n          newIdx = this._findNewUpperIndex(object, idx, intersecting);\n          removeFromArray(this._objects, object);\n          this._objects.splice(newIdx, 0, object);\n        }\n      }\n      this.renderOnAddRemove && this.requestRenderAll();\n      return this;\n    },\n\n    /**\n     * @private\n     */\n    _findNewUpperIndex: function(object, idx, intersecting) {\n      var newIdx, i, len;\n\n      if (intersecting) {\n        newIdx = idx;\n\n        // traverse up the stack looking for the nearest intersecting object\n        for (i = idx + 1, len = this._objects.length; i < len; ++i) {\n\n          var isIntersecting = object.intersectsWithObject(this._objects[i]) ||\n                               object.isContainedWithinObject(this._objects[i]) ||\n                               this._objects[i].isContainedWithinObject(object);\n\n          if (isIntersecting) {\n            newIdx = i;\n            break;\n          }\n        }\n      }\n      else {\n        newIdx = idx + 1;\n      }\n\n      return newIdx;\n    },\n\n    /**\n     * Moves an object to specified level in stack of drawn objects\n     * @param {fabric.Object} object Object to send\n     * @param {Number} index Position to move to\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    moveTo: function (object, index) {\n      removeFromArray(this._objects, object);\n      this._objects.splice(index, 0, object);\n      return this.renderOnAddRemove && this.requestRenderAll();\n    },\n\n    /**\n     * Clears a canvas element and dispose objects\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    dispose: function () {\n      // cancel eventually ongoing renders\n      if (this.isRendering) {\n        fabric.util.cancelAnimFrame(this.isRendering);\n        this.isRendering = 0;\n      }\n      this.forEachObject(function(object) {\n        object.dispose && object.dispose();\n      });\n      this._objects = [];\n      if (this.backgroundImage && this.backgroundImage.dispose) {\n        this.backgroundImage.dispose();\n      }\n      this.backgroundImage = null;\n      if (this.overlayImage && this.overlayImage.dispose) {\n        this.overlayImage.dispose();\n      }\n      this.overlayImage = null;\n      this._iTextInstances = null;\n      this.contextContainer = null;\n      // restore canvas style\n      this.lowerCanvasEl.classList.remove('lower-canvas');\n      fabric.util.setStyle(this.lowerCanvasEl, this._originalCanvasStyle);\n      delete this._originalCanvasStyle;\n      // restore canvas size to original size in case retina scaling was applied\n      this.lowerCanvasEl.setAttribute('width', this.width);\n      this.lowerCanvasEl.setAttribute('height', this.height);\n      fabric.util.cleanUpJsdomNode(this.lowerCanvasEl);\n      this.lowerCanvasEl = undefined;\n      return this;\n    },\n\n    /**\n     * Returns a string representation of an instance\n     * @return {String} string representation of an instance\n     */\n    toString: function () {\n      return '#<fabric.Canvas (' + this.complexity() + '): ' +\n               '{ objects: ' + this._objects.length + ' }>';\n    }\n  });\n\n  extend(fabric.StaticCanvas.prototype, fabric.Observable);\n  extend(fabric.StaticCanvas.prototype, fabric.Collection);\n  extend(fabric.StaticCanvas.prototype, fabric.DataURLExporter);\n\n  extend(fabric.StaticCanvas, /** @lends fabric.StaticCanvas */ {\n\n    /**\n     * @static\n     * @type String\n     * @default\n     */\n    EMPTY_JSON: '{\"objects\": [], \"background\": \"white\"}',\n\n    /**\n     * Provides a way to check support of some of the canvas methods\n     * (either those of HTMLCanvasElement itself, or rendering context)\n     *\n     * @param {String} methodName Method to check support for;\n     *                            Could be one of \"setLineDash\"\n     * @return {Boolean | null} `true` if method is supported (or at least exists),\n     *                          `null` if canvas element or context can not be initialized\n     */\n    supports: function (methodName) {\n      var el = createCanvasElement();\n\n      if (!el || !el.getContext) {\n        return null;\n      }\n\n      var ctx = el.getContext('2d');\n      if (!ctx) {\n        return null;\n      }\n\n      switch (methodName) {\n\n        case 'setLineDash':\n          return typeof ctx.setLineDash !== 'undefined';\n\n        default:\n          return null;\n      }\n    }\n  });\n\n  /**\n   * Returns Object representation of canvas\n   * this alias is provided because if you call JSON.stringify on an instance,\n   * the toJSON object will be invoked if it exists.\n   * Having a toJSON method means you can do JSON.stringify(myCanvas)\n   * @function\n   * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n   * @return {Object} JSON compatible object\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-3#serialization}\n   * @see {@link http://jsfiddle.net/fabricjs/pec86/|jsFiddle demo}\n   * @example <caption>JSON without additional properties</caption>\n   * var json = canvas.toJSON();\n   * @example <caption>JSON with additional properties included</caption>\n   * var json = canvas.toJSON(['lockMovementX', 'lockMovementY', 'lockRotation', 'lockScalingX', 'lockScalingY']);\n   * @example <caption>JSON without default values</caption>\n   * canvas.includeDefaultValues = false;\n   * var json = canvas.toJSON();\n   */\n  fabric.StaticCanvas.prototype.toJSON = fabric.StaticCanvas.prototype.toObject;\n\n  if (fabric.isLikelyNode) {\n    fabric.StaticCanvas.prototype.createPNGStream = function() {\n      var impl = getNodeCanvas(this.lowerCanvasEl);\n      return impl && impl.createPNGStream();\n    };\n    fabric.StaticCanvas.prototype.createJPEGStream = function(opts) {\n      var impl = getNodeCanvas(this.lowerCanvasEl);\n      return impl && impl.createJPEGStream(opts);\n    };\n  }\n})();\n\n\n/**\n * BaseBrush class\n * @class fabric.BaseBrush\n * @see {@link http://fabricjs.com/freedrawing|Freedrawing demo}\n */\nfabric.BaseBrush = fabric.util.createClass(/** @lends fabric.BaseBrush.prototype */ {\n\n  /**\n   * Color of a brush\n   * @type String\n   * @default\n   */\n  color: 'rgb(0, 0, 0)',\n\n  /**\n   * Width of a brush, has to be a Number, no string literals\n   * @type Number\n   * @default\n   */\n  width: 1,\n\n  /**\n   * Shadow object representing shadow of this shape.\n   * <b>Backwards incompatibility note:</b> This property replaces \"shadowColor\" (String), \"shadowOffsetX\" (Number),\n   * \"shadowOffsetY\" (Number) and \"shadowBlur\" (Number) since v1.2.12\n   * @type fabric.Shadow\n   * @default\n   */\n  shadow: null,\n\n  /**\n   * Line endings style of a brush (one of \"butt\", \"round\", \"square\")\n   * @type String\n   * @default\n   */\n  strokeLineCap: 'round',\n\n  /**\n   * Corner style of a brush (one of \"bevel\", \"round\", \"miter\")\n   * @type String\n   * @default\n   */\n  strokeLineJoin: 'round',\n\n  /**\n   * Maximum miter length (used for strokeLineJoin = \"miter\") of a brush's\n   * @type Number\n   * @default\n   */\n  strokeMiterLimit:         10,\n\n  /**\n   * Stroke Dash Array.\n   * @type Array\n   * @default\n   */\n  strokeDashArray: null,\n\n  /**\n   * When `true`, the free drawing is limited to the whiteboard size. Default to false.\n   * @type Boolean\n   * @default false\n  */\n\n  limitedToCanvasSize: false,\n\n\n  /**\n   * Sets brush styles\n   * @private\n   * @param {CanvasRenderingContext2D} ctx\n   */\n  _setBrushStyles: function (ctx) {\n    ctx.strokeStyle = this.color;\n    ctx.lineWidth = this.width;\n    ctx.lineCap = this.strokeLineCap;\n    ctx.miterLimit = this.strokeMiterLimit;\n    ctx.lineJoin = this.strokeLineJoin;\n    ctx.setLineDash(this.strokeDashArray || []);\n  },\n\n  /**\n   * Sets the transformation on given context\n   * @param {RenderingContext2d} ctx context to render on\n   * @private\n   */\n  _saveAndTransform: function(ctx) {\n    var v = this.canvas.viewportTransform;\n    ctx.save();\n    ctx.transform(v[0], v[1], v[2], v[3], v[4], v[5]);\n  },\n\n  /**\n   * Sets brush shadow styles\n   * @private\n   */\n  _setShadow: function() {\n    if (!this.shadow) {\n      return;\n    }\n\n    var canvas = this.canvas,\n        shadow = this.shadow,\n        ctx = canvas.contextTop,\n        zoom = canvas.getZoom();\n    if (canvas && canvas._isRetinaScaling()) {\n      zoom *= fabric.devicePixelRatio;\n    }\n\n    ctx.shadowColor = shadow.color;\n    ctx.shadowBlur = shadow.blur * zoom;\n    ctx.shadowOffsetX = shadow.offsetX * zoom;\n    ctx.shadowOffsetY = shadow.offsetY * zoom;\n  },\n\n  needsFullRender: function() {\n    var color = new fabric.Color(this.color);\n    return color.getAlpha() < 1 || !!this.shadow;\n  },\n\n  /**\n   * Removes brush shadow styles\n   * @private\n   */\n  _resetShadow: function() {\n    var ctx = this.canvas.contextTop;\n\n    ctx.shadowColor = '';\n    ctx.shadowBlur = ctx.shadowOffsetX = ctx.shadowOffsetY = 0;\n  },\n\n  /**\n   * Check is pointer is outside canvas boundaries\n   * @param {Object} pointer\n   * @private\n  */\n  _isOutSideCanvas: function(pointer) {\n    return pointer.x < 0 || pointer.x > this.canvas.getWidth() || pointer.y < 0 || pointer.y > this.canvas.getHeight();\n  }\n});\n\n\n(function() {\n  /**\n   * PencilBrush class\n   * @class fabric.PencilBrush\n   * @extends fabric.BaseBrush\n   */\n  fabric.PencilBrush = fabric.util.createClass(fabric.BaseBrush, /** @lends fabric.PencilBrush.prototype */ {\n\n    /**\n     * Discard points that are less than `decimate` pixel distant from each other\n     * @type Number\n     * @default 0.4\n     */\n    decimate: 0.4,\n\n    /**\n     * Draws a straight line between last recorded point to current pointer\n     * Used for `shift` functionality\n     *\n     * @type boolean\n     * @default false\n     */\n    drawStraightLine: false,\n\n    /**\n     * The event modifier key that makes the brush draw a straight line.\n     * If `null` or 'none' or any other string that is not a modifier key the feature is disabled.\n     * @type {'altKey' | 'shiftKey' | 'ctrlKey' | 'none' | undefined | null}\n     */\n    straightLineKey: 'shiftKey',\n\n    /**\n     * Constructor\n     * @param {fabric.Canvas} canvas\n     * @return {fabric.PencilBrush} Instance of a pencil brush\n     */\n    initialize: function(canvas) {\n      this.canvas = canvas;\n      this._points = [];\n    },\n\n    needsFullRender: function () {\n      return this.callSuper('needsFullRender') || this._hasStraightLine;\n    },\n\n    /**\n     * Invoked inside on mouse down and mouse move\n     * @param {Object} pointer\n     */\n    _drawSegment: function (ctx, p1, p2) {\n      var midPoint = p1.midPointFrom(p2);\n      ctx.quadraticCurveTo(p1.x, p1.y, midPoint.x, midPoint.y);\n      return midPoint;\n    },\n\n    /**\n     * Invoked on mouse down\n     * @param {Object} pointer\n     */\n    onMouseDown: function(pointer, options) {\n      if (!this.canvas._isMainEvent(options.e)) {\n        return;\n      }\n      this.drawStraightLine = options.e[this.straightLineKey];\n      this._prepareForDrawing(pointer);\n      // capture coordinates immediately\n      // this allows to draw dots (when movement never occurs)\n      this._captureDrawingPath(pointer);\n      this._render();\n    },\n\n    /**\n     * Invoked on mouse move\n     * @param {Object} pointer\n     */\n    onMouseMove: function(pointer, options) {\n      if (!this.canvas._isMainEvent(options.e)) {\n        return;\n      }\n      this.drawStraightLine = options.e[this.straightLineKey];\n      if (this.limitedToCanvasSize === true && this._isOutSideCanvas(pointer)) {\n        return;\n      }\n      if (this._captureDrawingPath(pointer) && this._points.length > 1) {\n        if (this.needsFullRender()) {\n          // redraw curve\n          // clear top canvas\n          this.canvas.clearContext(this.canvas.contextTop);\n          this._render();\n        }\n        else {\n          var points = this._points, length = points.length, ctx = this.canvas.contextTop;\n          // draw the curve update\n          this._saveAndTransform(ctx);\n          if (this.oldEnd) {\n            ctx.beginPath();\n            ctx.moveTo(this.oldEnd.x, this.oldEnd.y);\n          }\n          this.oldEnd = this._drawSegment(ctx, points[length - 2], points[length - 1], true);\n          ctx.stroke();\n          ctx.restore();\n        }\n      }\n    },\n\n    /**\n     * Invoked on mouse up\n     */\n    onMouseUp: function(options) {\n      if (!this.canvas._isMainEvent(options.e)) {\n        return true;\n      }\n      this.drawStraightLine = false;\n      this.oldEnd = undefined;\n      this._finalizeAndAddPath();\n      return false;\n    },\n\n    /**\n     * @private\n     * @param {Object} pointer Actual mouse position related to the canvas.\n     */\n    _prepareForDrawing: function(pointer) {\n\n      var p = new fabric.Point(pointer.x, pointer.y);\n\n      this._reset();\n      this._addPoint(p);\n      this.canvas.contextTop.moveTo(p.x, p.y);\n    },\n\n    /**\n     * @private\n     * @param {fabric.Point} point Point to be added to points array\n     */\n    _addPoint: function(point) {\n      if (this._points.length > 1 && point.eq(this._points[this._points.length - 1])) {\n        return false;\n      }\n      if (this.drawStraightLine && this._points.length > 1) {\n        this._hasStraightLine = true;\n        this._points.pop();\n      }\n      this._points.push(point);\n      return true;\n    },\n\n    /**\n     * Clear points array and set contextTop canvas style.\n     * @private\n     */\n    _reset: function() {\n      this._points = [];\n      this._setBrushStyles(this.canvas.contextTop);\n      this._setShadow();\n      this._hasStraightLine = false;\n    },\n\n    /**\n     * @private\n     * @param {Object} pointer Actual mouse position related to the canvas.\n     */\n    _captureDrawingPath: function(pointer) {\n      var pointerPoint = new fabric.Point(pointer.x, pointer.y);\n      return this._addPoint(pointerPoint);\n    },\n\n    /**\n     * Draw a smooth path on the topCanvas using quadraticCurveTo\n     * @private\n     * @param {CanvasRenderingContext2D} [ctx]\n     */\n    _render: function(ctx) {\n      var i, len,\n          p1 = this._points[0],\n          p2 = this._points[1];\n      ctx = ctx || this.canvas.contextTop;\n      this._saveAndTransform(ctx);\n      ctx.beginPath();\n      //if we only have 2 points in the path and they are the same\n      //it means that the user only clicked the canvas without moving the mouse\n      //then we should be drawing a dot. A path isn't drawn between two identical dots\n      //that's why we set them apart a bit\n      if (this._points.length === 2 && p1.x === p2.x && p1.y === p2.y) {\n        var width = this.width / 1000;\n        p1 = new fabric.Point(p1.x, p1.y);\n        p2 = new fabric.Point(p2.x, p2.y);\n        p1.x -= width;\n        p2.x += width;\n      }\n      ctx.moveTo(p1.x, p1.y);\n\n      for (i = 1, len = this._points.length; i < len; i++) {\n        // we pick the point between pi + 1 & pi + 2 as the\n        // end point and p1 as our control point.\n        this._drawSegment(ctx, p1, p2);\n        p1 = this._points[i];\n        p2 = this._points[i + 1];\n      }\n      // Draw last line as a straight line while\n      // we wait for the next point to be able to calculate\n      // the bezier control point\n      ctx.lineTo(p1.x, p1.y);\n      ctx.stroke();\n      ctx.restore();\n    },\n\n    /**\n     * Converts points to SVG path\n     * @param {Array} points Array of points\n     * @return {(string|number)[][]} SVG path commands\n     */\n    convertPointsToSVGPath: function (points) {\n      var correction = this.width / 1000;\n      return fabric.util.getSmoothPathFromPoints(points, correction);\n    },\n\n    /**\n     * @private\n     * @param {(string|number)[][]} pathData SVG path commands\n     * @returns {boolean}\n     */\n    _isEmptySVGPath: function (pathData) {\n      var pathString = fabric.util.joinPath(pathData);\n      return pathString === 'M 0 0 Q 0 0 0 0 L 0 0';\n    },\n\n    /**\n     * Creates fabric.Path object to add on canvas\n     * @param {(string|number)[][]} pathData Path data\n     * @return {fabric.Path} Path to add on canvas\n     */\n    createPath: function(pathData) {\n      var path = new fabric.Path(pathData, {\n        fill: null,\n        stroke: this.color,\n        strokeWidth: this.width,\n        strokeLineCap: this.strokeLineCap,\n        strokeMiterLimit: this.strokeMiterLimit,\n        strokeLineJoin: this.strokeLineJoin,\n        strokeDashArray: this.strokeDashArray,\n      });\n      if (this.shadow) {\n        this.shadow.affectStroke = true;\n        path.shadow = new fabric.Shadow(this.shadow);\n      }\n\n      return path;\n    },\n\n    /**\n     * Decimate points array with the decimate value\n     */\n    decimatePoints: function(points, distance) {\n      if (points.length <= 2) {\n        return points;\n      }\n      var zoom = this.canvas.getZoom(), adjustedDistance = Math.pow(distance / zoom, 2),\n          i, l = points.length - 1, lastPoint = points[0], newPoints = [lastPoint],\n          cDistance;\n      for (i = 1; i < l - 1; i++) {\n        cDistance = Math.pow(lastPoint.x - points[i].x, 2) + Math.pow(lastPoint.y - points[i].y, 2);\n        if (cDistance >= adjustedDistance) {\n          lastPoint = points[i];\n          newPoints.push(lastPoint);\n        }\n      }\n      /**\n       * Add the last point from the original line to the end of the array.\n       * This ensures decimate doesn't delete the last point on the line, and ensures the line is > 1 point.\n       */\n      newPoints.push(points[l]);\n      return newPoints;\n    },\n\n    /**\n     * On mouseup after drawing the path on contextTop canvas\n     * we use the points captured to create an new fabric path object\n     * and add it to the fabric canvas.\n     */\n    _finalizeAndAddPath: function() {\n      var ctx = this.canvas.contextTop;\n      ctx.closePath();\n      if (this.decimate) {\n        this._points = this.decimatePoints(this._points, this.decimate);\n      }\n      var pathData = this.convertPointsToSVGPath(this._points);\n      if (this._isEmptySVGPath(pathData)) {\n        // do not create 0 width/height paths, as they are\n        // rendered inconsistently across browsers\n        // Firefox 4, for example, renders a dot,\n        // whereas Chrome 10 renders nothing\n        this.canvas.requestRenderAll();\n        return;\n      }\n\n      var path = this.createPath(pathData);\n      this.canvas.clearContext(this.canvas.contextTop);\n      this.canvas.fire('before:path:created', { path: path });\n      this.canvas.add(path);\n      this.canvas.requestRenderAll();\n      path.setCoords();\n      this._resetShadow();\n\n\n      // fire event 'path' created\n      this.canvas.fire('path:created', { path: path });\n    }\n  });\n})();\n\n\n/**\n * CircleBrush class\n * @class fabric.CircleBrush\n */\nfabric.CircleBrush = fabric.util.createClass(fabric.BaseBrush, /** @lends fabric.CircleBrush.prototype */ {\n\n  /**\n   * Width of a brush\n   * @type Number\n   * @default\n   */\n  width: 10,\n\n  /**\n   * Constructor\n   * @param {fabric.Canvas} canvas\n   * @return {fabric.CircleBrush} Instance of a circle brush\n   */\n  initialize: function(canvas) {\n    this.canvas = canvas;\n    this.points = [];\n  },\n\n  /**\n   * Invoked inside on mouse down and mouse move\n   * @param {Object} pointer\n   */\n  drawDot: function(pointer) {\n    var point = this.addPoint(pointer),\n        ctx = this.canvas.contextTop;\n    this._saveAndTransform(ctx);\n    this.dot(ctx, point);\n    ctx.restore();\n  },\n\n  dot: function(ctx, point) {\n    ctx.fillStyle = point.fill;\n    ctx.beginPath();\n    ctx.arc(point.x, point.y, point.radius, 0, Math.PI * 2, false);\n    ctx.closePath();\n    ctx.fill();\n  },\n\n  /**\n   * Invoked on mouse down\n   */\n  onMouseDown: function(pointer) {\n    this.points.length = 0;\n    this.canvas.clearContext(this.canvas.contextTop);\n    this._setShadow();\n    this.drawDot(pointer);\n  },\n\n  /**\n   * Render the full state of the brush\n   * @private\n   */\n  _render: function() {\n    var ctx  = this.canvas.contextTop, i, len,\n        points = this.points;\n    this._saveAndTransform(ctx);\n    for (i = 0, len = points.length; i < len; i++) {\n      this.dot(ctx, points[i]);\n    }\n    ctx.restore();\n  },\n\n  /**\n   * Invoked on mouse move\n   * @param {Object} pointer\n   */\n  onMouseMove: function(pointer) {\n    if (this.limitedToCanvasSize === true && this._isOutSideCanvas(pointer)) {\n      return;\n    }\n    if (this.needsFullRender()) {\n      this.canvas.clearContext(this.canvas.contextTop);\n      this.addPoint(pointer);\n      this._render();\n    }\n    else {\n      this.drawDot(pointer);\n    }\n  },\n\n  /**\n   * Invoked on mouse up\n   */\n  onMouseUp: function() {\n    var originalRenderOnAddRemove = this.canvas.renderOnAddRemove, i, len;\n    this.canvas.renderOnAddRemove = false;\n\n    var circles = [];\n\n    for (i = 0, len = this.points.length; i < len; i++) {\n      var point = this.points[i],\n          circle = new fabric.Circle({\n            radius: point.radius,\n            left: point.x,\n            top: point.y,\n            originX: 'center',\n            originY: 'center',\n            fill: point.fill\n          });\n\n      this.shadow && (circle.shadow = new fabric.Shadow(this.shadow));\n\n      circles.push(circle);\n    }\n    var group = new fabric.Group(circles);\n    group.canvas = this.canvas;\n\n    this.canvas.fire('before:path:created', { path: group });\n    this.canvas.add(group);\n    this.canvas.fire('path:created', { path: group });\n\n    this.canvas.clearContext(this.canvas.contextTop);\n    this._resetShadow();\n    this.canvas.renderOnAddRemove = originalRenderOnAddRemove;\n    this.canvas.requestRenderAll();\n  },\n\n  /**\n   * @param {Object} pointer\n   * @return {fabric.Point} Just added pointer point\n   */\n  addPoint: function(pointer) {\n    var pointerPoint = new fabric.Point(pointer.x, pointer.y),\n\n        circleRadius = fabric.util.getRandomInt(\n          Math.max(0, this.width - 20), this.width + 20) / 2,\n\n        circleColor = new fabric.Color(this.color)\n          .setAlpha(fabric.util.getRandomInt(0, 100) / 100)\n          .toRgba();\n\n    pointerPoint.radius = circleRadius;\n    pointerPoint.fill = circleColor;\n\n    this.points.push(pointerPoint);\n\n    return pointerPoint;\n  }\n});\n\n\n/**\n * SprayBrush class\n * @class fabric.SprayBrush\n */\nfabric.SprayBrush = fabric.util.createClass( fabric.BaseBrush, /** @lends fabric.SprayBrush.prototype */ {\n\n  /**\n   * Width of a spray\n   * @type Number\n   * @default\n   */\n  width:              10,\n\n  /**\n   * Density of a spray (number of dots per chunk)\n   * @type Number\n   * @default\n   */\n  density:            20,\n\n  /**\n   * Width of spray dots\n   * @type Number\n   * @default\n   */\n  dotWidth:           1,\n\n  /**\n   * Width variance of spray dots\n   * @type Number\n   * @default\n   */\n  dotWidthVariance:   1,\n\n  /**\n   * Whether opacity of a dot should be random\n   * @type Boolean\n   * @default\n   */\n  randomOpacity:        false,\n\n  /**\n   * Whether overlapping dots (rectangles) should be removed (for performance reasons)\n   * @type Boolean\n   * @default\n   */\n  optimizeOverlapping:  true,\n\n  /**\n   * Constructor\n   * @param {fabric.Canvas} canvas\n   * @return {fabric.SprayBrush} Instance of a spray brush\n   */\n  initialize: function(canvas) {\n    this.canvas = canvas;\n    this.sprayChunks = [];\n  },\n\n  /**\n   * Invoked on mouse down\n   * @param {Object} pointer\n   */\n  onMouseDown: function(pointer) {\n    this.sprayChunks.length = 0;\n    this.canvas.clearContext(this.canvas.contextTop);\n    this._setShadow();\n\n    this.addSprayChunk(pointer);\n    this.render(this.sprayChunkPoints);\n  },\n\n  /**\n   * Invoked on mouse move\n   * @param {Object} pointer\n   */\n  onMouseMove: function(pointer) {\n    if (this.limitedToCanvasSize === true && this._isOutSideCanvas(pointer)) {\n      return;\n    }\n    this.addSprayChunk(pointer);\n    this.render(this.sprayChunkPoints);\n  },\n\n  /**\n   * Invoked on mouse up\n   */\n  onMouseUp: function() {\n    var originalRenderOnAddRemove = this.canvas.renderOnAddRemove;\n    this.canvas.renderOnAddRemove = false;\n\n    var rects = [];\n\n    for (var i = 0, ilen = this.sprayChunks.length; i < ilen; i++) {\n      var sprayChunk = this.sprayChunks[i];\n\n      for (var j = 0, jlen = sprayChunk.length; j < jlen; j++) {\n\n        var rect = new fabric.Rect({\n          width: sprayChunk[j].width,\n          height: sprayChunk[j].width,\n          left: sprayChunk[j].x + 1,\n          top: sprayChunk[j].y + 1,\n          originX: 'center',\n          originY: 'center',\n          fill: this.color\n        });\n        rects.push(rect);\n      }\n    }\n\n    if (this.optimizeOverlapping) {\n      rects = this._getOptimizedRects(rects);\n    }\n\n    var group = new fabric.Group(rects);\n    this.shadow && group.set('shadow', new fabric.Shadow(this.shadow));\n    this.canvas.fire('before:path:created', { path: group });\n    this.canvas.add(group);\n    this.canvas.fire('path:created', { path: group });\n\n    this.canvas.clearContext(this.canvas.contextTop);\n    this._resetShadow();\n    this.canvas.renderOnAddRemove = originalRenderOnAddRemove;\n    this.canvas.requestRenderAll();\n  },\n\n  /**\n   * @private\n   * @param {Array} rects\n   */\n  _getOptimizedRects: function(rects) {\n\n    // avoid creating duplicate rects at the same coordinates\n    var uniqueRects = { }, key, i, len;\n\n    for (i = 0, len = rects.length; i < len; i++) {\n      key = rects[i].left + '' + rects[i].top;\n      if (!uniqueRects[key]) {\n        uniqueRects[key] = rects[i];\n      }\n    }\n    var uniqueRectsArray = [];\n    for (key in uniqueRects) {\n      uniqueRectsArray.push(uniqueRects[key]);\n    }\n\n    return uniqueRectsArray;\n  },\n\n  /**\n   * Render new chunk of spray brush\n   */\n  render: function(sprayChunk) {\n    var ctx = this.canvas.contextTop, i, len;\n    ctx.fillStyle = this.color;\n\n    this._saveAndTransform(ctx);\n\n    for (i = 0, len = sprayChunk.length; i < len; i++) {\n      var point = sprayChunk[i];\n      if (typeof point.opacity !== 'undefined') {\n        ctx.globalAlpha = point.opacity;\n      }\n      ctx.fillRect(point.x, point.y, point.width, point.width);\n    }\n    ctx.restore();\n  },\n\n  /**\n   * Render all spray chunks\n   */\n  _render: function() {\n    var ctx = this.canvas.contextTop, i, ilen;\n    ctx.fillStyle = this.color;\n\n    this._saveAndTransform(ctx);\n\n    for (i = 0, ilen = this.sprayChunks.length; i < ilen; i++) {\n      this.render(this.sprayChunks[i]);\n    }\n    ctx.restore();\n  },\n\n  /**\n   * @param {Object} pointer\n   */\n  addSprayChunk: function(pointer) {\n    this.sprayChunkPoints = [];\n\n    var x, y, width, radius = this.width / 2, i;\n\n    for (i = 0; i < this.density; i++) {\n\n      x = fabric.util.getRandomInt(pointer.x - radius, pointer.x + radius);\n      y = fabric.util.getRandomInt(pointer.y - radius, pointer.y + radius);\n\n      if (this.dotWidthVariance) {\n        width = fabric.util.getRandomInt(\n          // bottom clamp width to 1\n          Math.max(1, this.dotWidth - this.dotWidthVariance),\n          this.dotWidth + this.dotWidthVariance);\n      }\n      else {\n        width = this.dotWidth;\n      }\n\n      var point = new fabric.Point(x, y);\n      point.width = width;\n\n      if (this.randomOpacity) {\n        point.opacity = fabric.util.getRandomInt(0, 100) / 100;\n      }\n\n      this.sprayChunkPoints.push(point);\n    }\n\n    this.sprayChunks.push(this.sprayChunkPoints);\n  }\n});\n\n\n/**\n * PatternBrush class\n * @class fabric.PatternBrush\n * @extends fabric.BaseBrush\n */\nfabric.PatternBrush = fabric.util.createClass(fabric.PencilBrush, /** @lends fabric.PatternBrush.prototype */ {\n\n  getPatternSrc: function() {\n\n    var dotWidth = 20,\n        dotDistance = 5,\n        patternCanvas = fabric.util.createCanvasElement(),\n        patternCtx = patternCanvas.getContext('2d');\n\n    patternCanvas.width = patternCanvas.height = dotWidth + dotDistance;\n\n    patternCtx.fillStyle = this.color;\n    patternCtx.beginPath();\n    patternCtx.arc(dotWidth / 2, dotWidth / 2, dotWidth / 2, 0, Math.PI * 2, false);\n    patternCtx.closePath();\n    patternCtx.fill();\n\n    return patternCanvas;\n  },\n\n  getPatternSrcFunction: function() {\n    return String(this.getPatternSrc).replace('this.color', '\"' + this.color + '\"');\n  },\n\n  /**\n   * Creates \"pattern\" instance property\n   * @param {CanvasRenderingContext2D} ctx\n   */\n  getPattern: function(ctx) {\n    return ctx.createPattern(this.source || this.getPatternSrc(), 'repeat');\n  },\n\n  /**\n   * Sets brush styles\n   * @param {CanvasRenderingContext2D} ctx\n   */\n  _setBrushStyles: function(ctx) {\n    this.callSuper('_setBrushStyles', ctx);\n    ctx.strokeStyle = this.getPattern(ctx);\n  },\n\n  /**\n   * Creates path\n   */\n  createPath: function(pathData) {\n    var path = this.callSuper('createPath', pathData),\n        topLeft = path._getLeftTopCoords().scalarAdd(path.strokeWidth / 2);\n\n    path.stroke = new fabric.Pattern({\n      source: this.source || this.getPatternSrcFunction(),\n      offsetX: -topLeft.x,\n      offsetY: -topLeft.y\n    });\n    return path;\n  }\n});\n\n\n(function() {\n\n  var getPointer = fabric.util.getPointer,\n      degreesToRadians = fabric.util.degreesToRadians,\n      isTouchEvent = fabric.util.isTouchEvent;\n\n  /**\n   * Canvas class\n   * @class fabric.Canvas\n   * @extends fabric.StaticCanvas\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-1#canvas}\n   * @see {@link fabric.Canvas#initialize} for constructor definition\n   *\n   * @fires object:modified at the end of a transform or any change when statefull is true\n   * @fires object:rotating while an object is being rotated from the control\n   * @fires object:scaling while an object is being scaled by controls\n   * @fires object:moving while an object is being dragged\n   * @fires object:skewing while an object is being skewed from the controls\n   *\n   * @fires before:transform before a transform is is started\n   * @fires before:selection:cleared\n   * @fires selection:cleared\n   * @fires selection:updated\n   * @fires selection:created\n   *\n   * @fires path:created after a drawing operation ends and the path is added\n   * @fires mouse:down\n   * @fires mouse:move\n   * @fires mouse:up\n   * @fires mouse:down:before  on mouse down, before the inner fabric logic runs\n   * @fires mouse:move:before on mouse move, before the inner fabric logic runs\n   * @fires mouse:up:before on mouse up, before the inner fabric logic runs\n   * @fires mouse:over\n   * @fires mouse:out\n   * @fires mouse:dblclick whenever a native dbl click event fires on the canvas.\n   *\n   * @fires dragover\n   * @fires dragenter\n   * @fires dragleave\n   * @fires drop:before before drop event. same native event. This is added to handle edge cases\n   * @fires drop\n   * @fires after:render at the end of the render process, receives the context in the callback\n   * @fires before:render at start the render process, receives the context in the callback\n   *\n   */\n  fabric.Canvas = fabric.util.createClass(fabric.StaticCanvas, /** @lends fabric.Canvas.prototype */ {\n\n    /**\n     * Constructor\n     * @param {HTMLElement | String} el &lt;canvas> element to initialize instance on\n     * @param {Object} [options] Options object\n     * @return {Object} thisArg\n     */\n    initialize: function(el, options) {\n      options || (options = { });\n      this.renderAndResetBound = this.renderAndReset.bind(this);\n      this.requestRenderAllBound = this.requestRenderAll.bind(this);\n      this._initStatic(el, options);\n      this._initInteractive();\n      this._createCacheCanvas();\n    },\n\n    /**\n     * When true, objects can be transformed by one side (unproportionally)\n     * when dragged on the corners that normally would not do that.\n     * @type Boolean\n     * @default\n     * @since fabric 4.0 // changed name and default value\n     */\n    uniformScaling:      true,\n\n    /**\n     * Indicates which key switches uniform scaling.\n     * values: 'altKey', 'shiftKey', 'ctrlKey'.\n     * If `null` or 'none' or any other string that is not a modifier key\n     * feature is disabled.\n     * totally wrong named. this sounds like `uniform scaling`\n     * if Canvas.uniformScaling is true, pressing this will set it to false\n     * and viceversa.\n     * @since 1.6.2\n     * @type String\n     * @default\n     */\n    uniScaleKey:           'shiftKey',\n\n    /**\n     * When true, objects use center point as the origin of scale transformation.\n     * <b>Backwards incompatibility note:</b> This property replaces \"centerTransform\" (Boolean).\n     * @since 1.3.4\n     * @type Boolean\n     * @default\n     */\n    centeredScaling:        false,\n\n    /**\n     * When true, objects use center point as the origin of rotate transformation.\n     * <b>Backwards incompatibility note:</b> This property replaces \"centerTransform\" (Boolean).\n     * @since 1.3.4\n     * @type Boolean\n     * @default\n     */\n    centeredRotation:       false,\n\n    /**\n     * Indicates which key enable centered Transform\n     * values: 'altKey', 'shiftKey', 'ctrlKey'.\n     * If `null` or 'none' or any other string that is not a modifier key\n     * feature is disabled feature disabled.\n     * @since 1.6.2\n     * @type String\n     * @default\n     */\n    centeredKey:           'altKey',\n\n    /**\n     * Indicates which key enable alternate action on corner\n     * values: 'altKey', 'shiftKey', 'ctrlKey'.\n     * If `null` or 'none' or any other string that is not a modifier key\n     * feature is disabled feature disabled.\n     * @since 1.6.2\n     * @type String\n     * @default\n     */\n    altActionKey:           'shiftKey',\n\n    /**\n     * Indicates that canvas is interactive. This property should not be changed.\n     * @type Boolean\n     * @default\n     */\n    interactive:            true,\n\n    /**\n     * Indicates whether group selection should be enabled\n     * @type Boolean\n     * @default\n     */\n    selection:              true,\n\n    /**\n     * Indicates which key or keys enable multiple click selection\n     * Pass value as a string or array of strings\n     * values: 'altKey', 'shiftKey', 'ctrlKey'.\n     * If `null` or empty or containing any other string that is not a modifier key\n     * feature is disabled.\n     * @since 1.6.2\n     * @type String|Array\n     * @default\n     */\n    selectionKey:           'shiftKey',\n\n    /**\n     * Indicates which key enable alternative selection\n     * in case of target overlapping with active object\n     * values: 'altKey', 'shiftKey', 'ctrlKey'.\n     * For a series of reason that come from the general expectations on how\n     * things should work, this feature works only for preserveObjectStacking true.\n     * If `null` or 'none' or any other string that is not a modifier key\n     * feature is disabled.\n     * @since 1.6.5\n     * @type null|String\n     * @default\n     */\n    altSelectionKey:           null,\n\n    /**\n     * Color of selection\n     * @type String\n     * @default\n     */\n    selectionColor:         'rgba(100, 100, 255, 0.3)', // blue\n\n    /**\n     * Default dash array pattern\n     * If not empty the selection border is dashed\n     * @type Array\n     */\n    selectionDashArray:     [],\n\n    /**\n     * Color of the border of selection (usually slightly darker than color of selection itself)\n     * @type String\n     * @default\n     */\n    selectionBorderColor:   'rgba(255, 255, 255, 0.3)',\n\n    /**\n     * Width of a line used in object/group selection\n     * @type Number\n     * @default\n     */\n    selectionLineWidth:     1,\n\n    /**\n     * Select only shapes that are fully contained in the dragged selection rectangle.\n     * @type Boolean\n     * @default\n     */\n    selectionFullyContained: false,\n\n    /**\n     * Default cursor value used when hovering over an object on canvas\n     * @type String\n     * @default\n     */\n    hoverCursor:            'move',\n\n    /**\n     * Default cursor value used when moving an object on canvas\n     * @type String\n     * @default\n     */\n    moveCursor:             'move',\n\n    /**\n     * Default cursor value used for the entire canvas\n     * @type String\n     * @default\n     */\n    defaultCursor:          'default',\n\n    /**\n     * Cursor value used during free drawing\n     * @type String\n     * @default\n     */\n    freeDrawingCursor:      'crosshair',\n\n    /**\n     * Cursor value used for disabled elements ( corners with disabled action )\n     * @type String\n     * @since 2.0.0\n     * @default\n     */\n    notAllowedCursor:         'not-allowed',\n\n    /**\n     * Default element class that's given to wrapper (div) element of canvas\n     * @type String\n     * @default\n     */\n    containerClass:         'canvas-container',\n\n    /**\n     * When true, object detection happens on per-pixel basis rather than on per-bounding-box\n     * @type Boolean\n     * @default\n     */\n    perPixelTargetFind:     false,\n\n    /**\n     * Number of pixels around target pixel to tolerate (consider active) during object detection\n     * @type Number\n     * @default\n     */\n    targetFindTolerance:    0,\n\n    /**\n     * When true, target detection is skipped. Target detection will return always undefined.\n     * click selection won't work anymore, events will fire with no targets.\n     * if something is selected before setting it to true, it will be deselected at the first click.\n     * area selection will still work. check the `selection` property too.\n     * if you deactivate both, you should look into staticCanvas.\n     * @type Boolean\n     * @default\n     */\n    skipTargetFind:         false,\n\n    /**\n     * When true, mouse events on canvas (mousedown/mousemove/mouseup) result in free drawing.\n     * After mousedown, mousemove creates a shape,\n     * and then mouseup finalizes it and adds an instance of `fabric.Path` onto canvas.\n     * @tutorial {@link http://fabricjs.com/fabric-intro-part-4#free_drawing}\n     * @type Boolean\n     * @default\n     */\n    isDrawingMode:          false,\n\n    /**\n     * Indicates whether objects should remain in current stack position when selected.\n     * When false objects are brought to top and rendered as part of the selection group\n     * @type Boolean\n     * @default\n     */\n    preserveObjectStacking: false,\n\n    /**\n     * Indicates the angle that an object will lock to while rotating.\n     * @type Number\n     * @since 1.6.7\n     * @default\n     */\n    snapAngle: 0,\n\n    /**\n     * Indicates the distance from the snapAngle the rotation will lock to the snapAngle.\n     * When `null`, the snapThreshold will default to the snapAngle.\n     * @type null|Number\n     * @since 1.6.7\n     * @default\n     */\n    snapThreshold: null,\n\n    /**\n     * Indicates if the right click on canvas can output the context menu or not\n     * @type Boolean\n     * @since 1.6.5\n     * @default\n     */\n    stopContextMenu: false,\n\n    /**\n     * Indicates if the canvas can fire right click events\n     * @type Boolean\n     * @since 1.6.5\n     * @default\n     */\n    fireRightClick: false,\n\n    /**\n     * Indicates if the canvas can fire middle click events\n     * @type Boolean\n     * @since 1.7.8\n     * @default\n     */\n    fireMiddleClick: false,\n\n    /**\n     * Keep track of the subTargets for Mouse Events\n     * @type fabric.Object[]\n     */\n    targets: [],\n\n    /**\n     * When the option is enabled, PointerEvent is used instead of MouseEvent.\n     * @type Boolean\n     * @default\n     */\n    enablePointerEvents: false,\n\n    /**\n     * Keep track of the hovered target\n     * @type fabric.Object\n     * @private\n     */\n    _hoveredTarget: null,\n\n    /**\n     * hold the list of nested targets hovered\n     * @type fabric.Object[]\n     * @private\n     */\n    _hoveredTargets: [],\n\n    /**\n     * @private\n     */\n    _initInteractive: function() {\n      this._currentTransform = null;\n      this._groupSelector = null;\n      this._initWrapperElement();\n      this._createUpperCanvas();\n      this._initEventListeners();\n\n      this._initRetinaScaling();\n\n      this.freeDrawingBrush = fabric.PencilBrush && new fabric.PencilBrush(this);\n\n      this.calcOffset();\n    },\n\n    /**\n     * Divides objects in two groups, one to render immediately\n     * and one to render as activeGroup.\n     * @return {Array} objects to render immediately and pushes the other in the activeGroup.\n     */\n    _chooseObjectsToRender: function() {\n      var activeObjects = this.getActiveObjects(),\n          object, objsToRender, activeGroupObjects;\n\n      if (activeObjects.length > 0 && !this.preserveObjectStacking) {\n        objsToRender = [];\n        activeGroupObjects = [];\n        for (var i = 0, length = this._objects.length; i < length; i++) {\n          object = this._objects[i];\n          if (activeObjects.indexOf(object) === -1 ) {\n            objsToRender.push(object);\n          }\n          else {\n            activeGroupObjects.push(object);\n          }\n        }\n        if (activeObjects.length > 1) {\n          this._activeObject._objects = activeGroupObjects;\n        }\n        objsToRender.push.apply(objsToRender, activeGroupObjects);\n      }\n      else {\n        objsToRender = this._objects;\n      }\n      return objsToRender;\n    },\n\n    /**\n     * Renders both the top canvas and the secondary container canvas.\n     * @return {fabric.Canvas} instance\n     * @chainable\n     */\n    renderAll: function () {\n      if (this.contextTopDirty && !this._groupSelector && !this.isDrawingMode) {\n        this.clearContext(this.contextTop);\n        this.contextTopDirty = false;\n      }\n      if (this.hasLostContext) {\n        this.renderTopLayer(this.contextTop);\n        this.hasLostContext = false;\n      }\n      var canvasToDrawOn = this.contextContainer;\n      this.renderCanvas(canvasToDrawOn, this._chooseObjectsToRender());\n      return this;\n    },\n\n    renderTopLayer: function(ctx) {\n      ctx.save();\n      if (this.isDrawingMode && this._isCurrentlyDrawing) {\n        this.freeDrawingBrush && this.freeDrawingBrush._render();\n        this.contextTopDirty = true;\n      }\n      // we render the top context - last object\n      if (this.selection && this._groupSelector) {\n        this._drawSelection(ctx);\n        this.contextTopDirty = true;\n      }\n      ctx.restore();\n    },\n\n    /**\n     * Method to render only the top canvas.\n     * Also used to render the group selection box.\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    renderTop: function () {\n      var ctx = this.contextTop;\n      this.clearContext(ctx);\n      this.renderTopLayer(ctx);\n      this.fire('after:render');\n      return this;\n    },\n\n    /**\n     * @private\n     */\n    _normalizePointer: function (object, pointer) {\n      var m = object.calcTransformMatrix(),\n          invertedM = fabric.util.invertTransform(m),\n          vptPointer = this.restorePointerVpt(pointer);\n      return fabric.util.transformPoint(vptPointer, invertedM);\n    },\n\n    /**\n     * Returns true if object is transparent at a certain location\n     * @param {fabric.Object} target Object to check\n     * @param {Number} x Left coordinate\n     * @param {Number} y Top coordinate\n     * @return {Boolean}\n     */\n    isTargetTransparent: function (target, x, y) {\n      // in case the target is the activeObject, we cannot execute this optimization\n      // because we need to draw controls too.\n      if (target.shouldCache() && target._cacheCanvas && target !== this._activeObject) {\n        var normalizedPointer = this._normalizePointer(target, {x: x, y: y}),\n            targetRelativeX = Math.max(target.cacheTranslationX + (normalizedPointer.x * target.zoomX), 0),\n            targetRelativeY = Math.max(target.cacheTranslationY + (normalizedPointer.y * target.zoomY), 0);\n\n        var isTransparent = fabric.util.isTransparent(\n          target._cacheContext, Math.round(targetRelativeX), Math.round(targetRelativeY), this.targetFindTolerance);\n\n        return isTransparent;\n      }\n\n      var ctx = this.contextCache,\n          originalColor = target.selectionBackgroundColor, v = this.viewportTransform;\n\n      target.selectionBackgroundColor = '';\n\n      this.clearContext(ctx);\n\n      ctx.save();\n      ctx.transform(v[0], v[1], v[2], v[3], v[4], v[5]);\n      target.render(ctx);\n      ctx.restore();\n\n      target.selectionBackgroundColor = originalColor;\n\n      var isTransparent = fabric.util.isTransparent(\n        ctx, x, y, this.targetFindTolerance);\n\n      return isTransparent;\n    },\n\n    /**\n     * takes an event and determines if selection key has been pressed\n     * @private\n     * @param {Event} e Event object\n     */\n    _isSelectionKeyPressed: function(e) {\n      var selectionKeyPressed = false;\n\n      if (Array.isArray(this.selectionKey)) {\n        selectionKeyPressed = !!this.selectionKey.find(function(key) { return e[key] === true; });\n      }\n      else {\n        selectionKeyPressed = e[this.selectionKey];\n      }\n\n      return selectionKeyPressed;\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object\n     * @param {fabric.Object} target\n     */\n    _shouldClearSelection: function (e, target) {\n      var activeObjects = this.getActiveObjects(),\n          activeObject = this._activeObject;\n\n      return (\n        !target\n        ||\n        (target &&\n          activeObject &&\n          activeObjects.length > 1 &&\n          activeObjects.indexOf(target) === -1 &&\n          activeObject !== target &&\n          !this._isSelectionKeyPressed(e))\n        ||\n        (target && !target.evented)\n        ||\n        (target &&\n          !target.selectable &&\n          activeObject &&\n          activeObject !== target)\n      );\n    },\n\n    /**\n     * centeredScaling from object can't override centeredScaling from canvas.\n     * this should be fixed, since object setting should take precedence over canvas.\n     * also this should be something that will be migrated in the control properties.\n     * as ability to define the origin of the transformation that the control provide.\n     * @private\n     * @param {fabric.Object} target\n     * @param {String} action\n     * @param {Boolean} altKey\n     */\n    _shouldCenterTransform: function (target, action, altKey) {\n      if (!target) {\n        return;\n      }\n\n      var centerTransform;\n\n      if (action === 'scale' || action === 'scaleX' || action === 'scaleY' || action === 'resizing') {\n        centerTransform = this.centeredScaling || target.centeredScaling;\n      }\n      else if (action === 'rotate') {\n        centerTransform = this.centeredRotation || target.centeredRotation;\n      }\n\n      return centerTransform ? !altKey : altKey;\n    },\n\n    /**\n     * should disappear before release 4.0\n     * @private\n     */\n    _getOriginFromCorner: function(target, corner) {\n      var origin = {\n        x: target.originX,\n        y: target.originY\n      };\n\n      if (corner === 'ml' || corner === 'tl' || corner === 'bl') {\n        origin.x = 'right';\n      }\n      else if (corner === 'mr' || corner === 'tr' || corner === 'br') {\n        origin.x = 'left';\n      }\n\n      if (corner === 'tl' || corner === 'mt' || corner === 'tr') {\n        origin.y = 'bottom';\n      }\n      else if (corner === 'bl' || corner === 'mb' || corner === 'br') {\n        origin.y = 'top';\n      }\n      return origin;\n    },\n\n    /**\n     * @private\n     * @param {Boolean} alreadySelected true if target is already selected\n     * @param {String} corner a string representing the corner ml, mr, tl ...\n     * @param {Event} e Event object\n     * @param {fabric.Object} [target] inserted back to help overriding. Unused\n     */\n    _getActionFromCorner: function(alreadySelected, corner, e, target) {\n      if (!corner || !alreadySelected) {\n        return 'drag';\n      }\n      var control = target.controls[corner];\n      return control.getActionName(e, control, target);\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object\n     * @param {fabric.Object} target\n     */\n    _setupCurrentTransform: function (e, target, alreadySelected) {\n      if (!target) {\n        return;\n      }\n\n      var pointer = this.getPointer(e), corner = target.__corner,\n          control = target.controls[corner],\n          actionHandler = (alreadySelected && corner) ?\n            control.getActionHandler(e, target, control) : fabric.controlsUtils.dragHandler,\n          action = this._getActionFromCorner(alreadySelected, corner, e, target),\n          origin = this._getOriginFromCorner(target, corner),\n          altKey = e[this.centeredKey],\n          transform = {\n            target: target,\n            action: action,\n            actionHandler: actionHandler,\n            corner: corner,\n            scaleX: target.scaleX,\n            scaleY: target.scaleY,\n            skewX: target.skewX,\n            skewY: target.skewY,\n            // used by transation\n            offsetX: pointer.x - target.left,\n            offsetY: pointer.y - target.top,\n            originX: origin.x,\n            originY: origin.y,\n            ex: pointer.x,\n            ey: pointer.y,\n            lastX: pointer.x,\n            lastY: pointer.y,\n            // unsure they are useful anymore.\n            // left: target.left,\n            // top: target.top,\n            theta: degreesToRadians(target.angle),\n            // end of unsure\n            width: target.width * target.scaleX,\n            shiftKey: e.shiftKey,\n            altKey: altKey,\n            original: fabric.util.saveObjectTransform(target),\n          };\n\n      if (this._shouldCenterTransform(target, action, altKey)) {\n        transform.originX = 'center';\n        transform.originY = 'center';\n      }\n      transform.original.originX = origin.x;\n      transform.original.originY = origin.y;\n      this._currentTransform = transform;\n      this._beforeTransform(e);\n    },\n\n    /**\n     * Set the cursor type of the canvas element\n     * @param {String} value Cursor type of the canvas element.\n     * @see http://www.w3.org/TR/css3-ui/#cursor\n     */\n    setCursor: function (value) {\n      this.upperCanvasEl.style.cursor = value;\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx to draw the selection on\n     */\n    _drawSelection: function (ctx) {\n      var selector = this._groupSelector,\n          viewportStart = new fabric.Point(selector.ex, selector.ey),\n          start = fabric.util.transformPoint(viewportStart, this.viewportTransform),\n          viewportExtent = new fabric.Point(selector.ex + selector.left, selector.ey + selector.top),\n          extent = fabric.util.transformPoint(viewportExtent, this.viewportTransform),\n          minX = Math.min(start.x, extent.x),\n          minY = Math.min(start.y, extent.y),\n          maxX = Math.max(start.x, extent.x),\n          maxY = Math.max(start.y, extent.y),\n          strokeOffset = this.selectionLineWidth / 2;\n\n      if (this.selectionColor) {\n        ctx.fillStyle = this.selectionColor;\n        ctx.fillRect(minX, minY, maxX - minX, maxY - minY);\n      }\n\n      if (!this.selectionLineWidth || !this.selectionBorderColor) {\n        return;\n      }\n      ctx.lineWidth = this.selectionLineWidth;\n      ctx.strokeStyle = this.selectionBorderColor;\n\n      minX += strokeOffset;\n      minY += strokeOffset;\n      maxX -= strokeOffset;\n      maxY -= strokeOffset;\n      // selection border\n      fabric.Object.prototype._setLineDash.call(this, ctx, this.selectionDashArray);\n      ctx.strokeRect(minX, minY, maxX - minX, maxY - minY);\n    },\n\n    /**\n     * Method that determines what object we are clicking on\n     * the skipGroup parameter is for internal use, is needed for shift+click action\n     * 11/09/2018 TODO: would be cool if findTarget could discern between being a full target\n     * or the outside part of the corner.\n     * @param {Event} e mouse event\n     * @param {Boolean} skipGroup when true, activeGroup is skipped and only objects are traversed through\n     * @return {fabric.Object} the target found\n     */\n    findTarget: function (e, skipGroup) {\n      if (this.skipTargetFind) {\n        return;\n      }\n\n      var ignoreZoom = true,\n          pointer = this.getPointer(e, ignoreZoom),\n          activeObject = this._activeObject,\n          aObjects = this.getActiveObjects(),\n          activeTarget, activeTargetSubs,\n          isTouch = isTouchEvent(e),\n          shouldLookForActive = (aObjects.length > 1 && !skipGroup) || aObjects.length === 1;\n\n      // first check current group (if one exists)\n      // active group does not check sub targets like normal groups.\n      // if active group just exits.\n      this.targets = [];\n\n      // if we hit the corner of an activeObject, let's return that.\n      if (shouldLookForActive && activeObject._findTargetCorner(pointer, isTouch)) {\n        return activeObject;\n      }\n      if (aObjects.length > 1 && !skipGroup && activeObject === this._searchPossibleTargets([activeObject], pointer)) {\n        return activeObject;\n      }\n      if (aObjects.length === 1 &&\n        activeObject === this._searchPossibleTargets([activeObject], pointer)) {\n        if (!this.preserveObjectStacking) {\n          return activeObject;\n        }\n        else {\n          activeTarget = activeObject;\n          activeTargetSubs = this.targets;\n          this.targets = [];\n        }\n      }\n      var target = this._searchPossibleTargets(this._objects, pointer);\n      if (e[this.altSelectionKey] && target && activeTarget && target !== activeTarget) {\n        target = activeTarget;\n        this.targets = activeTargetSubs;\n      }\n      return target;\n    },\n\n    /**\n     * Checks point is inside the object.\n     * @param {Object} [pointer] x,y object of point coordinates we want to check.\n     * @param {fabric.Object} obj Object to test against\n     * @param {Object} [globalPointer] x,y object of point coordinates relative to canvas used to search per pixel target.\n     * @return {Boolean} true if point is contained within an area of given object\n     * @private\n     */\n    _checkTarget: function(pointer, obj, globalPointer) {\n      if (obj &&\n          obj.visible &&\n          obj.evented &&\n          // http://www.geog.ubc.ca/courses/klink/gis.notes/ncgia/u32.html\n          // http://idav.ucdavis.edu/~okreylos/TAship/Spring2000/PointInPolygon.html\n          obj.containsPoint(pointer)\n      ) {\n        if ((this.perPixelTargetFind || obj.perPixelTargetFind) && !obj.isEditing) {\n          var isTransparent = this.isTargetTransparent(obj, globalPointer.x, globalPointer.y);\n          if (!isTransparent) {\n            return true;\n          }\n        }\n        else {\n          return true;\n        }\n      }\n    },\n\n    /**\n     * Function used to search inside objects an object that contains pointer in bounding box or that contains pointerOnCanvas when painted\n     * @param {Array} [objects] objects array to look into\n     * @param {Object} [pointer] x,y object of point coordinates we want to check.\n     * @return {fabric.Object} object that contains pointer\n     * @private\n     */\n    _searchPossibleTargets: function(objects, pointer) {\n      // Cache all targets where their bounding box contains point.\n      var target, i = objects.length, subTarget;\n      // Do not check for currently grouped objects, since we check the parent group itself.\n      // until we call this function specifically to search inside the activeGroup\n      while (i--) {\n        var objToCheck = objects[i];\n        var pointerToUse = objToCheck.group ?\n          this._normalizePointer(objToCheck.group, pointer) : pointer;\n        if (this._checkTarget(pointerToUse, objToCheck, pointer)) {\n          target = objects[i];\n          if (target.subTargetCheck && target instanceof fabric.Group) {\n            subTarget = this._searchPossibleTargets(target._objects, pointer);\n            subTarget && this.targets.push(subTarget);\n          }\n          break;\n        }\n      }\n      return target;\n    },\n\n    /**\n     * Returns pointer coordinates without the effect of the viewport\n     * @param {Object} pointer with \"x\" and \"y\" number values\n     * @return {Object} object with \"x\" and \"y\" number values\n     */\n    restorePointerVpt: function(pointer) {\n      return fabric.util.transformPoint(\n        pointer,\n        fabric.util.invertTransform(this.viewportTransform)\n      );\n    },\n\n    /**\n     * Returns pointer coordinates relative to canvas.\n     * Can return coordinates with or without viewportTransform.\n     * ignoreZoom false gives back coordinates that represent\n     * the point clicked on canvas element.\n     * ignoreZoom true gives back coordinates after being processed\n     * by the viewportTransform ( sort of coordinates of what is displayed\n     * on the canvas where you are clicking.\n     * ignoreZoom true = HTMLElement coordinates relative to top,left\n     * ignoreZoom false, default = fabric space coordinates, the same used for shape position\n     * To interact with your shapes top and left you want to use ignoreZoom true\n     * most of the time, while ignoreZoom false will give you coordinates\n     * compatible with the object.oCoords system.\n     * of the time.\n     * @param {Event} e\n     * @param {Boolean} ignoreZoom\n     * @return {Object} object with \"x\" and \"y\" number values\n     */\n    getPointer: function (e, ignoreZoom) {\n      // return cached values if we are in the event processing chain\n      if (this._absolutePointer && !ignoreZoom) {\n        return this._absolutePointer;\n      }\n      if (this._pointer && ignoreZoom) {\n        return this._pointer;\n      }\n\n      var pointer = getPointer(e),\n          upperCanvasEl = this.upperCanvasEl,\n          bounds = upperCanvasEl.getBoundingClientRect(),\n          boundsWidth = bounds.width || 0,\n          boundsHeight = bounds.height || 0,\n          cssScale;\n\n      if (!boundsWidth || !boundsHeight ) {\n        if ('top' in bounds && 'bottom' in bounds) {\n          boundsHeight = Math.abs( bounds.top - bounds.bottom );\n        }\n        if ('right' in bounds && 'left' in bounds) {\n          boundsWidth = Math.abs( bounds.right - bounds.left );\n        }\n      }\n\n      this.calcOffset();\n      pointer.x = pointer.x - this._offset.left;\n      pointer.y = pointer.y - this._offset.top;\n      if (!ignoreZoom) {\n        pointer = this.restorePointerVpt(pointer);\n      }\n\n      var retinaScaling = this.getRetinaScaling();\n      if (retinaScaling !== 1) {\n        pointer.x /= retinaScaling;\n        pointer.y /= retinaScaling;\n      }\n\n      if (boundsWidth === 0 || boundsHeight === 0) {\n        // If bounds are not available (i.e. not visible), do not apply scale.\n        cssScale = { width: 1, height: 1 };\n      }\n      else {\n        cssScale = {\n          width: upperCanvasEl.width / boundsWidth,\n          height: upperCanvasEl.height / boundsHeight\n        };\n      }\n\n      return {\n        x: pointer.x * cssScale.width,\n        y: pointer.y * cssScale.height\n      };\n    },\n\n    /**\n     * @private\n     * @throws {CANVAS_INIT_ERROR} If canvas can not be initialized\n     */\n    _createUpperCanvas: function () {\n      var lowerCanvasClass = this.lowerCanvasEl.className.replace(/\\s*lower-canvas\\s*/, ''),\n          lowerCanvasEl = this.lowerCanvasEl, upperCanvasEl = this.upperCanvasEl;\n\n      // there is no need to create a new upperCanvas element if we have already one.\n      if (upperCanvasEl) {\n        upperCanvasEl.className = '';\n      }\n      else {\n        upperCanvasEl = this._createCanvasElement();\n        this.upperCanvasEl = upperCanvasEl;\n      }\n      fabric.util.addClass(upperCanvasEl, 'upper-canvas ' + lowerCanvasClass);\n\n      this.wrapperEl.appendChild(upperCanvasEl);\n\n      this._copyCanvasStyle(lowerCanvasEl, upperCanvasEl);\n      this._applyCanvasStyle(upperCanvasEl);\n      this.contextTop = upperCanvasEl.getContext('2d');\n    },\n\n    /**\n     * Returns context of top canvas where interactions are drawn\n     * @returns {CanvasRenderingContext2D}\n     */\n    getTopContext: function () {\n      return this.contextTop;\n    },\n\n    /**\n     * @private\n     */\n    _createCacheCanvas: function () {\n      this.cacheCanvasEl = this._createCanvasElement();\n      this.cacheCanvasEl.setAttribute('width', this.width);\n      this.cacheCanvasEl.setAttribute('height', this.height);\n      this.contextCache = this.cacheCanvasEl.getContext('2d');\n    },\n\n    /**\n     * @private\n     */\n    _initWrapperElement: function () {\n      this.wrapperEl = fabric.util.wrapElement(this.lowerCanvasEl, 'div', {\n        'class': this.containerClass\n      });\n      fabric.util.setStyle(this.wrapperEl, {\n        width: this.width + 'px',\n        height: this.height + 'px',\n        position: 'relative'\n      });\n      fabric.util.makeElementUnselectable(this.wrapperEl);\n    },\n\n    /**\n     * @private\n     * @param {HTMLElement} element canvas element to apply styles on\n     */\n    _applyCanvasStyle: function (element) {\n      var width = this.width || element.width,\n          height = this.height || element.height;\n\n      fabric.util.setStyle(element, {\n        position: 'absolute',\n        width: width + 'px',\n        height: height + 'px',\n        left: 0,\n        top: 0,\n        'touch-action': this.allowTouchScrolling ? 'manipulation' : 'none',\n        '-ms-touch-action': this.allowTouchScrolling ? 'manipulation' : 'none'\n      });\n      element.width = width;\n      element.height = height;\n      fabric.util.makeElementUnselectable(element);\n    },\n\n    /**\n     * Copy the entire inline style from one element (fromEl) to another (toEl)\n     * @private\n     * @param {Element} fromEl Element style is copied from\n     * @param {Element} toEl Element copied style is applied to\n     */\n    _copyCanvasStyle: function (fromEl, toEl) {\n      toEl.style.cssText = fromEl.style.cssText;\n    },\n\n    /**\n     * Returns context of canvas where object selection is drawn\n     * @return {CanvasRenderingContext2D}\n     */\n    getSelectionContext: function() {\n      return this.contextTop;\n    },\n\n    /**\n     * Returns &lt;canvas> element on which object selection is drawn\n     * @return {HTMLCanvasElement}\n     */\n    getSelectionElement: function () {\n      return this.upperCanvasEl;\n    },\n\n    /**\n     * Returns currently active object\n     * @return {fabric.Object} active object\n     */\n    getActiveObject: function () {\n      return this._activeObject;\n    },\n\n    /**\n     * Returns an array with the current selected objects\n     * @return {fabric.Object} active object\n     */\n    getActiveObjects: function () {\n      var active = this._activeObject;\n      if (active) {\n        if (active.type === 'activeSelection' && active._objects) {\n          return active._objects.slice(0);\n        }\n        else {\n          return [active];\n        }\n      }\n      return [];\n    },\n\n    /**\n     * @private\n     * @param {fabric.Object} obj Object that was removed\n     */\n    _onObjectRemoved: function(obj) {\n      // removing active object should fire \"selection:cleared\" events\n      if (obj === this._activeObject) {\n        this.fire('before:selection:cleared', { target: obj });\n        this._discardActiveObject();\n        this.fire('selection:cleared', { target: obj });\n        obj.fire('deselected');\n      }\n      if (obj === this._hoveredTarget){\n        this._hoveredTarget = null;\n        this._hoveredTargets = [];\n      }\n      this.callSuper('_onObjectRemoved', obj);\n    },\n\n    /**\n     * @private\n     * Compares the old activeObject with the current one and fires correct events\n     * @param {fabric.Object} obj old activeObject\n     */\n    _fireSelectionEvents: function(oldObjects, e) {\n      var somethingChanged = false, objects = this.getActiveObjects(),\n          added = [], removed = [];\n      oldObjects.forEach(function(oldObject) {\n        if (objects.indexOf(oldObject) === -1) {\n          somethingChanged = true;\n          oldObject.fire('deselected', {\n            e: e,\n            target: oldObject\n          });\n          removed.push(oldObject);\n        }\n      });\n      objects.forEach(function(object) {\n        if (oldObjects.indexOf(object) === -1) {\n          somethingChanged = true;\n          object.fire('selected', {\n            e: e,\n            target: object\n          });\n          added.push(object);\n        }\n      });\n      if (oldObjects.length > 0 && objects.length > 0) {\n        somethingChanged && this.fire('selection:updated', {\n          e: e,\n          selected: added,\n          deselected: removed,\n        });\n      }\n      else if (objects.length > 0) {\n        this.fire('selection:created', {\n          e: e,\n          selected: added,\n        });\n      }\n      else if (oldObjects.length > 0) {\n        this.fire('selection:cleared', {\n          e: e,\n          deselected: removed,\n        });\n      }\n    },\n\n    /**\n     * Sets given object as the only active object on canvas\n     * @param {fabric.Object} object Object to set as an active one\n     * @param {Event} [e] Event (passed along when firing \"object:selected\")\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    setActiveObject: function (object, e) {\n      var currentActives = this.getActiveObjects();\n      this._setActiveObject(object, e);\n      this._fireSelectionEvents(currentActives, e);\n      return this;\n    },\n\n    /**\n     * This is a private method for now.\n     * This is supposed to be equivalent to setActiveObject but without firing\n     * any event. There is commitment to have this stay this way.\n     * This is the functional part of setActiveObject.\n     * @private\n     * @param {Object} object to set as active\n     * @param {Event} [e] Event (passed along when firing \"object:selected\")\n     * @return {Boolean} true if the selection happened\n     */\n    _setActiveObject: function(object, e) {\n      if (this._activeObject === object) {\n        return false;\n      }\n      if (!this._discardActiveObject(e, object)) {\n        return false;\n      }\n      if (object.onSelect({ e: e })) {\n        return false;\n      }\n      this._activeObject = object;\n      return true;\n    },\n\n    /**\n     * This is a private method for now.\n     * This is supposed to be equivalent to discardActiveObject but without firing\n     * any events. There is commitment to have this stay this way.\n     * This is the functional part of discardActiveObject.\n     * @param {Event} [e] Event (passed along when firing \"object:deselected\")\n     * @param {Object} object to set as active\n     * @return {Boolean} true if the selection happened\n     * @private\n     */\n    _discardActiveObject: function(e, object) {\n      var obj = this._activeObject;\n      if (obj) {\n        // onDeselect return TRUE to cancel selection;\n        if (obj.onDeselect({ e: e, object: object })) {\n          return false;\n        }\n        this._activeObject = null;\n      }\n      return true;\n    },\n\n    /**\n     * Discards currently active object and fire events. If the function is called by fabric\n     * as a consequence of a mouse event, the event is passed as a parameter and\n     * sent to the fire function for the custom events. When used as a method the\n     * e param does not have any application.\n     * @param {event} e\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    discardActiveObject: function (e) {\n      var currentActives = this.getActiveObjects(), activeObject = this.getActiveObject();\n      if (currentActives.length) {\n        this.fire('before:selection:cleared', { target: activeObject, e: e });\n      }\n      this._discardActiveObject(e);\n      this._fireSelectionEvents(currentActives, e);\n      return this;\n    },\n\n    /**\n     * Clears a canvas element and removes all event listeners\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    dispose: function () {\n      var wrapper = this.wrapperEl;\n      this.removeListeners();\n      wrapper.removeChild(this.upperCanvasEl);\n      wrapper.removeChild(this.lowerCanvasEl);\n      this.contextCache = null;\n      this.contextTop = null;\n      ['upperCanvasEl', 'cacheCanvasEl'].forEach((function(element) {\n        fabric.util.cleanUpJsdomNode(this[element]);\n        this[element] = undefined;\n      }).bind(this));\n      if (wrapper.parentNode) {\n        wrapper.parentNode.replaceChild(this.lowerCanvasEl, this.wrapperEl);\n      }\n      delete this.wrapperEl;\n      fabric.StaticCanvas.prototype.dispose.call(this);\n      return this;\n    },\n\n    /**\n     * Clears all contexts (background, main, top) of an instance\n     * @return {fabric.Canvas} thisArg\n     * @chainable\n     */\n    clear: function () {\n      // this.discardActiveGroup();\n      this.discardActiveObject();\n      this.clearContext(this.contextTop);\n      return this.callSuper('clear');\n    },\n\n    /**\n     * Draws objects' controls (borders/controls)\n     * @param {CanvasRenderingContext2D} ctx Context to render controls on\n     */\n    drawControls: function(ctx) {\n      var activeObject = this._activeObject;\n\n      if (activeObject) {\n        activeObject._renderControls(ctx);\n      }\n    },\n\n    /**\n     * @private\n     */\n    _toObject: function(instance, methodName, propertiesToInclude) {\n      //If the object is part of the current selection group, it should\n      //be transformed appropriately\n      //i.e. it should be serialised as it would appear if the selection group\n      //were to be destroyed.\n      var originalProperties = this._realizeGroupTransformOnObject(instance),\n          object = this.callSuper('_toObject', instance, methodName, propertiesToInclude);\n      //Undo the damage we did by changing all of its properties\n      this._unwindGroupTransformOnObject(instance, originalProperties);\n      return object;\n    },\n\n    /**\n     * Realises an object's group transformation on it\n     * @private\n     * @param {fabric.Object} [instance] the object to transform (gets mutated)\n     * @returns the original values of instance which were changed\n     */\n    _realizeGroupTransformOnObject: function(instance) {\n      if (instance.group && instance.group.type === 'activeSelection' && this._activeObject === instance.group) {\n        var layoutProps = ['angle', 'flipX', 'flipY', 'left', 'scaleX', 'scaleY', 'skewX', 'skewY', 'top'];\n        //Copy all the positionally relevant properties across now\n        var originalValues = {};\n        layoutProps.forEach(function(prop) {\n          originalValues[prop] = instance[prop];\n        });\n        fabric.util.addTransformToObject(instance, this._activeObject.calcOwnMatrix());\n        return originalValues;\n      }\n      else {\n        return null;\n      }\n    },\n\n    /**\n     * Restores the changed properties of instance\n     * @private\n     * @param {fabric.Object} [instance] the object to un-transform (gets mutated)\n     * @param {Object} [originalValues] the original values of instance, as returned by _realizeGroupTransformOnObject\n     */\n    _unwindGroupTransformOnObject: function(instance, originalValues) {\n      if (originalValues) {\n        instance.set(originalValues);\n      }\n    },\n\n    /**\n     * @private\n     */\n    _setSVGObject: function(markup, instance, reviver) {\n      //If the object is in a selection group, simulate what would happen to that\n      //object when the group is deselected\n      var originalProperties = this._realizeGroupTransformOnObject(instance);\n      this.callSuper('_setSVGObject', markup, instance, reviver);\n      this._unwindGroupTransformOnObject(instance, originalProperties);\n    },\n\n    setViewportTransform: function (vpt) {\n      if (this.renderOnAddRemove && this._activeObject && this._activeObject.isEditing) {\n        this._activeObject.clearContextTop();\n      }\n      fabric.StaticCanvas.prototype.setViewportTransform.call(this, vpt);\n    }\n  });\n\n  // copying static properties manually to work around Opera's bug,\n  // where \"prototype\" property is enumerable and overrides existing prototype\n  for (var prop in fabric.StaticCanvas) {\n    if (prop !== 'prototype') {\n      fabric.Canvas[prop] = fabric.StaticCanvas[prop];\n    }\n  }\n})();\n\n\n(function() {\n\n  var addListener = fabric.util.addListener,\n      removeListener = fabric.util.removeListener,\n      RIGHT_CLICK = 3, MIDDLE_CLICK = 2, LEFT_CLICK = 1,\n      addEventOptions = { passive: false };\n\n  function checkClick(e, value) {\n    return e.button && (e.button === value - 1);\n  }\n\n  fabric.util.object.extend(fabric.Canvas.prototype, /** @lends fabric.Canvas.prototype */ {\n\n    /**\n     * Contains the id of the touch event that owns the fabric transform\n     * @type Number\n     * @private\n     */\n    mainTouchId: null,\n\n    /**\n     * Adds mouse listeners to canvas\n     * @private\n     */\n    _initEventListeners: function () {\n      // in case we initialized the class twice. This should not happen normally\n      // but in some kind of applications where the canvas element may be changed\n      // this is a workaround to having double listeners.\n      this.removeListeners();\n      this._bindEvents();\n      this.addOrRemove(addListener, 'add');\n    },\n\n    /**\n     * return an event prefix pointer or mouse.\n     * @private\n     */\n    _getEventPrefix: function () {\n      return this.enablePointerEvents ? 'pointer' : 'mouse';\n    },\n\n    addOrRemove: function(functor, eventjsFunctor) {\n      var canvasElement = this.upperCanvasEl,\n          eventTypePrefix = this._getEventPrefix();\n      functor(fabric.window, 'resize', this._onResize);\n      functor(canvasElement, eventTypePrefix + 'down', this._onMouseDown);\n      functor(canvasElement, eventTypePrefix + 'move', this._onMouseMove, addEventOptions);\n      functor(canvasElement, eventTypePrefix + 'out', this._onMouseOut);\n      functor(canvasElement, eventTypePrefix + 'enter', this._onMouseEnter);\n      functor(canvasElement, 'wheel', this._onMouseWheel);\n      functor(canvasElement, 'contextmenu', this._onContextMenu);\n      functor(canvasElement, 'dblclick', this._onDoubleClick);\n      functor(canvasElement, 'dragover', this._onDragOver);\n      functor(canvasElement, 'dragenter', this._onDragEnter);\n      functor(canvasElement, 'dragleave', this._onDragLeave);\n      functor(canvasElement, 'drop', this._onDrop);\n      if (!this.enablePointerEvents) {\n        functor(canvasElement, 'touchstart', this._onTouchStart, addEventOptions);\n      }\n      if (typeof eventjs !== 'undefined' && eventjsFunctor in eventjs) {\n        eventjs[eventjsFunctor](canvasElement, 'gesture', this._onGesture);\n        eventjs[eventjsFunctor](canvasElement, 'drag', this._onDrag);\n        eventjs[eventjsFunctor](canvasElement, 'orientation', this._onOrientationChange);\n        eventjs[eventjsFunctor](canvasElement, 'shake', this._onShake);\n        eventjs[eventjsFunctor](canvasElement, 'longpress', this._onLongPress);\n      }\n    },\n\n    /**\n     * Removes all event listeners\n     */\n    removeListeners: function() {\n      this.addOrRemove(removeListener, 'remove');\n      // if you dispose on a mouseDown, before mouse up, you need to clean document to...\n      var eventTypePrefix = this._getEventPrefix();\n      removeListener(fabric.document, eventTypePrefix + 'up', this._onMouseUp);\n      removeListener(fabric.document, 'touchend', this._onTouchEnd, addEventOptions);\n      removeListener(fabric.document, eventTypePrefix + 'move', this._onMouseMove, addEventOptions);\n      removeListener(fabric.document, 'touchmove', this._onMouseMove, addEventOptions);\n    },\n\n    /**\n     * @private\n     */\n    _bindEvents: function() {\n      if (this.eventsBound) {\n        // for any reason we pass here twice we do not want to bind events twice.\n        return;\n      }\n      this._onMouseDown = this._onMouseDown.bind(this);\n      this._onTouchStart = this._onTouchStart.bind(this);\n      this._onMouseMove = this._onMouseMove.bind(this);\n      this._onMouseUp = this._onMouseUp.bind(this);\n      this._onTouchEnd = this._onTouchEnd.bind(this);\n      this._onResize = this._onResize.bind(this);\n      this._onGesture = this._onGesture.bind(this);\n      this._onDrag = this._onDrag.bind(this);\n      this._onShake = this._onShake.bind(this);\n      this._onLongPress = this._onLongPress.bind(this);\n      this._onOrientationChange = this._onOrientationChange.bind(this);\n      this._onMouseWheel = this._onMouseWheel.bind(this);\n      this._onMouseOut = this._onMouseOut.bind(this);\n      this._onMouseEnter = this._onMouseEnter.bind(this);\n      this._onContextMenu = this._onContextMenu.bind(this);\n      this._onDoubleClick = this._onDoubleClick.bind(this);\n      this._onDragOver = this._onDragOver.bind(this);\n      this._onDragEnter = this._simpleEventHandler.bind(this, 'dragenter');\n      this._onDragLeave = this._simpleEventHandler.bind(this, 'dragleave');\n      this._onDrop = this._onDrop.bind(this);\n      this.eventsBound = true;\n    },\n\n    /**\n     * @private\n     * @param {Event} [e] Event object fired on Event.js gesture\n     * @param {Event} [self] Inner Event object\n     */\n    _onGesture: function(e, self) {\n      this.__onTransformGesture && this.__onTransformGesture(e, self);\n    },\n\n    /**\n     * @private\n     * @param {Event} [e] Event object fired on Event.js drag\n     * @param {Event} [self] Inner Event object\n     */\n    _onDrag: function(e, self) {\n      this.__onDrag && this.__onDrag(e, self);\n    },\n\n    /**\n     * @private\n     * @param {Event} [e] Event object fired on wheel event\n     */\n    _onMouseWheel: function(e) {\n      this.__onMouseWheel(e);\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mousedown\n     */\n    _onMouseOut: function(e) {\n      var target = this._hoveredTarget;\n      this.fire('mouse:out', { target: target, e: e });\n      this._hoveredTarget = null;\n      target && target.fire('mouseout', { e: e });\n\n      var _this = this;\n      this._hoveredTargets.forEach(function(_target){\n        _this.fire('mouse:out', { target: target, e: e });\n        _target && target.fire('mouseout', { e: e });\n      });\n      this._hoveredTargets = [];\n\n      if (this._iTextInstances) {\n        this._iTextInstances.forEach(function(obj) {\n          if (obj.isEditing) {\n            obj.hiddenTextarea.focus();\n          }\n        });\n      }\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mouseenter\n     */\n    _onMouseEnter: function(e) {\n      // This find target and consequent 'mouse:over' is used to\n      // clear old instances on hovered target.\n      // calling findTarget has the side effect of killing target.__corner.\n      // as a short term fix we are not firing this if we are currently transforming.\n      // as a long term fix we need to separate the action of finding a target with the\n      // side effects we added to it.\n      if (!this._currentTransform && !this.findTarget(e)) {\n        this.fire('mouse:over', { target: null, e: e });\n        this._hoveredTarget = null;\n        this._hoveredTargets = [];\n      }\n    },\n\n    /**\n     * @private\n     * @param {Event} [e] Event object fired on Event.js orientation change\n     * @param {Event} [self] Inner Event object\n     */\n    _onOrientationChange: function(e, self) {\n      this.__onOrientationChange && this.__onOrientationChange(e, self);\n    },\n\n    /**\n     * @private\n     * @param {Event} [e] Event object fired on Event.js shake\n     * @param {Event} [self] Inner Event object\n     */\n    _onShake: function(e, self) {\n      this.__onShake && this.__onShake(e, self);\n    },\n\n    /**\n     * @private\n     * @param {Event} [e] Event object fired on Event.js shake\n     * @param {Event} [self] Inner Event object\n     */\n    _onLongPress: function(e, self) {\n      this.__onLongPress && this.__onLongPress(e, self);\n    },\n\n    /**\n     * prevent default to allow drop event to be fired\n     * @private\n     * @param {Event} [e] Event object fired on Event.js shake\n     */\n    _onDragOver: function(e) {\n      e.preventDefault();\n      var target = this._simpleEventHandler('dragover', e);\n      this._fireEnterLeaveEvents(target, e);\n    },\n\n    /**\n     * `drop:before` is a an event that allow you to schedule logic\n     * before the `drop` event. Prefer `drop` event always, but if you need\n     * to run some drop-disabling logic on an event, since there is no way\n     * to handle event handlers ordering, use `drop:before`\n     * @param {Event} e\n     */\n    _onDrop: function (e) {\n      this._simpleEventHandler('drop:before', e);\n      return this._simpleEventHandler('drop', e);\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mousedown\n     */\n    _onContextMenu: function (e) {\n      if (this.stopContextMenu) {\n        e.stopPropagation();\n        e.preventDefault();\n      }\n      return false;\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mousedown\n     */\n    _onDoubleClick: function (e) {\n      this._cacheTransformEventData(e);\n      this._handleEvent(e, 'dblclick');\n      this._resetTransformEventData(e);\n    },\n\n    /**\n     * Return a the id of an event.\n     * returns either the pointerId or the identifier or 0 for the mouse event\n     * @private\n     * @param {Event} evt Event object\n     */\n    getPointerId: function(evt) {\n      var changedTouches = evt.changedTouches;\n\n      if (changedTouches) {\n        return changedTouches[0] && changedTouches[0].identifier;\n      }\n\n      if (this.enablePointerEvents) {\n        return evt.pointerId;\n      }\n\n      return -1;\n    },\n\n    /**\n     * Determines if an event has the id of the event that is considered main\n     * @private\n     * @param {evt} event Event object\n     */\n    _isMainEvent: function(evt) {\n      if (evt.isPrimary === true) {\n        return true;\n      }\n      if (evt.isPrimary === false) {\n        return false;\n      }\n      if (evt.type === 'touchend' && evt.touches.length === 0) {\n        return true;\n      }\n      if (evt.changedTouches) {\n        return evt.changedTouches[0].identifier === this.mainTouchId;\n      }\n      return true;\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mousedown\n     */\n    _onTouchStart: function(e) {\n      !this.allowTouchScrolling && e.preventDefault && e.preventDefault();\n      if (this.mainTouchId === null) {\n        this.mainTouchId = this.getPointerId(e);\n      }\n      this.__onMouseDown(e);\n      this._resetTransformEventData();\n      var canvasElement = this.upperCanvasEl,\n          eventTypePrefix = this._getEventPrefix();\n      addListener(fabric.document, 'touchend', this._onTouchEnd, addEventOptions);\n      addListener(fabric.document, 'touchmove', this._onMouseMove, addEventOptions);\n      // Unbind mousedown to prevent double triggers from touch devices\n      removeListener(canvasElement, eventTypePrefix + 'down', this._onMouseDown);\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mousedown\n     */\n    _onMouseDown: function (e) {\n      this.__onMouseDown(e);\n      this._resetTransformEventData();\n      var canvasElement = this.upperCanvasEl,\n          eventTypePrefix = this._getEventPrefix();\n      removeListener(canvasElement, eventTypePrefix + 'move', this._onMouseMove, addEventOptions);\n      addListener(fabric.document, eventTypePrefix + 'up', this._onMouseUp);\n      addListener(fabric.document, eventTypePrefix + 'move', this._onMouseMove, addEventOptions);\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mousedown\n     */\n    _onTouchEnd: function(e) {\n      if (e.touches.length > 0) {\n        // if there are still touches stop here\n        return;\n      }\n      this.__onMouseUp(e);\n      this._resetTransformEventData();\n      this.mainTouchId = null;\n      var eventTypePrefix = this._getEventPrefix();\n      removeListener(fabric.document, 'touchend', this._onTouchEnd, addEventOptions);\n      removeListener(fabric.document, 'touchmove', this._onMouseMove, addEventOptions);\n      var _this = this;\n      if (this._willAddMouseDown) {\n        clearTimeout(this._willAddMouseDown);\n      }\n      this._willAddMouseDown = setTimeout(function() {\n        // Wait 400ms before rebinding mousedown to prevent double triggers\n        // from touch devices\n        addListener(_this.upperCanvasEl, eventTypePrefix + 'down', _this._onMouseDown);\n        _this._willAddMouseDown = 0;\n      }, 400);\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mouseup\n     */\n    _onMouseUp: function (e) {\n      this.__onMouseUp(e);\n      this._resetTransformEventData();\n      var canvasElement = this.upperCanvasEl,\n          eventTypePrefix = this._getEventPrefix();\n      if (this._isMainEvent(e)) {\n        removeListener(fabric.document, eventTypePrefix + 'up', this._onMouseUp);\n        removeListener(fabric.document, eventTypePrefix + 'move', this._onMouseMove, addEventOptions);\n        addListener(canvasElement, eventTypePrefix + 'move', this._onMouseMove, addEventOptions);\n      }\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mousemove\n     */\n    _onMouseMove: function (e) {\n      !this.allowTouchScrolling && e.preventDefault && e.preventDefault();\n      this.__onMouseMove(e);\n    },\n\n    /**\n     * @private\n     */\n    _onResize: function () {\n      this.calcOffset();\n    },\n\n    /**\n     * Decides whether the canvas should be redrawn in mouseup and mousedown events.\n     * @private\n     * @param {Object} target\n     */\n    _shouldRender: function(target) {\n      var activeObject = this._activeObject;\n\n      if (\n        !!activeObject !== !!target ||\n        (activeObject && target && (activeObject !== target))\n      ) {\n        // this covers: switch of target, from target to no target, selection of target\n        // multiSelection with key and mouse\n        return true;\n      }\n      else if (activeObject && activeObject.isEditing) {\n        // if we mouse up/down over a editing textbox a cursor change,\n        // there is no need to re render\n        return false;\n      }\n      return false;\n    },\n\n    /**\n     * Method that defines the actions when mouse is released on canvas.\n     * The method resets the currentTransform parameters, store the image corner\n     * position in the image object and render the canvas on top.\n     * @private\n     * @param {Event} e Event object fired on mouseup\n     */\n    __onMouseUp: function (e) {\n      var target, transform = this._currentTransform,\n          groupSelector = this._groupSelector, shouldRender = false,\n          isClick = (!groupSelector || (groupSelector.left === 0 && groupSelector.top === 0));\n      this._cacheTransformEventData(e);\n      target = this._target;\n      this._handleEvent(e, 'up:before');\n      // if right/middle click just fire events and return\n      // target undefined will make the _handleEvent search the target\n      if (checkClick(e, RIGHT_CLICK)) {\n        if (this.fireRightClick) {\n          this._handleEvent(e, 'up', RIGHT_CLICK, isClick);\n        }\n        return;\n      }\n\n      if (checkClick(e, MIDDLE_CLICK)) {\n        if (this.fireMiddleClick) {\n          this._handleEvent(e, 'up', MIDDLE_CLICK, isClick);\n        }\n        this._resetTransformEventData();\n        return;\n      }\n\n      if (this.isDrawingMode && this._isCurrentlyDrawing) {\n        this._onMouseUpInDrawingMode(e);\n        return;\n      }\n\n      if (!this._isMainEvent(e)) {\n        return;\n      }\n      if (transform) {\n        this._finalizeCurrentTransform(e);\n        shouldRender = transform.actionPerformed;\n      }\n      if (!isClick) {\n        var targetWasActive = target === this._activeObject;\n        this._maybeGroupObjects(e);\n        if (!shouldRender) {\n          shouldRender = (\n            this._shouldRender(target) ||\n            (!targetWasActive && target === this._activeObject)\n          );\n        }\n      }\n      var corner, pointer;\n      if (target) {\n        corner = target._findTargetCorner(\n          this.getPointer(e, true),\n          fabric.util.isTouchEvent(e)\n        );\n        if (target.selectable && target !== this._activeObject && target.activeOn === 'up') {\n          this.setActiveObject(target, e);\n          shouldRender = true;\n        }\n        else {\n          var control = target.controls[corner],\n              mouseUpHandler = control && control.getMouseUpHandler(e, target, control);\n          if (mouseUpHandler) {\n            pointer = this.getPointer(e);\n            mouseUpHandler(e, transform, pointer.x, pointer.y);\n          }\n        }\n        target.isMoving = false;\n      }\n      // if we are ending up a transform on a different control or a new object\n      // fire the original mouse up from the corner that started the transform\n      if (transform && (transform.target !== target || transform.corner !== corner)) {\n        var originalControl = transform.target && transform.target.controls[transform.corner],\n            originalMouseUpHandler = originalControl && originalControl.getMouseUpHandler(e, target, control);\n        pointer = pointer || this.getPointer(e);\n        originalMouseUpHandler && originalMouseUpHandler(e, transform, pointer.x, pointer.y);\n      }\n      this._setCursorFromEvent(e, target);\n      this._handleEvent(e, 'up', LEFT_CLICK, isClick);\n      this._groupSelector = null;\n      this._currentTransform = null;\n      // reset the target information about which corner is selected\n      target && (target.__corner = 0);\n      if (shouldRender) {\n        this.requestRenderAll();\n      }\n      else if (!isClick) {\n        this.renderTop();\n      }\n    },\n\n    /**\n     * @private\n     * Handle event firing for target and subtargets\n     * @param {Event} e event from mouse\n     * @param {String} eventType event to fire (up, down or move)\n     * @return {Fabric.Object} target return the the target found, for internal reasons.\n     */\n    _simpleEventHandler: function(eventType, e) {\n      var target = this.findTarget(e),\n          targets = this.targets,\n          options = {\n            e: e,\n            target: target,\n            subTargets: targets,\n          };\n      this.fire(eventType, options);\n      target && target.fire(eventType, options);\n      if (!targets) {\n        return target;\n      }\n      for (var i = 0; i < targets.length; i++) {\n        targets[i].fire(eventType, options);\n      }\n      return target;\n    },\n\n    /**\n     * @private\n     * Handle event firing for target and subtargets\n     * @param {Event} e event from mouse\n     * @param {String} eventType event to fire (up, down or move)\n     * @param {fabric.Object} targetObj receiving event\n     * @param {Number} [button] button used in the event 1 = left, 2 = middle, 3 = right\n     * @param {Boolean} isClick for left button only, indicates that the mouse up happened without move.\n     */\n    _handleEvent: function(e, eventType, button, isClick) {\n      var target = this._target,\n          targets = this.targets || [],\n          options = {\n            e: e,\n            target: target,\n            subTargets: targets,\n            button: button || LEFT_CLICK,\n            isClick: isClick || false,\n            pointer: this._pointer,\n            absolutePointer: this._absolutePointer,\n            transform: this._currentTransform\n          };\n      if (eventType === 'up') {\n        options.currentTarget = this.findTarget(e);\n        options.currentSubTargets = this.targets;\n      }\n      this.fire('mouse:' + eventType, options);\n      target && target.fire('mouse' + eventType, options);\n      for (var i = 0; i < targets.length; i++) {\n        targets[i].fire('mouse' + eventType, options);\n      }\n    },\n\n    /**\n     * @private\n     * @param {Event} e send the mouse event that generate the finalize down, so it can be used in the event\n     */\n    _finalizeCurrentTransform: function(e) {\n\n      var transform = this._currentTransform,\n          target = transform.target,\n          options = {\n            e: e,\n            target: target,\n            transform: transform,\n            action: transform.action,\n          };\n\n      if (target._scaling) {\n        target._scaling = false;\n      }\n\n      target.setCoords();\n\n      if (transform.actionPerformed || (this.stateful && target.hasStateChanged())) {\n        this._fire('modified', options);\n      }\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mousedown\n     */\n    _onMouseDownInDrawingMode: function(e) {\n      this._isCurrentlyDrawing = true;\n      if (this.getActiveObject()) {\n        this.discardActiveObject(e).requestRenderAll();\n      }\n      var pointer = this.getPointer(e);\n      this.freeDrawingBrush.onMouseDown(pointer, { e: e, pointer: pointer });\n      this._handleEvent(e, 'down');\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mousemove\n     */\n    _onMouseMoveInDrawingMode: function(e) {\n      if (this._isCurrentlyDrawing) {\n        var pointer = this.getPointer(e);\n        this.freeDrawingBrush.onMouseMove(pointer, { e: e, pointer: pointer });\n      }\n      this.setCursor(this.freeDrawingCursor);\n      this._handleEvent(e, 'move');\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object fired on mouseup\n     */\n    _onMouseUpInDrawingMode: function(e) {\n      var pointer = this.getPointer(e);\n      this._isCurrentlyDrawing = this.freeDrawingBrush.onMouseUp({ e: e, pointer: pointer });\n      this._handleEvent(e, 'up');\n    },\n\n    /**\n     * Method that defines the actions when mouse is clicked on canvas.\n     * The method inits the currentTransform parameters and renders all the\n     * canvas so the current image can be placed on the top canvas and the rest\n     * in on the container one.\n     * @private\n     * @param {Event} e Event object fired on mousedown\n     */\n    __onMouseDown: function (e) {\n      this._cacheTransformEventData(e);\n      this._handleEvent(e, 'down:before');\n      var target = this._target;\n      // if right click just fire events\n      if (checkClick(e, RIGHT_CLICK)) {\n        if (this.fireRightClick) {\n          this._handleEvent(e, 'down', RIGHT_CLICK);\n        }\n        return;\n      }\n\n      if (checkClick(e, MIDDLE_CLICK)) {\n        if (this.fireMiddleClick) {\n          this._handleEvent(e, 'down', MIDDLE_CLICK);\n        }\n        return;\n      }\n\n      if (this.isDrawingMode) {\n        this._onMouseDownInDrawingMode(e);\n        return;\n      }\n\n      if (!this._isMainEvent(e)) {\n        return;\n      }\n\n      // ignore if some object is being transformed at this moment\n      if (this._currentTransform) {\n        return;\n      }\n\n      var pointer = this._pointer;\n      // save pointer for check in __onMouseUp event\n      this._previousPointer = pointer;\n      var shouldRender = this._shouldRender(target),\n          shouldGroup = this._shouldGroup(e, target);\n      if (this._shouldClearSelection(e, target)) {\n        this.discardActiveObject(e);\n      }\n      else if (shouldGroup) {\n        this._handleGrouping(e, target);\n        target = this._activeObject;\n      }\n\n      if (this.selection && (!target ||\n        (!target.selectable && !target.isEditing && target !== this._activeObject))) {\n        this._groupSelector = {\n          ex: this._absolutePointer.x,\n          ey: this._absolutePointer.y,\n          top: 0,\n          left: 0\n        };\n      }\n\n      if (target) {\n        var alreadySelected = target === this._activeObject;\n        if (target.selectable && target.activeOn === 'down') {\n          this.setActiveObject(target, e);\n        }\n        var corner = target._findTargetCorner(\n          this.getPointer(e, true),\n          fabric.util.isTouchEvent(e)\n        );\n        target.__corner = corner;\n        if (target === this._activeObject && (corner || !shouldGroup)) {\n          this._setupCurrentTransform(e, target, alreadySelected);\n          var control = target.controls[corner],\n              pointer = this.getPointer(e),\n              mouseDownHandler = control && control.getMouseDownHandler(e, target, control);\n          if (mouseDownHandler) {\n            mouseDownHandler(e, this._currentTransform, pointer.x, pointer.y);\n          }\n        }\n      }\n      this._handleEvent(e, 'down');\n      // we must renderAll so that we update the visuals\n      (shouldRender || shouldGroup) && this.requestRenderAll();\n    },\n\n    /**\n     * reset cache form common information needed during event processing\n     * @private\n     */\n    _resetTransformEventData: function() {\n      this._target = null;\n      this._pointer = null;\n      this._absolutePointer = null;\n    },\n\n    /**\n     * Cache common information needed during event processing\n     * @private\n     * @param {Event} e Event object fired on event\n     */\n    _cacheTransformEventData: function(e) {\n      // reset in order to avoid stale caching\n      this._resetTransformEventData();\n      this._pointer = this.getPointer(e, true);\n      this._absolutePointer = this.restorePointerVpt(this._pointer);\n      this._target = this._currentTransform ? this._currentTransform.target : this.findTarget(e) || null;\n    },\n\n    /**\n     * @private\n     */\n    _beforeTransform: function(e) {\n      var t = this._currentTransform;\n      this.stateful && t.target.saveState();\n      this.fire('before:transform', {\n        e: e,\n        transform: t,\n      });\n    },\n\n    /**\n     * Method that defines the actions when mouse is hovering the canvas.\n     * The currentTransform parameter will define whether the user is rotating/scaling/translating\n     * an image or neither of them (only hovering). A group selection is also possible and would cancel\n     * all any other type of action.\n     * In case of an image transformation only the top canvas will be rendered.\n     * @private\n     * @param {Event} e Event object fired on mousemove\n     */\n    __onMouseMove: function (e) {\n      this._handleEvent(e, 'move:before');\n      this._cacheTransformEventData(e);\n      var target, pointer;\n\n      if (this.isDrawingMode) {\n        this._onMouseMoveInDrawingMode(e);\n        return;\n      }\n\n      if (!this._isMainEvent(e)) {\n        return;\n      }\n\n      var groupSelector = this._groupSelector;\n\n      // We initially clicked in an empty area, so we draw a box for multiple selection\n      if (groupSelector) {\n        pointer = this._absolutePointer;\n\n        groupSelector.left = pointer.x - groupSelector.ex;\n        groupSelector.top = pointer.y - groupSelector.ey;\n\n        this.renderTop();\n      }\n      else if (!this._currentTransform) {\n        target = this.findTarget(e) || null;\n        this._setCursorFromEvent(e, target);\n        this._fireOverOutEvents(target, e);\n      }\n      else {\n        this._transformObject(e);\n      }\n      this._handleEvent(e, 'move');\n      this._resetTransformEventData();\n    },\n\n    /**\n     * Manage the mouseout, mouseover events for the fabric object on the canvas\n     * @param {Fabric.Object} target the target where the target from the mousemove event\n     * @param {Event} e Event object fired on mousemove\n     * @private\n     */\n    _fireOverOutEvents: function(target, e) {\n      var _hoveredTarget = this._hoveredTarget,\n          _hoveredTargets = this._hoveredTargets, targets = this.targets,\n          length = Math.max(_hoveredTargets.length, targets.length);\n\n      this.fireSyntheticInOutEvents(target, e, {\n        oldTarget: _hoveredTarget,\n        evtOut: 'mouseout',\n        canvasEvtOut: 'mouse:out',\n        evtIn: 'mouseover',\n        canvasEvtIn: 'mouse:over',\n      });\n      for (var i = 0; i < length; i++){\n        this.fireSyntheticInOutEvents(targets[i], e, {\n          oldTarget: _hoveredTargets[i],\n          evtOut: 'mouseout',\n          evtIn: 'mouseover',\n        });\n      }\n      this._hoveredTarget = target;\n      this._hoveredTargets = this.targets.concat();\n    },\n\n    /**\n     * Manage the dragEnter, dragLeave events for the fabric objects on the canvas\n     * @param {Fabric.Object} target the target where the target from the onDrag event\n     * @param {Event} e Event object fired on ondrag\n     * @private\n     */\n    _fireEnterLeaveEvents: function(target, e) {\n      var _draggedoverTarget = this._draggedoverTarget,\n          _hoveredTargets = this._hoveredTargets, targets = this.targets,\n          length = Math.max(_hoveredTargets.length, targets.length);\n\n      this.fireSyntheticInOutEvents(target, e, {\n        oldTarget: _draggedoverTarget,\n        evtOut: 'dragleave',\n        evtIn: 'dragenter',\n      });\n      for (var i = 0; i < length; i++) {\n        this.fireSyntheticInOutEvents(targets[i], e, {\n          oldTarget: _hoveredTargets[i],\n          evtOut: 'dragleave',\n          evtIn: 'dragenter',\n        });\n      }\n      this._draggedoverTarget = target;\n    },\n\n    /**\n     * Manage the synthetic in/out events for the fabric objects on the canvas\n     * @param {Fabric.Object} target the target where the target from the supported events\n     * @param {Event} e Event object fired\n     * @param {Object} config configuration for the function to work\n     * @param {String} config.targetName property on the canvas where the old target is stored\n     * @param {String} [config.canvasEvtOut] name of the event to fire at canvas level for out\n     * @param {String} config.evtOut name of the event to fire for out\n     * @param {String} [config.canvasEvtIn] name of the event to fire at canvas level for in\n     * @param {String} config.evtIn name of the event to fire for in\n     * @private\n     */\n    fireSyntheticInOutEvents: function(target, e, config) {\n      var inOpt, outOpt, oldTarget = config.oldTarget, outFires, inFires,\n          targetChanged = oldTarget !== target, canvasEvtIn = config.canvasEvtIn, canvasEvtOut = config.canvasEvtOut;\n      if (targetChanged) {\n        inOpt = { e: e, target: target, previousTarget: oldTarget };\n        outOpt = { e: e, target: oldTarget, nextTarget: target };\n      }\n      inFires = target && targetChanged;\n      outFires = oldTarget && targetChanged;\n      if (outFires) {\n        canvasEvtOut && this.fire(canvasEvtOut, outOpt);\n        oldTarget.fire(config.evtOut, outOpt);\n      }\n      if (inFires) {\n        canvasEvtIn && this.fire(canvasEvtIn, inOpt);\n        target.fire(config.evtIn, inOpt);\n      }\n    },\n\n    /**\n     * Method that defines actions when an Event Mouse Wheel\n     * @param {Event} e Event object fired on mouseup\n     */\n    __onMouseWheel: function(e) {\n      this._cacheTransformEventData(e);\n      this._handleEvent(e, 'wheel');\n      this._resetTransformEventData();\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event fired on mousemove\n     */\n    _transformObject: function(e) {\n      var pointer = this.getPointer(e),\n          transform = this._currentTransform;\n\n      transform.reset = false;\n      transform.shiftKey = e.shiftKey;\n      transform.altKey = e[this.centeredKey];\n\n      this._performTransformAction(e, transform, pointer);\n      transform.actionPerformed && this.requestRenderAll();\n    },\n\n    /**\n     * @private\n     */\n    _performTransformAction: function(e, transform, pointer) {\n      var x = pointer.x,\n          y = pointer.y,\n          action = transform.action,\n          actionPerformed = false,\n          actionHandler = transform.actionHandler;\n          // this object could be created from the function in the control handlers\n\n\n      if (actionHandler) {\n        actionPerformed = actionHandler(e, transform, x, y);\n      }\n      if (action === 'drag' && actionPerformed) {\n        transform.target.isMoving = true;\n        this.setCursor(transform.target.moveCursor || this.moveCursor);\n      }\n      transform.actionPerformed = transform.actionPerformed || actionPerformed;\n    },\n\n    /**\n     * @private\n     */\n    _fire: fabric.controlsUtils.fireEvent,\n\n    /**\n     * Sets the cursor depending on where the canvas is being hovered.\n     * Note: very buggy in Opera\n     * @param {Event} e Event object\n     * @param {Object} target Object that the mouse is hovering, if so.\n     */\n    _setCursorFromEvent: function (e, target) {\n      if (!target) {\n        this.setCursor(this.defaultCursor);\n        return false;\n      }\n      var hoverCursor = target.hoverCursor || this.hoverCursor,\n          activeSelection = this._activeObject && this._activeObject.type === 'activeSelection' ?\n            this._activeObject : null,\n          // only show proper corner when group selection is not active\n          corner = (!activeSelection || !activeSelection.contains(target))\n          // here we call findTargetCorner always with undefined for the touch parameter.\n          // we assume that if you are using a cursor you do not need to interact with\n          // the bigger touch area.\n                    && target._findTargetCorner(this.getPointer(e, true));\n\n      if (!corner) {\n        if (target.subTargetCheck){\n          // hoverCursor should come from top-most subTarget,\n          // so we walk the array backwards\n          this.targets.concat().reverse().map(function(_target){\n            hoverCursor = _target.hoverCursor || hoverCursor;\n          });\n        }\n        this.setCursor(hoverCursor);\n      }\n      else {\n        this.setCursor(this.getCornerCursor(corner, target, e));\n      }\n    },\n\n    /**\n     * @private\n     */\n    getCornerCursor: function(corner, target, e) {\n      var control = target.controls[corner];\n      return control.cursorStyleHandler(e, control, target);\n    }\n  });\n})();\n\n\n(function() {\n\n  var min = Math.min,\n      max = Math.max;\n\n  fabric.util.object.extend(fabric.Canvas.prototype, /** @lends fabric.Canvas.prototype */ {\n\n    /**\n     * @private\n     * @param {Event} e Event object\n     * @param {fabric.Object} target\n     * @return {Boolean}\n     */\n    _shouldGroup: function(e, target) {\n      var activeObject = this._activeObject;\n      return activeObject && this._isSelectionKeyPressed(e) && target && target.selectable && this.selection &&\n            (activeObject !== target || activeObject.type === 'activeSelection') && !target.onSelect({ e: e });\n    },\n\n    /**\n     * @private\n     * @param {Event} e Event object\n     * @param {fabric.Object} target\n     */\n    _handleGrouping: function (e, target) {\n      var activeObject = this._activeObject;\n      // avoid multi select when shift click on a corner\n      if (activeObject.__corner) {\n        return;\n      }\n      if (target === activeObject) {\n        // if it's a group, find target again, using activeGroup objects\n        target = this.findTarget(e, true);\n        // if even object is not found or we are on activeObjectCorner, bail out\n        if (!target || !target.selectable) {\n          return;\n        }\n      }\n      if (activeObject && activeObject.type === 'activeSelection') {\n        this._updateActiveSelection(target, e);\n      }\n      else {\n        this._createActiveSelection(target, e);\n      }\n    },\n\n    /**\n     * @private\n     */\n    _updateActiveSelection: function(target, e) {\n      var activeSelection = this._activeObject,\n          currentActiveObjects = activeSelection._objects.slice(0);\n      if (activeSelection.contains(target)) {\n        activeSelection.removeWithUpdate(target);\n        this._hoveredTarget = target;\n        this._hoveredTargets = this.targets.concat();\n        if (activeSelection.size() === 1) {\n          // activate last remaining object\n          this._setActiveObject(activeSelection.item(0), e);\n        }\n      }\n      else {\n        activeSelection.addWithUpdate(target);\n        this._hoveredTarget = activeSelection;\n        this._hoveredTargets = this.targets.concat();\n      }\n      this._fireSelectionEvents(currentActiveObjects, e);\n    },\n\n    /**\n     * @private\n     */\n    _createActiveSelection: function(target, e) {\n      var currentActives = this.getActiveObjects(), group = this._createGroup(target);\n      this._hoveredTarget = group;\n      // ISSUE 4115: should we consider subTargets here?\n      // this._hoveredTargets = [];\n      // this._hoveredTargets = this.targets.concat();\n      this._setActiveObject(group, e);\n      this._fireSelectionEvents(currentActives, e);\n    },\n\n    /**\n     * @private\n     * @param {Object} target\n     */\n    _createGroup: function(target) {\n      var objects = this._objects,\n          isActiveLower = objects.indexOf(this._activeObject) < objects.indexOf(target),\n          groupObjects = isActiveLower\n            ? [this._activeObject, target]\n            : [target, this._activeObject];\n      this._activeObject.isEditing && this._activeObject.exitEditing();\n      return new fabric.ActiveSelection(groupObjects, {\n        canvas: this\n      });\n    },\n\n    /**\n     * @private\n     * @param {Event} e mouse event\n     */\n    _groupSelectedObjects: function (e) {\n\n      var group = this._collectObjects(e),\n          aGroup;\n\n      // do not create group for 1 element only\n      if (group.length === 1) {\n        this.setActiveObject(group[0], e);\n      }\n      else if (group.length > 1) {\n        aGroup = new fabric.ActiveSelection(group.reverse(), {\n          canvas: this\n        });\n        this.setActiveObject(aGroup, e);\n      }\n    },\n\n    /**\n     * @private\n     */\n    _collectObjects: function(e) {\n      var group = [],\n          currentObject,\n          x1 = this._groupSelector.ex,\n          y1 = this._groupSelector.ey,\n          x2 = x1 + this._groupSelector.left,\n          y2 = y1 + this._groupSelector.top,\n          selectionX1Y1 = new fabric.Point(min(x1, x2), min(y1, y2)),\n          selectionX2Y2 = new fabric.Point(max(x1, x2), max(y1, y2)),\n          allowIntersect = !this.selectionFullyContained,\n          isClick = x1 === x2 && y1 === y2;\n      // we iterate reverse order to collect top first in case of click.\n      for (var i = this._objects.length; i--; ) {\n        currentObject = this._objects[i];\n\n        if (!currentObject || !currentObject.selectable || !currentObject.visible) {\n          continue;\n        }\n\n        if ((allowIntersect && currentObject.intersectsWithRect(selectionX1Y1, selectionX2Y2, true)) ||\n            currentObject.isContainedWithinRect(selectionX1Y1, selectionX2Y2, true) ||\n            (allowIntersect && currentObject.containsPoint(selectionX1Y1, null, true)) ||\n            (allowIntersect && currentObject.containsPoint(selectionX2Y2, null, true))\n        ) {\n          group.push(currentObject);\n          // only add one object if it's a click\n          if (isClick) {\n            break;\n          }\n        }\n      }\n\n      if (group.length > 1) {\n        group = group.filter(function(object) {\n          return !object.onSelect({ e: e });\n        });\n      }\n\n      return group;\n    },\n\n    /**\n     * @private\n     */\n    _maybeGroupObjects: function(e) {\n      if (this.selection && this._groupSelector) {\n        this._groupSelectedObjects(e);\n      }\n      this.setCursor(this.defaultCursor);\n      // clear selection and current transformation\n      this._groupSelector = null;\n    }\n  });\n\n})();\n\n\n(function () {\n  fabric.util.object.extend(fabric.StaticCanvas.prototype, /** @lends fabric.StaticCanvas.prototype */ {\n\n    /**\n     * Exports canvas element to a dataurl image. Note that when multiplier is used, cropping is scaled appropriately\n     * @param {Object} [options] Options object\n     * @param {String} [options.format=png] The format of the output image. Either \"jpeg\" or \"png\"\n     * @param {Number} [options.quality=1] Quality level (0..1). Only used for jpeg.\n     * @param {Number} [options.multiplier=1] Multiplier to scale by, to have consistent\n     * @param {Number} [options.left] Cropping left offset. Introduced in v1.2.14\n     * @param {Number} [options.top] Cropping top offset. Introduced in v1.2.14\n     * @param {Number} [options.width] Cropping width. Introduced in v1.2.14\n     * @param {Number} [options.height] Cropping height. Introduced in v1.2.14\n     * @param {Boolean} [options.enableRetinaScaling] Enable retina scaling for clone image. Introduce in 2.0.0\n     * @return {String} Returns a data: URL containing a representation of the object in the format specified by options.format\n     * @see {@link http://jsfiddle.net/fabricjs/NfZVb/|jsFiddle demo}\n     * @example <caption>Generate jpeg dataURL with lower quality</caption>\n     * var dataURL = canvas.toDataURL({\n     *   format: 'jpeg',\n     *   quality: 0.8\n     * });\n     * @example <caption>Generate cropped png dataURL (clipping of canvas)</caption>\n     * var dataURL = canvas.toDataURL({\n     *   format: 'png',\n     *   left: 100,\n     *   top: 100,\n     *   width: 200,\n     *   height: 200\n     * });\n     * @example <caption>Generate double scaled png dataURL</caption>\n     * var dataURL = canvas.toDataURL({\n     *   format: 'png',\n     *   multiplier: 2\n     * });\n     */\n    toDataURL: function (options) {\n      options || (options = { });\n\n      var format = options.format || 'png',\n          quality = options.quality || 1,\n          multiplier = (options.multiplier || 1) * (options.enableRetinaScaling ? this.getRetinaScaling() : 1),\n          canvasEl = this.toCanvasElement(multiplier, options);\n      return fabric.util.toDataURL(canvasEl, format, quality);\n    },\n\n    /**\n     * Create a new HTMLCanvas element painted with the current canvas content.\n     * No need to resize the actual one or repaint it.\n     * Will transfer object ownership to a new canvas, paint it, and set everything back.\n     * This is an intermediary step used to get to a dataUrl but also it is useful to\n     * create quick image copies of a canvas without passing for the dataUrl string\n     * @param {Number} [multiplier] a zoom factor.\n     * @param {Object} [cropping] Cropping informations\n     * @param {Number} [cropping.left] Cropping left offset.\n     * @param {Number} [cropping.top] Cropping top offset.\n     * @param {Number} [cropping.width] Cropping width.\n     * @param {Number} [cropping.height] Cropping height.\n     */\n    toCanvasElement: function(multiplier, cropping) {\n      multiplier = multiplier || 1;\n      cropping = cropping || { };\n      var scaledWidth = (cropping.width || this.width) * multiplier,\n          scaledHeight = (cropping.height || this.height) * multiplier,\n          zoom = this.getZoom(),\n          originalWidth = this.width,\n          originalHeight = this.height,\n          newZoom = zoom * multiplier,\n          vp = this.viewportTransform,\n          translateX = (vp[4] - (cropping.left || 0)) * multiplier,\n          translateY = (vp[5] - (cropping.top || 0)) * multiplier,\n          originalInteractive = this.interactive,\n          newVp = [newZoom, 0, 0, newZoom, translateX, translateY],\n          originalRetina = this.enableRetinaScaling,\n          canvasEl = fabric.util.createCanvasElement(),\n          originalContextTop = this.contextTop;\n      canvasEl.width = scaledWidth;\n      canvasEl.height = scaledHeight;\n      this.contextTop = null;\n      this.enableRetinaScaling = false;\n      this.interactive = false;\n      this.viewportTransform = newVp;\n      this.width = scaledWidth;\n      this.height = scaledHeight;\n      this.calcViewportBoundaries();\n      this.renderCanvas(canvasEl.getContext('2d'), this._objects);\n      this.viewportTransform = vp;\n      this.width = originalWidth;\n      this.height = originalHeight;\n      this.calcViewportBoundaries();\n      this.interactive = originalInteractive;\n      this.enableRetinaScaling = originalRetina;\n      this.contextTop = originalContextTop;\n      return canvasEl;\n    },\n  });\n\n})();\n\n\nfabric.util.object.extend(fabric.StaticCanvas.prototype, /** @lends fabric.StaticCanvas.prototype */ {\n  /**\n   * Populates canvas with data from the specified JSON.\n   * JSON format must conform to the one of {@link fabric.Canvas#toJSON}\n   * @param {String|Object} json JSON string or object\n   * @param {Function} callback Callback, invoked when json is parsed\n   *                            and corresponding objects (e.g: {@link fabric.Image})\n   *                            are initialized\n   * @param {Function} [reviver] Method for further parsing of JSON elements, called after each fabric object created.\n   * @return {fabric.Canvas} instance\n   * @chainable\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-3#deserialization}\n   * @see {@link http://jsfiddle.net/fabricjs/fmgXt/|jsFiddle demo}\n   * @example <caption>loadFromJSON</caption>\n   * canvas.loadFromJSON(json, canvas.renderAll.bind(canvas));\n   * @example <caption>loadFromJSON with reviver</caption>\n   * canvas.loadFromJSON(json, canvas.renderAll.bind(canvas), function(o, object) {\n   *   // `o` = json object\n   *   // `object` = fabric.Object instance\n   *   // ... do some stuff ...\n   * });\n   */\n  loadFromJSON: function (json, callback, reviver) {\n    if (!json) {\n      return;\n    }\n\n    // serialize if it wasn't already\n    var serialized = (typeof json === 'string')\n      ? JSON.parse(json)\n      : fabric.util.object.clone(json);\n\n    var _this = this,\n        clipPath = serialized.clipPath,\n        renderOnAddRemove = this.renderOnAddRemove;\n\n    this.renderOnAddRemove = false;\n\n    delete serialized.clipPath;\n\n    this._enlivenObjects(serialized.objects, function (enlivenedObjects) {\n      _this.clear();\n      _this._setBgOverlay(serialized, function () {\n        if (clipPath) {\n          _this._enlivenObjects([clipPath], function (enlivenedCanvasClip) {\n            _this.clipPath = enlivenedCanvasClip[0];\n            _this.__setupCanvas.call(_this, serialized, enlivenedObjects, renderOnAddRemove, callback);\n          });\n        }\n        else {\n          _this.__setupCanvas.call(_this, serialized, enlivenedObjects, renderOnAddRemove, callback);\n        }\n      });\n    }, reviver);\n    return this;\n  },\n\n  /**\n   * @private\n   * @param {Object} serialized Object with background and overlay information\n   * @param {Array} restored canvas objects\n   * @param {Function} cached renderOnAddRemove callback\n   * @param {Function} callback Invoked after all background and overlay images/patterns loaded\n   */\n  __setupCanvas: function(serialized, enlivenedObjects, renderOnAddRemove, callback) {\n    var _this = this;\n    enlivenedObjects.forEach(function(obj, index) {\n      // we splice the array just in case some custom classes restored from JSON\n      // will add more object to canvas at canvas init.\n      _this.insertAt(obj, index);\n    });\n    this.renderOnAddRemove = renderOnAddRemove;\n    // remove parts i cannot set as options\n    delete serialized.objects;\n    delete serialized.backgroundImage;\n    delete serialized.overlayImage;\n    delete serialized.background;\n    delete serialized.overlay;\n    // this._initOptions does too many things to just\n    // call it. Normally loading an Object from JSON\n    // create the Object instance. Here the Canvas is\n    // already an instance and we are just loading things over it\n    this._setOptions(serialized);\n    this.renderAll();\n    callback && callback();\n  },\n\n  /**\n   * @private\n   * @param {Object} serialized Object with background and overlay information\n   * @param {Function} callback Invoked after all background and overlay images/patterns loaded\n   */\n  _setBgOverlay: function(serialized, callback) {\n    var loaded = {\n      backgroundColor: false,\n      overlayColor: false,\n      backgroundImage: false,\n      overlayImage: false\n    };\n\n    if (!serialized.backgroundImage && !serialized.overlayImage && !serialized.background && !serialized.overlay) {\n      callback && callback();\n      return;\n    }\n\n    var cbIfLoaded = function () {\n      if (loaded.backgroundImage && loaded.overlayImage && loaded.backgroundColor && loaded.overlayColor) {\n        callback && callback();\n      }\n    };\n\n    this.__setBgOverlay('backgroundImage', serialized.backgroundImage, loaded, cbIfLoaded);\n    this.__setBgOverlay('overlayImage', serialized.overlayImage, loaded, cbIfLoaded);\n    this.__setBgOverlay('backgroundColor', serialized.background, loaded, cbIfLoaded);\n    this.__setBgOverlay('overlayColor', serialized.overlay, loaded, cbIfLoaded);\n  },\n\n  /**\n   * @private\n   * @param {String} property Property to set (backgroundImage, overlayImage, backgroundColor, overlayColor)\n   * @param {(Object|String)} value Value to set\n   * @param {Object} loaded Set loaded property to true if property is set\n   * @param {Object} callback Callback function to invoke after property is set\n   */\n  __setBgOverlay: function(property, value, loaded, callback) {\n    var _this = this;\n\n    if (!value) {\n      loaded[property] = true;\n      callback && callback();\n      return;\n    }\n\n    if (property === 'backgroundImage' || property === 'overlayImage') {\n      fabric.util.enlivenObjects([value], function(enlivedObject){\n        _this[property] = enlivedObject[0];\n        loaded[property] = true;\n        callback && callback();\n      });\n    }\n    else {\n      this['set' + fabric.util.string.capitalize(property, true)](value, function() {\n        loaded[property] = true;\n        callback && callback();\n      });\n    }\n  },\n\n  /**\n   * @private\n   * @param {Array} objects\n   * @param {Function} callback\n   * @param {Function} [reviver]\n   */\n  _enlivenObjects: function (objects, callback, reviver) {\n    if (!objects || objects.length === 0) {\n      callback && callback([]);\n      return;\n    }\n\n    fabric.util.enlivenObjects(objects, function(enlivenedObjects) {\n      callback && callback(enlivenedObjects);\n    }, null, reviver);\n  },\n\n  /**\n   * @private\n   * @param {String} format\n   * @param {Function} callback\n   */\n  _toDataURL: function (format, callback) {\n    this.clone(function (clone) {\n      callback(clone.toDataURL(format));\n    });\n  },\n\n  /**\n   * @private\n   * @param {String} format\n   * @param {Number} multiplier\n   * @param {Function} callback\n   */\n  _toDataURLWithMultiplier: function (format, multiplier, callback) {\n    this.clone(function (clone) {\n      callback(clone.toDataURLWithMultiplier(format, multiplier));\n    });\n  },\n\n  /**\n   * Clones canvas instance\n   * @param {Object} [callback] Receives cloned instance as a first argument\n   * @param {Array} [properties] Array of properties to include in the cloned canvas and children\n   */\n  clone: function (callback, properties) {\n    var data = JSON.stringify(this.toJSON(properties));\n    this.cloneWithoutData(function(clone) {\n      clone.loadFromJSON(data, function() {\n        callback && callback(clone);\n      });\n    });\n  },\n\n  /**\n   * Clones canvas instance without cloning existing data.\n   * This essentially copies canvas dimensions, clipping properties, etc.\n   * but leaves data empty (so that you can populate it with your own)\n   * @param {Object} [callback] Receives cloned instance as a first argument\n   */\n  cloneWithoutData: function(callback) {\n    var el = fabric.util.createCanvasElement();\n\n    el.width = this.width;\n    el.height = this.height;\n\n    var clone = new fabric.Canvas(el);\n    if (this.backgroundImage) {\n      clone.setBackgroundImage(this.backgroundImage.src, function() {\n        clone.renderAll();\n        callback && callback(clone);\n      });\n      clone.backgroundImageOpacity = this.backgroundImageOpacity;\n      clone.backgroundImageStretch = this.backgroundImageStretch;\n    }\n    else {\n      callback && callback(clone);\n    }\n  }\n});\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      extend = fabric.util.object.extend,\n      clone = fabric.util.object.clone,\n      toFixed = fabric.util.toFixed,\n      capitalize = fabric.util.string.capitalize,\n      degreesToRadians = fabric.util.degreesToRadians,\n      objectCaching = !fabric.isLikelyNode,\n      ALIASING_LIMIT = 2;\n\n  if (fabric.Object) {\n    return;\n  }\n\n  /**\n   * Root object class from which all 2d shape classes inherit from\n   * @class fabric.Object\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-1#objects}\n   * @see {@link fabric.Object#initialize} for constructor definition\n   *\n   * @fires added\n   * @fires removed\n   *\n   * @fires selected\n   * @fires deselected\n   * @fires modified\n   * @fires modified\n   * @fires moved\n   * @fires scaled\n   * @fires rotated\n   * @fires skewed\n   *\n   * @fires rotating\n   * @fires scaling\n   * @fires moving\n   * @fires skewing\n   *\n   * @fires mousedown\n   * @fires mouseup\n   * @fires mouseover\n   * @fires mouseout\n   * @fires mousewheel\n   * @fires mousedblclick\n   *\n   * @fires dragover\n   * @fires dragenter\n   * @fires dragleave\n   * @fires drop\n   */\n  fabric.Object = fabric.util.createClass(fabric.CommonMethods, /** @lends fabric.Object.prototype */ {\n\n    /**\n     * Type of an object (rect, circle, path, etc.).\n     * Note that this property is meant to be read-only and not meant to be modified.\n     * If you modify, certain parts of Fabric (such as JSON loading) won't work correctly.\n     * @type String\n     * @default\n     */\n    type:                     'object',\n\n    /**\n     * Horizontal origin of transformation of an object (one of \"left\", \"right\", \"center\")\n     * See http://jsfiddle.net/1ow02gea/244/ on how originX/originY affect objects in groups\n     * @type String\n     * @default\n     */\n    originX:                  'left',\n\n    /**\n     * Vertical origin of transformation of an object (one of \"top\", \"bottom\", \"center\")\n     * See http://jsfiddle.net/1ow02gea/244/ on how originX/originY affect objects in groups\n     * @type String\n     * @default\n     */\n    originY:                  'top',\n\n    /**\n     * Top position of an object. Note that by default it's relative to object top. You can change this by setting originY={top/center/bottom}\n     * @type Number\n     * @default\n     */\n    top:                      0,\n\n    /**\n     * Left position of an object. Note that by default it's relative to object left. You can change this by setting originX={left/center/right}\n     * @type Number\n     * @default\n     */\n    left:                     0,\n\n    /**\n     * Object width\n     * @type Number\n     * @default\n     */\n    width:                    0,\n\n    /**\n     * Object height\n     * @type Number\n     * @default\n     */\n    height:                   0,\n\n    /**\n     * Object scale factor (horizontal)\n     * @type Number\n     * @default\n     */\n    scaleX:                   1,\n\n    /**\n     * Object scale factor (vertical)\n     * @type Number\n     * @default\n     */\n    scaleY:                   1,\n\n    /**\n     * When true, an object is rendered as flipped horizontally\n     * @type Boolean\n     * @default\n     */\n    flipX:                    false,\n\n    /**\n     * When true, an object is rendered as flipped vertically\n     * @type Boolean\n     * @default\n     */\n    flipY:                    false,\n\n    /**\n     * Opacity of an object\n     * @type Number\n     * @default\n     */\n    opacity:                  1,\n\n    /**\n     * Angle of rotation of an object (in degrees)\n     * @type Number\n     * @default\n     */\n    angle:                    0,\n\n    /**\n     * Angle of skew on x axes of an object (in degrees)\n     * @type Number\n     * @default\n     */\n    skewX:                    0,\n\n    /**\n     * Angle of skew on y axes of an object (in degrees)\n     * @type Number\n     * @default\n     */\n    skewY:                    0,\n\n    /**\n     * Size of object's controlling corners (in pixels)\n     * @type Number\n     * @default\n     */\n    cornerSize:               13,\n\n    /**\n     * Size of object's controlling corners when touch interaction is detected\n     * @type Number\n     * @default\n     */\n    touchCornerSize:               24,\n\n    /**\n     * When true, object's controlling corners are rendered as transparent inside (i.e. stroke instead of fill)\n     * @type Boolean\n     * @default\n     */\n    transparentCorners:       true,\n\n    /**\n     * Default cursor value used when hovering over this object on canvas\n     * @type String\n     * @default\n     */\n    hoverCursor:              null,\n\n    /**\n     * Default cursor value used when moving this object on canvas\n     * @type String\n     * @default\n     */\n    moveCursor:               null,\n\n    /**\n     * Padding between object and its controlling borders (in pixels)\n     * @type Number\n     * @default\n     */\n    padding:                  0,\n\n    /**\n     * Color of controlling borders of an object (when it's active)\n     * @type String\n     * @default\n     */\n    borderColor:              'rgb(178,204,255)',\n\n    /**\n     * Array specifying dash pattern of an object's borders (hasBorder must be true)\n     * @since 1.6.2\n     * @type Array\n     */\n    borderDashArray:          null,\n\n    /**\n     * Color of controlling corners of an object (when it's active)\n     * @type String\n     * @default\n     */\n    cornerColor:              'rgb(178,204,255)',\n\n    /**\n     * Color of controlling corners of an object (when it's active and transparentCorners false)\n     * @since 1.6.2\n     * @type String\n     * @default\n     */\n    cornerStrokeColor:        null,\n\n    /**\n     * Specify style of control, 'rect' or 'circle'\n     * @since 1.6.2\n     * @type String\n     */\n    cornerStyle:          'rect',\n\n    /**\n     * Array specifying dash pattern of an object's control (hasBorder must be true)\n     * @since 1.6.2\n     * @type Array\n     */\n    cornerDashArray:          null,\n\n    /**\n     * When true, this object will use center point as the origin of transformation\n     * when being scaled via the controls.\n     * <b>Backwards incompatibility note:</b> This property replaces \"centerTransform\" (Boolean).\n     * @since 1.3.4\n     * @type Boolean\n     * @default\n     */\n    centeredScaling:          false,\n\n    /**\n     * When true, this object will use center point as the origin of transformation\n     * when being rotated via the controls.\n     * <b>Backwards incompatibility note:</b> This property replaces \"centerTransform\" (Boolean).\n     * @since 1.3.4\n     * @type Boolean\n     * @default\n     */\n    centeredRotation:         true,\n\n    /**\n     * Color of object's fill\n     * takes css colors https://www.w3.org/TR/css-color-3/\n     * @type String\n     * @default\n     */\n    fill:                     'rgb(0,0,0)',\n\n    /**\n     * Fill rule used to fill an object\n     * accepted values are nonzero, evenodd\n     * <b>Backwards incompatibility note:</b> This property was used for setting globalCompositeOperation until v1.4.12 (use `fabric.Object#globalCompositeOperation` instead)\n     * @type String\n     * @default\n     */\n    fillRule:                 'nonzero',\n\n    /**\n     * Composite rule used for canvas globalCompositeOperation\n     * @type String\n     * @default\n     */\n    globalCompositeOperation: 'source-over',\n\n    /**\n     * Background color of an object.\n     * takes css colors https://www.w3.org/TR/css-color-3/\n     * @type String\n     * @default\n     */\n    backgroundColor:          '',\n\n    /**\n     * Selection Background color of an object. colored layer behind the object when it is active.\n     * does not mix good with globalCompositeOperation methods.\n     * @type String\n     * @default\n     */\n    selectionBackgroundColor:          '',\n\n    /**\n     * When defined, an object is rendered via stroke and this property specifies its color\n     * takes css colors https://www.w3.org/TR/css-color-3/\n     * @type String\n     * @default\n     */\n    stroke:                   null,\n\n    /**\n     * Width of a stroke used to render this object\n     * @type Number\n     * @default\n     */\n    strokeWidth:              1,\n\n    /**\n     * Array specifying dash pattern of an object's stroke (stroke must be defined)\n     * @type Array\n     */\n    strokeDashArray:          null,\n\n    /**\n     * Line offset of an object's stroke\n     * @type Number\n     * @default\n     */\n    strokeDashOffset: 0,\n\n    /**\n     * Line endings style of an object's stroke (one of \"butt\", \"round\", \"square\")\n     * @type String\n     * @default\n     */\n    strokeLineCap:            'butt',\n\n    /**\n     * Corner style of an object's stroke (one of \"bevel\", \"round\", \"miter\")\n     * @type String\n     * @default\n     */\n    strokeLineJoin:           'miter',\n\n    /**\n     * Maximum miter length (used for strokeLineJoin = \"miter\") of an object's stroke\n     * @type Number\n     * @default\n     */\n    strokeMiterLimit:         4,\n\n    /**\n     * Shadow object representing shadow of this shape\n     * @type fabric.Shadow\n     * @default\n     */\n    shadow:                   null,\n\n    /**\n     * Opacity of object's controlling borders when object is active and moving\n     * @type Number\n     * @default\n     */\n    borderOpacityWhenMoving:  0.4,\n\n    /**\n     * Scale factor of object's controlling borders\n     * bigger number will make a thicker border\n     * border is 1, so this is basically a border thickness\n     * since there is no way to change the border itself.\n     * @type Number\n     * @default\n     */\n    borderScaleFactor:        1,\n\n    /**\n     * Minimum allowed scale value of an object\n     * @type Number\n     * @default\n     */\n    minScaleLimit:            0,\n\n    /**\n     * When set to `false`, an object can not be selected for modification (using either point-click-based or group-based selection).\n     * But events still fire on it.\n     * @type Boolean\n     * @default\n     */\n    selectable:               true,\n\n    /**\n     * When set to `false`, an object can not be a target of events. All events propagate through it. Introduced in v1.3.4\n     * @type Boolean\n     * @default\n     */\n    evented:                  true,\n\n    /**\n     * When set to `false`, an object is not rendered on canvas\n     * @type Boolean\n     * @default\n     */\n    visible:                  true,\n\n    /**\n     * When set to `false`, object's controls are not displayed and can not be used to manipulate object\n     * @type Boolean\n     * @default\n     */\n    hasControls:              true,\n\n    /**\n     * When set to `false`, object's controlling borders are not rendered\n     * @type Boolean\n     * @default\n     */\n    hasBorders:               true,\n\n    /**\n     * When set to `true`, objects are \"found\" on canvas on per-pixel basis rather than according to bounding box\n     * @type Boolean\n     * @default\n     */\n    perPixelTargetFind:       false,\n\n    /**\n     * When `false`, default object's values are not included in its serialization\n     * @type Boolean\n     * @default\n     */\n    includeDefaultValues:     true,\n\n    /**\n     * When `true`, object horizontal movement is locked\n     * @type Boolean\n     * @default\n     */\n    lockMovementX:            false,\n\n    /**\n     * When `true`, object vertical movement is locked\n     * @type Boolean\n     * @default\n     */\n    lockMovementY:            false,\n\n    /**\n     * When `true`, object rotation is locked\n     * @type Boolean\n     * @default\n     */\n    lockRotation:             false,\n\n    /**\n     * When `true`, object horizontal scaling is locked\n     * @type Boolean\n     * @default\n     */\n    lockScalingX:             false,\n\n    /**\n     * When `true`, object vertical scaling is locked\n     * @type Boolean\n     * @default\n     */\n    lockScalingY:             false,\n\n    /**\n     * When `true`, object horizontal skewing is locked\n     * @type Boolean\n     * @default\n     */\n    lockSkewingX:             false,\n\n    /**\n     * When `true`, object vertical skewing is locked\n     * @type Boolean\n     * @default\n     */\n    lockSkewingY:             false,\n\n    /**\n     * When `true`, object cannot be flipped by scaling into negative values\n     * @type Boolean\n     * @default\n     */\n    lockScalingFlip:          false,\n\n    /**\n     * When `true`, object is not exported in OBJECT/JSON\n     * @since 1.6.3\n     * @type Boolean\n     * @default\n     */\n    excludeFromExport:        false,\n\n    /**\n     * When `true`, object is cached on an additional canvas.\n     * When `false`, object is not cached unless necessary ( clipPath )\n     * default to true\n     * @since 1.7.0\n     * @type Boolean\n     * @default true\n     */\n    objectCaching:            objectCaching,\n\n    /**\n     * When `true`, object properties are checked for cache invalidation. In some particular\n     * situation you may want this to be disabled ( spray brush, very big, groups)\n     * or if your application does not allow you to modify properties for groups child you want\n     * to disable it for groups.\n     * default to false\n     * since 1.7.0\n     * @type Boolean\n     * @default false\n     */\n    statefullCache:            false,\n\n    /**\n     * When `true`, cache does not get updated during scaling. The picture will get blocky if scaled\n     * too much and will be redrawn with correct details at the end of scaling.\n     * this setting is performance and application dependant.\n     * default to true\n     * since 1.7.0\n     * @type Boolean\n     * @default true\n     */\n    noScaleCache:              true,\n\n    /**\n     * When `false`, the stoke width will scale with the object.\n     * When `true`, the stroke will always match the exact pixel size entered for stroke width.\n     * this Property does not work on Text classes or drawing call that uses strokeText,fillText methods\n     * default to false\n     * @since 2.6.0\n     * @type Boolean\n     * @default false\n     * @type Boolean\n     * @default false\n     */\n    strokeUniform:              false,\n\n    /**\n     * When set to `true`, object's cache will be rerendered next render call.\n     * since 1.7.0\n     * @type Boolean\n     * @default true\n     */\n    dirty:                true,\n\n    /**\n     * keeps the value of the last hovered corner during mouse move.\n     * 0 is no corner, or 'mt', 'ml', 'mtr' etc..\n     * It should be private, but there is no harm in using it as\n     * a read-only property.\n     * @type number|string|any\n     * @default 0\n     */\n    __corner: 0,\n\n    /**\n     * Determines if the fill or the stroke is drawn first (one of \"fill\" or \"stroke\")\n     * @type String\n     * @default\n     */\n    paintFirst:           'fill',\n\n    /**\n     * When 'down', object is set to active on mousedown/touchstart\n     * When 'up', object is set to active on mouseup/touchend\n     * Experimental. Let's see if this breaks anything before supporting officially\n     * @private\n     * since 4.4.0\n     * @type String\n     * @default 'down'\n     */\n    activeOn:           'down',\n\n    /**\n     * List of properties to consider when checking if state\n     * of an object is changed (fabric.Object#hasStateChanged)\n     * as well as for history (undo/redo) purposes\n     * @type Array\n     */\n    stateProperties: (\n      'top left width height scaleX scaleY flipX flipY originX originY transformMatrix ' +\n      'stroke strokeWidth strokeDashArray strokeLineCap strokeDashOffset strokeLineJoin strokeMiterLimit ' +\n      'angle opacity fill globalCompositeOperation shadow visible backgroundColor ' +\n      'skewX skewY fillRule paintFirst clipPath strokeUniform'\n    ).split(' '),\n\n    /**\n     * List of properties to consider when checking if cache needs refresh\n     * Those properties are checked by statefullCache ON ( or lazy mode if we want ) or from single\n     * calls to Object.set(key, value). If the key is in this list, the object is marked as dirty\n     * and refreshed at the next render\n     * @type Array\n     */\n    cacheProperties: (\n      'fill stroke strokeWidth strokeDashArray width height paintFirst strokeUniform' +\n      ' strokeLineCap strokeDashOffset strokeLineJoin strokeMiterLimit backgroundColor clipPath'\n    ).split(' '),\n\n    /**\n     * List of properties to consider for animating colors.\n     * @type Array\n     */\n    colorProperties: (\n      'fill stroke backgroundColor'\n    ).split(' '),\n\n    /**\n     * a fabricObject that, without stroke define a clipping area with their shape. filled in black\n     * the clipPath object gets used when the object has rendered, and the context is placed in the center\n     * of the object cacheCanvas.\n     * If you want 0,0 of a clipPath to align with an object center, use clipPath.originX/Y to 'center'\n     * @type fabric.Object\n     */\n    clipPath: undefined,\n\n    /**\n     * Meaningful ONLY when the object is used as clipPath.\n     * if true, the clipPath will make the object clip to the outside of the clipPath\n     * since 2.4.0\n     * @type boolean\n     * @default false\n     */\n    inverted: false,\n\n    /**\n     * Meaningful ONLY when the object is used as clipPath.\n     * if true, the clipPath will have its top and left relative to canvas, and will\n     * not be influenced by the object transform. This will make the clipPath relative\n     * to the canvas, but clipping just a particular object.\n     * WARNING this is beta, this feature may change or be renamed.\n     * since 2.4.0\n     * @type boolean\n     * @default false\n     */\n    absolutePositioned: false,\n\n    /**\n     * Constructor\n     * @param {Object} [options] Options object\n     */\n    initialize: function(options) {\n      if (options) {\n        this.setOptions(options);\n      }\n    },\n\n    /**\n     * Create a the canvas used to keep the cached copy of the object\n     * @private\n     */\n    _createCacheCanvas: function() {\n      this._cacheProperties = {};\n      this._cacheCanvas = fabric.util.createCanvasElement();\n      this._cacheContext = this._cacheCanvas.getContext('2d');\n      this._updateCacheCanvas();\n      // if canvas gets created, is empty, so dirty.\n      this.dirty = true;\n    },\n\n    /**\n     * Limit the cache dimensions so that X * Y do not cross fabric.perfLimitSizeTotal\n     * and each side do not cross fabric.cacheSideLimit\n     * those numbers are configurable so that you can get as much detail as you want\n     * making bargain with performances.\n     * @param {Object} dims\n     * @param {Object} dims.width width of canvas\n     * @param {Object} dims.height height of canvas\n     * @param {Object} dims.zoomX zoomX zoom value to unscale the canvas before drawing cache\n     * @param {Object} dims.zoomY zoomY zoom value to unscale the canvas before drawing cache\n     * @return {Object}.width width of canvas\n     * @return {Object}.height height of canvas\n     * @return {Object}.zoomX zoomX zoom value to unscale the canvas before drawing cache\n     * @return {Object}.zoomY zoomY zoom value to unscale the canvas before drawing cache\n     */\n    _limitCacheSize: function(dims) {\n      var perfLimitSizeTotal = fabric.perfLimitSizeTotal,\n          width = dims.width, height = dims.height,\n          max = fabric.maxCacheSideLimit, min = fabric.minCacheSideLimit;\n      if (width <= max && height <= max && width * height <= perfLimitSizeTotal) {\n        if (width < min) {\n          dims.width = min;\n        }\n        if (height < min) {\n          dims.height = min;\n        }\n        return dims;\n      }\n      var ar = width / height, limitedDims = fabric.util.limitDimsByArea(ar, perfLimitSizeTotal),\n          capValue = fabric.util.capValue,\n          x = capValue(min, limitedDims.x, max),\n          y = capValue(min, limitedDims.y, max);\n      if (width > x) {\n        dims.zoomX /= width / x;\n        dims.width = x;\n        dims.capped = true;\n      }\n      if (height > y) {\n        dims.zoomY /= height / y;\n        dims.height = y;\n        dims.capped = true;\n      }\n      return dims;\n    },\n\n    /**\n     * Return the dimension and the zoom level needed to create a cache canvas\n     * big enough to host the object to be cached.\n     * @private\n     * @return {Object}.x width of object to be cached\n     * @return {Object}.y height of object to be cached\n     * @return {Object}.width width of canvas\n     * @return {Object}.height height of canvas\n     * @return {Object}.zoomX zoomX zoom value to unscale the canvas before drawing cache\n     * @return {Object}.zoomY zoomY zoom value to unscale the canvas before drawing cache\n     */\n    _getCacheCanvasDimensions: function() {\n      var objectScale = this.getTotalObjectScaling(),\n          // caculate dimensions without skewing\n          dim = this._getTransformedDimensions(0, 0),\n          neededX = dim.x * objectScale.scaleX / this.scaleX,\n          neededY = dim.y * objectScale.scaleY / this.scaleY;\n      return {\n        // for sure this ALIASING_LIMIT is slightly creating problem\n        // in situation in which the cache canvas gets an upper limit\n        // also objectScale contains already scaleX and scaleY\n        width: neededX + ALIASING_LIMIT,\n        height: neededY + ALIASING_LIMIT,\n        zoomX: objectScale.scaleX,\n        zoomY: objectScale.scaleY,\n        x: neededX,\n        y: neededY\n      };\n    },\n\n    /**\n     * Update width and height of the canvas for cache\n     * returns true or false if canvas needed resize.\n     * @private\n     * @return {Boolean} true if the canvas has been resized\n     */\n    _updateCacheCanvas: function() {\n      var targetCanvas = this.canvas;\n      if (this.noScaleCache && targetCanvas && targetCanvas._currentTransform) {\n        var target = targetCanvas._currentTransform.target,\n            action = targetCanvas._currentTransform.action;\n        if (this === target && action.slice && action.slice(0, 5) === 'scale') {\n          return false;\n        }\n      }\n      var canvas = this._cacheCanvas,\n          dims = this._limitCacheSize(this._getCacheCanvasDimensions()),\n          minCacheSize = fabric.minCacheSideLimit,\n          width = dims.width, height = dims.height, drawingWidth, drawingHeight,\n          zoomX = dims.zoomX, zoomY = dims.zoomY,\n          dimensionsChanged = width !== this.cacheWidth || height !== this.cacheHeight,\n          zoomChanged = this.zoomX !== zoomX || this.zoomY !== zoomY,\n          shouldRedraw = dimensionsChanged || zoomChanged,\n          additionalWidth = 0, additionalHeight = 0, shouldResizeCanvas = false;\n      if (dimensionsChanged) {\n        var canvasWidth = this._cacheCanvas.width,\n            canvasHeight = this._cacheCanvas.height,\n            sizeGrowing = width > canvasWidth || height > canvasHeight,\n            sizeShrinking = (width < canvasWidth * 0.9 || height < canvasHeight * 0.9) &&\n              canvasWidth > minCacheSize && canvasHeight > minCacheSize;\n        shouldResizeCanvas = sizeGrowing || sizeShrinking;\n        if (sizeGrowing && !dims.capped && (width > minCacheSize || height > minCacheSize)) {\n          additionalWidth = width * 0.1;\n          additionalHeight = height * 0.1;\n        }\n      }\n      if (this instanceof fabric.Text && this.path) {\n        shouldRedraw = true;\n        shouldResizeCanvas = true;\n        additionalWidth += this.getHeightOfLine(0) * this.zoomX;\n        additionalHeight += this.getHeightOfLine(0) * this.zoomY;\n      }\n      if (shouldRedraw) {\n        if (shouldResizeCanvas) {\n          canvas.width = Math.ceil(width + additionalWidth);\n          canvas.height = Math.ceil(height + additionalHeight);\n        }\n        else {\n          this._cacheContext.setTransform(1, 0, 0, 1, 0, 0);\n          this._cacheContext.clearRect(0, 0, canvas.width, canvas.height);\n        }\n        drawingWidth = dims.x / 2;\n        drawingHeight = dims.y / 2;\n        this.cacheTranslationX = Math.round(canvas.width / 2 - drawingWidth) + drawingWidth;\n        this.cacheTranslationY = Math.round(canvas.height / 2 - drawingHeight) + drawingHeight;\n        this.cacheWidth = width;\n        this.cacheHeight = height;\n        this._cacheContext.translate(this.cacheTranslationX, this.cacheTranslationY);\n        this._cacheContext.scale(zoomX, zoomY);\n        this.zoomX = zoomX;\n        this.zoomY = zoomY;\n        return true;\n      }\n      return false;\n    },\n\n    /**\n     * Sets object's properties from options\n     * @param {Object} [options] Options object\n     */\n    setOptions: function(options) {\n      this._setOptions(options);\n      this._initGradient(options.fill, 'fill');\n      this._initGradient(options.stroke, 'stroke');\n      this._initPattern(options.fill, 'fill');\n      this._initPattern(options.stroke, 'stroke');\n    },\n\n    /**\n     * Transforms context when rendering an object\n     * @param {CanvasRenderingContext2D} ctx Context\n     */\n    transform: function(ctx) {\n      var needFullTransform = (this.group && !this.group._transformDone) ||\n         (this.group && this.canvas && ctx === this.canvas.contextTop);\n      var m = this.calcTransformMatrix(!needFullTransform);\n      ctx.transform(m[0], m[1], m[2], m[3], m[4], m[5]);\n    },\n\n    /**\n     * Returns an object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      var NUM_FRACTION_DIGITS = fabric.Object.NUM_FRACTION_DIGITS,\n\n          object = {\n            type:                     this.type,\n            version:                  fabric.version,\n            originX:                  this.originX,\n            originY:                  this.originY,\n            left:                     toFixed(this.left, NUM_FRACTION_DIGITS),\n            top:                      toFixed(this.top, NUM_FRACTION_DIGITS),\n            width:                    toFixed(this.width, NUM_FRACTION_DIGITS),\n            height:                   toFixed(this.height, NUM_FRACTION_DIGITS),\n            fill:                     (this.fill && this.fill.toObject) ? this.fill.toObject() : this.fill,\n            stroke:                   (this.stroke && this.stroke.toObject) ? this.stroke.toObject() : this.stroke,\n            strokeWidth:              toFixed(this.strokeWidth, NUM_FRACTION_DIGITS),\n            strokeDashArray:          this.strokeDashArray ? this.strokeDashArray.concat() : this.strokeDashArray,\n            strokeLineCap:            this.strokeLineCap,\n            strokeDashOffset:         this.strokeDashOffset,\n            strokeLineJoin:           this.strokeLineJoin,\n            strokeUniform:            this.strokeUniform,\n            strokeMiterLimit:         toFixed(this.strokeMiterLimit, NUM_FRACTION_DIGITS),\n            scaleX:                   toFixed(this.scaleX, NUM_FRACTION_DIGITS),\n            scaleY:                   toFixed(this.scaleY, NUM_FRACTION_DIGITS),\n            angle:                    toFixed(this.angle, NUM_FRACTION_DIGITS),\n            flipX:                    this.flipX,\n            flipY:                    this.flipY,\n            opacity:                  toFixed(this.opacity, NUM_FRACTION_DIGITS),\n            shadow:                   (this.shadow && this.shadow.toObject) ? this.shadow.toObject() : this.shadow,\n            visible:                  this.visible,\n            backgroundColor:          this.backgroundColor,\n            fillRule:                 this.fillRule,\n            paintFirst:               this.paintFirst,\n            globalCompositeOperation: this.globalCompositeOperation,\n            skewX:                    toFixed(this.skewX, NUM_FRACTION_DIGITS),\n            skewY:                    toFixed(this.skewY, NUM_FRACTION_DIGITS),\n          };\n\n      if (this.clipPath && !this.clipPath.excludeFromExport) {\n        object.clipPath = this.clipPath.toObject(propertiesToInclude);\n        object.clipPath.inverted = this.clipPath.inverted;\n        object.clipPath.absolutePositioned = this.clipPath.absolutePositioned;\n      }\n\n      fabric.util.populateWithProperties(this, object, propertiesToInclude);\n      if (!this.includeDefaultValues) {\n        object = this._removeDefaultValues(object);\n      }\n\n      return object;\n    },\n\n    /**\n     * Returns (dataless) object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} Object representation of an instance\n     */\n    toDatalessObject: function(propertiesToInclude) {\n      // will be overwritten by subclasses\n      return this.toObject(propertiesToInclude);\n    },\n\n    /**\n     * @private\n     * @param {Object} object\n     */\n    _removeDefaultValues: function(object) {\n      var prototype = fabric.util.getKlass(object.type).prototype,\n          stateProperties = prototype.stateProperties;\n      stateProperties.forEach(function(prop) {\n        if (prop === 'left' || prop === 'top') {\n          return;\n        }\n        if (object[prop] === prototype[prop]) {\n          delete object[prop];\n        }\n        // basically a check for [] === []\n        if (Array.isArray(object[prop]) && Array.isArray(prototype[prop])\n          && object[prop].length === 0 && prototype[prop].length === 0) {\n          delete object[prop];\n        }\n      });\n\n      return object;\n    },\n\n    /**\n     * Returns a string representation of an instance\n     * @return {String}\n     */\n    toString: function() {\n      return '#<fabric.' + capitalize(this.type) + '>';\n    },\n\n    /**\n     * Return the object scale factor counting also the group scaling\n     * @return {Object} object with scaleX and scaleY properties\n     */\n    getObjectScaling: function() {\n      // if the object is a top level one, on the canvas, we go for simple aritmetic\n      // otherwise the complex method with angles will return approximations and decimals\n      // and will likely kill the cache when not needed\n      // https://github.com/fabricjs/fabric.js/issues/7157\n      if (!this.group) {\n        return {\n          scaleX: this.scaleX,\n          scaleY: this.scaleY,\n        };\n      }\n      // if we are inside a group total zoom calculation is complex, we defer to generic matrices\n      var options = fabric.util.qrDecompose(this.calcTransformMatrix());\n      return { scaleX: Math.abs(options.scaleX), scaleY: Math.abs(options.scaleY) };\n    },\n\n    /**\n     * Return the object scale factor counting also the group scaling, zoom and retina\n     * @return {Object} object with scaleX and scaleY properties\n     */\n    getTotalObjectScaling: function() {\n      var scale = this.getObjectScaling(), scaleX = scale.scaleX, scaleY = scale.scaleY;\n      if (this.canvas) {\n        var zoom = this.canvas.getZoom();\n        var retina = this.canvas.getRetinaScaling();\n        scaleX *= zoom * retina;\n        scaleY *= zoom * retina;\n      }\n      return { scaleX: scaleX, scaleY: scaleY };\n    },\n\n    /**\n     * Return the object opacity counting also the group property\n     * @return {Number}\n     */\n    getObjectOpacity: function() {\n      var opacity = this.opacity;\n      if (this.group) {\n        opacity *= this.group.getObjectOpacity();\n      }\n      return opacity;\n    },\n\n    /**\n     * @private\n     * @param {String} key\n     * @param {*} value\n     * @return {fabric.Object} thisArg\n     */\n    _set: function(key, value) {\n      var shouldConstrainValue = (key === 'scaleX' || key === 'scaleY'),\n          isChanged = this[key] !== value, groupNeedsUpdate = false;\n\n      if (shouldConstrainValue) {\n        value = this._constrainScale(value);\n      }\n      if (key === 'scaleX' && value < 0) {\n        this.flipX = !this.flipX;\n        value *= -1;\n      }\n      else if (key === 'scaleY' && value < 0) {\n        this.flipY = !this.flipY;\n        value *= -1;\n      }\n      else if (key === 'shadow' && value && !(value instanceof fabric.Shadow)) {\n        value = new fabric.Shadow(value);\n      }\n      else if (key === 'dirty' && this.group) {\n        this.group.set('dirty', value);\n      }\n\n      this[key] = value;\n\n      if (isChanged) {\n        groupNeedsUpdate = this.group && this.group.isOnACache();\n        if (this.cacheProperties.indexOf(key) > -1) {\n          this.dirty = true;\n          groupNeedsUpdate && this.group.set('dirty', true);\n        }\n        else if (groupNeedsUpdate && this.stateProperties.indexOf(key) > -1) {\n          this.group.set('dirty', true);\n        }\n      }\n      return this;\n    },\n\n    /**\n     * This callback function is called by the parent group of an object every\n     * time a non-delegated property changes on the group. It is passed the key\n     * and value as parameters. Not adding in this function's signature to avoid\n     * Travis build error about unused variables.\n     */\n    setOnGroup: function() {\n      // implemented by sub-classes, as needed.\n    },\n\n    /**\n     * Retrieves viewportTransform from Object's canvas if possible\n     * @method getViewportTransform\n     * @memberOf fabric.Object.prototype\n     * @return {Array}\n     */\n    getViewportTransform: function() {\n      if (this.canvas && this.canvas.viewportTransform) {\n        return this.canvas.viewportTransform;\n      }\n      return fabric.iMatrix.concat();\n    },\n\n    /*\n     * @private\n     * return if the object would be visible in rendering\n     * @memberOf fabric.Object.prototype\n     * @return {Boolean}\n     */\n    isNotVisible: function() {\n      return this.opacity === 0 ||\n        (!this.width && !this.height && this.strokeWidth === 0) ||\n        !this.visible;\n    },\n\n    /**\n     * Renders an object on a specified context\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    render: function(ctx) {\n      // do not render if width/height are zeros or object is not visible\n      if (this.isNotVisible()) {\n        return;\n      }\n      if (this.canvas && this.canvas.skipOffscreen && !this.group && !this.isOnScreen()) {\n        return;\n      }\n      ctx.save();\n      this._setupCompositeOperation(ctx);\n      this.drawSelectionBackground(ctx);\n      this.transform(ctx);\n      this._setOpacity(ctx);\n      this._setShadow(ctx, this);\n      if (this.shouldCache()) {\n        this.renderCache();\n        this.drawCacheOnCanvas(ctx);\n      }\n      else {\n        this._removeCacheCanvas();\n        this.dirty = false;\n        this.drawObject(ctx);\n        if (this.objectCaching && this.statefullCache) {\n          this.saveState({ propertySet: 'cacheProperties' });\n        }\n      }\n      ctx.restore();\n    },\n\n    renderCache: function(options) {\n      options = options || {};\n      if (!this._cacheCanvas || !this._cacheContext) {\n        this._createCacheCanvas();\n      }\n      if (this.isCacheDirty()) {\n        this.statefullCache && this.saveState({ propertySet: 'cacheProperties' });\n        this.drawObject(this._cacheContext, options.forClipping);\n        this.dirty = false;\n      }\n    },\n\n    /**\n     * Remove cacheCanvas and its dimensions from the objects\n     */\n    _removeCacheCanvas: function() {\n      this._cacheCanvas = null;\n      this._cacheContext = null;\n      this.cacheWidth = 0;\n      this.cacheHeight = 0;\n    },\n\n    /**\n     * return true if the object will draw a stroke\n     * Does not consider text styles. This is just a shortcut used at rendering time\n     * We want it to be an approximation and be fast.\n     * wrote to avoid extra caching, it has to return true when stroke happens,\n     * can guess when it will not happen at 100% chance, does not matter if it misses\n     * some use case where the stroke is invisible.\n     * @since 3.0.0\n     * @returns Boolean\n     */\n    hasStroke: function() {\n      return this.stroke && this.stroke !== 'transparent' && this.strokeWidth !== 0;\n    },\n\n    /**\n     * return true if the object will draw a fill\n     * Does not consider text styles. This is just a shortcut used at rendering time\n     * We want it to be an approximation and be fast.\n     * wrote to avoid extra caching, it has to return true when fill happens,\n     * can guess when it will not happen at 100% chance, does not matter if it misses\n     * some use case where the fill is invisible.\n     * @since 3.0.0\n     * @returns Boolean\n     */\n    hasFill: function() {\n      return this.fill && this.fill !== 'transparent';\n    },\n\n    /**\n     * When set to `true`, force the object to have its own cache, even if it is inside a group\n     * it may be needed when your object behave in a particular way on the cache and always needs\n     * its own isolated canvas to render correctly.\n     * Created to be overridden\n     * since 1.7.12\n     * @returns Boolean\n     */\n    needsItsOwnCache: function() {\n      if (this.paintFirst === 'stroke' &&\n        this.hasFill() && this.hasStroke() && typeof this.shadow === 'object') {\n        return true;\n      }\n      if (this.clipPath) {\n        return true;\n      }\n      return false;\n    },\n\n    /**\n     * Decide if the object should cache or not. Create its own cache level\n     * objectCaching is a global flag, wins over everything\n     * needsItsOwnCache should be used when the object drawing method requires\n     * a cache step. None of the fabric classes requires it.\n     * Generally you do not cache objects in groups because the group outside is cached.\n     * Read as: cache if is needed, or if the feature is enabled but we are not already caching.\n     * @return {Boolean}\n     */\n    shouldCache: function() {\n      this.ownCaching = this.needsItsOwnCache() || (\n        this.objectCaching &&\n        (!this.group || !this.group.isOnACache())\n      );\n      return this.ownCaching;\n    },\n\n    /**\n     * Check if this object or a child object will cast a shadow\n     * used by Group.shouldCache to know if child has a shadow recursively\n     * @return {Boolean}\n     */\n    willDrawShadow: function() {\n      return !!this.shadow && (this.shadow.offsetX !== 0 || this.shadow.offsetY !== 0);\n    },\n\n    /**\n     * Execute the drawing operation for an object clipPath\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {fabric.Object} clipPath\n     */\n    drawClipPathOnCache: function(ctx, clipPath) {\n      ctx.save();\n      // DEBUG: uncomment this line, comment the following\n      // ctx.globalAlpha = 0.4\n      if (clipPath.inverted) {\n        ctx.globalCompositeOperation = 'destination-out';\n      }\n      else {\n        ctx.globalCompositeOperation = 'destination-in';\n      }\n      //ctx.scale(1 / 2, 1 / 2);\n      if (clipPath.absolutePositioned) {\n        var m = fabric.util.invertTransform(this.calcTransformMatrix());\n        ctx.transform(m[0], m[1], m[2], m[3], m[4], m[5]);\n      }\n      clipPath.transform(ctx);\n      ctx.scale(1 / clipPath.zoomX, 1 / clipPath.zoomY);\n      ctx.drawImage(clipPath._cacheCanvas, -clipPath.cacheTranslationX, -clipPath.cacheTranslationY);\n      ctx.restore();\n    },\n\n    /**\n     * Execute the drawing operation for an object on a specified context\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    drawObject: function(ctx, forClipping) {\n      var originalFill = this.fill, originalStroke = this.stroke;\n      if (forClipping) {\n        this.fill = 'black';\n        this.stroke = '';\n        this._setClippingProperties(ctx);\n      }\n      else {\n        this._renderBackground(ctx);\n      }\n      this._render(ctx);\n      this._drawClipPath(ctx, this.clipPath);\n      this.fill = originalFill;\n      this.stroke = originalStroke;\n    },\n\n    /**\n     * Prepare clipPath state and cache and draw it on instance's cache\n     * @param {CanvasRenderingContext2D} ctx\n     * @param {fabric.Object} clipPath\n     */\n    _drawClipPath: function (ctx, clipPath) {\n      if (!clipPath) { return; }\n      // needed to setup a couple of variables\n      // path canvas gets overridden with this one.\n      // TODO find a better solution?\n      clipPath.canvas = this.canvas;\n      clipPath.shouldCache();\n      clipPath._transformDone = true;\n      clipPath.renderCache({ forClipping: true });\n      this.drawClipPathOnCache(ctx, clipPath);\n    },\n\n    /**\n     * Paint the cached copy of the object on the target context.\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    drawCacheOnCanvas: function(ctx) {\n      ctx.scale(1 / this.zoomX, 1 / this.zoomY);\n      ctx.drawImage(this._cacheCanvas, -this.cacheTranslationX, -this.cacheTranslationY);\n    },\n\n    /**\n     * Check if cache is dirty\n     * @param {Boolean} skipCanvas skip canvas checks because this object is painted\n     * on parent canvas.\n     */\n    isCacheDirty: function(skipCanvas) {\n      if (this.isNotVisible()) {\n        return false;\n      }\n      if (this._cacheCanvas && this._cacheContext && !skipCanvas && this._updateCacheCanvas()) {\n        // in this case the context is already cleared.\n        return true;\n      }\n      else {\n        if (this.dirty ||\n          (this.clipPath && this.clipPath.absolutePositioned) ||\n          (this.statefullCache && this.hasStateChanged('cacheProperties'))\n        ) {\n          if (this._cacheCanvas && this._cacheContext && !skipCanvas) {\n            var width = this.cacheWidth / this.zoomX;\n            var height = this.cacheHeight / this.zoomY;\n            this._cacheContext.clearRect(-width / 2, -height / 2, width, height);\n          }\n          return true;\n        }\n      }\n      return false;\n    },\n\n    /**\n     * Draws a background for the object big as its untransformed dimensions\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderBackground: function(ctx) {\n      if (!this.backgroundColor) {\n        return;\n      }\n      var dim = this._getNonTransformedDimensions();\n      ctx.fillStyle = this.backgroundColor;\n\n      ctx.fillRect(\n        -dim.x / 2,\n        -dim.y / 2,\n        dim.x,\n        dim.y\n      );\n      // if there is background color no other shadows\n      // should be casted\n      this._removeShadow(ctx);\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _setOpacity: function(ctx) {\n      if (this.group && !this.group._transformDone) {\n        ctx.globalAlpha = this.getObjectOpacity();\n      }\n      else {\n        ctx.globalAlpha *= this.opacity;\n      }\n    },\n\n    _setStrokeStyles: function(ctx, decl) {\n      var stroke = decl.stroke;\n      if (stroke) {\n        ctx.lineWidth = decl.strokeWidth;\n        ctx.lineCap = decl.strokeLineCap;\n        ctx.lineDashOffset = decl.strokeDashOffset;\n        ctx.lineJoin = decl.strokeLineJoin;\n        ctx.miterLimit = decl.strokeMiterLimit;\n        if (stroke.toLive) {\n          if (stroke.gradientUnits === 'percentage' || stroke.gradientTransform || stroke.patternTransform) {\n            // need to transform gradient in a pattern.\n            // this is a slow process. If you are hitting this codepath, and the object\n            // is not using caching, you should consider switching it on.\n            // we need a canvas as big as the current object caching canvas.\n            this._applyPatternForTransformedGradient(ctx, stroke);\n          }\n          else {\n            // is a simple gradient or pattern\n            ctx.strokeStyle = stroke.toLive(ctx, this);\n            this._applyPatternGradientTransform(ctx, stroke);\n          }\n        }\n        else {\n          // is a color\n          ctx.strokeStyle = decl.stroke;\n        }\n      }\n    },\n\n    _setFillStyles: function(ctx, decl) {\n      var fill = decl.fill;\n      if (fill) {\n        if (fill.toLive) {\n          ctx.fillStyle = fill.toLive(ctx, this);\n          this._applyPatternGradientTransform(ctx, decl.fill);\n        }\n        else {\n          ctx.fillStyle = fill;\n        }\n      }\n    },\n\n    _setClippingProperties: function(ctx) {\n      ctx.globalAlpha = 1;\n      ctx.strokeStyle = 'transparent';\n      ctx.fillStyle = '#000000';\n    },\n\n    /**\n     * @private\n     * Sets line dash\n     * @param {CanvasRenderingContext2D} ctx Context to set the dash line on\n     * @param {Array} dashArray array representing dashes\n     */\n    _setLineDash: function(ctx, dashArray) {\n      if (!dashArray || dashArray.length === 0) {\n        return;\n      }\n      // Spec requires the concatenation of two copies the dash list when the number of elements is odd\n      if (1 & dashArray.length) {\n        dashArray.push.apply(dashArray, dashArray);\n      }\n      ctx.setLineDash(dashArray);\n    },\n\n    /**\n     * Renders controls and borders for the object\n     * the context here is not transformed\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {Object} [styleOverride] properties to override the object style\n     */\n    _renderControls: function(ctx, styleOverride) {\n      var vpt = this.getViewportTransform(),\n          matrix = this.calcTransformMatrix(),\n          options, drawBorders, drawControls;\n      styleOverride = styleOverride || { };\n      drawBorders = typeof styleOverride.hasBorders !== 'undefined' ? styleOverride.hasBorders : this.hasBorders;\n      drawControls = typeof styleOverride.hasControls !== 'undefined' ? styleOverride.hasControls : this.hasControls;\n      matrix = fabric.util.multiplyTransformMatrices(vpt, matrix);\n      options = fabric.util.qrDecompose(matrix);\n      ctx.save();\n      ctx.translate(options.translateX, options.translateY);\n      ctx.lineWidth = 1 * this.borderScaleFactor;\n      if (!this.group) {\n        ctx.globalAlpha = this.isMoving ? this.borderOpacityWhenMoving : 1;\n      }\n      if (this.flipX) {\n        options.angle -= 180;\n      }\n      ctx.rotate(degreesToRadians(this.group ? options.angle : this.angle));\n      if (styleOverride.forActiveSelection || this.group) {\n        drawBorders && this.drawBordersInGroup(ctx, options, styleOverride);\n      }\n      else {\n        drawBorders && this.drawBorders(ctx, styleOverride);\n      }\n      drawControls && this.drawControls(ctx, styleOverride);\n      ctx.restore();\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _setShadow: function(ctx) {\n      if (!this.shadow) {\n        return;\n      }\n\n      var shadow = this.shadow, canvas = this.canvas, scaling,\n          multX = (canvas && canvas.viewportTransform[0]) || 1,\n          multY = (canvas && canvas.viewportTransform[3]) || 1;\n      if (shadow.nonScaling) {\n        scaling = { scaleX: 1, scaleY: 1 };\n      }\n      else {\n        scaling = this.getObjectScaling();\n      }\n      if (canvas && canvas._isRetinaScaling()) {\n        multX *= fabric.devicePixelRatio;\n        multY *= fabric.devicePixelRatio;\n      }\n      ctx.shadowColor = shadow.color;\n      ctx.shadowBlur = shadow.blur * fabric.browserShadowBlurConstant *\n        (multX + multY) * (scaling.scaleX + scaling.scaleY) / 4;\n      ctx.shadowOffsetX = shadow.offsetX * multX * scaling.scaleX;\n      ctx.shadowOffsetY = shadow.offsetY * multY * scaling.scaleY;\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _removeShadow: function(ctx) {\n      if (!this.shadow) {\n        return;\n      }\n\n      ctx.shadowColor = '';\n      ctx.shadowBlur = ctx.shadowOffsetX = ctx.shadowOffsetY = 0;\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {Object} filler fabric.Pattern or fabric.Gradient\n     * @return {Object} offset.offsetX offset for text rendering\n     * @return {Object} offset.offsetY offset for text rendering\n     */\n    _applyPatternGradientTransform: function(ctx, filler) {\n      if (!filler || !filler.toLive) {\n        return { offsetX: 0, offsetY: 0 };\n      }\n      var t = filler.gradientTransform || filler.patternTransform;\n      var offsetX = -this.width / 2 + filler.offsetX || 0,\n          offsetY = -this.height / 2 + filler.offsetY || 0;\n\n      if (filler.gradientUnits === 'percentage') {\n        ctx.transform(this.width, 0, 0, this.height, offsetX, offsetY);\n      }\n      else {\n        ctx.transform(1, 0, 0, 1, offsetX, offsetY);\n      }\n      if (t) {\n        ctx.transform(t[0], t[1], t[2], t[3], t[4], t[5]);\n      }\n      return { offsetX: offsetX, offsetY: offsetY };\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderPaintInOrder: function(ctx) {\n      if (this.paintFirst === 'stroke') {\n        this._renderStroke(ctx);\n        this._renderFill(ctx);\n      }\n      else {\n        this._renderFill(ctx);\n        this._renderStroke(ctx);\n      }\n    },\n\n    /**\n     * @private\n     * function that actually render something on the context.\n     * empty here to allow Obects to work on tests to benchmark fabric functionalites\n     * not related to rendering\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _render: function(/* ctx */) {\n\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderFill: function(ctx) {\n      if (!this.fill) {\n        return;\n      }\n\n      ctx.save();\n      this._setFillStyles(ctx, this);\n      if (this.fillRule === 'evenodd') {\n        ctx.fill('evenodd');\n      }\n      else {\n        ctx.fill();\n      }\n      ctx.restore();\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderStroke: function(ctx) {\n      if (!this.stroke || this.strokeWidth === 0) {\n        return;\n      }\n\n      if (this.shadow && !this.shadow.affectStroke) {\n        this._removeShadow(ctx);\n      }\n\n      ctx.save();\n      if (this.strokeUniform && this.group) {\n        var scaling = this.getObjectScaling();\n        ctx.scale(1 / scaling.scaleX, 1 / scaling.scaleY);\n      }\n      else if (this.strokeUniform) {\n        ctx.scale(1 / this.scaleX, 1 / this.scaleY);\n      }\n      this._setLineDash(ctx, this.strokeDashArray);\n      this._setStrokeStyles(ctx, this);\n      ctx.stroke();\n      ctx.restore();\n    },\n\n    /**\n     * This function try to patch the missing gradientTransform on canvas gradients.\n     * transforming a context to transform the gradient, is going to transform the stroke too.\n     * we want to transform the gradient but not the stroke operation, so we create\n     * a transformed gradient on a pattern and then we use the pattern instead of the gradient.\n     * this method has drwabacks: is slow, is in low resolution, needs a patch for when the size\n     * is limited.\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {fabric.Gradient} filler a fabric gradient instance\n     */\n    _applyPatternForTransformedGradient: function(ctx, filler) {\n      var dims = this._limitCacheSize(this._getCacheCanvasDimensions()),\n          pCanvas = fabric.util.createCanvasElement(), pCtx, retinaScaling = this.canvas.getRetinaScaling(),\n          width = dims.x / this.scaleX / retinaScaling, height = dims.y / this.scaleY / retinaScaling;\n      pCanvas.width = width;\n      pCanvas.height = height;\n      pCtx = pCanvas.getContext('2d');\n      pCtx.beginPath(); pCtx.moveTo(0, 0); pCtx.lineTo(width, 0); pCtx.lineTo(width, height);\n      pCtx.lineTo(0, height); pCtx.closePath();\n      pCtx.translate(width / 2, height / 2);\n      pCtx.scale(\n        dims.zoomX / this.scaleX / retinaScaling,\n        dims.zoomY / this.scaleY / retinaScaling\n      );\n      this._applyPatternGradientTransform(pCtx, filler);\n      pCtx.fillStyle = filler.toLive(ctx);\n      pCtx.fill();\n      ctx.translate(-this.width / 2 - this.strokeWidth / 2, -this.height / 2 - this.strokeWidth / 2);\n      ctx.scale(\n        retinaScaling * this.scaleX / dims.zoomX,\n        retinaScaling * this.scaleY / dims.zoomY\n      );\n      ctx.strokeStyle = pCtx.createPattern(pCanvas, 'no-repeat');\n    },\n\n    /**\n     * This function is an helper for svg import. it returns the center of the object in the svg\n     * untransformed coordinates\n     * @private\n     * @return {Object} center point from element coordinates\n     */\n    _findCenterFromElement: function() {\n      return { x: this.left + this.width / 2, y: this.top + this.height / 2 };\n    },\n\n    /**\n     * This function is an helper for svg import. it decompose the transformMatrix\n     * and assign properties to object.\n     * untransformed coordinates\n     * @private\n     * @chainable\n     */\n    _assignTransformMatrixProps: function() {\n      if (this.transformMatrix) {\n        var options = fabric.util.qrDecompose(this.transformMatrix);\n        this.flipX = false;\n        this.flipY = false;\n        this.set('scaleX', options.scaleX);\n        this.set('scaleY', options.scaleY);\n        this.angle = options.angle;\n        this.skewX = options.skewX;\n        this.skewY = 0;\n      }\n    },\n\n    /**\n     * This function is an helper for svg import. it removes the transform matrix\n     * and set to object properties that fabricjs can handle\n     * @private\n     * @param {Object} preserveAspectRatioOptions\n     * @return {thisArg}\n     */\n    _removeTransformMatrix: function(preserveAspectRatioOptions) {\n      var center = this._findCenterFromElement();\n      if (this.transformMatrix) {\n        this._assignTransformMatrixProps();\n        center = fabric.util.transformPoint(center, this.transformMatrix);\n      }\n      this.transformMatrix = null;\n      if (preserveAspectRatioOptions) {\n        this.scaleX *= preserveAspectRatioOptions.scaleX;\n        this.scaleY *= preserveAspectRatioOptions.scaleY;\n        this.cropX = preserveAspectRatioOptions.cropX;\n        this.cropY = preserveAspectRatioOptions.cropY;\n        center.x += preserveAspectRatioOptions.offsetLeft;\n        center.y += preserveAspectRatioOptions.offsetTop;\n        this.width = preserveAspectRatioOptions.width;\n        this.height = preserveAspectRatioOptions.height;\n      }\n      this.setPositionByOrigin(center, 'center', 'center');\n    },\n\n    /**\n     * Clones an instance, using a callback method will work for every object.\n     * @param {Function} callback Callback is invoked with a clone as a first argument\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     */\n    clone: function(callback, propertiesToInclude) {\n      var objectForm = this.toObject(propertiesToInclude);\n      if (this.constructor.fromObject) {\n        this.constructor.fromObject(objectForm, callback);\n      }\n      else {\n        fabric.Object._fromObject('Object', objectForm, callback);\n      }\n    },\n\n    /**\n     * Creates an instance of fabric.Image out of an object\n     * makes use of toCanvasElement.\n     * Once this method was based on toDataUrl and loadImage, so it also had a quality\n     * and format option. toCanvasElement is faster and produce no loss of quality.\n     * If you need to get a real Jpeg or Png from an object, using toDataURL is the right way to do it.\n     * toCanvasElement and then toBlob from the obtained canvas is also a good option.\n     * This method is sync now, but still support the callback because we did not want to break.\n     * When fabricJS 5.0 will be planned, this will probably be changed to not have a callback.\n     * @param {Function} callback callback, invoked with an instance as a first argument\n     * @param {Object} [options] for clone as image, passed to toDataURL\n     * @param {Number} [options.multiplier=1] Multiplier to scale by\n     * @param {Number} [options.left] Cropping left offset. Introduced in v1.2.14\n     * @param {Number} [options.top] Cropping top offset. Introduced in v1.2.14\n     * @param {Number} [options.width] Cropping width. Introduced in v1.2.14\n     * @param {Number} [options.height] Cropping height. Introduced in v1.2.14\n     * @param {Boolean} [options.enableRetinaScaling] Enable retina scaling for clone image. Introduce in 1.6.4\n     * @param {Boolean} [options.withoutTransform] Remove current object transform ( no scale , no angle, no flip, no skew ). Introduced in 2.3.4\n     * @param {Boolean} [options.withoutShadow] Remove current object shadow. Introduced in 2.4.2\n     * @return {fabric.Object} thisArg\n     */\n    cloneAsImage: function(callback, options) {\n      var canvasEl = this.toCanvasElement(options);\n      if (callback) {\n        callback(new fabric.Image(canvasEl));\n      }\n      return this;\n    },\n\n    /**\n     * Converts an object into a HTMLCanvas element\n     * @param {Object} options Options object\n     * @param {Number} [options.multiplier=1] Multiplier to scale by\n     * @param {Number} [options.left] Cropping left offset. Introduced in v1.2.14\n     * @param {Number} [options.top] Cropping top offset. Introduced in v1.2.14\n     * @param {Number} [options.width] Cropping width. Introduced in v1.2.14\n     * @param {Number} [options.height] Cropping height. Introduced in v1.2.14\n     * @param {Boolean} [options.enableRetinaScaling] Enable retina scaling for clone image. Introduce in 1.6.4\n     * @param {Boolean} [options.withoutTransform] Remove current object transform ( no scale , no angle, no flip, no skew ). Introduced in 2.3.4\n     * @param {Boolean} [options.withoutShadow] Remove current object shadow. Introduced in 2.4.2\n     * @return {HTMLCanvasElement} Returns DOM element <canvas> with the fabric.Object\n     */\n    toCanvasElement: function(options) {\n      options || (options = { });\n\n      var utils = fabric.util, origParams = utils.saveObjectTransform(this),\n          originalGroup = this.group,\n          originalShadow = this.shadow, abs = Math.abs,\n          multiplier = (options.multiplier || 1) * (options.enableRetinaScaling ? fabric.devicePixelRatio : 1);\n      delete this.group;\n      if (options.withoutTransform) {\n        utils.resetObjectTransform(this);\n      }\n      if (options.withoutShadow) {\n        this.shadow = null;\n      }\n\n      var el = fabric.util.createCanvasElement(),\n          // skip canvas zoom and calculate with setCoords now.\n          boundingRect = this.getBoundingRect(true, true),\n          shadow = this.shadow, scaling,\n          shadowOffset = { x: 0, y: 0 }, shadowBlur,\n          width, height;\n\n      if (shadow) {\n        shadowBlur = shadow.blur;\n        if (shadow.nonScaling) {\n          scaling = { scaleX: 1, scaleY: 1 };\n        }\n        else {\n          scaling = this.getObjectScaling();\n        }\n        // consider non scaling shadow.\n        shadowOffset.x = 2 * Math.round(abs(shadow.offsetX) + shadowBlur) * (abs(scaling.scaleX));\n        shadowOffset.y = 2 * Math.round(abs(shadow.offsetY) + shadowBlur) * (abs(scaling.scaleY));\n      }\n      width = boundingRect.width + shadowOffset.x;\n      height = boundingRect.height + shadowOffset.y;\n      // if the current width/height is not an integer\n      // we need to make it so.\n      el.width = Math.ceil(width);\n      el.height = Math.ceil(height);\n      var canvas = new fabric.StaticCanvas(el, {\n        enableRetinaScaling: false,\n        renderOnAddRemove: false,\n        skipOffscreen: false,\n      });\n      if (options.format === 'jpeg') {\n        canvas.backgroundColor = '#fff';\n      }\n      this.setPositionByOrigin(new fabric.Point(canvas.width / 2, canvas.height / 2), 'center', 'center');\n\n      var originalCanvas = this.canvas;\n      canvas.add(this);\n      var canvasEl = canvas.toCanvasElement(multiplier || 1, options);\n      this.shadow = originalShadow;\n      this.set('canvas', originalCanvas);\n      if (originalGroup) {\n        this.group = originalGroup;\n      }\n      this.set(origParams).setCoords();\n      // canvas.dispose will call image.dispose that will nullify the elements\n      // since this canvas is a simple element for the process, we remove references\n      // to objects in this way in order to avoid object trashing.\n      canvas._objects = [];\n      canvas.dispose();\n      canvas = null;\n\n      return canvasEl;\n    },\n\n    /**\n     * Converts an object into a data-url-like string\n     * @param {Object} options Options object\n     * @param {String} [options.format=png] The format of the output image. Either \"jpeg\" or \"png\"\n     * @param {Number} [options.quality=1] Quality level (0..1). Only used for jpeg.\n     * @param {Number} [options.multiplier=1] Multiplier to scale by\n     * @param {Number} [options.left] Cropping left offset. Introduced in v1.2.14\n     * @param {Number} [options.top] Cropping top offset. Introduced in v1.2.14\n     * @param {Number} [options.width] Cropping width. Introduced in v1.2.14\n     * @param {Number} [options.height] Cropping height. Introduced in v1.2.14\n     * @param {Boolean} [options.enableRetinaScaling] Enable retina scaling for clone image. Introduce in 1.6.4\n     * @param {Boolean} [options.withoutTransform] Remove current object transform ( no scale , no angle, no flip, no skew ). Introduced in 2.3.4\n     * @param {Boolean} [options.withoutShadow] Remove current object shadow. Introduced in 2.4.2\n     * @return {String} Returns a data: URL containing a representation of the object in the format specified by options.format\n     */\n    toDataURL: function(options) {\n      options || (options = { });\n      return fabric.util.toDataURL(this.toCanvasElement(options), options.format || 'png', options.quality || 1);\n    },\n\n    /**\n     * Returns true if specified type is identical to the type of an instance\n     * @param {String} type Type to check against\n     * @return {Boolean}\n     */\n    isType: function(type) {\n      return arguments.length > 1 ? Array.from(arguments).includes(this.type) : this.type === type;\n    },\n\n    /**\n     * Returns complexity of an instance\n     * @return {Number} complexity of this instance (is 1 unless subclassed)\n     */\n    complexity: function() {\n      return 1;\n    },\n\n    /**\n     * Returns a JSON representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} JSON\n     */\n    toJSON: function(propertiesToInclude) {\n      // delegate, not alias\n      return this.toObject(propertiesToInclude);\n    },\n\n    /**\n     * Sets \"angle\" of an instance with centered rotation\n     * @param {Number} angle Angle value (in degrees)\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    rotate: function(angle) {\n      var shouldCenterOrigin = (this.originX !== 'center' || this.originY !== 'center') && this.centeredRotation;\n\n      if (shouldCenterOrigin) {\n        this._setOriginToCenter();\n      }\n\n      this.set('angle', angle);\n\n      if (shouldCenterOrigin) {\n        this._resetOrigin();\n      }\n\n      return this;\n    },\n\n    /**\n     * Centers object horizontally on canvas to which it was added last.\n     * You might need to call `setCoords` on an object after centering, to update controls area.\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    centerH: function () {\n      this.canvas && this.canvas.centerObjectH(this);\n      return this;\n    },\n\n    /**\n     * Centers object horizontally on current viewport of canvas to which it was added last.\n     * You might need to call `setCoords` on an object after centering, to update controls area.\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    viewportCenterH: function () {\n      this.canvas && this.canvas.viewportCenterObjectH(this);\n      return this;\n    },\n\n    /**\n     * Centers object vertically on canvas to which it was added last.\n     * You might need to call `setCoords` on an object after centering, to update controls area.\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    centerV: function () {\n      this.canvas && this.canvas.centerObjectV(this);\n      return this;\n    },\n\n    /**\n     * Centers object vertically on current viewport of canvas to which it was added last.\n     * You might need to call `setCoords` on an object after centering, to update controls area.\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    viewportCenterV: function () {\n      this.canvas && this.canvas.viewportCenterObjectV(this);\n      return this;\n    },\n\n    /**\n     * Centers object vertically and horizontally on canvas to which is was added last\n     * You might need to call `setCoords` on an object after centering, to update controls area.\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    center: function () {\n      this.canvas && this.canvas.centerObject(this);\n      return this;\n    },\n\n    /**\n     * Centers object on current viewport of canvas to which it was added last.\n     * You might need to call `setCoords` on an object after centering, to update controls area.\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    viewportCenter: function () {\n      this.canvas && this.canvas.viewportCenterObject(this);\n      return this;\n    },\n\n    /**\n     * Returns coordinates of a pointer relative to an object\n     * @param {Event} e Event to operate upon\n     * @param {Object} [pointer] Pointer to operate upon (instead of event)\n     * @return {Object} Coordinates of a pointer (x, y)\n     */\n    getLocalPointer: function(e, pointer) {\n      pointer = pointer || this.canvas.getPointer(e);\n      var pClicked = new fabric.Point(pointer.x, pointer.y),\n          objectLeftTop = this._getLeftTopCoords();\n      if (this.angle) {\n        pClicked = fabric.util.rotatePoint(\n          pClicked, objectLeftTop, degreesToRadians(-this.angle));\n      }\n      return {\n        x: pClicked.x - objectLeftTop.x,\n        y: pClicked.y - objectLeftTop.y\n      };\n    },\n\n    /**\n     * Sets canvas globalCompositeOperation for specific object\n     * custom composition operation for the particular object can be specified using globalCompositeOperation property\n     * @param {CanvasRenderingContext2D} ctx Rendering canvas context\n     */\n    _setupCompositeOperation: function (ctx) {\n      if (this.globalCompositeOperation) {\n        ctx.globalCompositeOperation = this.globalCompositeOperation;\n      }\n    },\n\n    /**\n     * cancel instance's running animations\n     * override if necessary to dispose artifacts such as `clipPath`\n     */\n    dispose: function () {\n      if (fabric.runningAnimations) {\n        fabric.runningAnimations.cancelByTarget(this);\n      }\n    }\n  });\n\n  fabric.util.createAccessors && fabric.util.createAccessors(fabric.Object);\n\n  extend(fabric.Object.prototype, fabric.Observable);\n\n  /**\n   * Defines the number of fraction digits to use when serializing object values.\n   * You can use it to increase/decrease precision of such values like left, top, scaleX, scaleY, etc.\n   * @static\n   * @memberOf fabric.Object\n   * @constant\n   * @type Number\n   */\n  fabric.Object.NUM_FRACTION_DIGITS = 2;\n\n  /**\n   * Defines which properties should be enlivened from the object passed to {@link fabric.Object._fromObject}\n   * @static\n   * @memberOf fabric.Object\n   * @constant\n   * @type string[]\n   */\n  fabric.Object.ENLIVEN_PROPS = ['clipPath'];\n\n  fabric.Object._fromObject = function(className, object, callback, extraParam) {\n    var klass = fabric[className];\n    object = clone(object, true);\n    fabric.util.enlivenPatterns([object.fill, object.stroke], function(patterns) {\n      if (typeof patterns[0] !== 'undefined') {\n        object.fill = patterns[0];\n      }\n      if (typeof patterns[1] !== 'undefined') {\n        object.stroke = patterns[1];\n      }\n      fabric.util.enlivenObjectEnlivables(object, object, function () {\n        var instance = extraParam ? new klass(object[extraParam], object) : new klass(object);\n        callback && callback(instance);\n      });\n    });\n  };\n\n  /**\n   * Unique id used internally when creating SVG elements\n   * @static\n   * @memberOf fabric.Object\n   * @type Number\n   */\n  fabric.Object.__uid = 0;\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function() {\n\n  var degreesToRadians = fabric.util.degreesToRadians,\n      originXOffset = {\n        left: -0.5,\n        center: 0,\n        right: 0.5\n      },\n      originYOffset = {\n        top: -0.5,\n        center: 0,\n        bottom: 0.5\n      };\n\n  fabric.util.object.extend(fabric.Object.prototype, /** @lends fabric.Object.prototype */ {\n\n    /**\n     * Translates the coordinates from a set of origin to another (based on the object's dimensions)\n     * @param {fabric.Point} point The point which corresponds to the originX and originY params\n     * @param {String} fromOriginX Horizontal origin: 'left', 'center' or 'right'\n     * @param {String} fromOriginY Vertical origin: 'top', 'center' or 'bottom'\n     * @param {String} toOriginX Horizontal origin: 'left', 'center' or 'right'\n     * @param {String} toOriginY Vertical origin: 'top', 'center' or 'bottom'\n     * @return {fabric.Point}\n     */\n    translateToGivenOrigin: function(point, fromOriginX, fromOriginY, toOriginX, toOriginY) {\n      var x = point.x,\n          y = point.y,\n          offsetX, offsetY, dim;\n\n      if (typeof fromOriginX === 'string') {\n        fromOriginX = originXOffset[fromOriginX];\n      }\n      else {\n        fromOriginX -= 0.5;\n      }\n\n      if (typeof toOriginX === 'string') {\n        toOriginX = originXOffset[toOriginX];\n      }\n      else {\n        toOriginX -= 0.5;\n      }\n\n      offsetX = toOriginX - fromOriginX;\n\n      if (typeof fromOriginY === 'string') {\n        fromOriginY = originYOffset[fromOriginY];\n      }\n      else {\n        fromOriginY -= 0.5;\n      }\n\n      if (typeof toOriginY === 'string') {\n        toOriginY = originYOffset[toOriginY];\n      }\n      else {\n        toOriginY -= 0.5;\n      }\n\n      offsetY = toOriginY - fromOriginY;\n\n      if (offsetX || offsetY) {\n        dim = this._getTransformedDimensions();\n        x = point.x + offsetX * dim.x;\n        y = point.y + offsetY * dim.y;\n      }\n\n      return new fabric.Point(x, y);\n    },\n\n    /**\n     * Translates the coordinates from origin to center coordinates (based on the object's dimensions)\n     * @param {fabric.Point} point The point which corresponds to the originX and originY params\n     * @param {String} originX Horizontal origin: 'left', 'center' or 'right'\n     * @param {String} originY Vertical origin: 'top', 'center' or 'bottom'\n     * @return {fabric.Point}\n     */\n    translateToCenterPoint: function(point, originX, originY) {\n      var p = this.translateToGivenOrigin(point, originX, originY, 'center', 'center');\n      if (this.angle) {\n        return fabric.util.rotatePoint(p, point, degreesToRadians(this.angle));\n      }\n      return p;\n    },\n\n    /**\n     * Translates the coordinates from center to origin coordinates (based on the object's dimensions)\n     * @param {fabric.Point} center The point which corresponds to center of the object\n     * @param {String} originX Horizontal origin: 'left', 'center' or 'right'\n     * @param {String} originY Vertical origin: 'top', 'center' or 'bottom'\n     * @return {fabric.Point}\n     */\n    translateToOriginPoint: function(center, originX, originY) {\n      var p = this.translateToGivenOrigin(center, 'center', 'center', originX, originY);\n      if (this.angle) {\n        return fabric.util.rotatePoint(p, center, degreesToRadians(this.angle));\n      }\n      return p;\n    },\n\n    /**\n     * Returns the real center coordinates of the object\n     * @return {fabric.Point}\n     */\n    getCenterPoint: function() {\n      var leftTop = new fabric.Point(this.left, this.top);\n      return this.translateToCenterPoint(leftTop, this.originX, this.originY);\n    },\n\n    /**\n     * Returns the coordinates of the object based on center coordinates\n     * @param {fabric.Point} point The point which corresponds to the originX and originY params\n     * @return {fabric.Point}\n     */\n    // getOriginPoint: function(center) {\n    //   return this.translateToOriginPoint(center, this.originX, this.originY);\n    // },\n\n    /**\n     * Returns the coordinates of the object as if it has a different origin\n     * @param {String} originX Horizontal origin: 'left', 'center' or 'right'\n     * @param {String} originY Vertical origin: 'top', 'center' or 'bottom'\n     * @return {fabric.Point}\n     */\n    getPointByOrigin: function(originX, originY) {\n      var center = this.getCenterPoint();\n      return this.translateToOriginPoint(center, originX, originY);\n    },\n\n    /**\n     * Returns the point in local coordinates\n     * @param {fabric.Point} point The point relative to the global coordinate system\n     * @param {String} originX Horizontal origin: 'left', 'center' or 'right'\n     * @param {String} originY Vertical origin: 'top', 'center' or 'bottom'\n     * @return {fabric.Point}\n     */\n    toLocalPoint: function(point, originX, originY) {\n      var center = this.getCenterPoint(),\n          p, p2;\n\n      if (typeof originX !== 'undefined' && typeof originY !== 'undefined' ) {\n        p = this.translateToGivenOrigin(center, 'center', 'center', originX, originY);\n      }\n      else {\n        p = new fabric.Point(this.left, this.top);\n      }\n\n      p2 = new fabric.Point(point.x, point.y);\n      if (this.angle) {\n        p2 = fabric.util.rotatePoint(p2, center, -degreesToRadians(this.angle));\n      }\n      return p2.subtractEquals(p);\n    },\n\n    /**\n     * Returns the point in global coordinates\n     * @param {fabric.Point} The point relative to the local coordinate system\n     * @return {fabric.Point}\n     */\n    // toGlobalPoint: function(point) {\n    //   return fabric.util.rotatePoint(point, this.getCenterPoint(), degreesToRadians(this.angle)).addEquals(new fabric.Point(this.left, this.top));\n    // },\n\n    /**\n     * Sets the position of the object taking into consideration the object's origin\n     * @param {fabric.Point} pos The new position of the object\n     * @param {String} originX Horizontal origin: 'left', 'center' or 'right'\n     * @param {String} originY Vertical origin: 'top', 'center' or 'bottom'\n     * @return {void}\n     */\n    setPositionByOrigin: function(pos, originX, originY) {\n      var center = this.translateToCenterPoint(pos, originX, originY),\n          position = this.translateToOriginPoint(center, this.originX, this.originY);\n      this.set('left', position.x);\n      this.set('top', position.y);\n    },\n\n    /**\n     * @param {String} to One of 'left', 'center', 'right'\n     */\n    adjustPosition: function(to) {\n      var angle = degreesToRadians(this.angle),\n          hypotFull = this.getScaledWidth(),\n          xFull = fabric.util.cos(angle) * hypotFull,\n          yFull = fabric.util.sin(angle) * hypotFull,\n          offsetFrom, offsetTo;\n\n      //TODO: this function does not consider mixed situation like top, center.\n      if (typeof this.originX === 'string') {\n        offsetFrom = originXOffset[this.originX];\n      }\n      else {\n        offsetFrom = this.originX - 0.5;\n      }\n      if (typeof to === 'string') {\n        offsetTo = originXOffset[to];\n      }\n      else {\n        offsetTo = to - 0.5;\n      }\n      this.left += xFull * (offsetTo - offsetFrom);\n      this.top += yFull * (offsetTo - offsetFrom);\n      this.setCoords();\n      this.originX = to;\n    },\n\n    /**\n     * Sets the origin/position of the object to it's center point\n     * @private\n     * @return {void}\n     */\n    _setOriginToCenter: function() {\n      this._originalOriginX = this.originX;\n      this._originalOriginY = this.originY;\n\n      var center = this.getCenterPoint();\n\n      this.originX = 'center';\n      this.originY = 'center';\n\n      this.left = center.x;\n      this.top = center.y;\n    },\n\n    /**\n     * Resets the origin/position of the object to it's original origin\n     * @private\n     * @return {void}\n     */\n    _resetOrigin: function() {\n      var originPoint = this.translateToOriginPoint(\n        this.getCenterPoint(),\n        this._originalOriginX,\n        this._originalOriginY);\n\n      this.originX = this._originalOriginX;\n      this.originY = this._originalOriginY;\n\n      this.left = originPoint.x;\n      this.top = originPoint.y;\n\n      this._originalOriginX = null;\n      this._originalOriginY = null;\n    },\n\n    /**\n     * @private\n     */\n    _getLeftTopCoords: function() {\n      return this.translateToOriginPoint(this.getCenterPoint(), 'left', 'top');\n    },\n  });\n\n})();\n\n\n(function() {\n\n  function arrayFromCoords(coords) {\n    return [\n      new fabric.Point(coords.tl.x, coords.tl.y),\n      new fabric.Point(coords.tr.x, coords.tr.y),\n      new fabric.Point(coords.br.x, coords.br.y),\n      new fabric.Point(coords.bl.x, coords.bl.y)\n    ];\n  }\n\n  var util = fabric.util,\n      degreesToRadians = util.degreesToRadians,\n      multiplyMatrices = util.multiplyTransformMatrices,\n      transformPoint = util.transformPoint;\n\n  util.object.extend(fabric.Object.prototype, /** @lends fabric.Object.prototype */ {\n\n    /**\n     * Describe object's corner position in canvas element coordinates.\n     * properties are depending on control keys and padding the main controls.\n     * each property is an object with x, y and corner.\n     * The `corner` property contains in a similar manner the 4 points of the\n     * interactive area of the corner.\n     * The coordinates depends from the controls positionHandler and are used\n     * to draw and locate controls\n     * @memberOf fabric.Object.prototype\n     */\n    oCoords: null,\n\n    /**\n     * Describe object's corner position in canvas object absolute coordinates\n     * properties are tl,tr,bl,br and describe the four main corner.\n     * each property is an object with x, y, instance of Fabric.Point.\n     * The coordinates depends from this properties: width, height, scaleX, scaleY\n     * skewX, skewY, angle, strokeWidth, top, left.\n     * Those coordinates are useful to understand where an object is. They get updated\n     * with oCoords but they do not need to be updated when zoom or panning change.\n     * The coordinates get updated with @method setCoords.\n     * You can calculate them without updating with @method calcACoords();\n     * @memberOf fabric.Object.prototype\n     */\n    aCoords: null,\n\n    /**\n     * Describe object's corner position in canvas element coordinates.\n     * includes padding. Used of object detection.\n     * set and refreshed with setCoords.\n     * @memberOf fabric.Object.prototype\n     */\n    lineCoords: null,\n\n    /**\n     * storage for object transform matrix\n     */\n    ownMatrixCache: null,\n\n    /**\n     * storage for object full transform matrix\n     */\n    matrixCache: null,\n\n    /**\n     * custom controls interface\n     * controls are added by default_controls.js\n     */\n    controls: { },\n\n    /**\n     * return correct set of coordinates for intersection\n     * this will return either aCoords or lineCoords.\n     * @param {Boolean} absolute will return aCoords if true or lineCoords\n     * @return {Object} {tl, tr, br, bl} points\n     */\n    _getCoords: function(absolute, calculate) {\n      if (calculate) {\n        return (absolute ? this.calcACoords() : this.calcLineCoords());\n      }\n      if (!this.aCoords || !this.lineCoords) {\n        this.setCoords(true);\n      }\n      return (absolute ? this.aCoords : this.lineCoords);\n    },\n\n    /**\n     * return correct set of coordinates for intersection\n     * this will return either aCoords or lineCoords.\n     * The coords are returned in an array.\n     * @return {Array} [tl, tr, br, bl] of points\n     */\n    getCoords: function(absolute, calculate) {\n      return arrayFromCoords(this._getCoords(absolute, calculate));\n    },\n\n    /**\n     * Checks if object intersects with an area formed by 2 points\n     * @param {Object} pointTL top-left point of area\n     * @param {Object} pointBR bottom-right point of area\n     * @param {Boolean} [absolute] use coordinates without viewportTransform\n     * @param {Boolean} [calculate] use coordinates of current position instead of .oCoords\n     * @return {Boolean} true if object intersects with an area formed by 2 points\n     */\n    intersectsWithRect: function(pointTL, pointBR, absolute, calculate) {\n      var coords = this.getCoords(absolute, calculate),\n          intersection = fabric.Intersection.intersectPolygonRectangle(\n            coords,\n            pointTL,\n            pointBR\n          );\n      return intersection.status === 'Intersection';\n    },\n\n    /**\n     * Checks if object intersects with another object\n     * @param {Object} other Object to test\n     * @param {Boolean} [absolute] use coordinates without viewportTransform\n     * @param {Boolean} [calculate] use coordinates of current position instead of .oCoords\n     * @return {Boolean} true if object intersects with another object\n     */\n    intersectsWithObject: function(other, absolute, calculate) {\n      var intersection = fabric.Intersection.intersectPolygonPolygon(\n        this.getCoords(absolute, calculate),\n        other.getCoords(absolute, calculate)\n      );\n\n      return intersection.status === 'Intersection'\n        || other.isContainedWithinObject(this, absolute, calculate)\n        || this.isContainedWithinObject(other, absolute, calculate);\n    },\n\n    /**\n     * Checks if object is fully contained within area of another object\n     * @param {Object} other Object to test\n     * @param {Boolean} [absolute] use coordinates without viewportTransform\n     * @param {Boolean} [calculate] use coordinates of current position instead of .oCoords\n     * @return {Boolean} true if object is fully contained within area of another object\n     */\n    isContainedWithinObject: function(other, absolute, calculate) {\n      var points = this.getCoords(absolute, calculate),\n          otherCoords = absolute ? other.aCoords : other.lineCoords,\n          i = 0, lines = other._getImageLines(otherCoords);\n      for (; i < 4; i++) {\n        if (!other.containsPoint(points[i], lines)) {\n          return false;\n        }\n      }\n      return true;\n    },\n\n    /**\n     * Checks if object is fully contained within area formed by 2 points\n     * @param {Object} pointTL top-left point of area\n     * @param {Object} pointBR bottom-right point of area\n     * @param {Boolean} [absolute] use coordinates without viewportTransform\n     * @param {Boolean} [calculate] use coordinates of current position instead of .oCoords\n     * @return {Boolean} true if object is fully contained within area formed by 2 points\n     */\n    isContainedWithinRect: function(pointTL, pointBR, absolute, calculate) {\n      var boundingRect = this.getBoundingRect(absolute, calculate);\n\n      return (\n        boundingRect.left >= pointTL.x &&\n        boundingRect.left + boundingRect.width <= pointBR.x &&\n        boundingRect.top >= pointTL.y &&\n        boundingRect.top + boundingRect.height <= pointBR.y\n      );\n    },\n\n    /**\n     * Checks if point is inside the object\n     * @param {fabric.Point} point Point to check against\n     * @param {Object} [lines] object returned from @method _getImageLines\n     * @param {Boolean} [absolute] use coordinates without viewportTransform\n     * @param {Boolean} [calculate] use coordinates of current position instead of .oCoords\n     * @return {Boolean} true if point is inside the object\n     */\n    containsPoint: function(point, lines, absolute, calculate) {\n      var coords = this._getCoords(absolute, calculate),\n          lines = lines || this._getImageLines(coords),\n          xPoints = this._findCrossPoints(point, lines);\n      // if xPoints is odd then point is inside the object\n      return (xPoints !== 0 && xPoints % 2 === 1);\n    },\n\n    /**\n     * Checks if object is contained within the canvas with current viewportTransform\n     * the check is done stopping at first point that appears on screen\n     * @param {Boolean} [calculate] use coordinates of current position instead of .aCoords\n     * @return {Boolean} true if object is fully or partially contained within canvas\n     */\n    isOnScreen: function(calculate) {\n      if (!this.canvas) {\n        return false;\n      }\n      var pointTL = this.canvas.vptCoords.tl, pointBR = this.canvas.vptCoords.br;\n      var points = this.getCoords(true, calculate);\n      // if some point is on screen, the object is on screen.\n      if (points.some(function(point) {\n        return point.x <= pointBR.x && point.x >= pointTL.x &&\n        point.y <= pointBR.y && point.y >= pointTL.y;\n      })) {\n        return true;\n      }\n      // no points on screen, check intersection with absolute coordinates\n      if (this.intersectsWithRect(pointTL, pointBR, true, calculate)) {\n        return true;\n      }\n      return this._containsCenterOfCanvas(pointTL, pointBR, calculate);\n    },\n\n    /**\n     * Checks if the object contains the midpoint between canvas extremities\n     * Does not make sense outside the context of isOnScreen and isPartiallyOnScreen\n     * @private\n     * @param {Fabric.Point} pointTL Top Left point\n     * @param {Fabric.Point} pointBR Top Right point\n     * @param {Boolean} calculate use coordinates of current position instead of .oCoords\n     * @return {Boolean} true if the object contains the point\n     */\n    _containsCenterOfCanvas: function(pointTL, pointBR, calculate) {\n      // worst case scenario the object is so big that contains the screen\n      var centerPoint = { x: (pointTL.x + pointBR.x) / 2, y: (pointTL.y + pointBR.y) / 2 };\n      if (this.containsPoint(centerPoint, null, true, calculate)) {\n        return true;\n      }\n      return false;\n    },\n\n    /**\n     * Checks if object is partially contained within the canvas with current viewportTransform\n     * @param {Boolean} [calculate] use coordinates of current position instead of .oCoords\n     * @return {Boolean} true if object is partially contained within canvas\n     */\n    isPartiallyOnScreen: function(calculate) {\n      if (!this.canvas) {\n        return false;\n      }\n      var pointTL = this.canvas.vptCoords.tl, pointBR = this.canvas.vptCoords.br;\n      if (this.intersectsWithRect(pointTL, pointBR, true, calculate)) {\n        return true;\n      }\n      var allPointsAreOutside = this.getCoords(true, calculate).every(function(point) {\n        return (point.x >= pointBR.x || point.x <= pointTL.x) &&\n        (point.y >= pointBR.y || point.y <= pointTL.y);\n      });\n      return allPointsAreOutside && this._containsCenterOfCanvas(pointTL, pointBR, calculate);\n    },\n\n    /**\n     * Method that returns an object with the object edges in it, given the coordinates of the corners\n     * @private\n     * @param {Object} oCoords Coordinates of the object corners\n     */\n    _getImageLines: function(oCoords) {\n\n      var lines = {\n        topline: {\n          o: oCoords.tl,\n          d: oCoords.tr\n        },\n        rightline: {\n          o: oCoords.tr,\n          d: oCoords.br\n        },\n        bottomline: {\n          o: oCoords.br,\n          d: oCoords.bl\n        },\n        leftline: {\n          o: oCoords.bl,\n          d: oCoords.tl\n        }\n      };\n\n      // // debugging\n      // if (this.canvas.contextTop) {\n      //   this.canvas.contextTop.fillRect(lines.bottomline.d.x, lines.bottomline.d.y, 2, 2);\n      //   this.canvas.contextTop.fillRect(lines.bottomline.o.x, lines.bottomline.o.y, 2, 2);\n      //\n      //   this.canvas.contextTop.fillRect(lines.leftline.d.x, lines.leftline.d.y, 2, 2);\n      //   this.canvas.contextTop.fillRect(lines.leftline.o.x, lines.leftline.o.y, 2, 2);\n      //\n      //   this.canvas.contextTop.fillRect(lines.topline.d.x, lines.topline.d.y, 2, 2);\n      //   this.canvas.contextTop.fillRect(lines.topline.o.x, lines.topline.o.y, 2, 2);\n      //\n      //   this.canvas.contextTop.fillRect(lines.rightline.d.x, lines.rightline.d.y, 2, 2);\n      //   this.canvas.contextTop.fillRect(lines.rightline.o.x, lines.rightline.o.y, 2, 2);\n      // }\n\n      return lines;\n    },\n\n    /**\n     * Helper method to determine how many cross points are between the 4 object edges\n     * and the horizontal line determined by a point on canvas\n     * @private\n     * @param {fabric.Point} point Point to check\n     * @param {Object} lines Coordinates of the object being evaluated\n     */\n    // remove yi, not used but left code here just in case.\n    _findCrossPoints: function(point, lines) {\n      var b1, b2, a1, a2, xi, // yi,\n          xcount = 0,\n          iLine;\n\n      for (var lineKey in lines) {\n        iLine = lines[lineKey];\n        // optimisation 1: line below point. no cross\n        if ((iLine.o.y < point.y) && (iLine.d.y < point.y)) {\n          continue;\n        }\n        // optimisation 2: line above point. no cross\n        if ((iLine.o.y >= point.y) && (iLine.d.y >= point.y)) {\n          continue;\n        }\n        // optimisation 3: vertical line case\n        if ((iLine.o.x === iLine.d.x) && (iLine.o.x >= point.x)) {\n          xi = iLine.o.x;\n          // yi = point.y;\n        }\n        // calculate the intersection point\n        else {\n          b1 = 0;\n          b2 = (iLine.d.y - iLine.o.y) / (iLine.d.x - iLine.o.x);\n          a1 = point.y - b1 * point.x;\n          a2 = iLine.o.y - b2 * iLine.o.x;\n\n          xi = -(a1 - a2) / (b1 - b2);\n          // yi = a1 + b1 * xi;\n        }\n        // dont count xi < point.x cases\n        if (xi >= point.x) {\n          xcount += 1;\n        }\n        // optimisation 4: specific for square images\n        if (xcount === 2) {\n          break;\n        }\n      }\n      return xcount;\n    },\n\n    /**\n     * Returns coordinates of object's bounding rectangle (left, top, width, height)\n     * the box is intended as aligned to axis of canvas.\n     * @param {Boolean} [absolute] use coordinates without viewportTransform\n     * @param {Boolean} [calculate] use coordinates of current position instead of .oCoords / .aCoords\n     * @return {Object} Object with left, top, width, height properties\n     */\n    getBoundingRect: function(absolute, calculate) {\n      var coords = this.getCoords(absolute, calculate);\n      return util.makeBoundingBoxFromPoints(coords);\n    },\n\n    /**\n     * Returns width of an object's bounding box counting transformations\n     * before 2.0 it was named getWidth();\n     * @return {Number} width value\n     */\n    getScaledWidth: function() {\n      return this._getTransformedDimensions().x;\n    },\n\n    /**\n     * Returns height of an object bounding box counting transformations\n     * before 2.0 it was named getHeight();\n     * @return {Number} height value\n     */\n    getScaledHeight: function() {\n      return this._getTransformedDimensions().y;\n    },\n\n    /**\n     * Makes sure the scale is valid and modifies it if necessary\n     * @private\n     * @param {Number} value\n     * @return {Number}\n     */\n    _constrainScale: function(value) {\n      if (Math.abs(value) < this.minScaleLimit) {\n        if (value < 0) {\n          return -this.minScaleLimit;\n        }\n        else {\n          return this.minScaleLimit;\n        }\n      }\n      else if (value === 0) {\n        return 0.0001;\n      }\n      return value;\n    },\n\n    /**\n     * Scales an object (equally by x and y)\n     * @param {Number} value Scale factor\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    scale: function(value) {\n      this._set('scaleX', value);\n      this._set('scaleY', value);\n      return this.setCoords();\n    },\n\n    /**\n     * Scales an object to a given width, with respect to bounding box (scaling by x/y equally)\n     * @param {Number} value New width value\n     * @param {Boolean} absolute ignore viewport\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    scaleToWidth: function(value, absolute) {\n      // adjust to bounding rect factor so that rotated shapes would fit as well\n      var boundingRectFactor = this.getBoundingRect(absolute).width / this.getScaledWidth();\n      return this.scale(value / this.width / boundingRectFactor);\n    },\n\n    /**\n     * Scales an object to a given height, with respect to bounding box (scaling by x/y equally)\n     * @param {Number} value New height value\n     * @param {Boolean} absolute ignore viewport\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    scaleToHeight: function(value, absolute) {\n      // adjust to bounding rect factor so that rotated shapes would fit as well\n      var boundingRectFactor = this.getBoundingRect(absolute).height / this.getScaledHeight();\n      return this.scale(value / this.height / boundingRectFactor);\n    },\n\n    calcLineCoords: function() {\n      var vpt = this.getViewportTransform(),\n          padding = this.padding, angle = degreesToRadians(this.angle),\n          cos = util.cos(angle), sin = util.sin(angle),\n          cosP = cos * padding, sinP = sin * padding, cosPSinP = cosP + sinP,\n          cosPMinusSinP = cosP - sinP, aCoords = this.calcACoords();\n\n      var lineCoords = {\n        tl: transformPoint(aCoords.tl, vpt),\n        tr: transformPoint(aCoords.tr, vpt),\n        bl: transformPoint(aCoords.bl, vpt),\n        br: transformPoint(aCoords.br, vpt),\n      };\n\n      if (padding) {\n        lineCoords.tl.x -= cosPMinusSinP;\n        lineCoords.tl.y -= cosPSinP;\n        lineCoords.tr.x += cosPSinP;\n        lineCoords.tr.y -= cosPMinusSinP;\n        lineCoords.bl.x -= cosPSinP;\n        lineCoords.bl.y += cosPMinusSinP;\n        lineCoords.br.x += cosPMinusSinP;\n        lineCoords.br.y += cosPSinP;\n      }\n\n      return lineCoords;\n    },\n\n    calcOCoords: function() {\n      var rotateMatrix = this._calcRotateMatrix(),\n          translateMatrix = this._calcTranslateMatrix(),\n          vpt = this.getViewportTransform(),\n          startMatrix = multiplyMatrices(vpt, translateMatrix),\n          finalMatrix = multiplyMatrices(startMatrix, rotateMatrix),\n          finalMatrix = multiplyMatrices(finalMatrix, [1 / vpt[0], 0, 0, 1 / vpt[3], 0, 0]),\n          dim = this._calculateCurrentDimensions(),\n          coords = {};\n      this.forEachControl(function(control, key, fabricObject) {\n        coords[key] = control.positionHandler(dim, finalMatrix, fabricObject);\n      });\n\n      // debug code\n      // var canvas = this.canvas;\n      // setTimeout(function() {\n      //   canvas.contextTop.clearRect(0, 0, 700, 700);\n      //   canvas.contextTop.fillStyle = 'green';\n      //   Object.keys(coords).forEach(function(key) {\n      //     var control = coords[key];\n      //     canvas.contextTop.fillRect(control.x, control.y, 3, 3);\n      //   });\n      // }, 50);\n      return coords;\n    },\n\n    calcACoords: function() {\n      var rotateMatrix = this._calcRotateMatrix(),\n          translateMatrix = this._calcTranslateMatrix(),\n          finalMatrix = multiplyMatrices(translateMatrix, rotateMatrix),\n          dim = this._getTransformedDimensions(),\n          w = dim.x / 2, h = dim.y / 2;\n      return {\n        // corners\n        tl: transformPoint({ x: -w, y: -h }, finalMatrix),\n        tr: transformPoint({ x: w, y: -h }, finalMatrix),\n        bl: transformPoint({ x: -w, y: h }, finalMatrix),\n        br: transformPoint({ x: w, y: h }, finalMatrix)\n      };\n    },\n\n    /**\n     * Sets corner and controls position coordinates based on current angle, width and height, left and top.\n     * oCoords are used to find the corners\n     * aCoords are used to quickly find an object on the canvas\n     * lineCoords are used to quickly find object during pointer events.\n     * See {@link https://github.com/fabricjs/fabric.js/wiki/When-to-call-setCoords} and {@link http://fabricjs.com/fabric-gotchas}\n     *\n     * @param {Boolean} [skipCorners] skip calculation of oCoords.\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    setCoords: function(skipCorners) {\n      this.aCoords = this.calcACoords();\n      // in case we are in a group, for how the inner group target check works,\n      // lineCoords are exactly aCoords. Since the vpt gets absorbed by the normalized pointer.\n      this.lineCoords = this.group ? this.aCoords : this.calcLineCoords();\n      if (skipCorners) {\n        return this;\n      }\n      // set coordinates of the draggable boxes in the corners used to scale/rotate the image\n      this.oCoords = this.calcOCoords();\n      this._setCornerCoords && this._setCornerCoords();\n      return this;\n    },\n\n    /**\n     * calculate rotation matrix of an object\n     * @return {Array} rotation matrix for the object\n     */\n    _calcRotateMatrix: function() {\n      return util.calcRotateMatrix(this);\n    },\n\n    /**\n     * calculate the translation matrix for an object transform\n     * @return {Array} rotation matrix for the object\n     */\n    _calcTranslateMatrix: function() {\n      var center = this.getCenterPoint();\n      return [1, 0, 0, 1, center.x, center.y];\n    },\n\n    transformMatrixKey: function(skipGroup) {\n      var sep = '_', prefix = '';\n      if (!skipGroup && this.group) {\n        prefix = this.group.transformMatrixKey(skipGroup) + sep;\n      };\n      return prefix + this.top + sep + this.left + sep + this.scaleX + sep + this.scaleY +\n        sep + this.skewX + sep + this.skewY + sep + this.angle + sep + this.originX + sep + this.originY +\n        sep + this.width + sep + this.height + sep + this.strokeWidth + this.flipX + this.flipY;\n    },\n\n    /**\n     * calculate transform matrix that represents the current transformations from the\n     * object's properties.\n     * @param {Boolean} [skipGroup] return transform matrix for object not counting parent transformations\n     * There are some situation in which this is useful to avoid the fake rotation.\n     * @return {Array} transform matrix for the object\n     */\n    calcTransformMatrix: function(skipGroup) {\n      var matrix = this.calcOwnMatrix();\n      if (skipGroup || !this.group) {\n        return matrix;\n      }\n      var key = this.transformMatrixKey(skipGroup), cache = this.matrixCache || (this.matrixCache = {});\n      if (cache.key === key) {\n        return cache.value;\n      }\n      if (this.group) {\n        matrix = multiplyMatrices(this.group.calcTransformMatrix(false), matrix);\n      }\n      cache.key = key;\n      cache.value = matrix;\n      return matrix;\n    },\n\n    /**\n     * calculate transform matrix that represents the current transformations from the\n     * object's properties, this matrix does not include the group transformation\n     * @return {Array} transform matrix for the object\n     */\n    calcOwnMatrix: function() {\n      var key = this.transformMatrixKey(true), cache = this.ownMatrixCache || (this.ownMatrixCache = {});\n      if (cache.key === key) {\n        return cache.value;\n      }\n      var tMatrix = this._calcTranslateMatrix(),\n          options = {\n            angle: this.angle,\n            translateX: tMatrix[4],\n            translateY: tMatrix[5],\n            scaleX: this.scaleX,\n            scaleY: this.scaleY,\n            skewX: this.skewX,\n            skewY: this.skewY,\n            flipX: this.flipX,\n            flipY: this.flipY,\n          };\n      cache.key = key;\n      cache.value = util.composeMatrix(options);\n      return cache.value;\n    },\n\n    /*\n     * Calculate object dimensions from its properties\n     * @private\n     * @return {Object} .x width dimension\n     * @return {Object} .y height dimension\n     */\n    _getNonTransformedDimensions: function() {\n      var strokeWidth = this.strokeWidth,\n          w = this.width + strokeWidth,\n          h = this.height + strokeWidth;\n      return { x: w, y: h };\n    },\n\n    /*\n     * Calculate object bounding box dimensions from its properties scale, skew.\n     * @param {Number} skewX, a value to override current skewX\n     * @param {Number} skewY, a value to override current skewY\n     * @private\n     * @return {Object} .x width dimension\n     * @return {Object} .y height dimension\n     */\n    _getTransformedDimensions: function(skewX, skewY) {\n      if (typeof skewX === 'undefined') {\n        skewX = this.skewX;\n      }\n      if (typeof skewY === 'undefined') {\n        skewY = this.skewY;\n      }\n      var dimensions, dimX, dimY,\n          noSkew = skewX === 0 && skewY === 0;\n\n      if (this.strokeUniform) {\n        dimX = this.width;\n        dimY = this.height;\n      }\n      else {\n        dimensions = this._getNonTransformedDimensions();\n        dimX = dimensions.x;\n        dimY = dimensions.y;\n      }\n      if (noSkew) {\n        return this._finalizeDimensions(dimX * this.scaleX, dimY * this.scaleY);\n      }\n      var bbox = util.sizeAfterTransform(dimX, dimY, {\n        scaleX: this.scaleX,\n        scaleY: this.scaleY,\n        skewX: skewX,\n        skewY: skewY,\n      });\n      return this._finalizeDimensions(bbox.x, bbox.y);\n    },\n\n    /*\n     * Calculate object bounding box dimensions from its properties scale, skew.\n     * @param Number width width of the bbox\n     * @param Number height height of the bbox\n     * @private\n     * @return {Object} .x finalized width dimension\n     * @return {Object} .y finalized height dimension\n     */\n    _finalizeDimensions: function(width, height) {\n      return this.strokeUniform ?\n        { x: width + this.strokeWidth, y: height + this.strokeWidth }\n        :\n        { x: width, y: height };\n    },\n\n    /*\n     * Calculate object dimensions for controls box, including padding and canvas zoom.\n     * and active selection\n     * private\n     */\n    _calculateCurrentDimensions: function()  {\n      var vpt = this.getViewportTransform(),\n          dim = this._getTransformedDimensions(),\n          p = transformPoint(dim, vpt, true);\n      return p.scalarAdd(2 * this.padding);\n    },\n  });\n})();\n\n\nfabric.util.object.extend(fabric.Object.prototype, /** @lends fabric.Object.prototype */ {\n\n  /**\n   * Moves an object to the bottom of the stack of drawn objects\n   * @return {fabric.Object} thisArg\n   * @chainable\n   */\n  sendToBack: function() {\n    if (this.group) {\n      fabric.StaticCanvas.prototype.sendToBack.call(this.group, this);\n    }\n    else if (this.canvas) {\n      this.canvas.sendToBack(this);\n    }\n    return this;\n  },\n\n  /**\n   * Moves an object to the top of the stack of drawn objects\n   * @return {fabric.Object} thisArg\n   * @chainable\n   */\n  bringToFront: function() {\n    if (this.group) {\n      fabric.StaticCanvas.prototype.bringToFront.call(this.group, this);\n    }\n    else if (this.canvas) {\n      this.canvas.bringToFront(this);\n    }\n    return this;\n  },\n\n  /**\n   * Moves an object down in stack of drawn objects\n   * @param {Boolean} [intersecting] If `true`, send object behind next lower intersecting object\n   * @return {fabric.Object} thisArg\n   * @chainable\n   */\n  sendBackwards: function(intersecting) {\n    if (this.group) {\n      fabric.StaticCanvas.prototype.sendBackwards.call(this.group, this, intersecting);\n    }\n    else if (this.canvas) {\n      this.canvas.sendBackwards(this, intersecting);\n    }\n    return this;\n  },\n\n  /**\n   * Moves an object up in stack of drawn objects\n   * @param {Boolean} [intersecting] If `true`, send object in front of next upper intersecting object\n   * @return {fabric.Object} thisArg\n   * @chainable\n   */\n  bringForward: function(intersecting) {\n    if (this.group) {\n      fabric.StaticCanvas.prototype.bringForward.call(this.group, this, intersecting);\n    }\n    else if (this.canvas) {\n      this.canvas.bringForward(this, intersecting);\n    }\n    return this;\n  },\n\n  /**\n   * Moves an object to specified level in stack of drawn objects\n   * @param {Number} index New position of object\n   * @return {fabric.Object} thisArg\n   * @chainable\n   */\n  moveTo: function(index) {\n    if (this.group && this.group.type !== 'activeSelection') {\n      fabric.StaticCanvas.prototype.moveTo.call(this.group, this, index);\n    }\n    else if (this.canvas) {\n      this.canvas.moveTo(this, index);\n    }\n    return this;\n  }\n});\n\n\n/* _TO_SVG_START_ */\n(function() {\n  function getSvgColorString(prop, value) {\n    if (!value) {\n      return prop + ': none; ';\n    }\n    else if (value.toLive) {\n      return prop + ': url(#SVGID_' + value.id + '); ';\n    }\n    else {\n      var color = new fabric.Color(value),\n          str = prop + ': ' + color.toRgb() + '; ',\n          opacity = color.getAlpha();\n      if (opacity !== 1) {\n        //change the color in rgb + opacity\n        str += prop + '-opacity: ' + opacity.toString() + '; ';\n      }\n      return str;\n    }\n  }\n\n  var toFixed = fabric.util.toFixed;\n\n  fabric.util.object.extend(fabric.Object.prototype, /** @lends fabric.Object.prototype */ {\n    /**\n     * Returns styles-string for svg-export\n     * @param {Boolean} skipShadow a boolean to skip shadow filter output\n     * @return {String}\n     */\n    getSvgStyles: function(skipShadow) {\n\n      var fillRule = this.fillRule ? this.fillRule : 'nonzero',\n          strokeWidth = this.strokeWidth ? this.strokeWidth : '0',\n          strokeDashArray = this.strokeDashArray ? this.strokeDashArray.join(' ') : 'none',\n          strokeDashOffset = this.strokeDashOffset ? this.strokeDashOffset : '0',\n          strokeLineCap = this.strokeLineCap ? this.strokeLineCap : 'butt',\n          strokeLineJoin = this.strokeLineJoin ? this.strokeLineJoin : 'miter',\n          strokeMiterLimit = this.strokeMiterLimit ? this.strokeMiterLimit : '4',\n          opacity = typeof this.opacity !== 'undefined' ? this.opacity : '1',\n          visibility = this.visible ? '' : ' visibility: hidden;',\n          filter = skipShadow ? '' : this.getSvgFilter(),\n          fill = getSvgColorString('fill', this.fill),\n          stroke = getSvgColorString('stroke', this.stroke);\n\n      return [\n        stroke,\n        'stroke-width: ', strokeWidth, '; ',\n        'stroke-dasharray: ', strokeDashArray, '; ',\n        'stroke-linecap: ', strokeLineCap, '; ',\n        'stroke-dashoffset: ', strokeDashOffset, '; ',\n        'stroke-linejoin: ', strokeLineJoin, '; ',\n        'stroke-miterlimit: ', strokeMiterLimit, '; ',\n        fill,\n        'fill-rule: ', fillRule, '; ',\n        'opacity: ', opacity, ';',\n        filter,\n        visibility\n      ].join('');\n    },\n\n    /**\n     * Returns styles-string for svg-export\n     * @param {Object} style the object from which to retrieve style properties\n     * @param {Boolean} useWhiteSpace a boolean to include an additional attribute in the style.\n     * @return {String}\n     */\n    getSvgSpanStyles: function(style, useWhiteSpace) {\n      var term = '; ';\n      var fontFamily = style.fontFamily ?\n        'font-family: ' + (((style.fontFamily.indexOf('\\'') === -1 && style.fontFamily.indexOf('\"') === -1) ?\n          '\\'' + style.fontFamily  + '\\'' : style.fontFamily)) + term : '';\n      var strokeWidth = style.strokeWidth ? 'stroke-width: ' + style.strokeWidth + term : '',\n          fontFamily = fontFamily,\n          fontSize = style.fontSize ? 'font-size: ' + style.fontSize + 'px' + term : '',\n          fontStyle = style.fontStyle ? 'font-style: ' + style.fontStyle + term : '',\n          fontWeight = style.fontWeight ? 'font-weight: ' + style.fontWeight + term : '',\n          fill = style.fill ? getSvgColorString('fill', style.fill) : '',\n          stroke = style.stroke ? getSvgColorString('stroke', style.stroke) : '',\n          textDecoration = this.getSvgTextDecoration(style),\n          deltaY = style.deltaY ? 'baseline-shift: ' + (-style.deltaY) + '; ' : '';\n      if (textDecoration) {\n        textDecoration = 'text-decoration: ' + textDecoration + term;\n      }\n\n      return [\n        stroke,\n        strokeWidth,\n        fontFamily,\n        fontSize,\n        fontStyle,\n        fontWeight,\n        textDecoration,\n        fill,\n        deltaY,\n        useWhiteSpace ? 'white-space: pre; ' : ''\n      ].join('');\n    },\n\n    /**\n     * Returns text-decoration property for svg-export\n     * @param {Object} style the object from which to retrieve style properties\n     * @return {String}\n     */\n    getSvgTextDecoration: function(style) {\n      return ['overline', 'underline', 'line-through'].filter(function(decoration) {\n        return style[decoration.replace('-', '')];\n      }).join(' ');\n    },\n\n    /**\n     * Returns filter for svg shadow\n     * @return {String}\n     */\n    getSvgFilter: function() {\n      return this.shadow ? 'filter: url(#SVGID_' + this.shadow.id + ');' : '';\n    },\n\n    /**\n     * Returns id attribute for svg output\n     * @return {String}\n     */\n    getSvgCommons: function() {\n      return [\n        this.id ? 'id=\"' + this.id + '\" ' : '',\n        this.clipPath ? 'clip-path=\"url(#' + this.clipPath.clipPathId + ')\" ' : '',\n      ].join('');\n    },\n\n    /**\n     * Returns transform-string for svg-export\n     * @param {Boolean} use the full transform or the single object one.\n     * @return {String}\n     */\n    getSvgTransform: function(full, additionalTransform) {\n      var transform = full ? this.calcTransformMatrix() : this.calcOwnMatrix(),\n          svgTransform = 'transform=\"' + fabric.util.matrixToSVG(transform);\n      return svgTransform +\n        (additionalTransform || '') + '\" ';\n    },\n\n    _setSVGBg: function(textBgRects) {\n      if (this.backgroundColor) {\n        var NUM_FRACTION_DIGITS = fabric.Object.NUM_FRACTION_DIGITS;\n        textBgRects.push(\n          '\\t\\t<rect ',\n          this._getFillAttributes(this.backgroundColor),\n          ' x=\"',\n          toFixed(-this.width / 2, NUM_FRACTION_DIGITS),\n          '\" y=\"',\n          toFixed(-this.height / 2, NUM_FRACTION_DIGITS),\n          '\" width=\"',\n          toFixed(this.width, NUM_FRACTION_DIGITS),\n          '\" height=\"',\n          toFixed(this.height, NUM_FRACTION_DIGITS),\n          '\"></rect>\\n');\n      }\n    },\n\n    /**\n     * Returns svg representation of an instance\n     * @param {Function} [reviver] Method for further parsing of svg representation.\n     * @return {String} svg representation of an instance\n     */\n    toSVG: function(reviver) {\n      return this._createBaseSVGMarkup(this._toSVG(reviver), { reviver: reviver });\n    },\n\n    /**\n     * Returns svg clipPath representation of an instance\n     * @param {Function} [reviver] Method for further parsing of svg representation.\n     * @return {String} svg representation of an instance\n     */\n    toClipPathSVG: function(reviver) {\n      return '\\t' + this._createBaseClipPathSVGMarkup(this._toSVG(reviver), { reviver: reviver });\n    },\n\n    /**\n     * @private\n     */\n    _createBaseClipPathSVGMarkup: function(objectMarkup, options) {\n      options = options || {};\n      var reviver = options.reviver,\n          additionalTransform = options.additionalTransform || '',\n          commonPieces = [\n            this.getSvgTransform(true, additionalTransform),\n            this.getSvgCommons(),\n          ].join(''),\n          // insert commons in the markup, style and svgCommons\n          index = objectMarkup.indexOf('COMMON_PARTS');\n      objectMarkup[index] = commonPieces;\n      return reviver ? reviver(objectMarkup.join('')) : objectMarkup.join('');\n    },\n\n    /**\n     * @private\n     */\n    _createBaseSVGMarkup: function(objectMarkup, options) {\n      options = options || {};\n      var noStyle = options.noStyle,\n          reviver = options.reviver,\n          styleInfo = noStyle ? '' : 'style=\"' + this.getSvgStyles() + '\" ',\n          shadowInfo = options.withShadow ? 'style=\"' + this.getSvgFilter() + '\" ' : '',\n          clipPath = this.clipPath,\n          vectorEffect = this.strokeUniform ? 'vector-effect=\"non-scaling-stroke\" ' : '',\n          absoluteClipPath = clipPath && clipPath.absolutePositioned,\n          stroke = this.stroke, fill = this.fill, shadow = this.shadow,\n          commonPieces, markup = [], clipPathMarkup,\n          // insert commons in the markup, style and svgCommons\n          index = objectMarkup.indexOf('COMMON_PARTS'),\n          additionalTransform = options.additionalTransform;\n      if (clipPath) {\n        clipPath.clipPathId = 'CLIPPATH_' + fabric.Object.__uid++;\n        clipPathMarkup = '<clipPath id=\"' + clipPath.clipPathId + '\" >\\n' +\n          clipPath.toClipPathSVG(reviver) +\n          '</clipPath>\\n';\n      }\n      if (absoluteClipPath) {\n        markup.push(\n          '<g ', shadowInfo, this.getSvgCommons(), ' >\\n'\n        );\n      }\n      markup.push(\n        '<g ',\n        this.getSvgTransform(false),\n        !absoluteClipPath ? shadowInfo + this.getSvgCommons() : '',\n        ' >\\n'\n      );\n      commonPieces = [\n        styleInfo,\n        vectorEffect,\n        noStyle ? '' : this.addPaintOrder(), ' ',\n        additionalTransform ? 'transform=\"' + additionalTransform + '\" ' : '',\n      ].join('');\n      objectMarkup[index] = commonPieces;\n      if (fill && fill.toLive) {\n        markup.push(fill.toSVG(this));\n      }\n      if (stroke && stroke.toLive) {\n        markup.push(stroke.toSVG(this));\n      }\n      if (shadow) {\n        markup.push(shadow.toSVG(this));\n      }\n      if (clipPath) {\n        markup.push(clipPathMarkup);\n      }\n      markup.push(objectMarkup.join(''));\n      markup.push('</g>\\n');\n      absoluteClipPath && markup.push('</g>\\n');\n      return reviver ? reviver(markup.join('')) : markup.join('');\n    },\n\n    addPaintOrder: function() {\n      return this.paintFirst !== 'fill' ? ' paint-order=\"' + this.paintFirst + '\" ' : '';\n    }\n  });\n})();\n/* _TO_SVG_END_ */\n\n\n(function() {\n\n  var extend = fabric.util.object.extend,\n      originalSet = 'stateProperties';\n\n  /*\n    Depends on `stateProperties`\n  */\n  function saveProps(origin, destination, props) {\n    var tmpObj = { }, deep = true;\n    props.forEach(function(prop) {\n      tmpObj[prop] = origin[prop];\n    });\n\n    extend(origin[destination], tmpObj, deep);\n  }\n\n  function _isEqual(origValue, currentValue, firstPass) {\n    if (origValue === currentValue) {\n      // if the objects are identical, return\n      return true;\n    }\n    else if (Array.isArray(origValue)) {\n      if (!Array.isArray(currentValue) || origValue.length !== currentValue.length) {\n        return false;\n      }\n      for (var i = 0, len = origValue.length; i < len; i++) {\n        if (!_isEqual(origValue[i], currentValue[i])) {\n          return false;\n        }\n      }\n      return true;\n    }\n    else if (origValue && typeof origValue === 'object') {\n      var keys = Object.keys(origValue), key;\n      if (!currentValue ||\n          typeof currentValue !== 'object' ||\n          (!firstPass && keys.length !== Object.keys(currentValue).length)\n      ) {\n        return false;\n      }\n      for (var i = 0, len = keys.length; i < len; i++) {\n        key = keys[i];\n        // since clipPath is in the statefull cache list and the clipPath objects\n        // would be iterated as an object, this would lead to possible infinite recursion\n        // we do not want to compare those.\n        if (key === 'canvas' || key === 'group') {\n          continue;\n        }\n        if (!_isEqual(origValue[key], currentValue[key])) {\n          return false;\n        }\n      }\n      return true;\n    }\n  }\n\n\n  fabric.util.object.extend(fabric.Object.prototype, /** @lends fabric.Object.prototype */ {\n\n    /**\n     * Returns true if object state (one of its state properties) was changed\n     * @param {String} [propertySet] optional name for the set of property we want to save\n     * @return {Boolean} true if instance' state has changed since `{@link fabric.Object#saveState}` was called\n     */\n    hasStateChanged: function(propertySet) {\n      propertySet = propertySet || originalSet;\n      var dashedPropertySet = '_' + propertySet;\n      if (Object.keys(this[dashedPropertySet]).length < this[propertySet].length) {\n        return true;\n      }\n      return !_isEqual(this[dashedPropertySet], this, true);\n    },\n\n    /**\n     * Saves state of an object\n     * @param {Object} [options] Object with additional `stateProperties` array to include when saving state\n     * @return {fabric.Object} thisArg\n     */\n    saveState: function(options) {\n      var propertySet = options && options.propertySet || originalSet,\n          destination = '_' + propertySet;\n      if (!this[destination]) {\n        return this.setupState(options);\n      }\n      saveProps(this, destination, this[propertySet]);\n      if (options && options.stateProperties) {\n        saveProps(this, destination, options.stateProperties);\n      }\n      return this;\n    },\n\n    /**\n     * Setups state of an object\n     * @param {Object} [options] Object with additional `stateProperties` array to include when saving state\n     * @return {fabric.Object} thisArg\n     */\n    setupState: function(options) {\n      options = options || { };\n      var propertySet = options.propertySet || originalSet;\n      options.propertySet = propertySet;\n      this['_' + propertySet] = { };\n      this.saveState(options);\n      return this;\n    }\n  });\n})();\n\n\n(function() {\n\n  var degreesToRadians = fabric.util.degreesToRadians;\n\n  fabric.util.object.extend(fabric.Object.prototype, /** @lends fabric.Object.prototype */ {\n    /**\n     * Determines which corner has been clicked\n     * @private\n     * @param {Object} pointer The pointer indicating the mouse position\n     * @return {String|Boolean} corner code (tl, tr, bl, br, etc.), or false if nothing is found\n     */\n    _findTargetCorner: function(pointer, forTouch) {\n      // objects in group, anykind, are not self modificable,\n      // must not return an hovered corner.\n      if (!this.hasControls || this.group || (!this.canvas || this.canvas._activeObject !== this)) {\n        return false;\n      }\n\n      var ex = pointer.x,\n          ey = pointer.y,\n          xPoints,\n          lines, keys = Object.keys(this.oCoords),\n          j = keys.length - 1, i;\n      this.__corner = 0;\n\n      // cycle in reverse order so we pick first the one on top\n      for (; j >= 0; j--) {\n        i = keys[j];\n        if (!this.isControlVisible(i)) {\n          continue;\n        }\n\n        lines = this._getImageLines(forTouch ? this.oCoords[i].touchCorner : this.oCoords[i].corner);\n        // // debugging\n        //\n        // this.canvas.contextTop.fillRect(lines.bottomline.d.x, lines.bottomline.d.y, 2, 2);\n        // this.canvas.contextTop.fillRect(lines.bottomline.o.x, lines.bottomline.o.y, 2, 2);\n        //\n        // this.canvas.contextTop.fillRect(lines.leftline.d.x, lines.leftline.d.y, 2, 2);\n        // this.canvas.contextTop.fillRect(lines.leftline.o.x, lines.leftline.o.y, 2, 2);\n        //\n        // this.canvas.contextTop.fillRect(lines.topline.d.x, lines.topline.d.y, 2, 2);\n        // this.canvas.contextTop.fillRect(lines.topline.o.x, lines.topline.o.y, 2, 2);\n        //\n        // this.canvas.contextTop.fillRect(lines.rightline.d.x, lines.rightline.d.y, 2, 2);\n        // this.canvas.contextTop.fillRect(lines.rightline.o.x, lines.rightline.o.y, 2, 2);\n\n        xPoints = this._findCrossPoints({ x: ex, y: ey }, lines);\n        if (xPoints !== 0 && xPoints % 2 === 1) {\n          this.__corner = i;\n          return i;\n        }\n      }\n      return false;\n    },\n\n    /**\n     * Calls a function for each control. The function gets called,\n     * with the control, the object that is calling the iterator and the control's key\n     * @param {Function} fn function to iterate over the controls over\n     */\n    forEachControl: function(fn) {\n      for (var i in this.controls) {\n        fn(this.controls[i], i, this);\n      };\n    },\n\n    /**\n     * Sets the coordinates of the draggable boxes in the corners of\n     * the image used to scale/rotate it.\n     * note: if we would switch to ROUND corner area, all of this would disappear.\n     * everything would resolve to a single point and a pythagorean theorem for the distance\n     * @private\n     */\n    _setCornerCoords: function() {\n      var coords = this.oCoords;\n\n      for (var control in coords) {\n        var controlObject = this.controls[control];\n        coords[control].corner = controlObject.calcCornerCoords(\n          this.angle, this.cornerSize, coords[control].x, coords[control].y, false);\n        coords[control].touchCorner = controlObject.calcCornerCoords(\n          this.angle, this.touchCornerSize, coords[control].x, coords[control].y, true);\n      }\n    },\n\n    /**\n     * Draws a colored layer behind the object, inside its selection borders.\n     * Requires public options: padding, selectionBackgroundColor\n     * this function is called when the context is transformed\n     * has checks to be skipped when the object is on a staticCanvas\n     * @param {CanvasRenderingContext2D} ctx Context to draw on\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    drawSelectionBackground: function(ctx) {\n      if (!this.selectionBackgroundColor ||\n        (this.canvas && !this.canvas.interactive) ||\n        (this.canvas && this.canvas._activeObject !== this)\n      ) {\n        return this;\n      }\n      ctx.save();\n      var center = this.getCenterPoint(), wh = this._calculateCurrentDimensions(),\n          vpt = this.canvas.viewportTransform;\n      ctx.translate(center.x, center.y);\n      ctx.scale(1 / vpt[0], 1 / vpt[3]);\n      ctx.rotate(degreesToRadians(this.angle));\n      ctx.fillStyle = this.selectionBackgroundColor;\n      ctx.fillRect(-wh.x / 2, -wh.y / 2, wh.x, wh.y);\n      ctx.restore();\n      return this;\n    },\n\n    /**\n     * Draws borders of an object's bounding box.\n     * Requires public properties: width, height\n     * Requires public options: padding, borderColor\n     * @param {CanvasRenderingContext2D} ctx Context to draw on\n     * @param {Object} styleOverride object to override the object style\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    drawBorders: function(ctx, styleOverride) {\n      styleOverride = styleOverride || {};\n      var wh = this._calculateCurrentDimensions(),\n          strokeWidth = this.borderScaleFactor,\n          width = wh.x + strokeWidth,\n          height = wh.y + strokeWidth,\n          hasControls = typeof styleOverride.hasControls !== 'undefined' ?\n            styleOverride.hasControls : this.hasControls,\n          shouldStroke = false;\n\n      ctx.save();\n      ctx.strokeStyle = styleOverride.borderColor || this.borderColor;\n      this._setLineDash(ctx, styleOverride.borderDashArray || this.borderDashArray);\n\n      ctx.strokeRect(\n        -width / 2,\n        -height / 2,\n        width,\n        height\n      );\n\n      if (hasControls) {\n        ctx.beginPath();\n        this.forEachControl(function(control, key, fabricObject) {\n          // in this moment, the ctx is centered on the object.\n          // width and height of the above function are the size of the bbox.\n          if (control.withConnection && control.getVisibility(fabricObject, key)) {\n            // reset movement for each control\n            shouldStroke = true;\n            ctx.moveTo(control.x * width, control.y * height);\n            ctx.lineTo(\n              control.x * width + control.offsetX,\n              control.y * height + control.offsetY\n            );\n          }\n        });\n        if (shouldStroke) {\n          ctx.stroke();\n        }\n      }\n      ctx.restore();\n      return this;\n    },\n\n    /**\n     * Draws borders of an object's bounding box when it is inside a group.\n     * Requires public properties: width, height\n     * Requires public options: padding, borderColor\n     * @param {CanvasRenderingContext2D} ctx Context to draw on\n     * @param {object} options object representing current object parameters\n     * @param {Object} styleOverride object to override the object style\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    drawBordersInGroup: function(ctx, options, styleOverride) {\n      styleOverride = styleOverride || {};\n      var bbox = fabric.util.sizeAfterTransform(this.width, this.height, options),\n          strokeWidth = this.strokeWidth,\n          strokeUniform = this.strokeUniform,\n          borderScaleFactor = this.borderScaleFactor,\n          width =\n            bbox.x + strokeWidth * (strokeUniform ? this.canvas.getZoom() : options.scaleX) + borderScaleFactor,\n          height =\n            bbox.y + strokeWidth * (strokeUniform ? this.canvas.getZoom() : options.scaleY) + borderScaleFactor;\n      ctx.save();\n      this._setLineDash(ctx, styleOverride.borderDashArray || this.borderDashArray);\n      ctx.strokeStyle = styleOverride.borderColor || this.borderColor;\n      ctx.strokeRect(\n        -width / 2,\n        -height / 2,\n        width,\n        height\n      );\n\n      ctx.restore();\n      return this;\n    },\n\n    /**\n     * Draws corners of an object's bounding box.\n     * Requires public properties: width, height\n     * Requires public options: cornerSize, padding\n     * @param {CanvasRenderingContext2D} ctx Context to draw on\n     * @param {Object} styleOverride object to override the object style\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    drawControls: function(ctx, styleOverride) {\n      styleOverride = styleOverride || {};\n      ctx.save();\n      var retinaScaling = this.canvas.getRetinaScaling(), matrix, p;\n      ctx.setTransform(retinaScaling, 0, 0, retinaScaling, 0, 0);\n      ctx.strokeStyle = ctx.fillStyle = styleOverride.cornerColor || this.cornerColor;\n      if (!this.transparentCorners) {\n        ctx.strokeStyle = styleOverride.cornerStrokeColor || this.cornerStrokeColor;\n      }\n      this._setLineDash(ctx, styleOverride.cornerDashArray || this.cornerDashArray);\n      this.setCoords();\n      if (this.group) {\n        // fabricJS does not really support drawing controls inside groups,\n        // this piece of code here helps having at least the control in places.\n        // If an application needs to show some objects as selected because of some UI state\n        // can still call Object._renderControls() on any object they desire, independently of groups.\n        // using no padding, circular controls and hiding the rotating cursor is higly suggested,\n        matrix = this.group.calcTransformMatrix();\n      }\n      this.forEachControl(function(control, key, fabricObject) {\n        p = fabricObject.oCoords[key];\n        if (control.getVisibility(fabricObject, key)) {\n          if (matrix) {\n            p = fabric.util.transformPoint(p, matrix);\n          }\n          control.render(ctx, p.x, p.y, styleOverride, fabricObject);\n        }\n      });\n      ctx.restore();\n\n      return this;\n    },\n\n    /**\n     * Returns true if the specified control is visible, false otherwise.\n     * @param {String} controlKey The key of the control. Possible values are 'tl', 'tr', 'br', 'bl', 'ml', 'mt', 'mr', 'mb', 'mtr'.\n     * @returns {Boolean} true if the specified control is visible, false otherwise\n     */\n    isControlVisible: function(controlKey) {\n      return this.controls[controlKey] && this.controls[controlKey].getVisibility(this, controlKey);\n    },\n\n    /**\n     * Sets the visibility of the specified control.\n     * @param {String} controlKey The key of the control. Possible values are 'tl', 'tr', 'br', 'bl', 'ml', 'mt', 'mr', 'mb', 'mtr'.\n     * @param {Boolean} visible true to set the specified control visible, false otherwise\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    setControlVisible: function(controlKey, visible) {\n      if (!this._controlsVisibility) {\n        this._controlsVisibility = {};\n      }\n      this._controlsVisibility[controlKey] = visible;\n      return this;\n    },\n\n    /**\n     * Sets the visibility state of object controls.\n     * @param {Object} [options] Options object\n     * @param {Boolean} [options.bl] true to enable the bottom-left control, false to disable it\n     * @param {Boolean} [options.br] true to enable the bottom-right control, false to disable it\n     * @param {Boolean} [options.mb] true to enable the middle-bottom control, false to disable it\n     * @param {Boolean} [options.ml] true to enable the middle-left control, false to disable it\n     * @param {Boolean} [options.mr] true to enable the middle-right control, false to disable it\n     * @param {Boolean} [options.mt] true to enable the middle-top control, false to disable it\n     * @param {Boolean} [options.tl] true to enable the top-left control, false to disable it\n     * @param {Boolean} [options.tr] true to enable the top-right control, false to disable it\n     * @param {Boolean} [options.mtr] true to enable the middle-top-rotate control, false to disable it\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    setControlsVisibility: function(options) {\n      options || (options = { });\n\n      for (var p in options) {\n        this.setControlVisible(p, options[p]);\n      }\n      return this;\n    },\n\n\n    /**\n     * This callback function is called every time _discardActiveObject or _setActiveObject\n     * try to to deselect this object. If the function returns true, the process is cancelled\n     * @param {Object} [options] options sent from the upper functions\n     * @param {Event} [options.e] event if the process is generated by an event\n     */\n    onDeselect: function() {\n      // implemented by sub-classes, as needed.\n    },\n\n\n    /**\n     * This callback function is called every time _discardActiveObject or _setActiveObject\n     * try to to select this object. If the function returns true, the process is cancelled\n     * @param {Object} [options] options sent from the upper functions\n     * @param {Event} [options.e] event if the process is generated by an event\n     */\n    onSelect: function() {\n      // implemented by sub-classes, as needed.\n    }\n  });\n})();\n\n\nfabric.util.object.extend(fabric.StaticCanvas.prototype, /** @lends fabric.StaticCanvas.prototype */ {\n\n  /**\n   * Animation duration (in ms) for fx* methods\n   * @type Number\n   * @default\n   */\n  FX_DURATION: 500,\n\n  /**\n   * Centers object horizontally with animation.\n   * @param {fabric.Object} object Object to center\n   * @param {Object} [callbacks] Callbacks object with optional \"onComplete\" and/or \"onChange\" properties\n   * @param {Function} [callbacks.onComplete] Invoked on completion\n   * @param {Function} [callbacks.onChange] Invoked on every step of animation\n   * @return {fabric.AnimationContext} context\n   */\n  fxCenterObjectH: function (object, callbacks) {\n    callbacks = callbacks || { };\n\n    var empty = function() { },\n        onComplete = callbacks.onComplete || empty,\n        onChange = callbacks.onChange || empty,\n        _this = this;\n\n    return fabric.util.animate({\n      target: this,\n      startValue: object.left,\n      endValue: this.getCenterPoint().x,\n      duration: this.FX_DURATION,\n      onChange: function(value) {\n        object.set('left', value);\n        _this.requestRenderAll();\n        onChange();\n      },\n      onComplete: function() {\n        object.setCoords();\n        onComplete();\n      }\n    });\n  },\n\n  /**\n   * Centers object vertically with animation.\n   * @param {fabric.Object} object Object to center\n   * @param {Object} [callbacks] Callbacks object with optional \"onComplete\" and/or \"onChange\" properties\n   * @param {Function} [callbacks.onComplete] Invoked on completion\n   * @param {Function} [callbacks.onChange] Invoked on every step of animation\n   * @return {fabric.AnimationContext} context\n   */\n  fxCenterObjectV: function (object, callbacks) {\n    callbacks = callbacks || { };\n\n    var empty = function() { },\n        onComplete = callbacks.onComplete || empty,\n        onChange = callbacks.onChange || empty,\n        _this = this;\n\n    return fabric.util.animate({\n      target: this,\n      startValue: object.top,\n      endValue: this.getCenterPoint().y,\n      duration: this.FX_DURATION,\n      onChange: function(value) {\n        object.set('top', value);\n        _this.requestRenderAll();\n        onChange();\n      },\n      onComplete: function() {\n        object.setCoords();\n        onComplete();\n      }\n    });\n  },\n\n  /**\n   * Same as `fabric.Canvas#remove` but animated\n   * @param {fabric.Object} object Object to remove\n   * @param {Object} [callbacks] Callbacks object with optional \"onComplete\" and/or \"onChange\" properties\n   * @param {Function} [callbacks.onComplete] Invoked on completion\n   * @param {Function} [callbacks.onChange] Invoked on every step of animation\n   * @return {fabric.AnimationContext} context\n   */\n  fxRemove: function (object, callbacks) {\n    callbacks = callbacks || { };\n\n    var empty = function() { },\n        onComplete = callbacks.onComplete || empty,\n        onChange = callbacks.onChange || empty,\n        _this = this;\n\n    return fabric.util.animate({\n      target: this,\n      startValue: object.opacity,\n      endValue: 0,\n      duration: this.FX_DURATION,\n      onChange: function(value) {\n        object.set('opacity', value);\n        _this.requestRenderAll();\n        onChange();\n      },\n      onComplete: function () {\n        _this.remove(object);\n        onComplete();\n      }\n    });\n  }\n});\n\nfabric.util.object.extend(fabric.Object.prototype, /** @lends fabric.Object.prototype */ {\n  /**\n   * Animates object's properties\n   * @param {String|Object} property Property to animate (if string) or properties to animate (if object)\n   * @param {Number|Object} value Value to animate property to (if string was given first) or options object\n   * @return {fabric.Object} thisArg\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-2#animation}\n   * @return {fabric.AnimationContext | fabric.AnimationContext[]} animation context (or an array if passed multiple properties)\n   *\n   * As object \u2014 multiple properties\n   *\n   * object.animate({ left: ..., top: ... });\n   * object.animate({ left: ..., top: ... }, { duration: ... });\n   *\n   * As string \u2014 one property\n   *\n   * object.animate('left', ...);\n   * object.animate('left', { duration: ... });\n   *\n   */\n  animate: function () {\n    if (arguments[0] && typeof arguments[0] === 'object') {\n      var propsToAnimate = [], prop, skipCallbacks, out = [];\n      for (prop in arguments[0]) {\n        propsToAnimate.push(prop);\n      }\n      for (var i = 0, len = propsToAnimate.length; i < len; i++) {\n        prop = propsToAnimate[i];\n        skipCallbacks = i !== len - 1;\n        out.push(this._animate(prop, arguments[0][prop], arguments[1], skipCallbacks));\n      }\n      return out;\n    }\n    else {\n      return this._animate.apply(this, arguments);\n    }\n  },\n\n  /**\n   * @private\n   * @param {String} property Property to animate\n   * @param {String} to Value to animate to\n   * @param {Object} [options] Options object\n   * @param {Boolean} [skipCallbacks] When true, callbacks like onchange and oncomplete are not invoked\n   */\n  _animate: function(property, to, options, skipCallbacks) {\n    var _this = this, propPair;\n\n    to = to.toString();\n\n    if (!options) {\n      options = { };\n    }\n    else {\n      options = fabric.util.object.clone(options);\n    }\n\n    if (~property.indexOf('.')) {\n      propPair = property.split('.');\n    }\n\n    var propIsColor =\n      _this.colorProperties.indexOf(property) > -1 ||\n      (propPair && _this.colorProperties.indexOf(propPair[1]) > -1);\n\n    var currentValue = propPair\n      ? this.get(propPair[0])[propPair[1]]\n      : this.get(property);\n\n    if (!('from' in options)) {\n      options.from = currentValue;\n    }\n\n    if (!propIsColor) {\n      if (~to.indexOf('=')) {\n        to = currentValue + parseFloat(to.replace('=', ''));\n      }\n      else {\n        to = parseFloat(to);\n      }\n    }\n\n    var _options = {\n      target: this,\n      startValue: options.from,\n      endValue: to,\n      byValue: options.by,\n      easing: options.easing,\n      duration: options.duration,\n      abort: options.abort && function(value, valueProgress, timeProgress) {\n        return options.abort.call(_this, value, valueProgress, timeProgress);\n      },\n      onChange: function (value, valueProgress, timeProgress) {\n        if (propPair) {\n          _this[propPair[0]][propPair[1]] = value;\n        }\n        else {\n          _this.set(property, value);\n        }\n        if (skipCallbacks) {\n          return;\n        }\n        options.onChange && options.onChange(value, valueProgress, timeProgress);\n      },\n      onComplete: function (value, valueProgress, timeProgress) {\n        if (skipCallbacks) {\n          return;\n        }\n\n        _this.setCoords();\n        options.onComplete && options.onComplete(value, valueProgress, timeProgress);\n      }\n    };\n\n    if (propIsColor) {\n      return fabric.util.animateColor(_options.startValue, _options.endValue, _options.duration, _options);\n    }\n    else {\n      return fabric.util.animate(_options);\n    }\n  }\n});\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      extend = fabric.util.object.extend,\n      clone = fabric.util.object.clone,\n      coordProps = { x1: 1, x2: 1, y1: 1, y2: 1 };\n\n  if (fabric.Line) {\n    fabric.warn('fabric.Line is already defined');\n    return;\n  }\n\n  /**\n   * Line class\n   * @class fabric.Line\n   * @extends fabric.Object\n   * @see {@link fabric.Line#initialize} for constructor definition\n   */\n  fabric.Line = fabric.util.createClass(fabric.Object, /** @lends fabric.Line.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'line',\n\n    /**\n     * x value or first line edge\n     * @type Number\n     * @default\n     */\n    x1: 0,\n\n    /**\n     * y value or first line edge\n     * @type Number\n     * @default\n     */\n    y1: 0,\n\n    /**\n     * x value or second line edge\n     * @type Number\n     * @default\n     */\n    x2: 0,\n\n    /**\n     * y value or second line edge\n     * @type Number\n     * @default\n     */\n    y2: 0,\n\n    cacheProperties: fabric.Object.prototype.cacheProperties.concat('x1', 'x2', 'y1', 'y2'),\n\n    /**\n     * Constructor\n     * @param {Array} [points] Array of points\n     * @param {Object} [options] Options object\n     * @return {fabric.Line} thisArg\n     */\n    initialize: function(points, options) {\n      if (!points) {\n        points = [0, 0, 0, 0];\n      }\n\n      this.callSuper('initialize', options);\n\n      this.set('x1', points[0]);\n      this.set('y1', points[1]);\n      this.set('x2', points[2]);\n      this.set('y2', points[3]);\n\n      this._setWidthHeight(options);\n    },\n\n    /**\n     * @private\n     * @param {Object} [options] Options\n     */\n    _setWidthHeight: function(options) {\n      options || (options = { });\n\n      this.width = Math.abs(this.x2 - this.x1);\n      this.height = Math.abs(this.y2 - this.y1);\n\n      this.left = 'left' in options\n        ? options.left\n        : this._getLeftToOriginX();\n\n      this.top = 'top' in options\n        ? options.top\n        : this._getTopToOriginY();\n    },\n\n    /**\n     * @private\n     * @param {String} key\n     * @param {*} value\n     */\n    _set: function(key, value) {\n      this.callSuper('_set', key, value);\n      if (typeof coordProps[key] !== 'undefined') {\n        this._setWidthHeight();\n      }\n      return this;\n    },\n\n    /**\n     * @private\n     * @return {Number} leftToOriginX Distance from left edge of canvas to originX of Line.\n     */\n    _getLeftToOriginX: makeEdgeToOriginGetter(\n      { // property names\n        origin: 'originX',\n        axis1: 'x1',\n        axis2: 'x2',\n        dimension: 'width'\n      },\n      { // possible values of origin\n        nearest: 'left',\n        center: 'center',\n        farthest: 'right'\n      }\n    ),\n\n    /**\n     * @private\n     * @return {Number} topToOriginY Distance from top edge of canvas to originY of Line.\n     */\n    _getTopToOriginY: makeEdgeToOriginGetter(\n      { // property names\n        origin: 'originY',\n        axis1: 'y1',\n        axis2: 'y2',\n        dimension: 'height'\n      },\n      { // possible values of origin\n        nearest: 'top',\n        center: 'center',\n        farthest: 'bottom'\n      }\n    ),\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _render: function(ctx) {\n      ctx.beginPath();\n\n\n      var p = this.calcLinePoints();\n      ctx.moveTo(p.x1, p.y1);\n      ctx.lineTo(p.x2, p.y2);\n\n      ctx.lineWidth = this.strokeWidth;\n\n      // TODO: test this\n      // make sure setting \"fill\" changes color of a line\n      // (by copying fillStyle to strokeStyle, since line is stroked, not filled)\n      var origStrokeStyle = ctx.strokeStyle;\n      ctx.strokeStyle = this.stroke || ctx.fillStyle;\n      this.stroke && this._renderStroke(ctx);\n      ctx.strokeStyle = origStrokeStyle;\n    },\n\n    /**\n     * This function is an helper for svg import. it returns the center of the object in the svg\n     * untransformed coordinates\n     * @private\n     * @return {Object} center point from element coordinates\n     */\n    _findCenterFromElement: function() {\n      return {\n        x: (this.x1 + this.x2) / 2,\n        y: (this.y1 + this.y2) / 2,\n      };\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @method toObject\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      return extend(this.callSuper('toObject', propertiesToInclude), this.calcLinePoints());\n    },\n\n    /*\n     * Calculate object dimensions from its properties\n     * @private\n     */\n    _getNonTransformedDimensions: function() {\n      var dim = this.callSuper('_getNonTransformedDimensions');\n      if (this.strokeLineCap === 'butt') {\n        if (this.width === 0) {\n          dim.y -= this.strokeWidth;\n        }\n        if (this.height === 0) {\n          dim.x -= this.strokeWidth;\n        }\n      }\n      return dim;\n    },\n\n    /**\n     * Recalculates line points given width and height\n     * @private\n     */\n    calcLinePoints: function() {\n      var xMult = this.x1 <= this.x2 ? -1 : 1,\n          yMult = this.y1 <= this.y2 ? -1 : 1,\n          x1 = (xMult * this.width * 0.5),\n          y1 = (yMult * this.height * 0.5),\n          x2 = (xMult * this.width * -0.5),\n          y2 = (yMult * this.height * -0.5);\n\n      return {\n        x1: x1,\n        x2: x2,\n        y1: y1,\n        y2: y2\n      };\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns svg representation of an instance\n     * @return {Array} an array of strings with the specific svg representation\n     * of the instance\n     */\n    _toSVG: function() {\n      var p = this.calcLinePoints();\n      return [\n        '<line ', 'COMMON_PARTS',\n        'x1=\"', p.x1,\n        '\" y1=\"', p.y1,\n        '\" x2=\"', p.x2,\n        '\" y2=\"', p.y2,\n        '\" />\\n'\n      ];\n    },\n    /* _TO_SVG_END_ */\n  });\n\n  /* _FROM_SVG_START_ */\n  /**\n   * List of attribute names to account for when parsing SVG element (used by {@link fabric.Line.fromElement})\n   * @static\n   * @memberOf fabric.Line\n   * @see http://www.w3.org/TR/SVG/shapes.html#LineElement\n   */\n  fabric.Line.ATTRIBUTE_NAMES = fabric.SHARED_ATTRIBUTES.concat('x1 y1 x2 y2'.split(' '));\n\n  /**\n   * Returns fabric.Line instance from an SVG element\n   * @static\n   * @memberOf fabric.Line\n   * @param {SVGElement} element Element to parse\n   * @param {Object} [options] Options object\n   * @param {Function} [callback] callback function invoked after parsing\n   */\n  fabric.Line.fromElement = function(element, callback, options) {\n    options = options || { };\n    var parsedAttributes = fabric.parseAttributes(element, fabric.Line.ATTRIBUTE_NAMES),\n        points = [\n          parsedAttributes.x1 || 0,\n          parsedAttributes.y1 || 0,\n          parsedAttributes.x2 || 0,\n          parsedAttributes.y2 || 0\n        ];\n    callback(new fabric.Line(points, extend(parsedAttributes, options)));\n  };\n  /* _FROM_SVG_END_ */\n\n  /**\n   * Returns fabric.Line instance from an object representation\n   * @static\n   * @memberOf fabric.Line\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] invoked with new instance as first argument\n   */\n  fabric.Line.fromObject = function(object, callback) {\n    function _callback(instance) {\n      delete instance.points;\n      callback && callback(instance);\n    };\n    var options = clone(object, true);\n    options.points = [object.x1, object.y1, object.x2, object.y2];\n    fabric.Object._fromObject('Line', options, _callback, 'points');\n  };\n\n  /**\n   * Produces a function that calculates distance from canvas edge to Line origin.\n   */\n  function makeEdgeToOriginGetter(propertyNames, originValues) {\n    var origin = propertyNames.origin,\n        axis1 = propertyNames.axis1,\n        axis2 = propertyNames.axis2,\n        dimension = propertyNames.dimension,\n        nearest = originValues.nearest,\n        center = originValues.center,\n        farthest = originValues.farthest;\n\n    return function() {\n      switch (this.get(origin)) {\n        case nearest:\n          return Math.min(this.get(axis1), this.get(axis2));\n        case center:\n          return Math.min(this.get(axis1), this.get(axis2)) + (0.5 * this.get(dimension));\n        case farthest:\n          return Math.max(this.get(axis1), this.get(axis2));\n      }\n    };\n\n  }\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      degreesToRadians = fabric.util.degreesToRadians;\n\n  if (fabric.Circle) {\n    fabric.warn('fabric.Circle is already defined.');\n    return;\n  }\n\n  /**\n   * Circle class\n   * @class fabric.Circle\n   * @extends fabric.Object\n   * @see {@link fabric.Circle#initialize} for constructor definition\n   */\n  fabric.Circle = fabric.util.createClass(fabric.Object, /** @lends fabric.Circle.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'circle',\n\n    /**\n     * Radius of this circle\n     * @type Number\n     * @default\n     */\n    radius: 0,\n\n    /**\n     * degrees of start of the circle.\n     * probably will change to degrees in next major version\n     * @type Number 0 - 359\n     * @default 0\n     */\n    startAngle: 0,\n\n    /**\n     * End angle of the circle\n     * probably will change to degrees in next major version\n     * @type Number 1 - 360\n     * @default 360\n     */\n    endAngle: 360,\n\n    cacheProperties: fabric.Object.prototype.cacheProperties.concat('radius', 'startAngle', 'endAngle'),\n\n    /**\n     * @private\n     * @param {String} key\n     * @param {*} value\n     * @return {fabric.Circle} thisArg\n     */\n    _set: function(key, value) {\n      this.callSuper('_set', key, value);\n\n      if (key === 'radius') {\n        this.setRadius(value);\n      }\n\n      return this;\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      return this.callSuper('toObject', ['radius', 'startAngle', 'endAngle'].concat(propertiesToInclude));\n    },\n\n    /* _TO_SVG_START_ */\n\n    /**\n     * Returns svg representation of an instance\n     * @return {Array} an array of strings with the specific svg representation\n     * of the instance\n     */\n    _toSVG: function() {\n      var svgString, x = 0, y = 0,\n          angle = (this.endAngle - this.startAngle) % 360;\n\n      if (angle === 0) {\n        svgString = [\n          '<circle ', 'COMMON_PARTS',\n          'cx=\"' + x + '\" cy=\"' + y + '\" ',\n          'r=\"', this.radius,\n          '\" />\\n'\n        ];\n      }\n      else {\n        var start = degreesToRadians(this.startAngle),\n            end = degreesToRadians(this.endAngle),\n            radius = this.radius,\n            startX = fabric.util.cos(start) * radius,\n            startY = fabric.util.sin(start) * radius,\n            endX = fabric.util.cos(end) * radius,\n            endY = fabric.util.sin(end) * radius,\n            largeFlag = angle > 180 ? '1' : '0';\n        svgString = [\n          '<path d=\"M ' + startX + ' ' + startY,\n          ' A ' + radius + ' ' + radius,\n          ' 0 ', +largeFlag + ' 1', ' ' + endX + ' ' + endY,\n          '\" ', 'COMMON_PARTS', ' />\\n'\n        ];\n      }\n      return svgString;\n    },\n    /* _TO_SVG_END_ */\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx context to render on\n     */\n    _render: function(ctx) {\n      ctx.beginPath();\n      ctx.arc(\n        0,\n        0,\n        this.radius,\n        degreesToRadians(this.startAngle),\n        degreesToRadians(this.endAngle),\n        false\n      );\n      this._renderPaintInOrder(ctx);\n    },\n\n    /**\n     * Returns horizontal radius of an object (according to how an object is scaled)\n     * @return {Number}\n     */\n    getRadiusX: function() {\n      return this.get('radius') * this.get('scaleX');\n    },\n\n    /**\n     * Returns vertical radius of an object (according to how an object is scaled)\n     * @return {Number}\n     */\n    getRadiusY: function() {\n      return this.get('radius') * this.get('scaleY');\n    },\n\n    /**\n     * Sets radius of an object (and updates width accordingly)\n     * @return {fabric.Circle} thisArg\n     */\n    setRadius: function(value) {\n      this.radius = value;\n      return this.set('width', value * 2).set('height', value * 2);\n    },\n  });\n\n  /* _FROM_SVG_START_ */\n  /**\n   * List of attribute names to account for when parsing SVG element (used by {@link fabric.Circle.fromElement})\n   * @static\n   * @memberOf fabric.Circle\n   * @see: http://www.w3.org/TR/SVG/shapes.html#CircleElement\n   */\n  fabric.Circle.ATTRIBUTE_NAMES = fabric.SHARED_ATTRIBUTES.concat('cx cy r'.split(' '));\n\n  /**\n   * Returns {@link fabric.Circle} instance from an SVG element\n   * @static\n   * @memberOf fabric.Circle\n   * @param {SVGElement} element Element to parse\n   * @param {Function} [callback] Options callback invoked after parsing is finished\n   * @param {Object} [options] Options object\n   * @throws {Error} If value of `r` attribute is missing or invalid\n   */\n  fabric.Circle.fromElement = function(element, callback) {\n    var parsedAttributes = fabric.parseAttributes(element, fabric.Circle.ATTRIBUTE_NAMES);\n\n    if (!isValidRadius(parsedAttributes)) {\n      throw new Error('value of `r` attribute is required and can not be negative');\n    }\n\n    parsedAttributes.left = (parsedAttributes.left || 0) - parsedAttributes.radius;\n    parsedAttributes.top = (parsedAttributes.top || 0) - parsedAttributes.radius;\n    callback(new fabric.Circle(parsedAttributes));\n  };\n\n  /**\n   * @private\n   */\n  function isValidRadius(attributes) {\n    return (('radius' in attributes) && (attributes.radius >= 0));\n  }\n  /* _FROM_SVG_END_ */\n\n  /**\n   * Returns {@link fabric.Circle} instance from an object representation\n   * @static\n   * @memberOf fabric.Circle\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] invoked with new instance as first argument\n   * @return {void}\n   */\n  fabric.Circle.fromObject = function(object, callback) {\n    fabric.Object._fromObject('Circle', object, callback);\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { });\n\n  if (fabric.Triangle) {\n    fabric.warn('fabric.Triangle is already defined');\n    return;\n  }\n\n  /**\n   * Triangle class\n   * @class fabric.Triangle\n   * @extends fabric.Object\n   * @return {fabric.Triangle} thisArg\n   * @see {@link fabric.Triangle#initialize} for constructor definition\n   */\n  fabric.Triangle = fabric.util.createClass(fabric.Object, /** @lends fabric.Triangle.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'triangle',\n\n    /**\n     * Width is set to 100 to compensate the old initialize code that was setting it to 100\n     * @type Number\n     * @default\n     */\n    width: 100,\n\n    /**\n     * Height is set to 100 to compensate the old initialize code that was setting it to 100\n     * @type Number\n     * @default\n     */\n    height: 100,\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _render: function(ctx) {\n      var widthBy2 = this.width / 2,\n          heightBy2 = this.height / 2;\n\n      ctx.beginPath();\n      ctx.moveTo(-widthBy2, heightBy2);\n      ctx.lineTo(0, -heightBy2);\n      ctx.lineTo(widthBy2, heightBy2);\n      ctx.closePath();\n\n      this._renderPaintInOrder(ctx);\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns svg representation of an instance\n     * @return {Array} an array of strings with the specific svg representation\n     * of the instance\n     */\n    _toSVG: function() {\n      var widthBy2 = this.width / 2,\n          heightBy2 = this.height / 2,\n          points = [\n            -widthBy2 + ' ' + heightBy2,\n            '0 ' + -heightBy2,\n            widthBy2 + ' ' + heightBy2\n          ].join(',');\n      return [\n        '<polygon ', 'COMMON_PARTS',\n        'points=\"', points,\n        '\" />'\n      ];\n    },\n    /* _TO_SVG_END_ */\n  });\n\n  /**\n   * Returns {@link fabric.Triangle} instance from an object representation\n   * @static\n   * @memberOf fabric.Triangle\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] invoked with new instance as first argument\n   */\n  fabric.Triangle.fromObject = function(object, callback) {\n    return fabric.Object._fromObject('Triangle', object, callback);\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      piBy2   = Math.PI * 2;\n\n  if (fabric.Ellipse) {\n    fabric.warn('fabric.Ellipse is already defined.');\n    return;\n  }\n\n  /**\n   * Ellipse class\n   * @class fabric.Ellipse\n   * @extends fabric.Object\n   * @return {fabric.Ellipse} thisArg\n   * @see {@link fabric.Ellipse#initialize} for constructor definition\n   */\n  fabric.Ellipse = fabric.util.createClass(fabric.Object, /** @lends fabric.Ellipse.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'ellipse',\n\n    /**\n     * Horizontal radius\n     * @type Number\n     * @default\n     */\n    rx:   0,\n\n    /**\n     * Vertical radius\n     * @type Number\n     * @default\n     */\n    ry:   0,\n\n    cacheProperties: fabric.Object.prototype.cacheProperties.concat('rx', 'ry'),\n\n    /**\n     * Constructor\n     * @param {Object} [options] Options object\n     * @return {fabric.Ellipse} thisArg\n     */\n    initialize: function(options) {\n      this.callSuper('initialize', options);\n      this.set('rx', options && options.rx || 0);\n      this.set('ry', options && options.ry || 0);\n    },\n\n    /**\n     * @private\n     * @param {String} key\n     * @param {*} value\n     * @return {fabric.Ellipse} thisArg\n     */\n    _set: function(key, value) {\n      this.callSuper('_set', key, value);\n      switch (key) {\n\n        case 'rx':\n          this.rx = value;\n          this.set('width', value * 2);\n          break;\n\n        case 'ry':\n          this.ry = value;\n          this.set('height', value * 2);\n          break;\n\n      }\n      return this;\n    },\n\n    /**\n     * Returns horizontal radius of an object (according to how an object is scaled)\n     * @return {Number}\n     */\n    getRx: function() {\n      return this.get('rx') * this.get('scaleX');\n    },\n\n    /**\n     * Returns Vertical radius of an object (according to how an object is scaled)\n     * @return {Number}\n     */\n    getRy: function() {\n      return this.get('ry') * this.get('scaleY');\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      return this.callSuper('toObject', ['rx', 'ry'].concat(propertiesToInclude));\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns svg representation of an instance\n     * @return {Array} an array of strings with the specific svg representation\n     * of the instance\n     */\n    _toSVG: function() {\n      return [\n        '<ellipse ', 'COMMON_PARTS',\n        'cx=\"0\" cy=\"0\" ',\n        'rx=\"', this.rx,\n        '\" ry=\"', this.ry,\n        '\" />\\n'\n      ];\n    },\n    /* _TO_SVG_END_ */\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx context to render on\n     */\n    _render: function(ctx) {\n      ctx.beginPath();\n      ctx.save();\n      ctx.transform(1, 0, 0, this.ry / this.rx, 0, 0);\n      ctx.arc(\n        0,\n        0,\n        this.rx,\n        0,\n        piBy2,\n        false);\n      ctx.restore();\n      this._renderPaintInOrder(ctx);\n    },\n  });\n\n  /* _FROM_SVG_START_ */\n  /**\n   * List of attribute names to account for when parsing SVG element (used by {@link fabric.Ellipse.fromElement})\n   * @static\n   * @memberOf fabric.Ellipse\n   * @see http://www.w3.org/TR/SVG/shapes.html#EllipseElement\n   */\n  fabric.Ellipse.ATTRIBUTE_NAMES = fabric.SHARED_ATTRIBUTES.concat('cx cy rx ry'.split(' '));\n\n  /**\n   * Returns {@link fabric.Ellipse} instance from an SVG element\n   * @static\n   * @memberOf fabric.Ellipse\n   * @param {SVGElement} element Element to parse\n   * @param {Function} [callback] Options callback invoked after parsing is finished\n   * @return {fabric.Ellipse}\n   */\n  fabric.Ellipse.fromElement = function(element, callback) {\n\n    var parsedAttributes = fabric.parseAttributes(element, fabric.Ellipse.ATTRIBUTE_NAMES);\n\n    parsedAttributes.left = (parsedAttributes.left || 0) - parsedAttributes.rx;\n    parsedAttributes.top = (parsedAttributes.top || 0) - parsedAttributes.ry;\n    callback(new fabric.Ellipse(parsedAttributes));\n  };\n  /* _FROM_SVG_END_ */\n\n  /**\n   * Returns {@link fabric.Ellipse} instance from an object representation\n   * @static\n   * @memberOf fabric.Ellipse\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] invoked with new instance as first argument\n   * @return {void}\n   */\n  fabric.Ellipse.fromObject = function(object, callback) {\n    fabric.Object._fromObject('Ellipse', object, callback);\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      extend = fabric.util.object.extend;\n\n  if (fabric.Rect) {\n    fabric.warn('fabric.Rect is already defined');\n    return;\n  }\n\n  /**\n   * Rectangle class\n   * @class fabric.Rect\n   * @extends fabric.Object\n   * @return {fabric.Rect} thisArg\n   * @see {@link fabric.Rect#initialize} for constructor definition\n   */\n  fabric.Rect = fabric.util.createClass(fabric.Object, /** @lends fabric.Rect.prototype */ {\n\n    /**\n     * List of properties to consider when checking if state of an object is changed ({@link fabric.Object#hasStateChanged})\n     * as well as for history (undo/redo) purposes\n     * @type Array\n     */\n    stateProperties: fabric.Object.prototype.stateProperties.concat('rx', 'ry'),\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'rect',\n\n    /**\n     * Horizontal border radius\n     * @type Number\n     * @default\n     */\n    rx:   0,\n\n    /**\n     * Vertical border radius\n     * @type Number\n     * @default\n     */\n    ry:   0,\n\n    cacheProperties: fabric.Object.prototype.cacheProperties.concat('rx', 'ry'),\n\n    /**\n     * Constructor\n     * @param {Object} [options] Options object\n     * @return {Object} thisArg\n     */\n    initialize: function(options) {\n      this.callSuper('initialize', options);\n      this._initRxRy();\n    },\n\n    /**\n     * Initializes rx/ry attributes\n     * @private\n     */\n    _initRxRy: function() {\n      if (this.rx && !this.ry) {\n        this.ry = this.rx;\n      }\n      else if (this.ry && !this.rx) {\n        this.rx = this.ry;\n      }\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _render: function(ctx) {\n\n      // 1x1 case (used in spray brush) optimization was removed because\n      // with caching and higher zoom level this makes more damage than help\n\n      var rx = this.rx ? Math.min(this.rx, this.width / 2) : 0,\n          ry = this.ry ? Math.min(this.ry, this.height / 2) : 0,\n          w = this.width,\n          h = this.height,\n          x = -this.width / 2,\n          y = -this.height / 2,\n          isRounded = rx !== 0 || ry !== 0,\n          /* \"magic number\" for bezier approximations of arcs (http://itc.ktu.lt/itc354/Riskus354.pdf) */\n          k = 1 - 0.5522847498;\n      ctx.beginPath();\n\n      ctx.moveTo(x + rx, y);\n\n      ctx.lineTo(x + w - rx, y);\n      isRounded && ctx.bezierCurveTo(x + w - k * rx, y, x + w, y + k * ry, x + w, y + ry);\n\n      ctx.lineTo(x + w, y + h - ry);\n      isRounded && ctx.bezierCurveTo(x + w, y + h - k * ry, x + w - k * rx, y + h, x + w - rx, y + h);\n\n      ctx.lineTo(x + rx, y + h);\n      isRounded && ctx.bezierCurveTo(x + k * rx, y + h, x, y + h - k * ry, x, y + h - ry);\n\n      ctx.lineTo(x, y + ry);\n      isRounded && ctx.bezierCurveTo(x, y + k * ry, x + k * rx, y, x + rx, y);\n\n      ctx.closePath();\n\n      this._renderPaintInOrder(ctx);\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      return this.callSuper('toObject', ['rx', 'ry'].concat(propertiesToInclude));\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns svg representation of an instance\n     * @return {Array} an array of strings with the specific svg representation\n     * of the instance\n     */\n    _toSVG: function() {\n      var x = -this.width / 2, y = -this.height / 2;\n      return [\n        '<rect ', 'COMMON_PARTS',\n        'x=\"', x, '\" y=\"', y,\n        '\" rx=\"', this.rx, '\" ry=\"', this.ry,\n        '\" width=\"', this.width, '\" height=\"', this.height,\n        '\" />\\n'\n      ];\n    },\n    /* _TO_SVG_END_ */\n  });\n\n  /* _FROM_SVG_START_ */\n  /**\n   * List of attribute names to account for when parsing SVG element (used by `fabric.Rect.fromElement`)\n   * @static\n   * @memberOf fabric.Rect\n   * @see: http://www.w3.org/TR/SVG/shapes.html#RectElement\n   */\n  fabric.Rect.ATTRIBUTE_NAMES = fabric.SHARED_ATTRIBUTES.concat('x y rx ry width height'.split(' '));\n\n  /**\n   * Returns {@link fabric.Rect} instance from an SVG element\n   * @static\n   * @memberOf fabric.Rect\n   * @param {SVGElement} element Element to parse\n   * @param {Function} callback callback function invoked after parsing\n   * @param {Object} [options] Options object\n   */\n  fabric.Rect.fromElement = function(element, callback, options) {\n    if (!element) {\n      return callback(null);\n    }\n    options = options || { };\n\n    var parsedAttributes = fabric.parseAttributes(element, fabric.Rect.ATTRIBUTE_NAMES);\n    parsedAttributes.left = parsedAttributes.left || 0;\n    parsedAttributes.top  = parsedAttributes.top  || 0;\n    parsedAttributes.height  = parsedAttributes.height || 0;\n    parsedAttributes.width  = parsedAttributes.width || 0;\n    var rect = new fabric.Rect(extend((options ? fabric.util.object.clone(options) : { }), parsedAttributes));\n    rect.visible = rect.visible && rect.width > 0 && rect.height > 0;\n    callback(rect);\n  };\n  /* _FROM_SVG_END_ */\n\n  /**\n   * Returns {@link fabric.Rect} instance from an object representation\n   * @static\n   * @memberOf fabric.Rect\n   * @param {Object} object Object to create an instance from\n   * @param {Function} [callback] Callback to invoke when an fabric.Rect instance is created\n   */\n  fabric.Rect.fromObject = function(object, callback) {\n    return fabric.Object._fromObject('Rect', object, callback);\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      extend = fabric.util.object.extend,\n      min = fabric.util.array.min,\n      max = fabric.util.array.max,\n      toFixed = fabric.util.toFixed,\n      projectStrokeOnPoints = fabric.util.projectStrokeOnPoints;\n\n  if (fabric.Polyline) {\n    fabric.warn('fabric.Polyline is already defined');\n    return;\n  }\n\n  /**\n   * Polyline class\n   * @class fabric.Polyline\n   * @extends fabric.Object\n   * @see {@link fabric.Polyline#initialize} for constructor definition\n   */\n  fabric.Polyline = fabric.util.createClass(fabric.Object, /** @lends fabric.Polyline.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'polyline',\n\n    /**\n     * Points array\n     * @type Array\n     * @default\n     */\n    points: null,\n\n    /**\n     * WARNING: Feature in progress\n     * Calculate the exact bounding box taking in account strokeWidth on acute angles\n     * this will be turned to true by default on fabric 6.0\n     * maybe will be left in as an optimization since calculations may be slow\n     * @deprecated\n     * @type Boolean\n     * @default false\n     */\n    exactBoundingBox: false,\n\n    cacheProperties: fabric.Object.prototype.cacheProperties.concat('points'),\n\n    /**\n     * Constructor\n     * @param {Array} points Array of points (where each point is an object with x and y)\n     * @param {Object} [options] Options object\n     * @return {fabric.Polyline} thisArg\n     * @example\n     * var poly = new fabric.Polyline([\n     *     { x: 10, y: 10 },\n     *     { x: 50, y: 30 },\n     *     { x: 40, y: 70 },\n     *     { x: 60, y: 50 },\n     *     { x: 100, y: 150 },\n     *     { x: 40, y: 100 }\n     *   ], {\n     *   stroke: 'red',\n     *   left: 100,\n     *   top: 100\n     * });\n     */\n    initialize: function(points, options) {\n      options = options || {};\n      this.points = points || [];\n      this.callSuper('initialize', options);\n      this._setPositionDimensions(options);\n    },\n\n    /**\n     * @private\n     */\n    _projectStrokeOnPoints: function () {\n      return projectStrokeOnPoints(this.points, this, true);\n    },\n\n    _setPositionDimensions: function(options) {\n      var calcDim = this._calcDimensions(options), correctLeftTop,\n          correctSize = this.exactBoundingBox ? this.strokeWidth : 0;\n      this.width = calcDim.width - correctSize;\n      this.height = calcDim.height - correctSize;\n      if (!options.fromSVG) {\n        correctLeftTop = this.translateToGivenOrigin(\n          {\n            // this looks bad, but is one way to keep it optional for now.\n            x: calcDim.left - this.strokeWidth / 2 + correctSize / 2,\n            y: calcDim.top - this.strokeWidth / 2 + correctSize / 2\n          },\n          'left',\n          'top',\n          this.originX,\n          this.originY\n        );\n      }\n      if (typeof options.left === 'undefined') {\n        this.left = options.fromSVG ? calcDim.left : correctLeftTop.x;\n      }\n      if (typeof options.top === 'undefined') {\n        this.top = options.fromSVG ? calcDim.top : correctLeftTop.y;\n      }\n      this.pathOffset = {\n        x: calcDim.left + this.width / 2 + correctSize / 2,\n        y: calcDim.top + this.height / 2 + correctSize / 2\n      };\n    },\n\n    /**\n     * Calculate the polygon min and max point from points array,\n     * returning an object with left, top, width, height to measure the\n     * polygon size\n     * @return {Object} object.left X coordinate of the polygon leftmost point\n     * @return {Object} object.top Y coordinate of the polygon topmost point\n     * @return {Object} object.width distance between X coordinates of the polygon leftmost and rightmost point\n     * @return {Object} object.height distance between Y coordinates of the polygon topmost and bottommost point\n     * @private\n     */\n    _calcDimensions: function() {\n\n      var points = this.exactBoundingBox ? this._projectStrokeOnPoints() : this.points,\n          minX = min(points, 'x') || 0,\n          minY = min(points, 'y') || 0,\n          maxX = max(points, 'x') || 0,\n          maxY = max(points, 'y') || 0,\n          width = (maxX - minX),\n          height = (maxY - minY);\n\n      return {\n        left: minX,\n        top: minY,\n        width: width,\n        height: height,\n      };\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      return extend(this.callSuper('toObject', propertiesToInclude), {\n        points: this.points.concat()\n      });\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns svg representation of an instance\n     * @return {Array} an array of strings with the specific svg representation\n     * of the instance\n     */\n    _toSVG: function() {\n      var points = [], diffX = this.pathOffset.x, diffY = this.pathOffset.y,\n          NUM_FRACTION_DIGITS = fabric.Object.NUM_FRACTION_DIGITS;\n\n      for (var i = 0, len = this.points.length; i < len; i++) {\n        points.push(\n          toFixed(this.points[i].x - diffX, NUM_FRACTION_DIGITS), ',',\n          toFixed(this.points[i].y - diffY, NUM_FRACTION_DIGITS), ' '\n        );\n      }\n      return [\n        '<' + this.type + ' ', 'COMMON_PARTS',\n        'points=\"', points.join(''),\n        '\" />\\n'\n      ];\n    },\n    /* _TO_SVG_END_ */\n\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    commonRender: function(ctx) {\n      var point, len = this.points.length,\n          x = this.pathOffset.x,\n          y = this.pathOffset.y;\n\n      if (!len || isNaN(this.points[len - 1].y)) {\n        // do not draw if no points or odd points\n        // NaN comes from parseFloat of a empty string in parser\n        return false;\n      }\n      ctx.beginPath();\n      ctx.moveTo(this.points[0].x - x, this.points[0].y - y);\n      for (var i = 0; i < len; i++) {\n        point = this.points[i];\n        ctx.lineTo(point.x - x, point.y - y);\n      }\n      return true;\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _render: function(ctx) {\n      if (!this.commonRender(ctx)) {\n        return;\n      }\n      this._renderPaintInOrder(ctx);\n    },\n\n    /**\n     * Returns complexity of an instance\n     * @return {Number} complexity of this instance\n     */\n    complexity: function() {\n      return this.get('points').length;\n    }\n  });\n\n  /* _FROM_SVG_START_ */\n  /**\n   * List of attribute names to account for when parsing SVG element (used by {@link fabric.Polyline.fromElement})\n   * @static\n   * @memberOf fabric.Polyline\n   * @see: http://www.w3.org/TR/SVG/shapes.html#PolylineElement\n   */\n  fabric.Polyline.ATTRIBUTE_NAMES = fabric.SHARED_ATTRIBUTES.concat();\n\n  /**\n   * Returns fabric.Polyline instance from an SVG element\n   * @static\n   * @memberOf fabric.Polyline\n   * @param {SVGElement} element Element to parser\n   * @param {Function} callback callback function invoked after parsing\n   * @param {Object} [options] Options object\n   */\n  fabric.Polyline.fromElementGenerator = function(_class) {\n    return function(element, callback, options) {\n      if (!element) {\n        return callback(null);\n      }\n      options || (options = { });\n\n      var points = fabric.parsePointsAttribute(element.getAttribute('points')),\n          parsedAttributes = fabric.parseAttributes(element, fabric[_class].ATTRIBUTE_NAMES);\n      parsedAttributes.fromSVG = true;\n      callback(new fabric[_class](points, extend(parsedAttributes, options)));\n    };\n  };\n\n  fabric.Polyline.fromElement = fabric.Polyline.fromElementGenerator('Polyline');\n\n  /* _FROM_SVG_END_ */\n\n  /**\n   * Returns fabric.Polyline instance from an object representation\n   * @static\n   * @memberOf fabric.Polyline\n   * @param {Object} object Object to create an instance from\n   * @param {Function} [callback] Callback to invoke when an fabric.Path instance is created\n   */\n  fabric.Polyline.fromObject = function(object, callback) {\n    return fabric.Object._fromObject('Polyline', object, callback, 'points');\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = {}),\n      projectStrokeOnPoints = fabric.util.projectStrokeOnPoints;\n\n  if (fabric.Polygon) {\n    fabric.warn('fabric.Polygon is already defined');\n    return;\n  }\n\n  /**\n   * Polygon class\n   * @class fabric.Polygon\n   * @extends fabric.Polyline\n   * @see {@link fabric.Polygon#initialize} for constructor definition\n   */\n  fabric.Polygon = fabric.util.createClass(fabric.Polyline, /** @lends fabric.Polygon.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'polygon',\n\n    /**\n     * @private\n     */\n    _projectStrokeOnPoints: function () {\n      return projectStrokeOnPoints(this.points, this);\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _render: function(ctx) {\n      if (!this.commonRender(ctx)) {\n        return;\n      }\n      ctx.closePath();\n      this._renderPaintInOrder(ctx);\n    },\n\n  });\n\n  /* _FROM_SVG_START_ */\n  /**\n   * List of attribute names to account for when parsing SVG element (used by `fabric.Polygon.fromElement`)\n   * @static\n   * @memberOf fabric.Polygon\n   * @see: http://www.w3.org/TR/SVG/shapes.html#PolygonElement\n   */\n  fabric.Polygon.ATTRIBUTE_NAMES = fabric.SHARED_ATTRIBUTES.concat();\n\n  /**\n   * Returns {@link fabric.Polygon} instance from an SVG element\n   * @static\n   * @memberOf fabric.Polygon\n   * @param {SVGElement} element Element to parse\n   * @param {Function} callback callback function invoked after parsing\n   * @param {Object} [options] Options object\n   */\n  fabric.Polygon.fromElement = fabric.Polyline.fromElementGenerator('Polygon');\n  /* _FROM_SVG_END_ */\n\n  /**\n   * Returns fabric.Polygon instance from an object representation\n   * @static\n   * @memberOf fabric.Polygon\n   * @param {Object} object Object to create an instance from\n   * @param {Function} [callback] Callback to invoke when an fabric.Path instance is created\n   * @return {void}\n   */\n  fabric.Polygon.fromObject = function(object, callback) {\n    fabric.Object._fromObject('Polygon', object, callback, 'points');\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      min = fabric.util.array.min,\n      max = fabric.util.array.max,\n      extend = fabric.util.object.extend,\n      clone = fabric.util.object.clone,\n      toFixed = fabric.util.toFixed;\n\n  if (fabric.Path) {\n    fabric.warn('fabric.Path is already defined');\n    return;\n  }\n\n  /**\n   * Path class\n   * @class fabric.Path\n   * @extends fabric.Object\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-1#path_and_pathgroup}\n   * @see {@link fabric.Path#initialize} for constructor definition\n   */\n  fabric.Path = fabric.util.createClass(fabric.Object, /** @lends fabric.Path.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'path',\n\n    /**\n     * Array of path points\n     * @type Array\n     * @default\n     */\n    path: null,\n\n    cacheProperties: fabric.Object.prototype.cacheProperties.concat('path', 'fillRule'),\n\n    stateProperties: fabric.Object.prototype.stateProperties.concat('path'),\n\n    /**\n     * Constructor\n     * @param {Array|String} path Path data (sequence of coordinates and corresponding \"command\" tokens)\n     * @param {Object} [options] Options object\n     * @return {fabric.Path} thisArg\n     */\n    initialize: function (path, options) {\n      options = clone(options || {});\n      delete options.path;\n      this.callSuper('initialize', options);\n      this._setPath(path || [], options);\n    },\n\n    /**\n    * @private\n    * @param {Array|String} path Path data (sequence of coordinates and corresponding \"command\" tokens)\n    * @param {Object} [options] Options object\n    */\n    _setPath: function (path, options) {\n      this.path = fabric.util.makePathSimpler(\n        Array.isArray(path) ? path : fabric.util.parsePath(path)\n      );\n\n      fabric.Polyline.prototype._setPositionDimensions.call(this, options || {});\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx context to render path on\n     */\n    _renderPathCommands: function(ctx) {\n      var current, // current instruction\n          subpathStartX = 0,\n          subpathStartY = 0,\n          x = 0, // current x\n          y = 0, // current y\n          controlX = 0, // current control point x\n          controlY = 0, // current control point y\n          l = -this.pathOffset.x,\n          t = -this.pathOffset.y;\n\n      ctx.beginPath();\n\n      for (var i = 0, len = this.path.length; i < len; ++i) {\n\n        current = this.path[i];\n\n        switch (current[0]) { // first letter\n\n          case 'L': // lineto, absolute\n            x = current[1];\n            y = current[2];\n            ctx.lineTo(x + l, y + t);\n            break;\n\n          case 'M': // moveTo, absolute\n            x = current[1];\n            y = current[2];\n            subpathStartX = x;\n            subpathStartY = y;\n            ctx.moveTo(x + l, y + t);\n            break;\n\n          case 'C': // bezierCurveTo, absolute\n            x = current[5];\n            y = current[6];\n            controlX = current[3];\n            controlY = current[4];\n            ctx.bezierCurveTo(\n              current[1] + l,\n              current[2] + t,\n              controlX + l,\n              controlY + t,\n              x + l,\n              y + t\n            );\n            break;\n\n          case 'Q': // quadraticCurveTo, absolute\n            ctx.quadraticCurveTo(\n              current[1] + l,\n              current[2] + t,\n              current[3] + l,\n              current[4] + t\n            );\n            x = current[3];\n            y = current[4];\n            controlX = current[1];\n            controlY = current[2];\n            break;\n\n          case 'z':\n          case 'Z':\n            x = subpathStartX;\n            y = subpathStartY;\n            ctx.closePath();\n            break;\n        }\n      }\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx context to render path on\n     */\n    _render: function(ctx) {\n      this._renderPathCommands(ctx);\n      this._renderPaintInOrder(ctx);\n    },\n\n    /**\n     * Returns string representation of an instance\n     * @return {String} string representation of an instance\n     */\n    toString: function() {\n      return '#<fabric.Path (' + this.complexity() +\n        '): { \"top\": ' + this.top + ', \"left\": ' + this.left + ' }>';\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      return extend(this.callSuper('toObject', propertiesToInclude), {\n        path: this.path.map(function(item) { return item.slice(); }),\n      });\n    },\n\n    /**\n     * Returns dataless object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toDatalessObject: function(propertiesToInclude) {\n      var o = this.toObject(['sourcePath'].concat(propertiesToInclude));\n      if (o.sourcePath) {\n        delete o.path;\n      }\n      return o;\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns svg representation of an instance\n     * @return {Array} an array of strings with the specific svg representation\n     * of the instance\n     */\n    _toSVG: function() {\n      var path = fabric.util.joinPath(this.path);\n      return [\n        '<path ', 'COMMON_PARTS',\n        'd=\"', path,\n        '\" stroke-linecap=\"round\" ',\n        '/>\\n'\n      ];\n    },\n\n    _getOffsetTransform: function() {\n      var digits = fabric.Object.NUM_FRACTION_DIGITS;\n      return ' translate(' + toFixed(-this.pathOffset.x, digits) + ', ' +\n          toFixed(-this.pathOffset.y, digits) + ')';\n    },\n\n    /**\n     * Returns svg clipPath representation of an instance\n     * @param {Function} [reviver] Method for further parsing of svg representation.\n     * @return {String} svg representation of an instance\n     */\n    toClipPathSVG: function(reviver) {\n      var additionalTransform = this._getOffsetTransform();\n      return '\\t' + this._createBaseClipPathSVGMarkup(\n        this._toSVG(), { reviver: reviver, additionalTransform: additionalTransform }\n      );\n    },\n\n    /**\n     * Returns svg representation of an instance\n     * @param {Function} [reviver] Method for further parsing of svg representation.\n     * @return {String} svg representation of an instance\n     */\n    toSVG: function(reviver) {\n      var additionalTransform = this._getOffsetTransform();\n      return this._createBaseSVGMarkup(this._toSVG(), { reviver: reviver, additionalTransform: additionalTransform  });\n    },\n    /* _TO_SVG_END_ */\n\n    /**\n     * Returns number representation of an instance complexity\n     * @return {Number} complexity of this instance\n     */\n    complexity: function() {\n      return this.path.length;\n    },\n\n    /**\n     * @private\n     */\n    _calcDimensions: function() {\n\n      var aX = [],\n          aY = [],\n          current, // current instruction\n          subpathStartX = 0,\n          subpathStartY = 0,\n          x = 0, // current x\n          y = 0, // current y\n          bounds;\n\n      for (var i = 0, len = this.path.length; i < len; ++i) {\n\n        current = this.path[i];\n\n        switch (current[0]) { // first letter\n\n          case 'L': // lineto, absolute\n            x = current[1];\n            y = current[2];\n            bounds = [];\n            break;\n\n          case 'M': // moveTo, absolute\n            x = current[1];\n            y = current[2];\n            subpathStartX = x;\n            subpathStartY = y;\n            bounds = [];\n            break;\n\n          case 'C': // bezierCurveTo, absolute\n            bounds = fabric.util.getBoundsOfCurve(x, y,\n              current[1],\n              current[2],\n              current[3],\n              current[4],\n              current[5],\n              current[6]\n            );\n            x = current[5];\n            y = current[6];\n            break;\n\n          case 'Q': // quadraticCurveTo, absolute\n            bounds = fabric.util.getBoundsOfCurve(x, y,\n              current[1],\n              current[2],\n              current[1],\n              current[2],\n              current[3],\n              current[4]\n            );\n            x = current[3];\n            y = current[4];\n            break;\n\n          case 'z':\n          case 'Z':\n            x = subpathStartX;\n            y = subpathStartY;\n            break;\n        }\n        bounds.forEach(function (point) {\n          aX.push(point.x);\n          aY.push(point.y);\n        });\n        aX.push(x);\n        aY.push(y);\n      }\n\n      var minX = min(aX) || 0,\n          minY = min(aY) || 0,\n          maxX = max(aX) || 0,\n          maxY = max(aY) || 0,\n          deltaX = maxX - minX,\n          deltaY = maxY - minY;\n\n      return {\n        left: minX,\n        top: minY,\n        width: deltaX,\n        height: deltaY\n      };\n    }\n  });\n\n  /**\n   * Creates an instance of fabric.Path from an object\n   * @static\n   * @memberOf fabric.Path\n   * @param {Object} object\n   * @param {Function} [callback] Callback to invoke when an fabric.Path instance is created\n   */\n  fabric.Path.fromObject = function(object, callback) {\n    if (typeof object.sourcePath === 'string') {\n      var pathUrl = object.sourcePath;\n      fabric.loadSVGFromURL(pathUrl, function (elements) {\n        var path = elements[0];\n        path.setOptions(object);\n        callback && callback(path);\n      });\n    }\n    else {\n      fabric.Object._fromObject('Path', object, callback, 'path');\n    }\n  };\n\n  /* _FROM_SVG_START_ */\n  /**\n   * List of attribute names to account for when parsing SVG element (used by `fabric.Path.fromElement`)\n   * @static\n   * @memberOf fabric.Path\n   * @see http://www.w3.org/TR/SVG/paths.html#PathElement\n   */\n  fabric.Path.ATTRIBUTE_NAMES = fabric.SHARED_ATTRIBUTES.concat(['d']);\n\n  /**\n   * Creates an instance of fabric.Path from an SVG <path> element\n   * @static\n   * @memberOf fabric.Path\n   * @param {SVGElement} element to parse\n   * @param {Function} callback Callback to invoke when an fabric.Path instance is created\n   * @param {Object} [options] Options object\n   * @param {Function} [callback] Options callback invoked after parsing is finished\n   */\n  fabric.Path.fromElement = function(element, callback, options) {\n    var parsedAttributes = fabric.parseAttributes(element, fabric.Path.ATTRIBUTE_NAMES);\n    parsedAttributes.fromSVG = true;\n    callback(new fabric.Path(parsedAttributes.d, extend(parsedAttributes, options)));\n  };\n  /* _FROM_SVG_END_ */\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      min = fabric.util.array.min,\n      max = fabric.util.array.max;\n\n  if (fabric.Group) {\n    return;\n  }\n\n  /**\n   * Group class\n   * @class fabric.Group\n   * @extends fabric.Object\n   * @mixes fabric.Collection\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-3#groups}\n   * @see {@link fabric.Group#initialize} for constructor definition\n   */\n  fabric.Group = fabric.util.createClass(fabric.Object, fabric.Collection, /** @lends fabric.Group.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'group',\n\n    /**\n     * Width of stroke\n     * @type Number\n     * @default\n     */\n    strokeWidth: 0,\n\n    /**\n     * Indicates if click, mouseover, mouseout events & hoverCursor should also check for subtargets\n     * @type Boolean\n     * @default\n     */\n    subTargetCheck: false,\n\n    /**\n     * Groups are container, do not render anything on theyr own, ence no cache properties\n     * @type Array\n     * @default\n     */\n    cacheProperties: [],\n\n    /**\n     * setOnGroup is a method used for TextBox that is no more used since 2.0.0 The behavior is still\n     * available setting this boolean to true.\n     * @type Boolean\n     * @since 2.0.0\n     * @default\n     */\n    useSetOnGroup: false,\n\n    /**\n     * Constructor\n     * @param {Object} objects Group objects\n     * @param {Object} [options] Options object\n     * @param {Boolean} [isAlreadyGrouped] if true, objects have been grouped already.\n     * @return {Object} thisArg\n     */\n    initialize: function(objects, options, isAlreadyGrouped) {\n      options = options || {};\n      this._objects = [];\n      // if objects enclosed in a group have been grouped already,\n      // we cannot change properties of objects.\n      // Thus we need to set options to group without objects,\n      isAlreadyGrouped && this.callSuper('initialize', options);\n      this._objects = objects || [];\n      for (var i = this._objects.length; i--; ) {\n        this._objects[i].group = this;\n      }\n\n      if (!isAlreadyGrouped) {\n        var center = options && options.centerPoint;\n        // we want to set origins before calculating the bounding box.\n        // so that the topleft can be set with that in mind.\n        // if specific top and left are passed, are overwritten later\n        // with the callSuper('initialize', options)\n        if (options.originX !== undefined) {\n          this.originX = options.originX;\n        }\n        if (options.originY !== undefined) {\n          this.originY = options.originY;\n        }\n        // if coming from svg i do not want to calc bounds.\n        // i assume width and height are passed along options\n        center || this._calcBounds();\n        this._updateObjectsCoords(center);\n        delete options.centerPoint;\n        this.callSuper('initialize', options);\n      }\n      else {\n        this._updateObjectsACoords();\n      }\n\n      this.setCoords();\n    },\n\n    /**\n     * @private\n     */\n    _updateObjectsACoords: function() {\n      var skipControls = true;\n      for (var i = this._objects.length; i--; ){\n        this._objects[i].setCoords(skipControls);\n      }\n    },\n\n    /**\n     * @private\n     * @param {Boolean} [skipCoordsChange] if true, coordinates of objects enclosed in a group do not change\n     */\n    _updateObjectsCoords: function(center) {\n      var center = center || this.getCenterPoint();\n      for (var i = this._objects.length; i--; ){\n        this._updateObjectCoords(this._objects[i], center);\n      }\n    },\n\n    /**\n     * @private\n     * @param {Object} object\n     * @param {fabric.Point} center, current center of group.\n     */\n    _updateObjectCoords: function(object, center) {\n      var objectLeft = object.left,\n          objectTop = object.top,\n          skipControls = true;\n\n      object.set({\n        left: objectLeft - center.x,\n        top: objectTop - center.y\n      });\n      object.group = this;\n      object.setCoords(skipControls);\n    },\n\n    /**\n     * Returns string represenation of a group\n     * @return {String}\n     */\n    toString: function() {\n      return '#<fabric.Group: (' + this.complexity() + ')>';\n    },\n\n    /**\n     * Adds an object to a group; Then recalculates group's dimension, position.\n     * @param {Object} object\n     * @return {fabric.Group} thisArg\n     * @chainable\n     */\n    addWithUpdate: function(object) {\n      var nested = !!this.group;\n      this._restoreObjectsState();\n      fabric.util.resetObjectTransform(this);\n      if (object) {\n        if (nested) {\n          // if this group is inside another group, we need to pre transform the object\n          fabric.util.removeTransformFromObject(object, this.group.calcTransformMatrix());\n        }\n        this._objects.push(object);\n        object.group = this;\n        object._set('canvas', this.canvas);\n      }\n      this._calcBounds();\n      this._updateObjectsCoords();\n      this.dirty = true;\n      if (nested) {\n        this.group.addWithUpdate();\n      }\n      else {\n        this.setCoords();\n      }\n      return this;\n    },\n\n    /**\n     * Removes an object from a group; Then recalculates group's dimension, position.\n     * @param {Object} object\n     * @return {fabric.Group} thisArg\n     * @chainable\n     */\n    removeWithUpdate: function(object) {\n      this._restoreObjectsState();\n      fabric.util.resetObjectTransform(this);\n\n      this.remove(object);\n      this._calcBounds();\n      this._updateObjectsCoords();\n      this.setCoords();\n      this.dirty = true;\n      return this;\n    },\n\n    /**\n     * @private\n     */\n    _onObjectAdded: function(object) {\n      this.dirty = true;\n      object.group = this;\n      object._set('canvas', this.canvas);\n    },\n\n    /**\n     * @private\n     */\n    _onObjectRemoved: function(object) {\n      this.dirty = true;\n      delete object.group;\n    },\n\n    /**\n     * @private\n     */\n    _set: function(key, value) {\n      var i = this._objects.length;\n      if (this.useSetOnGroup) {\n        while (i--) {\n          this._objects[i].setOnGroup(key, value);\n        }\n      }\n      if (key === 'canvas') {\n        while (i--) {\n          this._objects[i]._set(key, value);\n        }\n      }\n      fabric.Object.prototype._set.call(this, key, value);\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      var _includeDefaultValues = this.includeDefaultValues;\n      var objsToObject = this._objects\n        .filter(function (obj) {\n          return !obj.excludeFromExport;\n        })\n        .map(function (obj) {\n          var originalDefaults = obj.includeDefaultValues;\n          obj.includeDefaultValues = _includeDefaultValues;\n          var _obj = obj.toObject(propertiesToInclude);\n          obj.includeDefaultValues = originalDefaults;\n          return _obj;\n        });\n      var obj = fabric.Object.prototype.toObject.call(this, propertiesToInclude);\n      obj.objects = objsToObject;\n      return obj;\n    },\n\n    /**\n     * Returns object representation of an instance, in dataless mode.\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} object representation of an instance\n     */\n    toDatalessObject: function(propertiesToInclude) {\n      var objsToObject, sourcePath = this.sourcePath;\n      if (sourcePath) {\n        objsToObject = sourcePath;\n      }\n      else {\n        var _includeDefaultValues = this.includeDefaultValues;\n        objsToObject = this._objects.map(function(obj) {\n          var originalDefaults = obj.includeDefaultValues;\n          obj.includeDefaultValues = _includeDefaultValues;\n          var _obj = obj.toDatalessObject(propertiesToInclude);\n          obj.includeDefaultValues = originalDefaults;\n          return _obj;\n        });\n      }\n      var obj = fabric.Object.prototype.toDatalessObject.call(this, propertiesToInclude);\n      obj.objects = objsToObject;\n      return obj;\n    },\n\n    /**\n     * Renders instance on a given context\n     * @param {CanvasRenderingContext2D} ctx context to render instance on\n     */\n    render: function(ctx) {\n      this._transformDone = true;\n      this.callSuper('render', ctx);\n      this._transformDone = false;\n    },\n\n    /**\n     * Decide if the object should cache or not. Create its own cache level\n     * needsItsOwnCache should be used when the object drawing method requires\n     * a cache step. None of the fabric classes requires it.\n     * Generally you do not cache objects in groups because the group is already cached.\n     * @return {Boolean}\n     */\n    shouldCache: function() {\n      var ownCache = fabric.Object.prototype.shouldCache.call(this);\n      if (ownCache) {\n        for (var i = 0, len = this._objects.length; i < len; i++) {\n          if (this._objects[i].willDrawShadow()) {\n            this.ownCaching = false;\n            return false;\n          }\n        }\n      }\n      return ownCache;\n    },\n\n    /**\n     * Check if this object or a child object will cast a shadow\n     * @return {Boolean}\n     */\n    willDrawShadow: function() {\n      if (fabric.Object.prototype.willDrawShadow.call(this)) {\n        return true;\n      }\n      for (var i = 0, len = this._objects.length; i < len; i++) {\n        if (this._objects[i].willDrawShadow()) {\n          return true;\n        }\n      }\n      return false;\n    },\n\n    /**\n     * Check if this group or its parent group are caching, recursively up\n     * @return {Boolean}\n     */\n    isOnACache: function() {\n      return this.ownCaching || (this.group && this.group.isOnACache());\n    },\n\n    /**\n     * Execute the drawing operation for an object on a specified context\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    drawObject: function(ctx) {\n      for (var i = 0, len = this._objects.length; i < len; i++) {\n        this._objects[i].render(ctx);\n      }\n      this._drawClipPath(ctx, this.clipPath);\n    },\n\n    /**\n     * Check if cache is dirty\n     */\n    isCacheDirty: function(skipCanvas) {\n      if (this.callSuper('isCacheDirty', skipCanvas)) {\n        return true;\n      }\n      if (!this.statefullCache) {\n        return false;\n      }\n      for (var i = 0, len = this._objects.length; i < len; i++) {\n        if (this._objects[i].isCacheDirty(true)) {\n          if (this._cacheCanvas) {\n            // if this group has not a cache canvas there is nothing to clean\n            var x = this.cacheWidth / this.zoomX, y = this.cacheHeight / this.zoomY;\n            this._cacheContext.clearRect(-x / 2, -y / 2, x, y);\n          }\n          return true;\n        }\n      }\n      return false;\n    },\n\n    /**\n     * Restores original state of each of group objects (original state is that which was before group was created).\n     * if the nested boolean is true, the original state will be restored just for the\n     * first group and not for all the group chain\n     * @private\n     * @param {Boolean} nested tell the function to restore object state up to the parent group and not more\n     * @return {fabric.Group} thisArg\n     * @chainable\n     */\n    _restoreObjectsState: function() {\n      var groupMatrix = this.calcOwnMatrix();\n      this._objects.forEach(function(object) {\n        // instead of using _this = this;\n        fabric.util.addTransformToObject(object, groupMatrix);\n        delete object.group;\n        object.setCoords();\n      });\n      return this;\n    },\n\n    /**\n     * Destroys a group (restoring state of its objects)\n     * @return {fabric.Group} thisArg\n     * @chainable\n     */\n    destroy: function() {\n      // when group is destroyed objects needs to get a repaint to be eventually\n      // displayed on canvas.\n      this._objects.forEach(function(object) {\n        object.set('dirty', true);\n      });\n      return this._restoreObjectsState();\n    },\n\n    dispose: function () {\n      this.callSuper('dispose');\n      this.forEachObject(function (object) {\n        object.dispose && object.dispose();\n      });\n      this._objects = [];\n    },\n\n    /**\n     * make a group an active selection, remove the group from canvas\n     * the group has to be on canvas for this to work.\n     * @return {fabric.ActiveSelection} thisArg\n     * @chainable\n     */\n    toActiveSelection: function() {\n      if (!this.canvas) {\n        return;\n      }\n      var objects = this._objects, canvas = this.canvas;\n      this._objects = [];\n      var options = this.toObject();\n      delete options.objects;\n      var activeSelection = new fabric.ActiveSelection([]);\n      activeSelection.set(options);\n      activeSelection.type = 'activeSelection';\n      canvas.remove(this);\n      objects.forEach(function(object) {\n        object.group = activeSelection;\n        object.dirty = true;\n        canvas.add(object);\n      });\n      activeSelection.canvas = canvas;\n      activeSelection._objects = objects;\n      canvas._activeObject = activeSelection;\n      activeSelection.setCoords();\n      return activeSelection;\n    },\n\n    /**\n     * Destroys a group (restoring state of its objects)\n     * @return {fabric.Group} thisArg\n     * @chainable\n     */\n    ungroupOnCanvas: function() {\n      return this._restoreObjectsState();\n    },\n\n    /**\n     * Sets coordinates of all objects inside group\n     * @return {fabric.Group} thisArg\n     * @chainable\n     */\n    setObjectsCoords: function() {\n      var skipControls = true;\n      this.forEachObject(function(object) {\n        object.setCoords(skipControls);\n      });\n      return this;\n    },\n\n    /**\n     * @private\n     */\n    _calcBounds: function(onlyWidthHeight) {\n      var aX = [],\n          aY = [],\n          o, prop, coords,\n          props = ['tr', 'br', 'bl', 'tl'],\n          i = 0, iLen = this._objects.length,\n          j, jLen = props.length;\n\n      for ( ; i < iLen; ++i) {\n        o = this._objects[i];\n        coords = o.calcACoords();\n        for (j = 0; j < jLen; j++) {\n          prop = props[j];\n          aX.push(coords[prop].x);\n          aY.push(coords[prop].y);\n        }\n        o.aCoords = coords;\n      }\n\n      this._getBounds(aX, aY, onlyWidthHeight);\n    },\n\n    /**\n     * @private\n     */\n    _getBounds: function(aX, aY, onlyWidthHeight) {\n      var minXY = new fabric.Point(min(aX), min(aY)),\n          maxXY = new fabric.Point(max(aX), max(aY)),\n          top = minXY.y || 0, left = minXY.x || 0,\n          width = (maxXY.x - minXY.x) || 0,\n          height = (maxXY.y - minXY.y) || 0;\n      this.width = width;\n      this.height = height;\n      if (!onlyWidthHeight) {\n        // the bounding box always finds the topleft most corner.\n        // whatever is the group origin, we set up here the left/top position.\n        this.setPositionByOrigin({ x: left, y: top }, 'left', 'top');\n      }\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns svg representation of an instance\n     * @param {Function} [reviver] Method for further parsing of svg representation.\n     * @return {String} svg representation of an instance\n     */\n    _toSVG: function(reviver) {\n      var svgString = ['<g ', 'COMMON_PARTS', ' >\\n'];\n\n      for (var i = 0, len = this._objects.length; i < len; i++) {\n        svgString.push('\\t\\t', this._objects[i].toSVG(reviver));\n      }\n      svgString.push('</g>\\n');\n      return svgString;\n    },\n\n    /**\n     * Returns styles-string for svg-export, specific version for group\n     * @return {String}\n     */\n    getSvgStyles: function() {\n      var opacity = typeof this.opacity !== 'undefined' && this.opacity !== 1 ?\n            'opacity: ' + this.opacity + ';' : '',\n          visibility = this.visible ? '' : ' visibility: hidden;';\n      return [\n        opacity,\n        this.getSvgFilter(),\n        visibility\n      ].join('');\n    },\n\n    /**\n     * Returns svg clipPath representation of an instance\n     * @param {Function} [reviver] Method for further parsing of svg representation.\n     * @return {String} svg representation of an instance\n     */\n    toClipPathSVG: function(reviver) {\n      var svgString = [];\n\n      for (var i = 0, len = this._objects.length; i < len; i++) {\n        svgString.push('\\t', this._objects[i].toClipPathSVG(reviver));\n      }\n\n      return this._createBaseClipPathSVGMarkup(svgString, { reviver: reviver });\n    },\n    /* _TO_SVG_END_ */\n  });\n\n  /**\n   * Returns {@link fabric.Group} instance from an object representation\n   * @static\n   * @memberOf fabric.Group\n   * @param {Object} object Object to create a group from\n   * @param {Function} [callback] Callback to invoke when an group instance is created\n   */\n  fabric.Group.fromObject = function(object, callback) {\n    var objects = object.objects,\n        options = fabric.util.object.clone(object, true);\n    delete options.objects;\n    if (typeof objects === 'string') {\n      // it has to be an url or something went wrong.\n      fabric.loadSVGFromURL(objects, function (elements) {\n        var group = fabric.util.groupSVGElements(elements, object, objects);\n        group.set(options);\n        callback && callback(group);\n      });\n      return;\n    }\n    fabric.util.enlivenObjects(objects, function (enlivenedObjects) {\n      var options = fabric.util.object.clone(object, true);\n      delete options.objects;\n      fabric.util.enlivenObjectEnlivables(object, options, function () {\n        callback && callback(new fabric.Group(enlivenedObjects, options, true));\n      });\n    });\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { });\n\n  if (fabric.ActiveSelection) {\n    return;\n  }\n\n  /**\n   * Group class\n   * @class fabric.ActiveSelection\n   * @extends fabric.Group\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-3#groups}\n   * @see {@link fabric.ActiveSelection#initialize} for constructor definition\n   */\n  fabric.ActiveSelection = fabric.util.createClass(fabric.Group, /** @lends fabric.ActiveSelection.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'activeSelection',\n\n    /**\n     * Constructor\n     * @param {Object} objects ActiveSelection objects\n     * @param {Object} [options] Options object\n     * @return {Object} thisArg\n     */\n    initialize: function(objects, options) {\n      options = options || {};\n      this._objects = objects || [];\n      for (var i = this._objects.length; i--; ) {\n        this._objects[i].group = this;\n      }\n\n      if (options.originX) {\n        this.originX = options.originX;\n      }\n      if (options.originY) {\n        this.originY = options.originY;\n      }\n      this._calcBounds();\n      this._updateObjectsCoords();\n      fabric.Object.prototype.initialize.call(this, options);\n      this.setCoords();\n    },\n\n    /**\n     * Change te activeSelection to a normal group,\n     * High level function that automatically adds it to canvas as\n     * active object. no events fired.\n     * @since 2.0.0\n     * @return {fabric.Group}\n     */\n    toGroup: function() {\n      var objects = this._objects.concat();\n      this._objects = [];\n      var options = fabric.Object.prototype.toObject.call(this);\n      var newGroup = new fabric.Group([]);\n      delete options.type;\n      newGroup.set(options);\n      objects.forEach(function(object) {\n        object.canvas.remove(object);\n        object.group = newGroup;\n      });\n      newGroup._objects = objects;\n      if (!this.canvas) {\n        return newGroup;\n      }\n      var canvas = this.canvas;\n      canvas.add(newGroup);\n      canvas._activeObject = newGroup;\n      newGroup.setCoords();\n      return newGroup;\n    },\n\n    /**\n     * If returns true, deselection is cancelled.\n     * @since 2.0.0\n     * @return {Boolean} [cancel]\n     */\n    onDeselect: function() {\n      this.destroy();\n      return false;\n    },\n\n    /**\n     * Returns string representation of a group\n     * @return {String}\n     */\n    toString: function() {\n      return '#<fabric.ActiveSelection: (' + this.complexity() + ')>';\n    },\n\n    /**\n     * Decide if the object should cache or not. Create its own cache level\n     * objectCaching is a global flag, wins over everything\n     * needsItsOwnCache should be used when the object drawing method requires\n     * a cache step. None of the fabric classes requires it.\n     * Generally you do not cache objects in groups because the group outside is cached.\n     * @return {Boolean}\n     */\n    shouldCache: function() {\n      return false;\n    },\n\n    /**\n     * Check if this group or its parent group are caching, recursively up\n     * @return {Boolean}\n     */\n    isOnACache: function() {\n      return false;\n    },\n\n    /**\n     * Renders controls and borders for the object\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {Object} [styleOverride] properties to override the object style\n     * @param {Object} [childrenOverride] properties to override the children overrides\n     */\n    _renderControls: function(ctx, styleOverride, childrenOverride) {\n      ctx.save();\n      ctx.globalAlpha = this.isMoving ? this.borderOpacityWhenMoving : 1;\n      this.callSuper('_renderControls', ctx, styleOverride);\n      childrenOverride = childrenOverride || { };\n      if (typeof childrenOverride.hasControls === 'undefined') {\n        childrenOverride.hasControls = false;\n      }\n      childrenOverride.forActiveSelection = true;\n      for (var i = 0, len = this._objects.length; i < len; i++) {\n        this._objects[i]._renderControls(ctx, childrenOverride);\n      }\n      ctx.restore();\n    },\n  });\n\n  /**\n   * Returns {@link fabric.ActiveSelection} instance from an object representation\n   * @static\n   * @memberOf fabric.ActiveSelection\n   * @param {Object} object Object to create a group from\n   * @param {Function} [callback] Callback to invoke when an ActiveSelection instance is created\n   */\n  fabric.ActiveSelection.fromObject = function(object, callback) {\n    fabric.util.enlivenObjects(object.objects, function(enlivenedObjects) {\n      delete object.objects;\n      callback && callback(new fabric.ActiveSelection(enlivenedObjects, object, true));\n    });\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var extend = fabric.util.object.extend;\n\n  if (!global.fabric) {\n    global.fabric = { };\n  }\n\n  if (global.fabric.Image) {\n    fabric.warn('fabric.Image is already defined.');\n    return;\n  }\n\n  /**\n   * Image class\n   * @class fabric.Image\n   * @extends fabric.Object\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-1#images}\n   * @see {@link fabric.Image#initialize} for constructor definition\n   */\n  fabric.Image = fabric.util.createClass(fabric.Object, /** @lends fabric.Image.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'image',\n\n    /**\n     * Width of a stroke.\n     * For image quality a stroke multiple of 2 gives better results.\n     * @type Number\n     * @default\n     */\n    strokeWidth: 0,\n\n    /**\n     * When calling {@link fabric.Image.getSrc}, return value from element src with `element.getAttribute('src')`.\n     * This allows for relative urls as image src.\n     * @since 2.7.0\n     * @type Boolean\n     * @default\n     */\n    srcFromAttribute: false,\n\n    /**\n     * private\n     * contains last value of scaleX to detect\n     * if the Image got resized after the last Render\n     * @type Number\n     */\n    _lastScaleX: 1,\n\n    /**\n     * private\n     * contains last value of scaleY to detect\n     * if the Image got resized after the last Render\n     * @type Number\n     */\n    _lastScaleY: 1,\n\n    /**\n     * private\n     * contains last value of scaling applied by the apply filter chain\n     * @type Number\n     */\n    _filterScalingX: 1,\n\n    /**\n     * private\n     * contains last value of scaling applied by the apply filter chain\n     * @type Number\n     */\n    _filterScalingY: 1,\n\n    /**\n     * minimum scale factor under which any resizeFilter is triggered to resize the image\n     * 0 will disable the automatic resize. 1 will trigger automatically always.\n     * number bigger than 1 are not implemented yet.\n     * @type Number\n     */\n    minimumScaleTrigger: 0.5,\n\n    /**\n     * List of properties to consider when checking if\n     * state of an object is changed ({@link fabric.Object#hasStateChanged})\n     * as well as for history (undo/redo) purposes\n     * @type Array\n     */\n    stateProperties: fabric.Object.prototype.stateProperties.concat('cropX', 'cropY'),\n\n    /**\n     * List of properties to consider when checking if cache needs refresh\n     * Those properties are checked by statefullCache ON ( or lazy mode if we want ) or from single\n     * calls to Object.set(key, value). If the key is in this list, the object is marked as dirty\n     * and refreshed at the next render\n     * @type Array\n     */\n    cacheProperties: fabric.Object.prototype.cacheProperties.concat('cropX', 'cropY'),\n\n    /**\n     * key used to retrieve the texture representing this image\n     * @since 2.0.0\n     * @type String\n     * @default\n     */\n    cacheKey: '',\n\n    /**\n     * Image crop in pixels from original image size.\n     * @since 2.0.0\n     * @type Number\n     * @default\n     */\n    cropX: 0,\n\n    /**\n     * Image crop in pixels from original image size.\n     * @since 2.0.0\n     * @type Number\n     * @default\n     */\n    cropY: 0,\n\n    /**\n     * Indicates whether this canvas will use image smoothing when painting this image.\n     * Also influence if the cacheCanvas for this image uses imageSmoothing\n     * @since 4.0.0-beta.11\n     * @type Boolean\n     * @default\n     */\n    imageSmoothing: true,\n\n    /**\n     * Constructor\n     * Image can be initialized with any canvas drawable or a string.\n     * The string should be a url and will be loaded as an image.\n     * Canvas and Image element work out of the box, while videos require extra code to work.\n     * Please check video element events for seeking.\n     * @param {HTMLImageElement | HTMLCanvasElement | HTMLVideoElement | String} element Image element\n     * @param {Object} [options] Options object\n     * @param {function} [callback] callback function to call after eventual filters applied.\n     * @return {fabric.Image} thisArg\n     */\n    initialize: function(element, options) {\n      options || (options = { });\n      this.filters = [];\n      this.cacheKey = 'texture' + fabric.Object.__uid++;\n      this.callSuper('initialize', options);\n      this._initElement(element, options);\n    },\n\n    /**\n     * Returns image element which this instance if based on\n     * @return {HTMLImageElement} Image element\n     */\n    getElement: function() {\n      return this._element || {};\n    },\n\n    /**\n     * Sets image element for this instance to a specified one.\n     * If filters defined they are applied to new image.\n     * You might need to call `canvas.renderAll` and `object.setCoords` after replacing, to render new image and update controls area.\n     * @param {HTMLImageElement} element\n     * @param {Object} [options] Options object\n     * @return {fabric.Image} thisArg\n     * @chainable\n     */\n    setElement: function(element, options) {\n      this.removeTexture(this.cacheKey);\n      this.removeTexture(this.cacheKey + '_filtered');\n      this._element = element;\n      this._originalElement = element;\n      this._initConfig(options);\n      if (this.filters.length !== 0) {\n        this.applyFilters();\n      }\n      // resizeFilters work on the already filtered copy.\n      // we need to apply resizeFilters AFTER normal filters.\n      // applyResizeFilters is run more often than normal filters\n      // and is triggered by user interactions rather than dev code\n      if (this.resizeFilter) {\n        this.applyResizeFilters();\n      }\n      return this;\n    },\n\n    /**\n     * Delete a single texture if in webgl mode\n     */\n    removeTexture: function(key) {\n      var backend = fabric.filterBackend;\n      if (backend && backend.evictCachesForKey) {\n        backend.evictCachesForKey(key);\n      }\n    },\n\n    /**\n     * Delete textures, reference to elements and eventually JSDOM cleanup\n     */\n    dispose: function () {\n      this.callSuper('dispose');\n      this.removeTexture(this.cacheKey);\n      this.removeTexture(this.cacheKey + '_filtered');\n      this._cacheContext = undefined;\n      ['_originalElement', '_element', '_filteredEl', '_cacheCanvas'].forEach((function(element) {\n        fabric.util.cleanUpJsdomNode(this[element]);\n        this[element] = undefined;\n      }).bind(this));\n    },\n\n    /**\n     * Get the crossOrigin value (of the corresponding image element)\n     */\n    getCrossOrigin: function() {\n      return this._originalElement && (this._originalElement.crossOrigin || null);\n    },\n\n    /**\n     * Returns original size of an image\n     * @return {Object} Object with \"width\" and \"height\" properties\n     */\n    getOriginalSize: function() {\n      var element = this.getElement();\n      return {\n        width: element.naturalWidth || element.width,\n        height: element.naturalHeight || element.height\n      };\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _stroke: function(ctx) {\n      if (!this.stroke || this.strokeWidth === 0) {\n        return;\n      }\n      var w = this.width / 2, h = this.height / 2;\n      ctx.beginPath();\n      ctx.moveTo(-w, -h);\n      ctx.lineTo(w, -h);\n      ctx.lineTo(w, h);\n      ctx.lineTo(-w, h);\n      ctx.lineTo(-w, -h);\n      ctx.closePath();\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      var filters = [];\n\n      this.filters.forEach(function(filterObj) {\n        if (filterObj) {\n          filters.push(filterObj.toObject());\n        }\n      });\n      var object = extend(\n        this.callSuper(\n          'toObject',\n          ['cropX', 'cropY'].concat(propertiesToInclude)\n        ), {\n          src: this.getSrc(),\n          crossOrigin: this.getCrossOrigin(),\n          filters: filters,\n        });\n      if (this.resizeFilter) {\n        object.resizeFilter = this.resizeFilter.toObject();\n      }\n      return object;\n    },\n\n    /**\n     * Returns true if an image has crop applied, inspecting values of cropX,cropY,width,height.\n     * @return {Boolean}\n     */\n    hasCrop: function() {\n      return this.cropX || this.cropY || this.width < this._element.width || this.height < this._element.height;\n    },\n\n    /* _TO_SVG_START_ */\n    /**\n     * Returns svg representation of an instance\n     * @return {Array} an array of strings with the specific svg representation\n     * of the instance\n     */\n    _toSVG: function() {\n      var svgString = [], imageMarkup = [], strokeSvg, element = this._element,\n          x = -this.width / 2, y = -this.height / 2, clipPath = '', imageRendering = '';\n      if (!element) {\n        return [];\n      }\n      if (this.hasCrop()) {\n        var clipPathId = fabric.Object.__uid++;\n        svgString.push(\n          '<clipPath id=\"imageCrop_' + clipPathId + '\">\\n',\n          '\\t<rect x=\"' + x + '\" y=\"' + y + '\" width=\"' + this.width + '\" height=\"' + this.height + '\" />\\n',\n          '</clipPath>\\n'\n        );\n        clipPath = ' clip-path=\"url(#imageCrop_' + clipPathId + ')\" ';\n      }\n      if (!this.imageSmoothing) {\n        imageRendering = '\" image-rendering=\"optimizeSpeed';\n      }\n      imageMarkup.push('\\t<image ', 'COMMON_PARTS', 'xlink:href=\"', this.getSvgSrc(true),\n        '\" x=\"', x - this.cropX, '\" y=\"', y - this.cropY,\n        // we're essentially moving origin of transformation from top/left corner to the center of the shape\n        // by wrapping it in container <g> element with actual transformation, then offsetting object to the top/left\n        // so that object's center aligns with container's left/top\n        '\" width=\"', element.width || element.naturalWidth,\n        '\" height=\"', element.height || element.height,\n        imageRendering,\n        '\"', clipPath,\n        '></image>\\n');\n\n      if (this.stroke || this.strokeDashArray) {\n        var origFill = this.fill;\n        this.fill = null;\n        strokeSvg = [\n          '\\t<rect ',\n          'x=\"', x, '\" y=\"', y,\n          '\" width=\"', this.width, '\" height=\"', this.height,\n          '\" style=\"', this.getSvgStyles(),\n          '\"/>\\n'\n        ];\n        this.fill = origFill;\n      }\n      if (this.paintFirst !== 'fill') {\n        svgString = svgString.concat(strokeSvg, imageMarkup);\n      }\n      else {\n        svgString = svgString.concat(imageMarkup, strokeSvg);\n      }\n      return svgString;\n    },\n    /* _TO_SVG_END_ */\n\n    /**\n     * Returns source of an image\n     * @param {Boolean} filtered indicates if the src is needed for svg\n     * @return {String} Source of an image\n     */\n    getSrc: function(filtered) {\n      var element = filtered ? this._element : this._originalElement;\n      if (element) {\n        if (element.toDataURL) {\n          return element.toDataURL();\n        }\n\n        if (this.srcFromAttribute) {\n          return element.getAttribute('src');\n        }\n        else {\n          return element.src;\n        }\n      }\n      else {\n        return this.src || '';\n      }\n    },\n\n    /**\n     * Sets source of an image\n     * @param {String} src Source string (URL)\n     * @param {Function} [callback] Callback is invoked when image has been loaded (and all filters have been applied)\n     * @param {Object} [options] Options object\n     * @param {String} [options.crossOrigin] crossOrigin value (one of \"\", \"anonymous\", \"use-credentials\")\n     * @see https://developer.mozilla.org/en-US/docs/HTML/CORS_settings_attributes\n     * @return {fabric.Image} thisArg\n     * @chainable\n     */\n    setSrc: function(src, callback, options) {\n      fabric.util.loadImage(src, function(img, isError) {\n        this.setElement(img, options);\n        this._setWidthHeight();\n        callback && callback(this, isError);\n      }, this, options && options.crossOrigin);\n      return this;\n    },\n\n    /**\n     * Returns string representation of an instance\n     * @return {String} String representation of an instance\n     */\n    toString: function() {\n      return '#<fabric.Image: { src: \"' + this.getSrc() + '\" }>';\n    },\n\n    applyResizeFilters: function() {\n      var filter = this.resizeFilter,\n          minimumScale = this.minimumScaleTrigger,\n          objectScale = this.getTotalObjectScaling(),\n          scaleX = objectScale.scaleX,\n          scaleY = objectScale.scaleY,\n          elementToFilter = this._filteredEl || this._originalElement;\n      if (this.group) {\n        this.set('dirty', true);\n      }\n      if (!filter || (scaleX > minimumScale && scaleY > minimumScale)) {\n        this._element = elementToFilter;\n        this._filterScalingX = 1;\n        this._filterScalingY = 1;\n        this._lastScaleX = scaleX;\n        this._lastScaleY = scaleY;\n        return;\n      }\n      if (!fabric.filterBackend) {\n        fabric.filterBackend = fabric.initFilterBackend();\n      }\n      var canvasEl = fabric.util.createCanvasElement(),\n          cacheKey = this._filteredEl ? (this.cacheKey + '_filtered') : this.cacheKey,\n          sourceWidth = elementToFilter.width, sourceHeight = elementToFilter.height;\n      canvasEl.width = sourceWidth;\n      canvasEl.height = sourceHeight;\n      this._element = canvasEl;\n      this._lastScaleX = filter.scaleX = scaleX;\n      this._lastScaleY = filter.scaleY = scaleY;\n      fabric.filterBackend.applyFilters(\n        [filter], elementToFilter, sourceWidth, sourceHeight, this._element, cacheKey);\n      this._filterScalingX = canvasEl.width / this._originalElement.width;\n      this._filterScalingY = canvasEl.height / this._originalElement.height;\n    },\n\n    /**\n     * Applies filters assigned to this image (from \"filters\" array) or from filter param\n     * @method applyFilters\n     * @param {Array} filters to be applied\n     * @param {Boolean} forResizing specify if the filter operation is a resize operation\n     * @return {thisArg} return the fabric.Image object\n     * @chainable\n     */\n    applyFilters: function(filters) {\n\n      filters = filters || this.filters || [];\n      filters = filters.filter(function(filter) { return filter && !filter.isNeutralState(); });\n      this.set('dirty', true);\n\n      // needs to clear out or WEBGL will not resize correctly\n      this.removeTexture(this.cacheKey + '_filtered');\n\n      if (filters.length === 0) {\n        this._element = this._originalElement;\n        this._filteredEl = null;\n        this._filterScalingX = 1;\n        this._filterScalingY = 1;\n        return this;\n      }\n\n      var imgElement = this._originalElement,\n          sourceWidth = imgElement.naturalWidth || imgElement.width,\n          sourceHeight = imgElement.naturalHeight || imgElement.height;\n\n      if (this._element === this._originalElement) {\n        // if the element is the same we need to create a new element\n        var canvasEl = fabric.util.createCanvasElement();\n        canvasEl.width = sourceWidth;\n        canvasEl.height = sourceHeight;\n        this._element = canvasEl;\n        this._filteredEl = canvasEl;\n      }\n      else {\n        // clear the existing element to get new filter data\n        // also dereference the eventual resized _element\n        this._element = this._filteredEl;\n        this._filteredEl.getContext('2d').clearRect(0, 0, sourceWidth, sourceHeight);\n        // we also need to resize again at next renderAll, so remove saved _lastScaleX/Y\n        this._lastScaleX = 1;\n        this._lastScaleY = 1;\n      }\n      if (!fabric.filterBackend) {\n        fabric.filterBackend = fabric.initFilterBackend();\n      }\n      fabric.filterBackend.applyFilters(\n        filters, this._originalElement, sourceWidth, sourceHeight, this._element, this.cacheKey);\n      if (this._originalElement.width !== this._element.width ||\n        this._originalElement.height !== this._element.height) {\n        this._filterScalingX = this._element.width / this._originalElement.width;\n        this._filterScalingY = this._element.height / this._originalElement.height;\n      }\n      return this;\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _render: function(ctx) {\n      fabric.util.setImageSmoothing(ctx, this.imageSmoothing);\n      if (this.isMoving !== true && this.resizeFilter && this._needsResize()) {\n        this.applyResizeFilters();\n      }\n      this._stroke(ctx);\n      this._renderPaintInOrder(ctx);\n    },\n\n    /**\n     * Paint the cached copy of the object on the target context.\n     * it will set the imageSmoothing for the draw operation\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    drawCacheOnCanvas: function(ctx) {\n      fabric.util.setImageSmoothing(ctx, this.imageSmoothing);\n      fabric.Object.prototype.drawCacheOnCanvas.call(this, ctx);\n    },\n\n    /**\n     * Decide if the object should cache or not. Create its own cache level\n     * needsItsOwnCache should be used when the object drawing method requires\n     * a cache step. None of the fabric classes requires it.\n     * Generally you do not cache objects in groups because the group outside is cached.\n     * This is the special image version where we would like to avoid caching where possible.\n     * Essentially images do not benefit from caching. They may require caching, and in that\n     * case we do it. Also caching an image usually ends in a loss of details.\n     * A full performance audit should be done.\n     * @return {Boolean}\n     */\n    shouldCache: function() {\n      return this.needsItsOwnCache();\n    },\n\n    _renderFill: function(ctx) {\n      var elementToDraw = this._element;\n      if (!elementToDraw) {\n        return;\n      }\n      var scaleX = this._filterScalingX, scaleY = this._filterScalingY,\n          w = this.width, h = this.height, min = Math.min, max = Math.max,\n          // crop values cannot be lesser than 0.\n          cropX = max(this.cropX, 0), cropY = max(this.cropY, 0),\n          elWidth = elementToDraw.naturalWidth || elementToDraw.width,\n          elHeight = elementToDraw.naturalHeight || elementToDraw.height,\n          sX = cropX * scaleX,\n          sY = cropY * scaleY,\n          // the width height cannot exceed element width/height, starting from the crop offset.\n          sW = min(w * scaleX, elWidth - sX),\n          sH = min(h * scaleY, elHeight - sY),\n          x = -w / 2, y = -h / 2,\n          maxDestW = min(w, elWidth / scaleX - cropX),\n          maxDestH = min(h, elHeight / scaleY - cropY);\n\n      elementToDraw && ctx.drawImage(elementToDraw, sX, sY, sW, sH, x, y, maxDestW, maxDestH);\n    },\n\n    /**\n     * needed to check if image needs resize\n     * @private\n     */\n    _needsResize: function() {\n      var scale = this.getTotalObjectScaling();\n      return (scale.scaleX !== this._lastScaleX || scale.scaleY !== this._lastScaleY);\n    },\n\n    /**\n     * @private\n     */\n    _resetWidthHeight: function() {\n      this.set(this.getOriginalSize());\n    },\n\n    /**\n     * The Image class's initialization method. This method is automatically\n     * called by the constructor.\n     * @private\n     * @param {HTMLImageElement|String} element The element representing the image\n     * @param {Object} [options] Options object\n     */\n    _initElement: function(element, options) {\n      this.setElement(fabric.util.getById(element), options);\n      fabric.util.addClass(this.getElement(), fabric.Image.CSS_CANVAS);\n    },\n\n    /**\n     * @private\n     * @param {Object} [options] Options object\n     */\n    _initConfig: function(options) {\n      options || (options = { });\n      this.setOptions(options);\n      this._setWidthHeight(options);\n    },\n\n    /**\n     * @private\n     * @param {Array} filters to be initialized\n     * @param {Function} callback Callback to invoke when all fabric.Image.filters instances are created\n     */\n    _initFilters: function(filters, callback) {\n      if (filters && filters.length) {\n        fabric.util.enlivenObjects(filters, function(enlivenedObjects) {\n          callback && callback(enlivenedObjects);\n        }, 'fabric.Image.filters');\n      }\n      else {\n        callback && callback();\n      }\n    },\n\n    /**\n     * @private\n     * Set the width and the height of the image object, using the element or the\n     * options.\n     * @param {Object} [options] Object with width/height properties\n     */\n    _setWidthHeight: function(options) {\n      options || (options = { });\n      var el = this.getElement();\n      this.width = options.width || el.naturalWidth || el.width || 0;\n      this.height = options.height || el.naturalHeight || el.height || 0;\n    },\n\n    /**\n     * Calculate offset for center and scale factor for the image in order to respect\n     * the preserveAspectRatio attribute\n     * @private\n     * @return {Object}\n     */\n    parsePreserveAspectRatioAttribute: function() {\n      var pAR = fabric.util.parsePreserveAspectRatioAttribute(this.preserveAspectRatio || ''),\n          rWidth = this._element.width, rHeight = this._element.height,\n          scaleX = 1, scaleY = 1, offsetLeft = 0, offsetTop = 0, cropX = 0, cropY = 0,\n          offset, pWidth = this.width, pHeight = this.height, parsedAttributes = { width: pWidth, height: pHeight };\n      if (pAR && (pAR.alignX !== 'none' || pAR.alignY !== 'none')) {\n        if (pAR.meetOrSlice === 'meet') {\n          scaleX = scaleY = fabric.util.findScaleToFit(this._element, parsedAttributes);\n          offset = (pWidth - rWidth * scaleX) / 2;\n          if (pAR.alignX === 'Min') {\n            offsetLeft = -offset;\n          }\n          if (pAR.alignX === 'Max') {\n            offsetLeft = offset;\n          }\n          offset = (pHeight - rHeight * scaleY) / 2;\n          if (pAR.alignY === 'Min') {\n            offsetTop = -offset;\n          }\n          if (pAR.alignY === 'Max') {\n            offsetTop = offset;\n          }\n        }\n        if (pAR.meetOrSlice === 'slice') {\n          scaleX = scaleY = fabric.util.findScaleToCover(this._element, parsedAttributes);\n          offset = rWidth - pWidth / scaleX;\n          if (pAR.alignX === 'Mid') {\n            cropX = offset / 2;\n          }\n          if (pAR.alignX === 'Max') {\n            cropX = offset;\n          }\n          offset = rHeight - pHeight / scaleY;\n          if (pAR.alignY === 'Mid') {\n            cropY = offset / 2;\n          }\n          if (pAR.alignY === 'Max') {\n            cropY = offset;\n          }\n          rWidth = pWidth / scaleX;\n          rHeight = pHeight / scaleY;\n        }\n      }\n      else {\n        scaleX = pWidth / rWidth;\n        scaleY = pHeight / rHeight;\n      }\n      return {\n        width: rWidth,\n        height: rHeight,\n        scaleX: scaleX,\n        scaleY: scaleY,\n        offsetLeft: offsetLeft,\n        offsetTop: offsetTop,\n        cropX: cropX,\n        cropY: cropY\n      };\n    }\n  });\n\n  /**\n   * Default CSS class name for canvas\n   * @static\n   * @type String\n   * @default\n   */\n  fabric.Image.CSS_CANVAS = 'canvas-img';\n\n  /**\n   * Alias for getSrc\n   * @static\n   */\n  fabric.Image.prototype.getSvgSrc = fabric.Image.prototype.getSrc;\n\n  /**\n   * Creates an instance of fabric.Image from its object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {Function} callback Callback to invoke when an image instance is created\n   */\n  fabric.Image.fromObject = function(_object, callback) {\n    var object = fabric.util.object.clone(_object);\n    fabric.util.loadImage(object.src, function(img, isError) {\n      if (isError) {\n        callback && callback(null, true);\n        return;\n      }\n      fabric.Image.prototype._initFilters.call(object, object.filters, function(filters) {\n        object.filters = filters || [];\n        fabric.Image.prototype._initFilters.call(object, [object.resizeFilter], function(resizeFilters) {\n          object.resizeFilter = resizeFilters[0];\n          fabric.util.enlivenObjectEnlivables(object, object, function () {\n            var image = new fabric.Image(img, object);\n            callback(image, false);\n          });\n        });\n      });\n    }, null, object.crossOrigin);\n  };\n\n  /**\n   * Creates an instance of fabric.Image from an URL string\n   * @static\n   * @param {String} url URL to create an image from\n   * @param {Function} [callback] Callback to invoke when image is created (newly created image is passed as a first argument). Second argument is a boolean indicating if an error occurred or not.\n   * @param {Object} [imgOptions] Options object\n   */\n  fabric.Image.fromURL = function(url, callback, imgOptions) {\n    fabric.util.loadImage(url, function(img, isError) {\n      callback && callback(new fabric.Image(img, imgOptions), isError);\n    }, null, imgOptions && imgOptions.crossOrigin);\n  };\n\n  /* _FROM_SVG_START_ */\n  /**\n   * List of attribute names to account for when parsing SVG element (used by {@link fabric.Image.fromElement})\n   * @static\n   * @see {@link http://www.w3.org/TR/SVG/struct.html#ImageElement}\n   */\n  fabric.Image.ATTRIBUTE_NAMES =\n    fabric.SHARED_ATTRIBUTES.concat(\n      'x y width height preserveAspectRatio xlink:href crossOrigin image-rendering'.split(' ')\n    );\n\n  /**\n   * Returns {@link fabric.Image} instance from an SVG element\n   * @static\n   * @param {SVGElement} element Element to parse\n   * @param {Object} [options] Options object\n   * @param {Function} callback Callback to execute when fabric.Image object is created\n   * @return {fabric.Image} Instance of fabric.Image\n   */\n  fabric.Image.fromElement = function(element, callback, options) {\n    var parsedAttributes = fabric.parseAttributes(element, fabric.Image.ATTRIBUTE_NAMES);\n    fabric.Image.fromURL(parsedAttributes['xlink:href'], callback,\n      extend((options ? fabric.util.object.clone(options) : { }), parsedAttributes));\n  };\n  /* _FROM_SVG_END_ */\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\nfabric.util.object.extend(fabric.Object.prototype, /** @lends fabric.Object.prototype */ {\n\n  /**\n   * @private\n   * @return {Number} angle value\n   */\n  _getAngleValueForStraighten: function() {\n    var angle = this.angle % 360;\n    if (angle > 0) {\n      return Math.round((angle - 1) / 90) * 90;\n    }\n    return Math.round(angle / 90) * 90;\n  },\n\n  /**\n   * Straightens an object (rotating it from current angle to one of 0, 90, 180, 270, etc. depending on which is closer)\n   * @return {fabric.Object} thisArg\n   * @chainable\n   */\n  straighten: function() {\n    return this.rotate(this._getAngleValueForStraighten());\n  },\n\n  /**\n   * Same as {@link fabric.Object.prototype.straighten} but with animation\n   * @param {Object} callbacks Object with callback functions\n   * @param {Function} [callbacks.onComplete] Invoked on completion\n   * @param {Function} [callbacks.onChange] Invoked on every step of animation\n   * @return {fabric.Object} thisArg\n   */\n  fxStraighten: function(callbacks) {\n    callbacks = callbacks || { };\n\n    var empty = function() { },\n        onComplete = callbacks.onComplete || empty,\n        onChange = callbacks.onChange || empty,\n        _this = this;\n\n    return fabric.util.animate({\n      target: this,\n      startValue: this.get('angle'),\n      endValue: this._getAngleValueForStraighten(),\n      duration: this.FX_DURATION,\n      onChange: function(value) {\n        _this.rotate(value);\n        onChange();\n      },\n      onComplete: function() {\n        _this.setCoords();\n        onComplete();\n      },\n    });\n  }\n});\n\nfabric.util.object.extend(fabric.StaticCanvas.prototype, /** @lends fabric.StaticCanvas.prototype */ {\n\n  /**\n   * Straightens object, then rerenders canvas\n   * @param {fabric.Object} object Object to straighten\n   * @return {fabric.Canvas} thisArg\n   * @chainable\n   */\n  straightenObject: function (object) {\n    object.straighten();\n    this.requestRenderAll();\n    return this;\n  },\n\n  /**\n   * Same as {@link fabric.Canvas.prototype.straightenObject}, but animated\n   * @param {fabric.Object} object Object to straighten\n   * @return {fabric.Canvas} thisArg\n   */\n  fxStraightenObject: function (object) {\n    return object.fxStraighten({\n      onChange: this.requestRenderAllBound\n    });\n  }\n});\n\n\n(function() {\n\n  'use strict';\n\n  /**\n   * Tests if webgl supports certain precision\n   * @param {WebGL} Canvas WebGL context to test on\n   * @param {String} Precision to test can be any of following: 'lowp', 'mediump', 'highp'\n   * @returns {Boolean} Whether the user's browser WebGL supports given precision.\n   */\n  function testPrecision(gl, precision){\n    var fragmentSource = 'precision ' + precision + ' float;\\nvoid main(){}';\n    var fragmentShader = gl.createShader(gl.FRAGMENT_SHADER);\n    gl.shaderSource(fragmentShader, fragmentSource);\n    gl.compileShader(fragmentShader);\n    if (!gl.getShaderParameter(fragmentShader, gl.COMPILE_STATUS)) {\n      return false;\n    }\n    return true;\n  }\n\n  /**\n   * Indicate whether this filtering backend is supported by the user's browser.\n   * @param {Number} tileSize check if the tileSize is supported\n   * @returns {Boolean} Whether the user's browser supports WebGL.\n   */\n  fabric.isWebglSupported = function(tileSize) {\n    if (fabric.isLikelyNode) {\n      return false;\n    }\n    tileSize = tileSize || fabric.WebglFilterBackend.prototype.tileSize;\n    var canvas = document.createElement('canvas');\n    var gl = canvas.getContext('webgl') || canvas.getContext('experimental-webgl');\n    var isSupported = false;\n    // eslint-disable-next-line\n    if (gl) {\n      fabric.maxTextureSize = gl.getParameter(gl.MAX_TEXTURE_SIZE);\n      isSupported = fabric.maxTextureSize >= tileSize;\n      var precisions = ['highp', 'mediump', 'lowp'];\n      for (var i = 0; i < 3; i++){\n        if (testPrecision(gl, precisions[i])){\n          fabric.webGlPrecision = precisions[i];\n          break;\n        };\n      }\n    }\n    this.isSupported = isSupported;\n    return isSupported;\n  };\n\n  fabric.WebglFilterBackend = WebglFilterBackend;\n\n  /**\n   * WebGL filter backend.\n   */\n  function WebglFilterBackend(options) {\n    if (options && options.tileSize) {\n      this.tileSize = options.tileSize;\n    }\n    this.setupGLContext(this.tileSize, this.tileSize);\n    this.captureGPUInfo();\n  };\n\n  WebglFilterBackend.prototype = /** @lends fabric.WebglFilterBackend.prototype */ {\n\n    tileSize: 2048,\n\n    /**\n     * Experimental. This object is a sort of repository of help layers used to avoid\n     * of recreating them during frequent filtering. If you are previewing a filter with\n     * a slider you probably do not want to create help layers every filter step.\n     * in this object there will be appended some canvases, created once, resized sometimes\n     * cleared never. Clearing is left to the developer.\n     **/\n    resources: {\n\n    },\n\n    /**\n     * Setup a WebGL context suitable for filtering, and bind any needed event handlers.\n     */\n    setupGLContext: function(width, height) {\n      this.dispose();\n      this.createWebGLCanvas(width, height);\n      // eslint-disable-next-line\n      this.aPosition = new Float32Array([0, 0, 0, 1, 1, 0, 1, 1]);\n      this.chooseFastestCopyGLTo2DMethod(width, height);\n    },\n\n    /**\n     * Pick a method to copy data from GL context to 2d canvas.  In some browsers using\n     * putImageData is faster than drawImage for that specific operation.\n     */\n    chooseFastestCopyGLTo2DMethod: function(width, height) {\n      var canMeasurePerf = typeof window.performance !== 'undefined', canUseImageData;\n      try {\n        new ImageData(1, 1);\n        canUseImageData = true;\n      }\n      catch (e) {\n        canUseImageData = false;\n      }\n      // eslint-disable-next-line no-undef\n      var canUseArrayBuffer = typeof ArrayBuffer !== 'undefined';\n      // eslint-disable-next-line no-undef\n      var canUseUint8Clamped = typeof Uint8ClampedArray !== 'undefined';\n\n      if (!(canMeasurePerf && canUseImageData && canUseArrayBuffer && canUseUint8Clamped)) {\n        return;\n      }\n\n      var targetCanvas = fabric.util.createCanvasElement();\n      // eslint-disable-next-line no-undef\n      var imageBuffer = new ArrayBuffer(width * height * 4);\n      if (fabric.forceGLPutImageData) {\n        this.imageBuffer = imageBuffer;\n        this.copyGLTo2D = copyGLTo2DPutImageData;\n        return;\n      }\n      var testContext = {\n        imageBuffer: imageBuffer,\n        destinationWidth: width,\n        destinationHeight: height,\n        targetCanvas: targetCanvas\n      };\n      var startTime, drawImageTime, putImageDataTime;\n      targetCanvas.width = width;\n      targetCanvas.height = height;\n\n      startTime = window.performance.now();\n      copyGLTo2DDrawImage.call(testContext, this.gl, testContext);\n      drawImageTime = window.performance.now() - startTime;\n\n      startTime = window.performance.now();\n      copyGLTo2DPutImageData.call(testContext, this.gl, testContext);\n      putImageDataTime = window.performance.now() - startTime;\n\n      if (drawImageTime > putImageDataTime) {\n        this.imageBuffer = imageBuffer;\n        this.copyGLTo2D = copyGLTo2DPutImageData;\n      }\n      else {\n        this.copyGLTo2D = copyGLTo2DDrawImage;\n      }\n    },\n\n    /**\n     * Create a canvas element and associated WebGL context and attaches them as\n     * class properties to the GLFilterBackend class.\n     */\n    createWebGLCanvas: function(width, height) {\n      var canvas = fabric.util.createCanvasElement();\n      canvas.width = width;\n      canvas.height = height;\n      var glOptions = {\n            alpha: true,\n            premultipliedAlpha: false,\n            depth: false,\n            stencil: false,\n            antialias: false\n          },\n          gl = canvas.getContext('webgl', glOptions);\n      if (!gl) {\n        gl = canvas.getContext('experimental-webgl', glOptions);\n      }\n      if (!gl) {\n        return;\n      }\n      gl.clearColor(0, 0, 0, 0);\n      // this canvas can fire webglcontextlost and webglcontextrestored\n      this.canvas = canvas;\n      this.gl = gl;\n    },\n\n    /**\n     * Attempts to apply the requested filters to the source provided, drawing the filtered output\n     * to the provided target canvas.\n     *\n     * @param {Array} filters The filters to apply.\n     * @param {HTMLImageElement|HTMLCanvasElement} source The source to be filtered.\n     * @param {Number} width The width of the source input.\n     * @param {Number} height The height of the source input.\n     * @param {HTMLCanvasElement} targetCanvas The destination for filtered output to be drawn.\n     * @param {String|undefined} cacheKey A key used to cache resources related to the source. If\n     * omitted, caching will be skipped.\n     */\n    applyFilters: function(filters, source, width, height, targetCanvas, cacheKey) {\n      var gl = this.gl;\n      var cachedTexture;\n      if (cacheKey) {\n        cachedTexture = this.getCachedTexture(cacheKey, source);\n      }\n      var pipelineState = {\n        originalWidth: source.width || source.originalWidth,\n        originalHeight: source.height || source.originalHeight,\n        sourceWidth: width,\n        sourceHeight: height,\n        destinationWidth: width,\n        destinationHeight: height,\n        context: gl,\n        sourceTexture: this.createTexture(gl, width, height, !cachedTexture && source),\n        targetTexture: this.createTexture(gl, width, height),\n        originalTexture: cachedTexture ||\n          this.createTexture(gl, width, height, !cachedTexture && source),\n        passes: filters.length,\n        webgl: true,\n        aPosition: this.aPosition,\n        programCache: this.programCache,\n        pass: 0,\n        filterBackend: this,\n        targetCanvas: targetCanvas\n      };\n      var tempFbo = gl.createFramebuffer();\n      gl.bindFramebuffer(gl.FRAMEBUFFER, tempFbo);\n      filters.forEach(function(filter) { filter && filter.applyTo(pipelineState); });\n      resizeCanvasIfNeeded(pipelineState);\n      this.copyGLTo2D(gl, pipelineState);\n      gl.bindTexture(gl.TEXTURE_2D, null);\n      gl.deleteTexture(pipelineState.sourceTexture);\n      gl.deleteTexture(pipelineState.targetTexture);\n      gl.deleteFramebuffer(tempFbo);\n      targetCanvas.getContext('2d').setTransform(1, 0, 0, 1, 0, 0);\n      return pipelineState;\n    },\n\n    /**\n     * Detach event listeners, remove references, and clean up caches.\n     */\n    dispose: function() {\n      if (this.canvas) {\n        this.canvas = null;\n        this.gl = null;\n      }\n      this.clearWebGLCaches();\n    },\n\n    /**\n     * Wipe out WebGL-related caches.\n     */\n    clearWebGLCaches: function() {\n      this.programCache = {};\n      this.textureCache = {};\n    },\n\n    /**\n     * Create a WebGL texture object.\n     *\n     * Accepts specific dimensions to initialize the texture to or a source image.\n     *\n     * @param {WebGLRenderingContext} gl The GL context to use for creating the texture.\n     * @param {Number} width The width to initialize the texture at.\n     * @param {Number} height The height to initialize the texture.\n     * @param {HTMLImageElement|HTMLCanvasElement} textureImageSource A source for the texture data.\n     * @returns {WebGLTexture}\n     */\n    createTexture: function(gl, width, height, textureImageSource) {\n      var texture = gl.createTexture();\n      gl.bindTexture(gl.TEXTURE_2D, texture);\n      gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MAG_FILTER, gl.NEAREST);\n      gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_MIN_FILTER, gl.NEAREST);\n      gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_S, gl.CLAMP_TO_EDGE);\n      gl.texParameteri(gl.TEXTURE_2D, gl.TEXTURE_WRAP_T, gl.CLAMP_TO_EDGE);\n      if (textureImageSource) {\n        gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, textureImageSource);\n      }\n      else {\n        gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, width, height, 0, gl.RGBA, gl.UNSIGNED_BYTE, null);\n      }\n      return texture;\n    },\n\n    /**\n     * Can be optionally used to get a texture from the cache array\n     *\n     * If an existing texture is not found, a new texture is created and cached.\n     *\n     * @param {String} uniqueId A cache key to use to find an existing texture.\n     * @param {HTMLImageElement|HTMLCanvasElement} textureImageSource A source to use to create the\n     * texture cache entry if one does not already exist.\n     */\n    getCachedTexture: function(uniqueId, textureImageSource) {\n      if (this.textureCache[uniqueId]) {\n        return this.textureCache[uniqueId];\n      }\n      else {\n        var texture = this.createTexture(\n          this.gl, textureImageSource.width, textureImageSource.height, textureImageSource);\n        this.textureCache[uniqueId] = texture;\n        return texture;\n      }\n    },\n\n    /**\n     * Clear out cached resources related to a source image that has been\n     * filtered previously.\n     *\n     * @param {String} cacheKey The cache key provided when the source image was filtered.\n     */\n    evictCachesForKey: function(cacheKey) {\n      if (this.textureCache[cacheKey]) {\n        this.gl.deleteTexture(this.textureCache[cacheKey]);\n        delete this.textureCache[cacheKey];\n      }\n    },\n\n    copyGLTo2D: copyGLTo2DDrawImage,\n\n    /**\n     * Attempt to extract GPU information strings from a WebGL context.\n     *\n     * Useful information when debugging or blacklisting specific GPUs.\n     *\n     * @returns {Object} A GPU info object with renderer and vendor strings.\n     */\n    captureGPUInfo: function() {\n      if (this.gpuInfo) {\n        return this.gpuInfo;\n      }\n      var gl = this.gl, gpuInfo = { renderer: '', vendor: '' };\n      if (!gl) {\n        return gpuInfo;\n      }\n      var ext = gl.getExtension('WEBGL_debug_renderer_info');\n      if (ext) {\n        var renderer = gl.getParameter(ext.UNMASKED_RENDERER_WEBGL);\n        var vendor = gl.getParameter(ext.UNMASKED_VENDOR_WEBGL);\n        if (renderer) {\n          gpuInfo.renderer = renderer.toLowerCase();\n        }\n        if (vendor) {\n          gpuInfo.vendor = vendor.toLowerCase();\n        }\n      }\n      this.gpuInfo = gpuInfo;\n      return gpuInfo;\n    },\n  };\n})();\n\nfunction resizeCanvasIfNeeded(pipelineState) {\n  var targetCanvas = pipelineState.targetCanvas,\n      width = targetCanvas.width, height = targetCanvas.height,\n      dWidth = pipelineState.destinationWidth,\n      dHeight = pipelineState.destinationHeight;\n\n  if (width !== dWidth || height !== dHeight) {\n    targetCanvas.width = dWidth;\n    targetCanvas.height = dHeight;\n  }\n}\n\n/**\n * Copy an input WebGL canvas on to an output 2D canvas.\n *\n * The WebGL canvas is assumed to be upside down, with the top-left pixel of the\n * desired output image appearing in the bottom-left corner of the WebGL canvas.\n *\n * @param {WebGLRenderingContext} sourceContext The WebGL context to copy from.\n * @param {HTMLCanvasElement} targetCanvas The 2D target canvas to copy on to.\n * @param {Object} pipelineState The 2D target canvas to copy on to.\n */\nfunction copyGLTo2DDrawImage(gl, pipelineState) {\n  var glCanvas = gl.canvas, targetCanvas = pipelineState.targetCanvas,\n      ctx = targetCanvas.getContext('2d');\n  ctx.translate(0, targetCanvas.height); // move it down again\n  ctx.scale(1, -1); // vertical flip\n  // where is my image on the big glcanvas?\n  var sourceY = glCanvas.height - targetCanvas.height;\n  ctx.drawImage(glCanvas, 0, sourceY, targetCanvas.width, targetCanvas.height, 0, 0,\n    targetCanvas.width, targetCanvas.height);\n}\n\n/**\n * Copy an input WebGL canvas on to an output 2D canvas using 2d canvas' putImageData\n * API. Measurably faster than using ctx.drawImage in Firefox (version 54 on OSX Sierra).\n *\n * @param {WebGLRenderingContext} sourceContext The WebGL context to copy from.\n * @param {HTMLCanvasElement} targetCanvas The 2D target canvas to copy on to.\n * @param {Object} pipelineState The 2D target canvas to copy on to.\n */\nfunction copyGLTo2DPutImageData(gl, pipelineState) {\n  var targetCanvas = pipelineState.targetCanvas, ctx = targetCanvas.getContext('2d'),\n      dWidth = pipelineState.destinationWidth,\n      dHeight = pipelineState.destinationHeight,\n      numBytes = dWidth * dHeight * 4;\n\n  // eslint-disable-next-line no-undef\n  var u8 = new Uint8Array(this.imageBuffer, 0, numBytes);\n  // eslint-disable-next-line no-undef\n  var u8Clamped = new Uint8ClampedArray(this.imageBuffer, 0, numBytes);\n\n  gl.readPixels(0, 0, dWidth, dHeight, gl.RGBA, gl.UNSIGNED_BYTE, u8);\n  var imgData = new ImageData(u8Clamped, dWidth, dHeight);\n  ctx.putImageData(imgData, 0, 0);\n}\n\n\n(function() {\n\n  'use strict';\n\n  var noop = function() {};\n\n  fabric.Canvas2dFilterBackend = Canvas2dFilterBackend;\n\n  /**\n   * Canvas 2D filter backend.\n   */\n  function Canvas2dFilterBackend() {};\n\n  Canvas2dFilterBackend.prototype = /** @lends fabric.Canvas2dFilterBackend.prototype */ {\n    evictCachesForKey: noop,\n    dispose: noop,\n    clearWebGLCaches: noop,\n\n    /**\n     * Experimental. This object is a sort of repository of help layers used to avoid\n     * of recreating them during frequent filtering. If you are previewing a filter with\n     * a slider you probably do not want to create help layers every filter step.\n     * in this object there will be appended some canvases, created once, resized sometimes\n     * cleared never. Clearing is left to the developer.\n     **/\n    resources: {\n\n    },\n\n    /**\n     * Apply a set of filters against a source image and draw the filtered output\n     * to the provided destination canvas.\n     *\n     * @param {EnhancedFilter} filters The filter to apply.\n     * @param {HTMLImageElement|HTMLCanvasElement} sourceElement The source to be filtered.\n     * @param {Number} sourceWidth The width of the source input.\n     * @param {Number} sourceHeight The height of the source input.\n     * @param {HTMLCanvasElement} targetCanvas The destination for filtered output to be drawn.\n     */\n    applyFilters: function(filters, sourceElement, sourceWidth, sourceHeight, targetCanvas) {\n      var ctx = targetCanvas.getContext('2d');\n      ctx.drawImage(sourceElement, 0, 0, sourceWidth, sourceHeight);\n      var imageData = ctx.getImageData(0, 0, sourceWidth, sourceHeight);\n      var originalImageData = ctx.getImageData(0, 0, sourceWidth, sourceHeight);\n      var pipelineState = {\n        sourceWidth: sourceWidth,\n        sourceHeight: sourceHeight,\n        imageData: imageData,\n        originalEl: sourceElement,\n        originalImageData: originalImageData,\n        canvasEl: targetCanvas,\n        ctx: ctx,\n        filterBackend: this,\n      };\n      filters.forEach(function(filter) { filter.applyTo(pipelineState); });\n      if (pipelineState.imageData.width !== sourceWidth || pipelineState.imageData.height !== sourceHeight) {\n        targetCanvas.width = pipelineState.imageData.width;\n        targetCanvas.height = pipelineState.imageData.height;\n      }\n      ctx.putImageData(pipelineState.imageData, 0, 0);\n      return pipelineState;\n    },\n\n  };\n})();\n\n\n/**\n * @namespace fabric.Image.filters\n * @memberOf fabric.Image\n * @tutorial {@link http://fabricjs.com/fabric-intro-part-2#image_filters}\n * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n */\nfabric.Image = fabric.Image || { };\nfabric.Image.filters = fabric.Image.filters || { };\n\n/**\n * Root filter class from which all filter classes inherit from\n * @class fabric.Image.filters.BaseFilter\n * @memberOf fabric.Image.filters\n */\nfabric.Image.filters.BaseFilter = fabric.util.createClass(/** @lends fabric.Image.filters.BaseFilter.prototype */ {\n\n  /**\n   * Filter type\n   * @param {String} type\n   * @default\n   */\n  type: 'BaseFilter',\n\n  /**\n   * Array of attributes to send with buffers. do not modify\n   * @private\n   */\n\n  vertexSource: 'attribute vec2 aPosition;\\n' +\n    'varying vec2 vTexCoord;\\n' +\n    'void main() {\\n' +\n      'vTexCoord = aPosition;\\n' +\n      'gl_Position = vec4(aPosition * 2.0 - 1.0, 0.0, 1.0);\\n' +\n    '}',\n\n  fragmentSource: 'precision highp float;\\n' +\n    'varying vec2 vTexCoord;\\n' +\n    'uniform sampler2D uTexture;\\n' +\n    'void main() {\\n' +\n      'gl_FragColor = texture2D(uTexture, vTexCoord);\\n' +\n    '}',\n\n  /**\n   * Constructor\n   * @param {Object} [options] Options object\n   */\n  initialize: function(options) {\n    if (options) {\n      this.setOptions(options);\n    }\n  },\n\n  /**\n   * Sets filter's properties from options\n   * @param {Object} [options] Options object\n   */\n  setOptions: function(options) {\n    for (var prop in options) {\n      this[prop] = options[prop];\n    }\n  },\n\n  /**\n   * Compile this filter's shader program.\n   *\n   * @param {WebGLRenderingContext} gl The GL canvas context to use for shader compilation.\n   * @param {String} fragmentSource fragmentShader source for compilation\n   * @param {String} vertexSource vertexShader source for compilation\n   */\n  createProgram: function(gl, fragmentSource, vertexSource) {\n    fragmentSource = fragmentSource || this.fragmentSource;\n    vertexSource = vertexSource || this.vertexSource;\n    if (fabric.webGlPrecision !== 'highp'){\n      fragmentSource = fragmentSource.replace(\n        /precision highp float/g,\n        'precision ' + fabric.webGlPrecision + ' float'\n      );\n    }\n    var vertexShader = gl.createShader(gl.VERTEX_SHADER);\n    gl.shaderSource(vertexShader, vertexSource);\n    gl.compileShader(vertexShader);\n    if (!gl.getShaderParameter(vertexShader, gl.COMPILE_STATUS)) {\n      throw new Error(\n        // eslint-disable-next-line prefer-template\n        'Vertex shader compile error for ' + this.type + ': ' +\n        gl.getShaderInfoLog(vertexShader)\n      );\n    }\n\n    var fragmentShader = gl.createShader(gl.FRAGMENT_SHADER);\n    gl.shaderSource(fragmentShader, fragmentSource);\n    gl.compileShader(fragmentShader);\n    if (!gl.getShaderParameter(fragmentShader, gl.COMPILE_STATUS)) {\n      throw new Error(\n        // eslint-disable-next-line prefer-template\n        'Fragment shader compile error for ' + this.type + ': ' +\n        gl.getShaderInfoLog(fragmentShader)\n      );\n    }\n\n    var program = gl.createProgram();\n    gl.attachShader(program, vertexShader);\n    gl.attachShader(program, fragmentShader);\n    gl.linkProgram(program);\n    if (!gl.getProgramParameter(program, gl.LINK_STATUS)) {\n      throw new Error(\n        // eslint-disable-next-line prefer-template\n        'Shader link error for \"${this.type}\" ' +\n        gl.getProgramInfoLog(program)\n      );\n    }\n\n    var attributeLocations = this.getAttributeLocations(gl, program);\n    var uniformLocations = this.getUniformLocations(gl, program) || { };\n    uniformLocations.uStepW = gl.getUniformLocation(program, 'uStepW');\n    uniformLocations.uStepH = gl.getUniformLocation(program, 'uStepH');\n    return {\n      program: program,\n      attributeLocations: attributeLocations,\n      uniformLocations: uniformLocations\n    };\n  },\n\n  /**\n   * Return a map of attribute names to WebGLAttributeLocation objects.\n   *\n   * @param {WebGLRenderingContext} gl The canvas context used to compile the shader program.\n   * @param {WebGLShaderProgram} program The shader program from which to take attribute locations.\n   * @returns {Object} A map of attribute names to attribute locations.\n   */\n  getAttributeLocations: function(gl, program) {\n    return {\n      aPosition: gl.getAttribLocation(program, 'aPosition'),\n    };\n  },\n\n  /**\n   * Return a map of uniform names to WebGLUniformLocation objects.\n   *\n   * Intended to be overridden by subclasses.\n   *\n   * @param {WebGLRenderingContext} gl The canvas context used to compile the shader program.\n   * @param {WebGLShaderProgram} program The shader program from which to take uniform locations.\n   * @returns {Object} A map of uniform names to uniform locations.\n   */\n  getUniformLocations: function (/* gl, program */) {\n    // in case i do not need any special uniform i need to return an empty object\n    return { };\n  },\n\n  /**\n   * Send attribute data from this filter to its shader program on the GPU.\n   *\n   * @param {WebGLRenderingContext} gl The canvas context used to compile the shader program.\n   * @param {Object} attributeLocations A map of shader attribute names to their locations.\n   */\n  sendAttributeData: function(gl, attributeLocations, aPositionData) {\n    var attributeLocation = attributeLocations.aPosition;\n    var buffer = gl.createBuffer();\n    gl.bindBuffer(gl.ARRAY_BUFFER, buffer);\n    gl.enableVertexAttribArray(attributeLocation);\n    gl.vertexAttribPointer(attributeLocation, 2, gl.FLOAT, false, 0, 0);\n    gl.bufferData(gl.ARRAY_BUFFER, aPositionData, gl.STATIC_DRAW);\n  },\n\n  _setupFrameBuffer: function(options) {\n    var gl = options.context, width, height;\n    if (options.passes > 1) {\n      width = options.destinationWidth;\n      height = options.destinationHeight;\n      if (options.sourceWidth !== width || options.sourceHeight !== height) {\n        gl.deleteTexture(options.targetTexture);\n        options.targetTexture = options.filterBackend.createTexture(gl, width, height);\n      }\n      gl.framebufferTexture2D(gl.FRAMEBUFFER, gl.COLOR_ATTACHMENT0, gl.TEXTURE_2D,\n        options.targetTexture, 0);\n    }\n    else {\n      // draw last filter on canvas and not to framebuffer.\n      gl.bindFramebuffer(gl.FRAMEBUFFER, null);\n      gl.finish();\n    }\n  },\n\n  _swapTextures: function(options) {\n    options.passes--;\n    options.pass++;\n    var temp = options.targetTexture;\n    options.targetTexture = options.sourceTexture;\n    options.sourceTexture = temp;\n  },\n\n  /**\n   * Generic isNeutral implementation for one parameter based filters.\n   * Used only in image applyFilters to discard filters that will not have an effect\n   * on the image\n   * Other filters may need their own version ( ColorMatrix, HueRotation, gamma, ComposedFilter )\n   * @param {Object} options\n   **/\n  isNeutralState: function(/* options */) {\n    var main = this.mainParameter,\n        _class = fabric.Image.filters[this.type].prototype;\n    if (main) {\n      if (Array.isArray(_class[main])) {\n        for (var i = _class[main].length; i--;) {\n          if (this[main][i] !== _class[main][i]) {\n            return false;\n          }\n        }\n        return true;\n      }\n      else {\n        return _class[main] === this[main];\n      }\n    }\n    else {\n      return false;\n    }\n  },\n\n  /**\n   * Apply this filter to the input image data provided.\n   *\n   * Determines whether to use WebGL or Canvas2D based on the options.webgl flag.\n   *\n   * @param {Object} options\n   * @param {Number} options.passes The number of filters remaining to be executed\n   * @param {Boolean} options.webgl Whether to use webgl to render the filter.\n   * @param {WebGLTexture} options.sourceTexture The texture setup as the source to be filtered.\n   * @param {WebGLTexture} options.targetTexture The texture where filtered output should be drawn.\n   * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n   * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n   */\n  applyTo: function(options) {\n    if (options.webgl) {\n      this._setupFrameBuffer(options);\n      this.applyToWebGL(options);\n      this._swapTextures(options);\n    }\n    else {\n      this.applyTo2d(options);\n    }\n  },\n\n  /**\n   * Retrieves the cached shader.\n   * @param {Object} options\n   * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n   * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n   */\n  retrieveShader: function(options) {\n    if (!options.programCache.hasOwnProperty(this.type)) {\n      options.programCache[this.type] = this.createProgram(options.context);\n    }\n    return options.programCache[this.type];\n  },\n\n  /**\n   * Apply this filter using webgl.\n   *\n   * @param {Object} options\n   * @param {Number} options.passes The number of filters remaining to be executed\n   * @param {Boolean} options.webgl Whether to use webgl to render the filter.\n   * @param {WebGLTexture} options.originalTexture The texture of the original input image.\n   * @param {WebGLTexture} options.sourceTexture The texture setup as the source to be filtered.\n   * @param {WebGLTexture} options.targetTexture The texture where filtered output should be drawn.\n   * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n   * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n   */\n  applyToWebGL: function(options) {\n    var gl = options.context;\n    var shader = this.retrieveShader(options);\n    if (options.pass === 0 && options.originalTexture) {\n      gl.bindTexture(gl.TEXTURE_2D, options.originalTexture);\n    }\n    else {\n      gl.bindTexture(gl.TEXTURE_2D, options.sourceTexture);\n    }\n    gl.useProgram(shader.program);\n    this.sendAttributeData(gl, shader.attributeLocations, options.aPosition);\n\n    gl.uniform1f(shader.uniformLocations.uStepW, 1 / options.sourceWidth);\n    gl.uniform1f(shader.uniformLocations.uStepH, 1 / options.sourceHeight);\n\n    this.sendUniformData(gl, shader.uniformLocations);\n    gl.viewport(0, 0, options.destinationWidth, options.destinationHeight);\n    gl.drawArrays(gl.TRIANGLE_STRIP, 0, 4);\n  },\n\n  bindAdditionalTexture: function(gl, texture, textureUnit) {\n    gl.activeTexture(textureUnit);\n    gl.bindTexture(gl.TEXTURE_2D, texture);\n    // reset active texture to 0 as usual\n    gl.activeTexture(gl.TEXTURE0);\n  },\n\n  unbindAdditionalTexture: function(gl, textureUnit) {\n    gl.activeTexture(textureUnit);\n    gl.bindTexture(gl.TEXTURE_2D, null);\n    gl.activeTexture(gl.TEXTURE0);\n  },\n\n  getMainParameter: function() {\n    return this[this.mainParameter];\n  },\n\n  setMainParameter: function(value) {\n    this[this.mainParameter] = value;\n  },\n\n  /**\n   * Send uniform data from this filter to its shader program on the GPU.\n   *\n   * Intended to be overridden by subclasses.\n   *\n   * @param {WebGLRenderingContext} gl The canvas context used to compile the shader program.\n   * @param {Object} uniformLocations A map of shader uniform names to their locations.\n   */\n  sendUniformData: function(/* gl, uniformLocations */) {\n    // Intentionally left blank.  Override me in subclasses.\n  },\n\n  /**\n   * If needed by a 2d filter, this functions can create an helper canvas to be used\n   * remember that options.targetCanvas is available for use till end of chain.\n   */\n  createHelpLayer: function(options) {\n    if (!options.helpLayer) {\n      var helpLayer = document.createElement('canvas');\n      helpLayer.width = options.sourceWidth;\n      helpLayer.height = options.sourceHeight;\n      options.helpLayer = helpLayer;\n    }\n  },\n\n  /**\n   * Returns object representation of an instance\n   * @return {Object} Object representation of an instance\n   */\n  toObject: function() {\n    var object = { type: this.type }, mainP = this.mainParameter;\n    if (mainP) {\n      object[mainP] = this[mainP];\n    }\n    return object;\n  },\n\n  /**\n   * Returns a JSON representation of an instance\n   * @return {Object} JSON\n   */\n  toJSON: function() {\n    // delegate, not alias\n    return this.toObject();\n  }\n});\n\nfabric.Image.filters.BaseFilter.fromObject = function(object, callback) {\n  var filter = new fabric.Image.filters[object.type](object);\n  callback && callback(filter);\n  return filter;\n};\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Color Matrix filter class\n   * @class fabric.Image.filters.ColorMatrix\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.ColorMatrix#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @see {@Link http://www.webwasp.co.uk/tutorials/219/Color_Matrix_Filter.php}\n   * @see {@Link http://phoboslab.org/log/2013/11/fast-image-filters-with-webgl}\n   * @example <caption>Kodachrome filter</caption>\n   * var filter = new fabric.Image.filters.ColorMatrix({\n   *  matrix: [\n       1.1285582396593525, -0.3967382283601348, -0.03992559172921793, 0, 63.72958762196502,\n       -0.16404339962244616, 1.0835251566291304, -0.05498805115633132, 0, 24.732407896706203,\n       -0.16786010706155763, -0.5603416277695248, 1.6014850761964943, 0, 35.62982807460946,\n       0, 0, 0, 1, 0\n      ]\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   */\n  filters.ColorMatrix = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.ColorMatrix.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'ColorMatrix',\n\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'uniform mat4 uColorMatrix;\\n' +\n      'uniform vec4 uConstants;\\n' +\n      'void main() {\\n' +\n        'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n        'color *= uColorMatrix;\\n' +\n        'color += uConstants;\\n' +\n        'gl_FragColor = color;\\n' +\n      '}',\n\n    /**\n     * Colormatrix for pixels.\n     * array of 20 floats. Numbers in positions 4, 9, 14, 19 loose meaning\n     * outside the -1, 1 range.\n     * 0.0039215686 is the part of 1 that get translated to 1 in 2d\n     * @param {Array} matrix array of 20 numbers.\n     * @default\n     */\n    matrix: [\n      1, 0, 0, 0, 0,\n      0, 1, 0, 0, 0,\n      0, 0, 1, 0, 0,\n      0, 0, 0, 1, 0\n    ],\n\n    mainParameter: 'matrix',\n\n    /**\n     * Lock the colormatrix on the color part, skipping alpha, mainly for non webgl scenario\n     * to save some calculation\n     * @type Boolean\n     * @default true\n     */\n    colorsOnly: true,\n\n    /**\n     * Constructor\n     * @param {Object} [options] Options object\n     */\n    initialize: function(options) {\n      this.callSuper('initialize', options);\n      // create a new array instead mutating the prototype with push\n      this.matrix = this.matrix.slice(0);\n    },\n\n    /**\n     * Apply the ColorMatrix operation to a Uint8Array representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8Array to be filtered.\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData,\n          data = imageData.data,\n          iLen = data.length,\n          m = this.matrix,\n          r, g, b, a, i, colorsOnly = this.colorsOnly;\n\n      for (i = 0; i < iLen; i += 4) {\n        r = data[i];\n        g = data[i + 1];\n        b = data[i + 2];\n        if (colorsOnly) {\n          data[i] = r * m[0] + g * m[1] + b * m[2] + m[4] * 255;\n          data[i + 1] = r * m[5] + g * m[6] + b * m[7] + m[9] * 255;\n          data[i + 2] = r * m[10] + g * m[11] + b * m[12] + m[14] * 255;\n        }\n        else {\n          a = data[i + 3];\n          data[i] = r * m[0] + g * m[1] + b * m[2] + a * m[3] + m[4] * 255;\n          data[i + 1] = r * m[5] + g * m[6] + b * m[7] + a * m[8] + m[9] * 255;\n          data[i + 2] = r * m[10] + g * m[11] + b * m[12] + a * m[13] + m[14] * 255;\n          data[i + 3] = r * m[15] + g * m[16] + b * m[17] + a * m[18] + m[19] * 255;\n        }\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uColorMatrix: gl.getUniformLocation(program, 'uColorMatrix'),\n        uConstants: gl.getUniformLocation(program, 'uConstants'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      var m = this.matrix,\n          matrix = [\n            m[0], m[1], m[2], m[3],\n            m[5], m[6], m[7], m[8],\n            m[10], m[11], m[12], m[13],\n            m[15], m[16], m[17], m[18]\n          ],\n          constants = [m[4], m[9], m[14], m[19]];\n      gl.uniformMatrix4fv(uniformLocations.uColorMatrix, false, matrix);\n      gl.uniform4fv(uniformLocations.uConstants, constants);\n    },\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] function to invoke after filter creation\n   * @return {fabric.Image.filters.ColorMatrix} Instance of fabric.Image.filters.ColorMatrix\n   */\n  fabric.Image.filters.ColorMatrix.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Brightness filter class\n   * @class fabric.Image.filters.Brightness\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.Brightness#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Brightness({\n   *   brightness: 0.05\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   */\n  filters.Brightness = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Brightness.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Brightness',\n\n    /**\n     * Fragment source for the brightness program\n     */\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform float uBrightness;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'void main() {\\n' +\n        'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n        'color.rgb += uBrightness;\\n' +\n        'gl_FragColor = color;\\n' +\n      '}',\n\n    /**\n     * Brightness value, from -1 to 1.\n     * translated to -255 to 255 for 2d\n     * 0.0039215686 is the part of 1 that get translated to 1 in 2d\n     * @param {Number} brightness\n     * @default\n     */\n    brightness: 0,\n\n    /**\n     * Describe the property that is the filter parameter\n     * @param {String} m\n     * @default\n     */\n    mainParameter: 'brightness',\n\n    /**\n    * Apply the Brightness operation to a Uint8ClampedArray representing the pixels of an image.\n    *\n    * @param {Object} options\n    * @param {ImageData} options.imageData The Uint8ClampedArray to be filtered.\n    */\n    applyTo2d: function(options) {\n      if (this.brightness === 0) {\n        return;\n      }\n      var imageData = options.imageData,\n          data = imageData.data, i, len = data.length,\n          brightness = Math.round(this.brightness * 255);\n      for (i = 0; i < len; i += 4) {\n        data[i] = data[i] + brightness;\n        data[i + 1] = data[i + 1] + brightness;\n        data[i + 2] = data[i + 2] + brightness;\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uBrightness: gl.getUniformLocation(program, 'uBrightness'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform1f(uniformLocations.uBrightness, this.brightness);\n    },\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Brightness} Instance of fabric.Image.filters.Brightness\n   */\n  fabric.Image.filters.Brightness.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      extend = fabric.util.object.extend,\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Adapted from <a href=\"http://www.html5rocks.com/en/tutorials/canvas/imagefilters/\">html5rocks article</a>\n   * @class fabric.Image.filters.Convolute\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.Convolute#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example <caption>Sharpen filter</caption>\n   * var filter = new fabric.Image.filters.Convolute({\n   *   matrix: [ 0, -1,  0,\n   *            -1,  5, -1,\n   *             0, -1,  0 ]\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   * canvas.renderAll();\n   * @example <caption>Blur filter</caption>\n   * var filter = new fabric.Image.filters.Convolute({\n   *   matrix: [ 1/9, 1/9, 1/9,\n   *             1/9, 1/9, 1/9,\n   *             1/9, 1/9, 1/9 ]\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   * canvas.renderAll();\n   * @example <caption>Emboss filter</caption>\n   * var filter = new fabric.Image.filters.Convolute({\n   *   matrix: [ 1,   1,  1,\n   *             1, 0.7, -1,\n   *            -1,  -1, -1 ]\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   * canvas.renderAll();\n   * @example <caption>Emboss filter with opaqueness</caption>\n   * var filter = new fabric.Image.filters.Convolute({\n   *   opaque: true,\n   *   matrix: [ 1,   1,  1,\n   *             1, 0.7, -1,\n   *            -1,  -1, -1 ]\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   * canvas.renderAll();\n   */\n  filters.Convolute = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Convolute.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Convolute',\n\n    /*\n     * Opaque value (true/false)\n     */\n    opaque: false,\n\n    /*\n     * matrix for the filter, max 9x9\n     */\n    matrix: [0, 0, 0, 0, 1, 0, 0, 0, 0],\n\n    /**\n     * Fragment source for the brightness program\n     */\n    fragmentSource: {\n      Convolute_3_1: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform float uMatrix[9];\\n' +\n        'uniform float uStepW;\\n' +\n        'uniform float uStepH;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = vec4(0, 0, 0, 0);\\n' +\n          'for (float h = 0.0; h < 3.0; h+=1.0) {\\n' +\n            'for (float w = 0.0; w < 3.0; w+=1.0) {\\n' +\n              'vec2 matrixPos = vec2(uStepW * (w - 1), uStepH * (h - 1));\\n' +\n              'color += texture2D(uTexture, vTexCoord + matrixPos) * uMatrix[int(h * 3.0 + w)];\\n' +\n            '}\\n' +\n          '}\\n' +\n          'gl_FragColor = color;\\n' +\n        '}',\n      Convolute_3_0: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform float uMatrix[9];\\n' +\n        'uniform float uStepW;\\n' +\n        'uniform float uStepH;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = vec4(0, 0, 0, 1);\\n' +\n          'for (float h = 0.0; h < 3.0; h+=1.0) {\\n' +\n            'for (float w = 0.0; w < 3.0; w+=1.0) {\\n' +\n              'vec2 matrixPos = vec2(uStepW * (w - 1.0), uStepH * (h - 1.0));\\n' +\n              'color.rgb += texture2D(uTexture, vTexCoord + matrixPos).rgb * uMatrix[int(h * 3.0 + w)];\\n' +\n            '}\\n' +\n          '}\\n' +\n          'float alpha = texture2D(uTexture, vTexCoord).a;\\n' +\n          'gl_FragColor = color;\\n' +\n          'gl_FragColor.a = alpha;\\n' +\n        '}',\n      Convolute_5_1: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform float uMatrix[25];\\n' +\n        'uniform float uStepW;\\n' +\n        'uniform float uStepH;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = vec4(0, 0, 0, 0);\\n' +\n          'for (float h = 0.0; h < 5.0; h+=1.0) {\\n' +\n            'for (float w = 0.0; w < 5.0; w+=1.0) {\\n' +\n              'vec2 matrixPos = vec2(uStepW * (w - 2.0), uStepH * (h - 2.0));\\n' +\n              'color += texture2D(uTexture, vTexCoord + matrixPos) * uMatrix[int(h * 5.0 + w)];\\n' +\n            '}\\n' +\n          '}\\n' +\n          'gl_FragColor = color;\\n' +\n        '}',\n      Convolute_5_0: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform float uMatrix[25];\\n' +\n        'uniform float uStepW;\\n' +\n        'uniform float uStepH;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = vec4(0, 0, 0, 1);\\n' +\n          'for (float h = 0.0; h < 5.0; h+=1.0) {\\n' +\n            'for (float w = 0.0; w < 5.0; w+=1.0) {\\n' +\n              'vec2 matrixPos = vec2(uStepW * (w - 2.0), uStepH * (h - 2.0));\\n' +\n              'color.rgb += texture2D(uTexture, vTexCoord + matrixPos).rgb * uMatrix[int(h * 5.0 + w)];\\n' +\n            '}\\n' +\n          '}\\n' +\n          'float alpha = texture2D(uTexture, vTexCoord).a;\\n' +\n          'gl_FragColor = color;\\n' +\n          'gl_FragColor.a = alpha;\\n' +\n        '}',\n      Convolute_7_1: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform float uMatrix[49];\\n' +\n        'uniform float uStepW;\\n' +\n        'uniform float uStepH;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = vec4(0, 0, 0, 0);\\n' +\n          'for (float h = 0.0; h < 7.0; h+=1.0) {\\n' +\n            'for (float w = 0.0; w < 7.0; w+=1.0) {\\n' +\n              'vec2 matrixPos = vec2(uStepW * (w - 3.0), uStepH * (h - 3.0));\\n' +\n              'color += texture2D(uTexture, vTexCoord + matrixPos) * uMatrix[int(h * 7.0 + w)];\\n' +\n            '}\\n' +\n          '}\\n' +\n          'gl_FragColor = color;\\n' +\n        '}',\n      Convolute_7_0: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform float uMatrix[49];\\n' +\n        'uniform float uStepW;\\n' +\n        'uniform float uStepH;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = vec4(0, 0, 0, 1);\\n' +\n          'for (float h = 0.0; h < 7.0; h+=1.0) {\\n' +\n            'for (float w = 0.0; w < 7.0; w+=1.0) {\\n' +\n              'vec2 matrixPos = vec2(uStepW * (w - 3.0), uStepH * (h - 3.0));\\n' +\n              'color.rgb += texture2D(uTexture, vTexCoord + matrixPos).rgb * uMatrix[int(h * 7.0 + w)];\\n' +\n            '}\\n' +\n          '}\\n' +\n          'float alpha = texture2D(uTexture, vTexCoord).a;\\n' +\n          'gl_FragColor = color;\\n' +\n          'gl_FragColor.a = alpha;\\n' +\n        '}',\n      Convolute_9_1: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform float uMatrix[81];\\n' +\n        'uniform float uStepW;\\n' +\n        'uniform float uStepH;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = vec4(0, 0, 0, 0);\\n' +\n          'for (float h = 0.0; h < 9.0; h+=1.0) {\\n' +\n            'for (float w = 0.0; w < 9.0; w+=1.0) {\\n' +\n              'vec2 matrixPos = vec2(uStepW * (w - 4.0), uStepH * (h - 4.0));\\n' +\n              'color += texture2D(uTexture, vTexCoord + matrixPos) * uMatrix[int(h * 9.0 + w)];\\n' +\n            '}\\n' +\n          '}\\n' +\n          'gl_FragColor = color;\\n' +\n        '}',\n      Convolute_9_0: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform float uMatrix[81];\\n' +\n        'uniform float uStepW;\\n' +\n        'uniform float uStepH;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = vec4(0, 0, 0, 1);\\n' +\n          'for (float h = 0.0; h < 9.0; h+=1.0) {\\n' +\n            'for (float w = 0.0; w < 9.0; w+=1.0) {\\n' +\n              'vec2 matrixPos = vec2(uStepW * (w - 4.0), uStepH * (h - 4.0));\\n' +\n              'color.rgb += texture2D(uTexture, vTexCoord + matrixPos).rgb * uMatrix[int(h * 9.0 + w)];\\n' +\n            '}\\n' +\n          '}\\n' +\n          'float alpha = texture2D(uTexture, vTexCoord).a;\\n' +\n          'gl_FragColor = color;\\n' +\n          'gl_FragColor.a = alpha;\\n' +\n        '}',\n    },\n\n    /**\n     * Constructor\n     * @memberOf fabric.Image.filters.Convolute.prototype\n     * @param {Object} [options] Options object\n     * @param {Boolean} [options.opaque=false] Opaque value (true/false)\n     * @param {Array} [options.matrix] Filter matrix\n     */\n\n\n    /**\n    * Retrieves the cached shader.\n    * @param {Object} options\n    * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n    * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n    */\n    retrieveShader: function(options) {\n      var size = Math.sqrt(this.matrix.length);\n      var cacheKey = this.type + '_' + size + '_' + (this.opaque ? 1 : 0);\n      var shaderSource = this.fragmentSource[cacheKey];\n      if (!options.programCache.hasOwnProperty(cacheKey)) {\n        options.programCache[cacheKey] = this.createProgram(options.context, shaderSource);\n      }\n      return options.programCache[cacheKey];\n    },\n\n    /**\n     * Apply the Brightness operation to a Uint8ClampedArray representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8ClampedArray to be filtered.\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData,\n          data = imageData.data,\n          weights = this.matrix,\n          side = Math.round(Math.sqrt(weights.length)),\n          halfSide = Math.floor(side / 2),\n          sw = imageData.width,\n          sh = imageData.height,\n          output = options.ctx.createImageData(sw, sh),\n          dst = output.data,\n          // go through the destination image pixels\n          alphaFac = this.opaque ? 1 : 0,\n          r, g, b, a, dstOff,\n          scx, scy, srcOff, wt,\n          x, y, cx, cy;\n\n      for (y = 0; y < sh; y++) {\n        for (x = 0; x < sw; x++) {\n          dstOff = (y * sw + x) * 4;\n          // calculate the weighed sum of the source image pixels that\n          // fall under the convolution matrix\n          r = 0; g = 0; b = 0; a = 0;\n\n          for (cy = 0; cy < side; cy++) {\n            for (cx = 0; cx < side; cx++) {\n              scy = y + cy - halfSide;\n              scx = x + cx - halfSide;\n\n              // eslint-disable-next-line max-depth\n              if (scy < 0 || scy >= sh || scx < 0 || scx >= sw) {\n                continue;\n              }\n\n              srcOff = (scy * sw + scx) * 4;\n              wt = weights[cy * side + cx];\n\n              r += data[srcOff] * wt;\n              g += data[srcOff + 1] * wt;\n              b += data[srcOff + 2] * wt;\n              // eslint-disable-next-line max-depth\n              if (!alphaFac) {\n                a += data[srcOff + 3] * wt;\n              }\n            }\n          }\n          dst[dstOff] = r;\n          dst[dstOff + 1] = g;\n          dst[dstOff + 2] = b;\n          if (!alphaFac) {\n            dst[dstOff + 3] = a;\n          }\n          else {\n            dst[dstOff + 3] = data[dstOff + 3];\n          }\n        }\n      }\n      options.imageData = output;\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uMatrix: gl.getUniformLocation(program, 'uMatrix'),\n        uOpaque: gl.getUniformLocation(program, 'uOpaque'),\n        uHalfSize: gl.getUniformLocation(program, 'uHalfSize'),\n        uSize: gl.getUniformLocation(program, 'uSize'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform1fv(uniformLocations.uMatrix, this.matrix);\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function() {\n      return extend(this.callSuper('toObject'), {\n        opaque: this.opaque,\n        matrix: this.matrix\n      });\n    }\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Convolute} Instance of fabric.Image.filters.Convolute\n   */\n  fabric.Image.filters.Convolute.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Grayscale image filter class\n   * @class fabric.Image.filters.Grayscale\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Grayscale();\n   * object.filters.push(filter);\n   * object.applyFilters();\n   */\n  filters.Grayscale = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Grayscale.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Grayscale',\n\n    fragmentSource: {\n      average: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n          'float average = (color.r + color.b + color.g) / 3.0;\\n' +\n          'gl_FragColor = vec4(average, average, average, color.a);\\n' +\n        '}',\n      lightness: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform int uMode;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 col = texture2D(uTexture, vTexCoord);\\n' +\n          'float average = (max(max(col.r, col.g),col.b) + min(min(col.r, col.g),col.b)) / 2.0;\\n' +\n          'gl_FragColor = vec4(average, average, average, col.a);\\n' +\n        '}',\n      luminosity: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform int uMode;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 col = texture2D(uTexture, vTexCoord);\\n' +\n          'float average = 0.21 * col.r + 0.72 * col.g + 0.07 * col.b;\\n' +\n          'gl_FragColor = vec4(average, average, average, col.a);\\n' +\n        '}',\n    },\n\n\n    /**\n     * Grayscale mode, between 'average', 'lightness', 'luminosity'\n     * @param {String} type\n     * @default\n     */\n    mode: 'average',\n\n    mainParameter: 'mode',\n\n    /**\n     * Apply the Grayscale operation to a Uint8Array representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8Array to be filtered.\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData,\n          data = imageData.data, i,\n          len = data.length, value,\n          mode = this.mode;\n      for (i = 0; i < len; i += 4) {\n        if (mode === 'average') {\n          value = (data[i] + data[i + 1] + data[i + 2]) / 3;\n        }\n        else if (mode === 'lightness') {\n          value = (Math.min(data[i], data[i + 1], data[i + 2]) +\n            Math.max(data[i], data[i + 1], data[i + 2])) / 2;\n        }\n        else if (mode === 'luminosity') {\n          value = 0.21 * data[i] + 0.72 * data[i + 1] + 0.07 * data[i + 2];\n        }\n        data[i] = value;\n        data[i + 1] = value;\n        data[i + 2] = value;\n      }\n    },\n\n    /**\n     * Retrieves the cached shader.\n     * @param {Object} options\n     * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n     * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n     */\n    retrieveShader: function(options) {\n      var cacheKey = this.type + '_' + this.mode;\n      if (!options.programCache.hasOwnProperty(cacheKey)) {\n        var shaderSource = this.fragmentSource[this.mode];\n        options.programCache[cacheKey] = this.createProgram(options.context, shaderSource);\n      }\n      return options.programCache[cacheKey];\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uMode: gl.getUniformLocation(program, 'uMode'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      // default average mode.\n      var mode = 1;\n      gl.uniform1i(uniformLocations.uMode, mode);\n    },\n\n    /**\n     * Grayscale filter isNeutralState implementation\n     * The filter is never neutral\n     * on the image\n     **/\n    isNeutralState: function() {\n      return false;\n    },\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Grayscale} Instance of fabric.Image.filters.Grayscale\n   */\n  fabric.Image.filters.Grayscale.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Invert filter class\n   * @class fabric.Image.filters.Invert\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Invert();\n   * object.filters.push(filter);\n   * object.applyFilters(canvas.renderAll.bind(canvas));\n   */\n  filters.Invert = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Invert.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Invert',\n\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform int uInvert;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'void main() {\\n' +\n        'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n        'if (uInvert == 1) {\\n' +\n          'gl_FragColor = vec4(1.0 - color.r,1.0 -color.g,1.0 -color.b,color.a);\\n' +\n        '} else {\\n' +\n          'gl_FragColor = color;\\n' +\n        '}\\n' +\n      '}',\n\n    /**\n     * Filter invert. if false, does nothing\n     * @param {Boolean} invert\n     * @default\n     */\n    invert: true,\n\n    mainParameter: 'invert',\n\n    /**\n     * Apply the Invert operation to a Uint8Array representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8Array to be filtered.\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData,\n          data = imageData.data, i,\n          len = data.length;\n      for (i = 0; i < len; i += 4) {\n        data[i] = 255 - data[i];\n        data[i + 1] = 255 - data[i + 1];\n        data[i + 2] = 255 - data[i + 2];\n      }\n    },\n\n    /**\n     * Invert filter isNeutralState implementation\n     * Used only in image applyFilters to discard filters that will not have an effect\n     * on the image\n     * @param {Object} options\n     **/\n    isNeutralState: function() {\n      return !this.invert;\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uInvert: gl.getUniformLocation(program, 'uInvert'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform1i(uniformLocations.uInvert, this.invert);\n    },\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Invert} Instance of fabric.Image.filters.Invert\n   */\n  fabric.Image.filters.Invert.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      extend = fabric.util.object.extend,\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Noise filter class\n   * @class fabric.Image.filters.Noise\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.Noise#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Noise({\n   *   noise: 700\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   * canvas.renderAll();\n   */\n  filters.Noise = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Noise.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Noise',\n\n    /**\n     * Fragment source for the noise program\n     */\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform float uStepH;\\n' +\n      'uniform float uNoise;\\n' +\n      'uniform float uSeed;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'float rand(vec2 co, float seed, float vScale) {\\n' +\n        'return fract(sin(dot(co.xy * vScale ,vec2(12.9898 , 78.233))) * 43758.5453 * (seed + 0.01) / 2.0);\\n' +\n      '}\\n' +\n      'void main() {\\n' +\n        'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n        'color.rgb += (0.5 - rand(vTexCoord, uSeed, 0.1 / uStepH)) * uNoise;\\n' +\n        'gl_FragColor = color;\\n' +\n      '}',\n\n    /**\n     * Describe the property that is the filter parameter\n     * @param {String} m\n     * @default\n     */\n    mainParameter: 'noise',\n\n    /**\n     * Noise value, from\n     * @param {Number} noise\n     * @default\n     */\n    noise: 0,\n\n    /**\n     * Apply the Brightness operation to a Uint8ClampedArray representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8ClampedArray to be filtered.\n     */\n    applyTo2d: function(options) {\n      if (this.noise === 0) {\n        return;\n      }\n      var imageData = options.imageData,\n          data = imageData.data, i, len = data.length,\n          noise = this.noise, rand;\n\n      for (i = 0, len = data.length; i < len; i += 4) {\n\n        rand = (0.5 - Math.random()) * noise;\n\n        data[i] += rand;\n        data[i + 1] += rand;\n        data[i + 2] += rand;\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uNoise: gl.getUniformLocation(program, 'uNoise'),\n        uSeed: gl.getUniformLocation(program, 'uSeed'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform1f(uniformLocations.uNoise, this.noise / 255);\n      gl.uniform1f(uniformLocations.uSeed, Math.random());\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function() {\n      return extend(this.callSuper('toObject'), {\n        noise: this.noise\n      });\n    }\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {Function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Noise} Instance of fabric.Image.filters.Noise\n   */\n  fabric.Image.filters.Noise.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Pixelate filter class\n   * @class fabric.Image.filters.Pixelate\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.Pixelate#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Pixelate({\n   *   blocksize: 8\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   */\n  filters.Pixelate = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Pixelate.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Pixelate',\n\n    blocksize: 4,\n\n    mainParameter: 'blocksize',\n\n    /**\n     * Fragment source for the Pixelate program\n     */\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform float uBlocksize;\\n' +\n      'uniform float uStepW;\\n' +\n      'uniform float uStepH;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'void main() {\\n' +\n        'float blockW = uBlocksize * uStepW;\\n' +\n        'float blockH = uBlocksize * uStepW;\\n' +\n        'int posX = int(vTexCoord.x / blockW);\\n' +\n        'int posY = int(vTexCoord.y / blockH);\\n' +\n        'float fposX = float(posX);\\n' +\n        'float fposY = float(posY);\\n' +\n        'vec2 squareCoords = vec2(fposX * blockW, fposY * blockH);\\n' +\n        'vec4 color = texture2D(uTexture, squareCoords);\\n' +\n        'gl_FragColor = color;\\n' +\n      '}',\n\n    /**\n     * Apply the Pixelate operation to a Uint8ClampedArray representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8ClampedArray to be filtered.\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData,\n          data = imageData.data,\n          iLen = imageData.height,\n          jLen = imageData.width,\n          index, i, j, r, g, b, a,\n          _i, _j, _iLen, _jLen;\n\n      for (i = 0; i < iLen; i += this.blocksize) {\n        for (j = 0; j < jLen; j += this.blocksize) {\n\n          index = (i * 4) * jLen + (j * 4);\n\n          r = data[index];\n          g = data[index + 1];\n          b = data[index + 2];\n          a = data[index + 3];\n\n          _iLen = Math.min(i + this.blocksize, iLen);\n          _jLen = Math.min(j + this.blocksize, jLen);\n          for (_i = i; _i < _iLen; _i++) {\n            for (_j = j; _j < _jLen; _j++) {\n              index = (_i * 4) * jLen + (_j * 4);\n              data[index] = r;\n              data[index + 1] = g;\n              data[index + 2] = b;\n              data[index + 3] = a;\n            }\n          }\n        }\n      }\n    },\n\n    /**\n     * Indicate when the filter is not gonna apply changes to the image\n     **/\n    isNeutralState: function() {\n      return this.blocksize === 1;\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uBlocksize: gl.getUniformLocation(program, 'uBlocksize'),\n        uStepW: gl.getUniformLocation(program, 'uStepW'),\n        uStepH: gl.getUniformLocation(program, 'uStepH'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform1f(uniformLocations.uBlocksize, this.blocksize);\n    },\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {Function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Pixelate} Instance of fabric.Image.filters.Pixelate\n   */\n  fabric.Image.filters.Pixelate.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      extend = fabric.util.object.extend,\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Remove white filter class\n   * @class fabric.Image.filters.RemoveColor\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.RemoveColor#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.RemoveColor({\n   *   threshold: 0.2,\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   * canvas.renderAll();\n   */\n  filters.RemoveColor = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.RemoveColor.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'RemoveColor',\n\n    /**\n     * Color to remove, in any format understood by fabric.Color.\n     * @param {String} type\n     * @default\n     */\n    color: '#FFFFFF',\n\n    /**\n     * Fragment source for the brightness program\n     */\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform vec4 uLow;\\n' +\n      'uniform vec4 uHigh;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'void main() {\\n' +\n        'gl_FragColor = texture2D(uTexture, vTexCoord);\\n' +\n        'if(all(greaterThan(gl_FragColor.rgb,uLow.rgb)) && all(greaterThan(uHigh.rgb,gl_FragColor.rgb))) {\\n' +\n          'gl_FragColor.a = 0.0;\\n' +\n        '}\\n' +\n      '}',\n\n    /**\n     * distance to actual color, as value up or down from each r,g,b\n     * between 0 and 1\n     **/\n    distance: 0.02,\n\n    /**\n     * For color to remove inside distance, use alpha channel for a smoother deletion\n     * NOT IMPLEMENTED YET\n     **/\n    useAlpha: false,\n\n    /**\n     * Constructor\n     * @memberOf fabric.Image.filters.RemoveWhite.prototype\n     * @param {Object} [options] Options object\n     * @param {Number} [options.color=#RRGGBB] Threshold value\n     * @param {Number} [options.distance=10] Distance value\n     */\n\n    /**\n     * Applies filter to canvas element\n     * @param {Object} canvasEl Canvas element to apply filter to\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData,\n          data = imageData.data, i,\n          distance = this.distance * 255,\n          r, g, b,\n          source = new fabric.Color(this.color).getSource(),\n          lowC = [\n            source[0] - distance,\n            source[1] - distance,\n            source[2] - distance,\n          ],\n          highC = [\n            source[0] + distance,\n            source[1] + distance,\n            source[2] + distance,\n          ];\n\n\n      for (i = 0; i < data.length; i += 4) {\n        r = data[i];\n        g = data[i + 1];\n        b = data[i + 2];\n\n        if (r > lowC[0] &&\n            g > lowC[1] &&\n            b > lowC[2] &&\n            r < highC[0] &&\n            g < highC[1] &&\n            b < highC[2]) {\n          data[i + 3] = 0;\n        }\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uLow: gl.getUniformLocation(program, 'uLow'),\n        uHigh: gl.getUniformLocation(program, 'uHigh'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      var source = new fabric.Color(this.color).getSource(),\n          distance = parseFloat(this.distance),\n          lowC = [\n            0 + source[0] / 255 - distance,\n            0 + source[1] / 255 - distance,\n            0 + source[2] / 255 - distance,\n            1\n          ],\n          highC = [\n            source[0] / 255 + distance,\n            source[1] / 255 + distance,\n            source[2] / 255 + distance,\n            1\n          ];\n      gl.uniform4fv(uniformLocations.uLow, lowC);\n      gl.uniform4fv(uniformLocations.uHigh, highC);\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function() {\n      return extend(this.callSuper('toObject'), {\n        color: this.color,\n        distance: this.distance\n      });\n    }\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {Function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.RemoveColor} Instance of fabric.Image.filters.RemoveWhite\n   */\n  fabric.Image.filters.RemoveColor.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  var matrices = {\n    Brownie: [\n      0.59970,0.34553,-0.27082,0,0.186,\n      -0.03770,0.86095,0.15059,0,-0.1449,\n      0.24113,-0.07441,0.44972,0,-0.02965,\n      0,0,0,1,0\n    ],\n    Vintage: [\n      0.62793,0.32021,-0.03965,0,0.03784,\n      0.02578,0.64411,0.03259,0,0.02926,\n      0.04660,-0.08512,0.52416,0,0.02023,\n      0,0,0,1,0\n    ],\n    Kodachrome: [\n      1.12855,-0.39673,-0.03992,0,0.24991,\n      -0.16404,1.08352,-0.05498,0,0.09698,\n      -0.16786,-0.56034,1.60148,0,0.13972,\n      0,0,0,1,0\n    ],\n    Technicolor: [\n      1.91252,-0.85453,-0.09155,0,0.04624,\n      -0.30878,1.76589,-0.10601,0,-0.27589,\n      -0.23110,-0.75018,1.84759,0,0.12137,\n      0,0,0,1,0\n    ],\n    Polaroid: [\n      1.438,-0.062,-0.062,0,0,\n      -0.122,1.378,-0.122,0,0,\n      -0.016,-0.016,1.483,0,0,\n      0,0,0,1,0\n    ],\n    Sepia: [\n      0.393, 0.769, 0.189, 0, 0,\n      0.349, 0.686, 0.168, 0, 0,\n      0.272, 0.534, 0.131, 0, 0,\n      0, 0, 0, 1, 0\n    ],\n    BlackWhite: [\n      1.5, 1.5, 1.5, 0, -1,\n      1.5, 1.5, 1.5, 0, -1,\n      1.5, 1.5, 1.5, 0, -1,\n      0, 0, 0, 1, 0,\n    ]\n  };\n\n  for (var key in matrices) {\n    filters[key] = createClass(filters.ColorMatrix, /** @lends fabric.Image.filters.Sepia.prototype */ {\n\n      /**\n       * Filter type\n       * @param {String} type\n       * @default\n       */\n      type: key,\n\n      /**\n       * Colormatrix for the effect\n       * array of 20 floats. Numbers in positions 4, 9, 14, 19 loose meaning\n       * outside the -1, 1 range.\n       * @param {Array} matrix array of 20 numbers.\n       * @default\n       */\n      matrix: matrices[key],\n\n      /**\n       * Lock the matrix export for this kind of static, parameter less filters.\n       */\n      mainParameter: false,\n      /**\n       * Lock the colormatrix on the color part, skipping alpha\n       */\n      colorsOnly: true,\n\n    });\n    fabric.Image.filters[key].fromObject = fabric.Image.filters.BaseFilter.fromObject;\n  }\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n  'use strict';\n\n  var fabric = global.fabric,\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Color Blend filter class\n   * @class fabric.Image.filter.BlendColor\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @example\n   * var filter = new fabric.Image.filters.BlendColor({\n   *  color: '#000',\n   *  mode: 'multiply'\n   * });\n   *\n   * var filter = new fabric.Image.filters.BlendImage({\n   *  image: fabricImageObject,\n   *  mode: 'multiply',\n   *  alpha: 0.5\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   * canvas.renderAll();\n   */\n\n  filters.BlendColor = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Blend.prototype */ {\n    type: 'BlendColor',\n\n    /**\n     * Color to make the blend operation with. default to a reddish color since black or white\n     * gives always strong result.\n     * @type String\n     * @default\n     **/\n    color: '#F95C63',\n\n    /**\n     * Blend mode for the filter: one of multiply, add, diff, screen, subtract,\n     * darken, lighten, overlay, exclusion, tint.\n     * @type String\n     * @default\n     **/\n    mode: 'multiply',\n\n    /**\n     * alpha value. represent the strength of the blend color operation.\n     * @type Number\n     * @default\n     **/\n    alpha: 1,\n\n    /**\n     * Fragment source for the Multiply program\n     */\n    fragmentSource: {\n      multiply: 'gl_FragColor.rgb *= uColor.rgb;\\n',\n      screen: 'gl_FragColor.rgb = 1.0 - (1.0 - gl_FragColor.rgb) * (1.0 - uColor.rgb);\\n',\n      add: 'gl_FragColor.rgb += uColor.rgb;\\n',\n      diff: 'gl_FragColor.rgb = abs(gl_FragColor.rgb - uColor.rgb);\\n',\n      subtract: 'gl_FragColor.rgb -= uColor.rgb;\\n',\n      lighten: 'gl_FragColor.rgb = max(gl_FragColor.rgb, uColor.rgb);\\n',\n      darken: 'gl_FragColor.rgb = min(gl_FragColor.rgb, uColor.rgb);\\n',\n      exclusion: 'gl_FragColor.rgb += uColor.rgb - 2.0 * (uColor.rgb * gl_FragColor.rgb);\\n',\n      overlay: 'if (uColor.r < 0.5) {\\n' +\n          'gl_FragColor.r *= 2.0 * uColor.r;\\n' +\n        '} else {\\n' +\n          'gl_FragColor.r = 1.0 - 2.0 * (1.0 - gl_FragColor.r) * (1.0 - uColor.r);\\n' +\n        '}\\n' +\n        'if (uColor.g < 0.5) {\\n' +\n          'gl_FragColor.g *= 2.0 * uColor.g;\\n' +\n        '} else {\\n' +\n          'gl_FragColor.g = 1.0 - 2.0 * (1.0 - gl_FragColor.g) * (1.0 - uColor.g);\\n' +\n        '}\\n' +\n        'if (uColor.b < 0.5) {\\n' +\n          'gl_FragColor.b *= 2.0 * uColor.b;\\n' +\n        '} else {\\n' +\n          'gl_FragColor.b = 1.0 - 2.0 * (1.0 - gl_FragColor.b) * (1.0 - uColor.b);\\n' +\n        '}\\n',\n      tint: 'gl_FragColor.rgb *= (1.0 - uColor.a);\\n' +\n        'gl_FragColor.rgb += uColor.rgb;\\n',\n    },\n\n    /**\n     * build the fragment source for the filters, joining the common part with\n     * the specific one.\n     * @param {String} mode the mode of the filter, a key of this.fragmentSource\n     * @return {String} the source to be compiled\n     * @private\n     */\n    buildSource: function(mode) {\n      return 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform vec4 uColor;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'void main() {\\n' +\n          'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n          'gl_FragColor = color;\\n' +\n          'if (color.a > 0.0) {\\n' +\n            this.fragmentSource[mode] +\n          '}\\n' +\n        '}';\n    },\n\n    /**\n     * Retrieves the cached shader.\n     * @param {Object} options\n     * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n     * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n     */\n    retrieveShader: function(options) {\n      var cacheKey = this.type + '_' + this.mode, shaderSource;\n      if (!options.programCache.hasOwnProperty(cacheKey)) {\n        shaderSource = this.buildSource(this.mode);\n        options.programCache[cacheKey] = this.createProgram(options.context, shaderSource);\n      }\n      return options.programCache[cacheKey];\n    },\n\n    /**\n     * Apply the Blend operation to a Uint8ClampedArray representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8ClampedArray to be filtered.\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData,\n          data = imageData.data, iLen = data.length,\n          tr, tg, tb,\n          r, g, b,\n          source, alpha1 = 1 - this.alpha;\n\n      source = new fabric.Color(this.color).getSource();\n      tr = source[0] * this.alpha;\n      tg = source[1] * this.alpha;\n      tb = source[2] * this.alpha;\n\n      for (var i = 0; i < iLen; i += 4) {\n\n        r = data[i];\n        g = data[i + 1];\n        b = data[i + 2];\n\n        switch (this.mode) {\n          case 'multiply':\n            data[i] = r * tr / 255;\n            data[i + 1] = g * tg / 255;\n            data[i + 2] = b * tb / 255;\n            break;\n          case 'screen':\n            data[i] = 255 - (255 - r) * (255 - tr) / 255;\n            data[i + 1] = 255 - (255 - g) * (255 - tg) / 255;\n            data[i + 2] = 255 - (255 - b) * (255 - tb) / 255;\n            break;\n          case 'add':\n            data[i] = r + tr;\n            data[i + 1] = g + tg;\n            data[i + 2] = b + tb;\n            break;\n          case 'diff':\n          case 'difference':\n            data[i] = Math.abs(r - tr);\n            data[i + 1] = Math.abs(g - tg);\n            data[i + 2] = Math.abs(b - tb);\n            break;\n          case 'subtract':\n            data[i] = r - tr;\n            data[i + 1] = g - tg;\n            data[i + 2] = b - tb;\n            break;\n          case 'darken':\n            data[i] = Math.min(r, tr);\n            data[i + 1] = Math.min(g, tg);\n            data[i + 2] = Math.min(b, tb);\n            break;\n          case 'lighten':\n            data[i] = Math.max(r, tr);\n            data[i + 1] = Math.max(g, tg);\n            data[i + 2] = Math.max(b, tb);\n            break;\n          case 'overlay':\n            data[i] = tr < 128 ? (2 * r * tr / 255) : (255 - 2 * (255 - r) * (255 - tr) / 255);\n            data[i + 1] = tg < 128 ? (2 * g * tg / 255) : (255 - 2 * (255 - g) * (255 - tg) / 255);\n            data[i + 2] = tb < 128 ? (2 * b * tb / 255) : (255 - 2 * (255 - b) * (255 - tb) / 255);\n            break;\n          case 'exclusion':\n            data[i] = tr + r - ((2 * tr * r) / 255);\n            data[i + 1] = tg + g - ((2 * tg * g) / 255);\n            data[i + 2] = tb + b - ((2 * tb * b) / 255);\n            break;\n          case 'tint':\n            data[i] = tr + r * alpha1;\n            data[i + 1] = tg + g * alpha1;\n            data[i + 2] = tb + b * alpha1;\n        }\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uColor: gl.getUniformLocation(program, 'uColor'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      var source = new fabric.Color(this.color).getSource();\n      source[0] = this.alpha * source[0] / 255;\n      source[1] = this.alpha * source[1] / 255;\n      source[2] = this.alpha * source[2] / 255;\n      source[3] = this.alpha;\n      gl.uniform4fv(uniformLocations.uColor, source);\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function() {\n      return {\n        type: this.type,\n        color: this.color,\n        mode: this.mode,\n        alpha: this.alpha\n      };\n    }\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.BlendColor} Instance of fabric.Image.filters.BlendColor\n   */\n  fabric.Image.filters.BlendColor.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n  'use strict';\n\n  var fabric = global.fabric,\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Image Blend filter class\n   * @class fabric.Image.filter.BlendImage\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @example\n   * var filter = new fabric.Image.filters.BlendColor({\n   *  color: '#000',\n   *  mode: 'multiply'\n   * });\n   *\n   * var filter = new fabric.Image.filters.BlendImage({\n   *  image: fabricImageObject,\n   *  mode: 'multiply',\n   *  alpha: 0.5\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   * canvas.renderAll();\n   */\n\n  filters.BlendImage = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.BlendImage.prototype */ {\n    type: 'BlendImage',\n\n    /**\n     * Color to make the blend operation with. default to a reddish color since black or white\n     * gives always strong result.\n     **/\n    image: null,\n\n    /**\n     * Blend mode for the filter (one of \"multiply\", \"mask\")\n     * @type String\n     * @default\n     **/\n    mode: 'multiply',\n\n    /**\n     * alpha value. represent the strength of the blend image operation.\n     * not implemented.\n     **/\n    alpha: 1,\n\n    vertexSource: 'attribute vec2 aPosition;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'varying vec2 vTexCoord2;\\n' +\n      'uniform mat3 uTransformMatrix;\\n' +\n      'void main() {\\n' +\n        'vTexCoord = aPosition;\\n' +\n        'vTexCoord2 = (uTransformMatrix * vec3(aPosition, 1.0)).xy;\\n' +\n        'gl_Position = vec4(aPosition * 2.0 - 1.0, 0.0, 1.0);\\n' +\n      '}',\n\n    /**\n     * Fragment source for the Multiply program\n     */\n    fragmentSource: {\n      multiply: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform sampler2D uImage;\\n' +\n        'uniform vec4 uColor;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'varying vec2 vTexCoord2;\\n' +\n        'void main() {\\n' +\n          'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n          'vec4 color2 = texture2D(uImage, vTexCoord2);\\n' +\n          'color.rgba *= color2.rgba;\\n' +\n          'gl_FragColor = color;\\n' +\n        '}',\n      mask: 'precision highp float;\\n' +\n        'uniform sampler2D uTexture;\\n' +\n        'uniform sampler2D uImage;\\n' +\n        'uniform vec4 uColor;\\n' +\n        'varying vec2 vTexCoord;\\n' +\n        'varying vec2 vTexCoord2;\\n' +\n        'void main() {\\n' +\n          'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n          'vec4 color2 = texture2D(uImage, vTexCoord2);\\n' +\n          'color.a = color2.a;\\n' +\n          'gl_FragColor = color;\\n' +\n        '}',\n    },\n\n    /**\n     * Retrieves the cached shader.\n     * @param {Object} options\n     * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n     * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n     */\n    retrieveShader: function(options) {\n      var cacheKey = this.type + '_' + this.mode;\n      var shaderSource = this.fragmentSource[this.mode];\n      if (!options.programCache.hasOwnProperty(cacheKey)) {\n        options.programCache[cacheKey] = this.createProgram(options.context, shaderSource);\n      }\n      return options.programCache[cacheKey];\n    },\n\n    applyToWebGL: function(options) {\n      // load texture to blend.\n      var gl = options.context,\n          texture = this.createTexture(options.filterBackend, this.image);\n      this.bindAdditionalTexture(gl, texture, gl.TEXTURE1);\n      this.callSuper('applyToWebGL', options);\n      this.unbindAdditionalTexture(gl, gl.TEXTURE1);\n    },\n\n    createTexture: function(backend, image) {\n      return backend.getCachedTexture(image.cacheKey, image._element);\n    },\n\n    /**\n     * Calculate a transformMatrix to adapt the image to blend over\n     * @param {Object} options\n     * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n     * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n     */\n    calculateMatrix: function() {\n      var image = this.image,\n          width = image._element.width,\n          height = image._element.height;\n      return [\n        1 / image.scaleX, 0, 0,\n        0, 1 / image.scaleY, 0,\n        -image.left / width, -image.top / height, 1\n      ];\n    },\n\n    /**\n     * Apply the Blend operation to a Uint8ClampedArray representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8ClampedArray to be filtered.\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData,\n          resources = options.filterBackend.resources,\n          data = imageData.data, iLen = data.length,\n          width = imageData.width,\n          height = imageData.height,\n          tr, tg, tb, ta,\n          r, g, b, a,\n          canvas1, context, image = this.image, blendData;\n\n      if (!resources.blendImage) {\n        resources.blendImage = fabric.util.createCanvasElement();\n      }\n      canvas1 = resources.blendImage;\n      context = canvas1.getContext('2d');\n      if (canvas1.width !== width || canvas1.height !== height) {\n        canvas1.width = width;\n        canvas1.height = height;\n      }\n      else {\n        context.clearRect(0, 0, width, height);\n      }\n      context.setTransform(image.scaleX, 0, 0, image.scaleY, image.left, image.top);\n      context.drawImage(image._element, 0, 0, width, height);\n      blendData = context.getImageData(0, 0, width, height).data;\n      for (var i = 0; i < iLen; i += 4) {\n\n        r = data[i];\n        g = data[i + 1];\n        b = data[i + 2];\n        a = data[i + 3];\n\n        tr = blendData[i];\n        tg = blendData[i + 1];\n        tb = blendData[i + 2];\n        ta = blendData[i + 3];\n\n        switch (this.mode) {\n          case 'multiply':\n            data[i] = r * tr / 255;\n            data[i + 1] = g * tg / 255;\n            data[i + 2] = b * tb / 255;\n            data[i + 3] = a * ta / 255;\n            break;\n          case 'mask':\n            data[i + 3] = ta;\n            break;\n        }\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uTransformMatrix: gl.getUniformLocation(program, 'uTransformMatrix'),\n        uImage: gl.getUniformLocation(program, 'uImage'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      var matrix = this.calculateMatrix();\n      gl.uniform1i(uniformLocations.uImage, 1); // texture unit 1.\n      gl.uniformMatrix3fv(uniformLocations.uTransformMatrix, false, matrix);\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function() {\n      return {\n        type: this.type,\n        image: this.image && this.image.toObject(),\n        mode: this.mode,\n        alpha: this.alpha\n      };\n    }\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} callback to be invoked after filter creation\n   * @return {fabric.Image.filters.BlendImage} Instance of fabric.Image.filters.BlendImage\n   */\n  fabric.Image.filters.BlendImage.fromObject = function(object, callback) {\n    fabric.Image.fromObject(object.image, function(image) {\n      var options = fabric.util.object.clone(object);\n      options.image = image;\n      callback(new fabric.Image.filters.BlendImage(options));\n    });\n  };\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }), pow = Math.pow, floor = Math.floor,\n      sqrt = Math.sqrt, abs = Math.abs, round = Math.round, sin = Math.sin,\n      ceil = Math.ceil,\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Resize image filter class\n   * @class fabric.Image.filters.Resize\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Resize();\n   * object.filters.push(filter);\n   * object.applyFilters(canvas.renderAll.bind(canvas));\n   */\n  filters.Resize = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Resize.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Resize',\n\n    /**\n     * Resize type\n     * for webgl resizeType is just lanczos, for canvas2d can be:\n     * bilinear, hermite, sliceHack, lanczos.\n     * @param {String} resizeType\n     * @default\n     */\n    resizeType: 'hermite',\n\n    /**\n     * Scale factor for resizing, x axis\n     * @param {Number} scaleX\n     * @default\n     */\n    scaleX: 1,\n\n    /**\n     * Scale factor for resizing, y axis\n     * @param {Number} scaleY\n     * @default\n     */\n    scaleY: 1,\n\n    /**\n     * LanczosLobes parameter for lanczos filter, valid for resizeType lanczos\n     * @param {Number} lanczosLobes\n     * @default\n     */\n    lanczosLobes: 3,\n\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uDelta: gl.getUniformLocation(program, 'uDelta'),\n        uTaps: gl.getUniformLocation(program, 'uTaps'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform2fv(uniformLocations.uDelta, this.horizontal ? [1 / this.width, 0] : [0, 1 / this.height]);\n      gl.uniform1fv(uniformLocations.uTaps, this.taps);\n    },\n\n    /**\n     * Retrieves the cached shader.\n     * @param {Object} options\n     * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n     * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n     */\n    retrieveShader: function(options) {\n      var filterWindow = this.getFilterWindow(), cacheKey = this.type + '_' + filterWindow;\n      if (!options.programCache.hasOwnProperty(cacheKey)) {\n        var fragmentShader = this.generateShader(filterWindow);\n        options.programCache[cacheKey] = this.createProgram(options.context, fragmentShader);\n      }\n      return options.programCache[cacheKey];\n    },\n\n    getFilterWindow: function() {\n      var scale = this.tempScale;\n      return Math.ceil(this.lanczosLobes / scale);\n    },\n\n    getTaps: function() {\n      var lobeFunction = this.lanczosCreate(this.lanczosLobes), scale = this.tempScale,\n          filterWindow = this.getFilterWindow(), taps = new Array(filterWindow);\n      for (var i = 1; i <= filterWindow; i++) {\n        taps[i - 1] = lobeFunction(i * scale);\n      }\n      return taps;\n    },\n\n    /**\n     * Generate vertex and shader sources from the necessary steps numbers\n     * @param {Number} filterWindow\n     */\n    generateShader: function(filterWindow) {\n      var offsets = new Array(filterWindow),\n          fragmentShader = this.fragmentSourceTOP, filterWindow;\n\n      for (var i = 1; i <= filterWindow; i++) {\n        offsets[i - 1] = i + '.0 * uDelta';\n      }\n\n      fragmentShader += 'uniform float uTaps[' + filterWindow + '];\\n';\n      fragmentShader += 'void main() {\\n';\n      fragmentShader += '  vec4 color = texture2D(uTexture, vTexCoord);\\n';\n      fragmentShader += '  float sum = 1.0;\\n';\n\n      offsets.forEach(function(offset, i) {\n        fragmentShader += '  color += texture2D(uTexture, vTexCoord + ' + offset + ') * uTaps[' + i + '];\\n';\n        fragmentShader += '  color += texture2D(uTexture, vTexCoord - ' + offset + ') * uTaps[' + i + '];\\n';\n        fragmentShader += '  sum += 2.0 * uTaps[' + i + '];\\n';\n      });\n      fragmentShader += '  gl_FragColor = color / sum;\\n';\n      fragmentShader += '}';\n      return fragmentShader;\n    },\n\n    fragmentSourceTOP: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform vec2 uDelta;\\n' +\n      'varying vec2 vTexCoord;\\n',\n\n    /**\n     * Apply the resize filter to the image\n     * Determines whether to use WebGL or Canvas2D based on the options.webgl flag.\n     *\n     * @param {Object} options\n     * @param {Number} options.passes The number of filters remaining to be executed\n     * @param {Boolean} options.webgl Whether to use webgl to render the filter.\n     * @param {WebGLTexture} options.sourceTexture The texture setup as the source to be filtered.\n     * @param {WebGLTexture} options.targetTexture The texture where filtered output should be drawn.\n     * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n     * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n     */\n    applyTo: function(options) {\n      if (options.webgl) {\n        options.passes++;\n        this.width = options.sourceWidth;\n        this.horizontal = true;\n        this.dW = Math.round(this.width * this.scaleX);\n        this.dH = options.sourceHeight;\n        this.tempScale = this.dW / this.width;\n        this.taps = this.getTaps();\n        options.destinationWidth = this.dW;\n        this._setupFrameBuffer(options);\n        this.applyToWebGL(options);\n        this._swapTextures(options);\n        options.sourceWidth = options.destinationWidth;\n\n        this.height = options.sourceHeight;\n        this.horizontal = false;\n        this.dH = Math.round(this.height * this.scaleY);\n        this.tempScale = this.dH / this.height;\n        this.taps = this.getTaps();\n        options.destinationHeight = this.dH;\n        this._setupFrameBuffer(options);\n        this.applyToWebGL(options);\n        this._swapTextures(options);\n        options.sourceHeight = options.destinationHeight;\n      }\n      else {\n        this.applyTo2d(options);\n      }\n    },\n\n    isNeutralState: function() {\n      return this.scaleX === 1 && this.scaleY === 1;\n    },\n\n    lanczosCreate: function(lobes) {\n      return function(x) {\n        if (x >= lobes || x <= -lobes) {\n          return 0.0;\n        }\n        if (x < 1.19209290E-07 && x > -1.19209290E-07) {\n          return 1.0;\n        }\n        x *= Math.PI;\n        var xx = x / lobes;\n        return (sin(x) / x) * sin(xx) / xx;\n      };\n    },\n\n    /**\n     * Applies filter to canvas element\n     * @memberOf fabric.Image.filters.Resize.prototype\n     * @param {Object} canvasEl Canvas element to apply filter to\n     * @param {Number} scaleX\n     * @param {Number} scaleY\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData,\n          scaleX = this.scaleX,\n          scaleY = this.scaleY;\n\n      this.rcpScaleX = 1 / scaleX;\n      this.rcpScaleY = 1 / scaleY;\n\n      var oW = imageData.width, oH = imageData.height,\n          dW = round(oW * scaleX), dH = round(oH * scaleY),\n          newData;\n\n      if (this.resizeType === 'sliceHack') {\n        newData = this.sliceByTwo(options, oW, oH, dW, dH);\n      }\n      else if (this.resizeType === 'hermite') {\n        newData = this.hermiteFastResize(options, oW, oH, dW, dH);\n      }\n      else if (this.resizeType === 'bilinear') {\n        newData = this.bilinearFiltering(options, oW, oH, dW, dH);\n      }\n      else if (this.resizeType === 'lanczos') {\n        newData = this.lanczosResize(options, oW, oH, dW, dH);\n      }\n      options.imageData = newData;\n    },\n\n    /**\n     * Filter sliceByTwo\n     * @param {Object} canvasEl Canvas element to apply filter to\n     * @param {Number} oW Original Width\n     * @param {Number} oH Original Height\n     * @param {Number} dW Destination Width\n     * @param {Number} dH Destination Height\n     * @returns {ImageData}\n     */\n    sliceByTwo: function(options, oW, oH, dW, dH) {\n      var imageData = options.imageData,\n          mult = 0.5, doneW = false, doneH = false, stepW = oW * mult,\n          stepH = oH * mult, resources = fabric.filterBackend.resources,\n          tmpCanvas, ctx, sX = 0, sY = 0, dX = oW, dY = 0;\n      if (!resources.sliceByTwo) {\n        resources.sliceByTwo = document.createElement('canvas');\n      }\n      tmpCanvas = resources.sliceByTwo;\n      if (tmpCanvas.width < oW * 1.5 || tmpCanvas.height < oH) {\n        tmpCanvas.width = oW * 1.5;\n        tmpCanvas.height = oH;\n      }\n      ctx = tmpCanvas.getContext('2d');\n      ctx.clearRect(0, 0, oW * 1.5, oH);\n      ctx.putImageData(imageData, 0, 0);\n\n      dW = floor(dW);\n      dH = floor(dH);\n\n      while (!doneW || !doneH) {\n        oW = stepW;\n        oH = stepH;\n        if (dW < floor(stepW * mult)) {\n          stepW = floor(stepW * mult);\n        }\n        else {\n          stepW = dW;\n          doneW = true;\n        }\n        if (dH < floor(stepH * mult)) {\n          stepH = floor(stepH * mult);\n        }\n        else {\n          stepH = dH;\n          doneH = true;\n        }\n        ctx.drawImage(tmpCanvas, sX, sY, oW, oH, dX, dY, stepW, stepH);\n        sX = dX;\n        sY = dY;\n        dY += stepH;\n      }\n      return ctx.getImageData(sX, sY, dW, dH);\n    },\n\n    /**\n     * Filter lanczosResize\n     * @param {Object} canvasEl Canvas element to apply filter to\n     * @param {Number} oW Original Width\n     * @param {Number} oH Original Height\n     * @param {Number} dW Destination Width\n     * @param {Number} dH Destination Height\n     * @returns {ImageData}\n     */\n    lanczosResize: function(options, oW, oH, dW, dH) {\n\n      function process(u) {\n        var v, i, weight, idx, a, red, green,\n            blue, alpha, fX, fY;\n        center.x = (u + 0.5) * ratioX;\n        icenter.x = floor(center.x);\n        for (v = 0; v < dH; v++) {\n          center.y = (v + 0.5) * ratioY;\n          icenter.y = floor(center.y);\n          a = 0; red = 0; green = 0; blue = 0; alpha = 0;\n          for (i = icenter.x - range2X; i <= icenter.x + range2X; i++) {\n            if (i < 0 || i >= oW) {\n              continue;\n            }\n            fX = floor(1000 * abs(i - center.x));\n            if (!cacheLanc[fX]) {\n              cacheLanc[fX] = { };\n            }\n            for (var j = icenter.y - range2Y; j <= icenter.y + range2Y; j++) {\n              if (j < 0 || j >= oH) {\n                continue;\n              }\n              fY = floor(1000 * abs(j - center.y));\n              if (!cacheLanc[fX][fY]) {\n                cacheLanc[fX][fY] = lanczos(sqrt(pow(fX * rcpRatioX, 2) + pow(fY * rcpRatioY, 2)) / 1000);\n              }\n              weight = cacheLanc[fX][fY];\n              if (weight > 0) {\n                idx = (j * oW + i) * 4;\n                a += weight;\n                red += weight * srcData[idx];\n                green += weight * srcData[idx + 1];\n                blue += weight * srcData[idx + 2];\n                alpha += weight * srcData[idx + 3];\n              }\n            }\n          }\n          idx = (v * dW + u) * 4;\n          destData[idx] = red / a;\n          destData[idx + 1] = green / a;\n          destData[idx + 2] = blue / a;\n          destData[idx + 3] = alpha / a;\n        }\n\n        if (++u < dW) {\n          return process(u);\n        }\n        else {\n          return destImg;\n        }\n      }\n\n      var srcData = options.imageData.data,\n          destImg = options.ctx.createImageData(dW, dH),\n          destData = destImg.data,\n          lanczos = this.lanczosCreate(this.lanczosLobes),\n          ratioX = this.rcpScaleX, ratioY = this.rcpScaleY,\n          rcpRatioX = 2 / this.rcpScaleX, rcpRatioY = 2 / this.rcpScaleY,\n          range2X = ceil(ratioX * this.lanczosLobes / 2),\n          range2Y = ceil(ratioY * this.lanczosLobes / 2),\n          cacheLanc = { }, center = { }, icenter = { };\n\n      return process(0);\n    },\n\n    /**\n     * bilinearFiltering\n     * @param {Object} canvasEl Canvas element to apply filter to\n     * @param {Number} oW Original Width\n     * @param {Number} oH Original Height\n     * @param {Number} dW Destination Width\n     * @param {Number} dH Destination Height\n     * @returns {ImageData}\n     */\n    bilinearFiltering: function(options, oW, oH, dW, dH) {\n      var a, b, c, d, x, y, i, j, xDiff, yDiff, chnl,\n          color, offset = 0, origPix, ratioX = this.rcpScaleX,\n          ratioY = this.rcpScaleY,\n          w4 = 4 * (oW - 1), img = options.imageData,\n          pixels = img.data, destImage = options.ctx.createImageData(dW, dH),\n          destPixels = destImage.data;\n      for (i = 0; i < dH; i++) {\n        for (j = 0; j < dW; j++) {\n          x = floor(ratioX * j);\n          y = floor(ratioY * i);\n          xDiff = ratioX * j - x;\n          yDiff = ratioY * i - y;\n          origPix = 4 * (y * oW + x);\n\n          for (chnl = 0; chnl < 4; chnl++) {\n            a = pixels[origPix + chnl];\n            b = pixels[origPix + 4 + chnl];\n            c = pixels[origPix + w4 + chnl];\n            d = pixels[origPix + w4 + 4 + chnl];\n            color = a * (1 - xDiff) * (1 - yDiff) + b * xDiff * (1 - yDiff) +\n                    c * yDiff * (1 - xDiff) + d * xDiff * yDiff;\n            destPixels[offset++] = color;\n          }\n        }\n      }\n      return destImage;\n    },\n\n    /**\n     * hermiteFastResize\n     * @param {Object} canvasEl Canvas element to apply filter to\n     * @param {Number} oW Original Width\n     * @param {Number} oH Original Height\n     * @param {Number} dW Destination Width\n     * @param {Number} dH Destination Height\n     * @returns {ImageData}\n     */\n    hermiteFastResize: function(options, oW, oH, dW, dH) {\n      var ratioW = this.rcpScaleX, ratioH = this.rcpScaleY,\n          ratioWHalf = ceil(ratioW / 2),\n          ratioHHalf = ceil(ratioH / 2),\n          img = options.imageData, data = img.data,\n          img2 = options.ctx.createImageData(dW, dH), data2 = img2.data;\n      for (var j = 0; j < dH; j++) {\n        for (var i = 0; i < dW; i++) {\n          var x2 = (i + j * dW) * 4, weight = 0, weights = 0, weightsAlpha = 0,\n              gxR = 0, gxG = 0, gxB = 0, gxA = 0, centerY = (j + 0.5) * ratioH;\n          for (var yy = floor(j * ratioH); yy < (j + 1) * ratioH; yy++) {\n            var dy = abs(centerY - (yy + 0.5)) / ratioHHalf,\n                centerX = (i + 0.5) * ratioW, w0 = dy * dy;\n            for (var xx = floor(i * ratioW); xx < (i + 1) * ratioW; xx++) {\n              var dx = abs(centerX - (xx + 0.5)) / ratioWHalf,\n                  w = sqrt(w0 + dx * dx);\n              /* eslint-disable max-depth */\n              if (w > 1 && w < -1) {\n                continue;\n              }\n              //hermite filter\n              weight = 2 * w * w * w - 3 * w * w + 1;\n              if (weight > 0) {\n                dx = 4 * (xx + yy * oW);\n                //alpha\n                gxA += weight * data[dx + 3];\n                weightsAlpha += weight;\n                //colors\n                if (data[dx + 3] < 255) {\n                  weight = weight * data[dx + 3] / 250;\n                }\n                gxR += weight * data[dx];\n                gxG += weight * data[dx + 1];\n                gxB += weight * data[dx + 2];\n                weights += weight;\n              }\n              /* eslint-enable max-depth */\n            }\n          }\n          data2[x2] = gxR / weights;\n          data2[x2 + 1] = gxG / weights;\n          data2[x2 + 2] = gxB / weights;\n          data2[x2 + 3] = gxA / weightsAlpha;\n        }\n      }\n      return img2;\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function() {\n      return {\n        type: this.type,\n        scaleX: this.scaleX,\n        scaleY: this.scaleY,\n        resizeType: this.resizeType,\n        lanczosLobes: this.lanczosLobes\n      };\n    }\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {Function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Resize} Instance of fabric.Image.filters.Resize\n   */\n  fabric.Image.filters.Resize.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Contrast filter class\n   * @class fabric.Image.filters.Contrast\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.Contrast#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Contrast({\n   *   contrast: 0.25\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   */\n  filters.Contrast = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Contrast.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Contrast',\n\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform float uContrast;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'void main() {\\n' +\n        'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n        'float contrastF = 1.015 * (uContrast + 1.0) / (1.0 * (1.015 - uContrast));\\n' +\n        'color.rgb = contrastF * (color.rgb - 0.5) + 0.5;\\n' +\n        'gl_FragColor = color;\\n' +\n      '}',\n\n    /**\n     * contrast value, range from -1 to 1.\n     * @param {Number} contrast\n     * @default 0\n     */\n    contrast: 0,\n\n    mainParameter: 'contrast',\n\n    /**\n     * Constructor\n     * @memberOf fabric.Image.filters.Contrast.prototype\n     * @param {Object} [options] Options object\n     * @param {Number} [options.contrast=0] Value to contrast the image up (-1...1)\n     */\n\n    /**\n      * Apply the Contrast operation to a Uint8Array representing the pixels of an image.\n      *\n      * @param {Object} options\n      * @param {ImageData} options.imageData The Uint8Array to be filtered.\n      */\n    applyTo2d: function(options) {\n      if (this.contrast === 0) {\n        return;\n      }\n      var imageData = options.imageData, i, len,\n          data = imageData.data, len = data.length,\n          contrast = Math.floor(this.contrast * 255),\n          contrastF = 259 * (contrast + 255) / (255 * (259 - contrast));\n\n      for (i = 0; i < len; i += 4) {\n        data[i] = contrastF * (data[i] - 128) + 128;\n        data[i + 1] = contrastF * (data[i + 1] - 128) + 128;\n        data[i + 2] = contrastF * (data[i + 2] - 128) + 128;\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uContrast: gl.getUniformLocation(program, 'uContrast'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform1f(uniformLocations.uContrast, this.contrast);\n    },\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Contrast} Instance of fabric.Image.filters.Contrast\n   */\n  fabric.Image.filters.Contrast.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Saturate filter class\n   * @class fabric.Image.filters.Saturation\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.Saturation#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Saturation({\n   *   saturation: 1\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   */\n  filters.Saturation = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Saturation.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Saturation',\n\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform float uSaturation;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'void main() {\\n' +\n        'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n        'float rgMax = max(color.r, color.g);\\n' +\n        'float rgbMax = max(rgMax, color.b);\\n' +\n        'color.r += rgbMax != color.r ? (rgbMax - color.r) * uSaturation : 0.00;\\n' +\n        'color.g += rgbMax != color.g ? (rgbMax - color.g) * uSaturation : 0.00;\\n' +\n        'color.b += rgbMax != color.b ? (rgbMax - color.b) * uSaturation : 0.00;\\n' +\n        'gl_FragColor = color;\\n' +\n      '}',\n\n    /**\n     * Saturation value, from -1 to 1.\n     * Increases/decreases the color saturation.\n     * A value of 0 has no effect.\n     *\n     * @param {Number} saturation\n     * @default\n     */\n    saturation: 0,\n\n    mainParameter: 'saturation',\n\n    /**\n     * Constructor\n     * @memberOf fabric.Image.filters.Saturate.prototype\n     * @param {Object} [options] Options object\n     * @param {Number} [options.saturate=0] Value to saturate the image (-1...1)\n     */\n\n    /**\n     * Apply the Saturation operation to a Uint8ClampedArray representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8ClampedArray to be filtered.\n     */\n    applyTo2d: function(options) {\n      if (this.saturation === 0) {\n        return;\n      }\n      var imageData = options.imageData,\n          data = imageData.data, len = data.length,\n          adjust = -this.saturation, i, max;\n\n      for (i = 0; i < len; i += 4) {\n        max = Math.max(data[i], data[i + 1], data[i + 2]);\n        data[i] += max !== data[i] ? (max - data[i]) * adjust : 0;\n        data[i + 1] += max !== data[i + 1] ? (max - data[i + 1]) * adjust : 0;\n        data[i + 2] += max !== data[i + 2] ? (max - data[i + 2]) * adjust : 0;\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uSaturation: gl.getUniformLocation(program, 'uSaturation'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform1f(uniformLocations.uSaturation, -this.saturation);\n    },\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {Function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Saturation} Instance of fabric.Image.filters.Saturate\n   */\n  fabric.Image.filters.Saturation.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Vibrance filter class\n   * @class fabric.Image.filters.Vibrance\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.Vibrance#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Vibrance({\n   *   vibrance: 1\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   */\n  filters.Vibrance = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Vibrance.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Vibrance',\n\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform float uVibrance;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'void main() {\\n' +\n        'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n        'float max = max(color.r, max(color.g, color.b));\\n' +\n        'float avg = (color.r + color.g + color.b) / 3.0;\\n' +\n        'float amt = (abs(max - avg) * 2.0) * uVibrance;\\n' +\n        'color.r += max != color.r ? (max - color.r) * amt : 0.00;\\n' +\n        'color.g += max != color.g ? (max - color.g) * amt : 0.00;\\n' +\n        'color.b += max != color.b ? (max - color.b) * amt : 0.00;\\n' +\n        'gl_FragColor = color;\\n' +\n      '}',\n\n    /**\n     * Vibrance value, from -1 to 1.\n     * Increases/decreases the saturation of more muted colors with less effect on saturated colors.\n     * A value of 0 has no effect.\n     *\n     * @param {Number} vibrance\n     * @default\n     */\n    vibrance: 0,\n\n    mainParameter: 'vibrance',\n\n    /**\n     * Constructor\n     * @memberOf fabric.Image.filters.Vibrance.prototype\n     * @param {Object} [options] Options object\n     * @param {Number} [options.vibrance=0] Vibrance value for the image (between -1 and 1)\n     */\n\n    /**\n     * Apply the Vibrance operation to a Uint8ClampedArray representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8ClampedArray to be filtered.\n     */\n    applyTo2d: function(options) {\n      if (this.vibrance === 0) {\n        return;\n      }\n      var imageData = options.imageData,\n          data = imageData.data, len = data.length,\n          adjust = -this.vibrance, i, max, avg, amt;\n\n      for (i = 0; i < len; i += 4) {\n        max = Math.max(data[i], data[i + 1], data[i + 2]);\n        avg = (data[i] + data[i + 1] + data[i + 2]) / 3;\n        amt = ((Math.abs(max - avg) * 2 / 255) * adjust);\n        data[i] += max !== data[i] ? (max - data[i]) * amt : 0;\n        data[i + 1] += max !== data[i + 1] ? (max - data[i + 1]) * amt : 0;\n        data[i + 2] += max !== data[i + 2] ? (max - data[i + 2]) * amt : 0;\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uVibrance: gl.getUniformLocation(program, 'uVibrance'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform1f(uniformLocations.uVibrance, -this.vibrance);\n    },\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {Function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Vibrance} Instance of fabric.Image.filters.Vibrance\n   */\n  fabric.Image.filters.Vibrance.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Blur filter class\n   * @class fabric.Image.filters.Blur\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.Blur#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Blur({\n   *   blur: 0.5\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   * canvas.renderAll();\n   */\n  filters.Blur = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Blur.prototype */ {\n\n    type: 'Blur',\n\n    /*\n'gl_FragColor = vec4(0.0);',\n'gl_FragColor += texture2D(texture, vTexCoord + -7 * uDelta)*0.0044299121055113265;',\n'gl_FragColor += texture2D(texture, vTexCoord + -6 * uDelta)*0.00895781211794;',\n'gl_FragColor += texture2D(texture, vTexCoord + -5 * uDelta)*0.0215963866053;',\n'gl_FragColor += texture2D(texture, vTexCoord + -4 * uDelta)*0.0443683338718;',\n'gl_FragColor += texture2D(texture, vTexCoord + -3 * uDelta)*0.0776744219933;',\n'gl_FragColor += texture2D(texture, vTexCoord + -2 * uDelta)*0.115876621105;',\n'gl_FragColor += texture2D(texture, vTexCoord + -1 * uDelta)*0.147308056121;',\n'gl_FragColor += texture2D(texture, vTexCoord              )*0.159576912161;',\n'gl_FragColor += texture2D(texture, vTexCoord + 1 * uDelta)*0.147308056121;',\n'gl_FragColor += texture2D(texture, vTexCoord + 2 * uDelta)*0.115876621105;',\n'gl_FragColor += texture2D(texture, vTexCoord + 3 * uDelta)*0.0776744219933;',\n'gl_FragColor += texture2D(texture, vTexCoord + 4 * uDelta)*0.0443683338718;',\n'gl_FragColor += texture2D(texture, vTexCoord + 5 * uDelta)*0.0215963866053;',\n'gl_FragColor += texture2D(texture, vTexCoord + 6 * uDelta)*0.00895781211794;',\n'gl_FragColor += texture2D(texture, vTexCoord + 7 * uDelta)*0.0044299121055113265;',\n*/\n\n    /* eslint-disable max-len */\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform vec2 uDelta;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'const float nSamples = 15.0;\\n' +\n      'vec3 v3offset = vec3(12.9898, 78.233, 151.7182);\\n' +\n      'float random(vec3 scale) {\\n' +\n        /* use the fragment position for a different seed per-pixel */\n        'return fract(sin(dot(gl_FragCoord.xyz, scale)) * 43758.5453);\\n' +\n      '}\\n' +\n      'void main() {\\n' +\n        'vec4 color = vec4(0.0);\\n' +\n        'float total = 0.0;\\n' +\n        'float offset = random(v3offset);\\n' +\n        'for (float t = -nSamples; t <= nSamples; t++) {\\n' +\n          'float percent = (t + offset - 0.5) / nSamples;\\n' +\n          'float weight = 1.0 - abs(percent);\\n' +\n          'color += texture2D(uTexture, vTexCoord + uDelta * percent) * weight;\\n' +\n          'total += weight;\\n' +\n        '}\\n' +\n        'gl_FragColor = color / total;\\n' +\n      '}',\n    /* eslint-enable max-len */\n\n    /**\n     * blur value, in percentage of image dimensions.\n     * specific to keep the image blur constant at different resolutions\n     * range between 0 and 1.\n     * @type Number\n     * @default\n     */\n    blur: 0,\n\n    mainParameter: 'blur',\n\n    applyTo: function(options) {\n      if (options.webgl) {\n        // this aspectRatio is used to give the same blur to vertical and horizontal\n        this.aspectRatio = options.sourceWidth / options.sourceHeight;\n        options.passes++;\n        this._setupFrameBuffer(options);\n        this.horizontal = true;\n        this.applyToWebGL(options);\n        this._swapTextures(options);\n        this._setupFrameBuffer(options);\n        this.horizontal = false;\n        this.applyToWebGL(options);\n        this._swapTextures(options);\n      }\n      else {\n        this.applyTo2d(options);\n      }\n    },\n\n    applyTo2d: function(options) {\n      // paint canvasEl with current image data.\n      //options.ctx.putImageData(options.imageData, 0, 0);\n      options.imageData = this.simpleBlur(options);\n    },\n\n    simpleBlur: function(options) {\n      var resources = options.filterBackend.resources, canvas1, canvas2,\n          width = options.imageData.width,\n          height = options.imageData.height;\n\n      if (!resources.blurLayer1) {\n        resources.blurLayer1 = fabric.util.createCanvasElement();\n        resources.blurLayer2 = fabric.util.createCanvasElement();\n      }\n      canvas1 = resources.blurLayer1;\n      canvas2 = resources.blurLayer2;\n      if (canvas1.width !== width || canvas1.height !== height) {\n        canvas2.width = canvas1.width = width;\n        canvas2.height = canvas1.height = height;\n      }\n      var ctx1 = canvas1.getContext('2d'),\n          ctx2 = canvas2.getContext('2d'),\n          nSamples = 15,\n          random, percent, j, i,\n          blur = this.blur * 0.06 * 0.5;\n\n      // load first canvas\n      ctx1.putImageData(options.imageData, 0, 0);\n      ctx2.clearRect(0, 0, width, height);\n\n      for (i = -nSamples; i <= nSamples; i++) {\n        random = (Math.random() - 0.5) / 4;\n        percent = i / nSamples;\n        j = blur * percent * width + random;\n        ctx2.globalAlpha = 1 - Math.abs(percent);\n        ctx2.drawImage(canvas1, j, random);\n        ctx1.drawImage(canvas2, 0, 0);\n        ctx2.globalAlpha = 1;\n        ctx2.clearRect(0, 0, canvas2.width, canvas2.height);\n      }\n      for (i = -nSamples; i <= nSamples; i++) {\n        random = (Math.random() - 0.5) / 4;\n        percent = i / nSamples;\n        j = blur * percent * height + random;\n        ctx2.globalAlpha = 1 - Math.abs(percent);\n        ctx2.drawImage(canvas1, random, j);\n        ctx1.drawImage(canvas2, 0, 0);\n        ctx2.globalAlpha = 1;\n        ctx2.clearRect(0, 0, canvas2.width, canvas2.height);\n      }\n      options.ctx.drawImage(canvas1, 0, 0);\n      var newImageData = options.ctx.getImageData(0, 0, canvas1.width, canvas1.height);\n      ctx1.globalAlpha = 1;\n      ctx1.clearRect(0, 0, canvas1.width, canvas1.height);\n      return newImageData;\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        delta: gl.getUniformLocation(program, 'uDelta'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      var delta = this.chooseRightDelta();\n      gl.uniform2fv(uniformLocations.delta, delta);\n    },\n\n    /**\n     * choose right value of image percentage to blur with\n     * @returns {Array} a numeric array with delta values\n     */\n    chooseRightDelta: function() {\n      var blurScale = 1, delta = [0, 0], blur;\n      if (this.horizontal) {\n        if (this.aspectRatio > 1) {\n          // image is wide, i want to shrink radius horizontal\n          blurScale = 1 / this.aspectRatio;\n        }\n      }\n      else {\n        if (this.aspectRatio < 1) {\n          // image is tall, i want to shrink radius vertical\n          blurScale = this.aspectRatio;\n        }\n      }\n      blur = blurScale * this.blur * 0.12;\n      if (this.horizontal) {\n        delta[0] = blur;\n      }\n      else {\n        delta[1] = blur;\n      }\n      return delta;\n    },\n  });\n\n  /**\n   * Deserialize a JSON definition of a BlurFilter into a concrete instance.\n   */\n  filters.Blur.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * Gamma filter class\n   * @class fabric.Image.filters.Gamma\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.Gamma#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.Gamma({\n   *   gamma: [1, 0.5, 2.1]\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   */\n  filters.Gamma = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Gamma.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'Gamma',\n\n    fragmentSource: 'precision highp float;\\n' +\n      'uniform sampler2D uTexture;\\n' +\n      'uniform vec3 uGamma;\\n' +\n      'varying vec2 vTexCoord;\\n' +\n      'void main() {\\n' +\n        'vec4 color = texture2D(uTexture, vTexCoord);\\n' +\n        'vec3 correction = (1.0 / uGamma);\\n' +\n        'color.r = pow(color.r, correction.r);\\n' +\n        'color.g = pow(color.g, correction.g);\\n' +\n        'color.b = pow(color.b, correction.b);\\n' +\n        'gl_FragColor = color;\\n' +\n        'gl_FragColor.rgb *= color.a;\\n' +\n      '}',\n\n    /**\n     * Gamma array value, from 0.01 to 2.2.\n     * @param {Array} gamma\n     * @default\n     */\n    gamma: [1, 1, 1],\n\n    /**\n     * Describe the property that is the filter parameter\n     * @param {String} m\n     * @default\n     */\n    mainParameter: 'gamma',\n\n    /**\n     * Constructor\n     * @param {Object} [options] Options object\n     */\n    initialize: function(options) {\n      this.gamma = [1, 1, 1];\n      filters.BaseFilter.prototype.initialize.call(this, options);\n    },\n\n    /**\n     * Apply the Gamma operation to a Uint8Array representing the pixels of an image.\n     *\n     * @param {Object} options\n     * @param {ImageData} options.imageData The Uint8Array to be filtered.\n     */\n    applyTo2d: function(options) {\n      var imageData = options.imageData, data = imageData.data,\n          gamma = this.gamma, len = data.length,\n          rInv = 1 / gamma[0], gInv = 1 / gamma[1],\n          bInv = 1 / gamma[2], i;\n\n      if (!this.rVals) {\n        // eslint-disable-next-line\n        this.rVals = new Uint8Array(256);\n        // eslint-disable-next-line\n        this.gVals = new Uint8Array(256);\n        // eslint-disable-next-line\n        this.bVals = new Uint8Array(256);\n      }\n\n      // This is an optimization - pre-compute a look-up table for each color channel\n      // instead of performing these pow calls for each pixel in the image.\n      for (i = 0, len = 256; i < len; i++) {\n        this.rVals[i] = Math.pow(i / 255, rInv) * 255;\n        this.gVals[i] = Math.pow(i / 255, gInv) * 255;\n        this.bVals[i] = Math.pow(i / 255, bInv) * 255;\n      }\n      for (i = 0, len = data.length; i < len; i += 4) {\n        data[i] = this.rVals[data[i]];\n        data[i + 1] = this.gVals[data[i + 1]];\n        data[i + 2] = this.bVals[data[i + 2]];\n      }\n    },\n\n    /**\n     * Return WebGL uniform locations for this filter's shader.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {WebGLShaderProgram} program This filter's compiled shader program.\n     */\n    getUniformLocations: function(gl, program) {\n      return {\n        uGamma: gl.getUniformLocation(program, 'uGamma'),\n      };\n    },\n\n    /**\n     * Send data from this filter to its shader program's uniforms.\n     *\n     * @param {WebGLRenderingContext} gl The GL canvas context used to compile this filter's shader.\n     * @param {Object} uniformLocations A map of string uniform names to WebGLUniformLocation objects\n     */\n    sendUniformData: function(gl, uniformLocations) {\n      gl.uniform3fv(uniformLocations.uGamma, this.gamma);\n    },\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.Gamma} Instance of fabric.Image.filters.Gamma\n   */\n  fabric.Image.filters.Gamma.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * A container class that knows how to apply a sequence of filters to an input image.\n   */\n  filters.Composed = createClass(filters.BaseFilter, /** @lends fabric.Image.filters.Composed.prototype */ {\n\n    type: 'Composed',\n\n    /**\n     * A non sparse array of filters to apply\n     */\n    subFilters: [],\n\n    /**\n     * Constructor\n     * @param {Object} [options] Options object\n     */\n    initialize: function(options) {\n      this.callSuper('initialize', options);\n      // create a new array instead mutating the prototype with push\n      this.subFilters = this.subFilters.slice(0);\n    },\n\n    /**\n     * Apply this container's filters to the input image provided.\n     *\n     * @param {Object} options\n     * @param {Number} options.passes The number of filters remaining to be applied.\n     */\n    applyTo: function(options) {\n      options.passes += this.subFilters.length - 1;\n      this.subFilters.forEach(function(filter) {\n        filter.applyTo(options);\n      });\n    },\n\n    /**\n     * Serialize this filter into JSON.\n     *\n     * @returns {Object} A JSON representation of this filter.\n     */\n    toObject: function() {\n      return fabric.util.object.extend(this.callSuper('toObject'), {\n        subFilters: this.subFilters.map(function(filter) { return filter.toObject(); }),\n      });\n    },\n\n    isNeutralState: function() {\n      return !this.subFilters.some(function(filter) { return !filter.isNeutralState(); });\n    }\n  });\n\n  /**\n   * Deserialize a JSON definition of a ComposedFilter into a concrete instance.\n   */\n  fabric.Image.filters.Composed.fromObject = function(object, callback) {\n    var filters = object.subFilters || [],\n        subFilters = filters.map(function(filter) {\n          return new fabric.Image.filters[filter.type](filter);\n        }),\n        instance = new fabric.Image.filters.Composed({ subFilters: subFilters });\n    callback && callback(instance);\n    return instance;\n  };\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric  = global.fabric || (global.fabric = { }),\n      filters = fabric.Image.filters,\n      createClass = fabric.util.createClass;\n\n  /**\n   * HueRotation filter class\n   * @class fabric.Image.filters.HueRotation\n   * @memberOf fabric.Image.filters\n   * @extends fabric.Image.filters.BaseFilter\n   * @see {@link fabric.Image.filters.HueRotation#initialize} for constructor definition\n   * @see {@link http://fabricjs.com/image-filters|ImageFilters demo}\n   * @example\n   * var filter = new fabric.Image.filters.HueRotation({\n   *   rotation: -0.5\n   * });\n   * object.filters.push(filter);\n   * object.applyFilters();\n   */\n  filters.HueRotation = createClass(filters.ColorMatrix, /** @lends fabric.Image.filters.HueRotation.prototype */ {\n\n    /**\n     * Filter type\n     * @param {String} type\n     * @default\n     */\n    type: 'HueRotation',\n\n    /**\n     * HueRotation value, from -1 to 1.\n     * the unit is radians\n     * @param {Number} myParameter\n     * @default\n     */\n    rotation: 0,\n\n    /**\n     * Describe the property that is the filter parameter\n     * @param {String} m\n     * @default\n     */\n    mainParameter: 'rotation',\n\n    calculateMatrix: function() {\n      var rad = this.rotation * Math.PI, cos = fabric.util.cos(rad), sin = fabric.util.sin(rad),\n          aThird = 1 / 3, aThirdSqtSin = Math.sqrt(aThird) * sin, OneMinusCos = 1 - cos;\n      this.matrix = [\n        1, 0, 0, 0, 0,\n        0, 1, 0, 0, 0,\n        0, 0, 1, 0, 0,\n        0, 0, 0, 1, 0\n      ];\n      this.matrix[0] = cos + OneMinusCos / 3;\n      this.matrix[1] = aThird * OneMinusCos - aThirdSqtSin;\n      this.matrix[2] = aThird * OneMinusCos + aThirdSqtSin;\n      this.matrix[5] = aThird * OneMinusCos + aThirdSqtSin;\n      this.matrix[6] = cos + aThird * OneMinusCos;\n      this.matrix[7] = aThird * OneMinusCos - aThirdSqtSin;\n      this.matrix[10] = aThird * OneMinusCos - aThirdSqtSin;\n      this.matrix[11] = aThird * OneMinusCos + aThirdSqtSin;\n      this.matrix[12] = cos + aThird * OneMinusCos;\n    },\n\n    /**\n     * HueRotation isNeutralState implementation\n     * Used only in image applyFilters to discard filters that will not have an effect\n     * on the image\n     * @param {Object} options\n     **/\n    isNeutralState: function(options) {\n      this.calculateMatrix();\n      return filters.BaseFilter.prototype.isNeutralState.call(this, options);\n    },\n\n    /**\n     * Apply this filter to the input image data provided.\n     *\n     * Determines whether to use WebGL or Canvas2D based on the options.webgl flag.\n     *\n     * @param {Object} options\n     * @param {Number} options.passes The number of filters remaining to be executed\n     * @param {Boolean} options.webgl Whether to use webgl to render the filter.\n     * @param {WebGLTexture} options.sourceTexture The texture setup as the source to be filtered.\n     * @param {WebGLTexture} options.targetTexture The texture where filtered output should be drawn.\n     * @param {WebGLRenderingContext} options.context The GL context used for rendering.\n     * @param {Object} options.programCache A map of compiled shader programs, keyed by filter type.\n     */\n    applyTo: function(options) {\n      this.calculateMatrix();\n      filters.BaseFilter.prototype.applyTo.call(this, options);\n    },\n\n  });\n\n  /**\n   * Returns filter instance from an object representation\n   * @static\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] to be invoked after filter creation\n   * @return {fabric.Image.filters.HueRotation} Instance of fabric.Image.filters.HueRotation\n   */\n  fabric.Image.filters.HueRotation.fromObject = fabric.Image.filters.BaseFilter.fromObject;\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function(global) {\n\n  'use strict';\n\n  var fabric = global.fabric || (global.fabric = { }),\n      clone = fabric.util.object.clone;\n\n  if (fabric.Text) {\n    fabric.warn('fabric.Text is already defined');\n    return;\n  }\n\n  var additionalProps =\n    ('fontFamily fontWeight fontSize text underline overline linethrough' +\n    ' textAlign fontStyle lineHeight textBackgroundColor charSpacing styles' +\n    ' direction path pathStartOffset pathSide pathAlign').split(' ');\n\n  /**\n   * Text class\n   * @class fabric.Text\n   * @extends fabric.Object\n   * @return {fabric.Text} thisArg\n   * @tutorial {@link http://fabricjs.com/fabric-intro-part-2#text}\n   * @see {@link fabric.Text#initialize} for constructor definition\n   */\n  fabric.Text = fabric.util.createClass(fabric.Object, /** @lends fabric.Text.prototype */ {\n\n    /**\n     * Properties which when set cause object to change dimensions\n     * @type Array\n     * @private\n     */\n    _dimensionAffectingProps: [\n      'fontSize',\n      'fontWeight',\n      'fontFamily',\n      'fontStyle',\n      'lineHeight',\n      'text',\n      'charSpacing',\n      'textAlign',\n      'styles',\n      'path',\n      'pathStartOffset',\n      'pathSide',\n      'pathAlign'\n    ],\n\n    /**\n     * @private\n     */\n    _reNewline: /\\r?\\n/,\n\n    /**\n     * Use this regular expression to filter for whitespaces that is not a new line.\n     * Mostly used when text is 'justify' aligned.\n     * @private\n     */\n    _reSpacesAndTabs: /[ \\t\\r]/g,\n\n    /**\n     * Use this regular expression to filter for whitespace that is not a new line.\n     * Mostly used when text is 'justify' aligned.\n     * @private\n     */\n    _reSpaceAndTab: /[ \\t\\r]/,\n\n    /**\n     * Use this regular expression to filter consecutive groups of non spaces.\n     * Mostly used when text is 'justify' aligned.\n     * @private\n     */\n    _reWords: /\\S+/g,\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type:                 'text',\n\n    /**\n     * Font size (in pixels)\n     * @type Number\n     * @default\n     */\n    fontSize:             40,\n\n    /**\n     * Font weight (e.g. bold, normal, 400, 600, 800)\n     * @type {(Number|String)}\n     * @default\n     */\n    fontWeight:           'normal',\n\n    /**\n     * Font family\n     * @type String\n     * @default\n     */\n    fontFamily:           'Times New Roman',\n\n    /**\n     * Text decoration underline.\n     * @type Boolean\n     * @default\n     */\n    underline:       false,\n\n    /**\n     * Text decoration overline.\n     * @type Boolean\n     * @default\n     */\n    overline:       false,\n\n    /**\n     * Text decoration linethrough.\n     * @type Boolean\n     * @default\n     */\n    linethrough:       false,\n\n    /**\n     * Text alignment. Possible values: \"left\", \"center\", \"right\", \"justify\",\n     * \"justify-left\", \"justify-center\" or \"justify-right\".\n     * @type String\n     * @default\n     */\n    textAlign:            'left',\n\n    /**\n     * Font style . Possible values: \"\", \"normal\", \"italic\" or \"oblique\".\n     * @type String\n     * @default\n     */\n    fontStyle:            'normal',\n\n    /**\n     * Line height\n     * @type Number\n     * @default\n     */\n    lineHeight:           1.16,\n\n    /**\n     * Superscript schema object (minimum overlap)\n     * @type {Object}\n     * @default\n     */\n    superscript: {\n      size:      0.60, // fontSize factor\n      baseline: -0.35  // baseline-shift factor (upwards)\n    },\n\n    /**\n     * Subscript schema object (minimum overlap)\n     * @type {Object}\n     * @default\n     */\n    subscript: {\n      size:      0.60, // fontSize factor\n      baseline:  0.11  // baseline-shift factor (downwards)\n    },\n\n    /**\n     * Background color of text lines\n     * @type String\n     * @default\n     */\n    textBackgroundColor:  '',\n\n    /**\n     * List of properties to consider when checking if\n     * state of an object is changed ({@link fabric.Object#hasStateChanged})\n     * as well as for history (undo/redo) purposes\n     * @type Array\n     */\n    stateProperties: fabric.Object.prototype.stateProperties.concat(additionalProps),\n\n    /**\n     * List of properties to consider when checking if cache needs refresh\n     * @type Array\n     */\n    cacheProperties: fabric.Object.prototype.cacheProperties.concat(additionalProps),\n\n    /**\n     * When defined, an object is rendered via stroke and this property specifies its color.\n     * <b>Backwards incompatibility note:</b> This property was named \"strokeStyle\" until v1.1.6\n     * @type String\n     * @default\n     */\n    stroke:               null,\n\n    /**\n     * Shadow object representing shadow of this shape.\n     * <b>Backwards incompatibility note:</b> This property was named \"textShadow\" (String) until v1.2.11\n     * @type fabric.Shadow\n     * @default\n     */\n    shadow:               null,\n\n    /**\n     * fabric.Path that the text should follow.\n     * since 4.6.0 the path will be drawn automatically.\n     * if you want to make the path visible, give it a stroke and strokeWidth or fill value\n     * if you want it to be hidden, assign visible = false to the path.\n     * This feature is in BETA, and SVG import/export is not yet supported.\n     * @type fabric.Path\n     * @example\n     * var textPath = new fabric.Text('Text on a path', {\n     *     top: 150,\n     *     left: 150,\n     *     textAlign: 'center',\n     *     charSpacing: -50,\n     *     path: new fabric.Path('M 0 0 C 50 -100 150 -100 200 0', {\n     *         strokeWidth: 1,\n     *         visible: false\n     *     }),\n     *     pathSide: 'left',\n     *     pathStartOffset: 0\n     * });\n     * @default\n     */\n    path:               null,\n\n    /**\n     * Offset amount for text path starting position\n     * Only used when text has a path\n     * @type Number\n     * @default\n     */\n    pathStartOffset:               0,\n\n    /**\n     * Which side of the path the text should be drawn on.\n     * Only used when text has a path\n     * @type {String} 'left|right'\n     * @default\n     */\n    pathSide:               'left',\n\n    /**\n     * How text is aligned to the path. This property determines\n     * the perpendicular position of each character relative to the path.\n     * (one of \"baseline\", \"center\", \"ascender\", \"descender\")\n     * This feature is in BETA, and its behavior may change\n     * @type String\n     * @default\n     */\n    pathAlign:               'baseline',\n\n    /**\n     * @private\n     */\n    _fontSizeFraction: 0.222,\n\n    /**\n     * @private\n     */\n    offsets: {\n      underline: 0.10,\n      linethrough: -0.315,\n      overline: -0.88\n    },\n\n    /**\n     * Text Line proportion to font Size (in pixels)\n     * @type Number\n     * @default\n     */\n    _fontSizeMult:             1.13,\n\n    /**\n     * additional space between characters\n     * expressed in thousands of em unit\n     * @type Number\n     * @default\n     */\n    charSpacing:             0,\n\n    /**\n     * Object containing character styles - top-level properties -> line numbers,\n     * 2nd-level properties - character numbers\n     * @type Object\n     * @default\n     */\n    styles: null,\n\n    /**\n     * Reference to a context to measure text char or couple of chars\n     * the cacheContext of the canvas will be used or a freshly created one if the object is not on canvas\n     * once created it will be referenced on fabric._measuringContext to avoid creating a canvas for every\n     * text object created.\n     * @type {CanvasRenderingContext2D}\n     * @default\n     */\n    _measuringContext: null,\n\n    /**\n     * Baseline shift, styles only, keep at 0 for the main text object\n     * @type {Number}\n     * @default\n     */\n    deltaY: 0,\n\n    /**\n     * WARNING: EXPERIMENTAL. NOT SUPPORTED YET\n     * determine the direction of the text.\n     * This has to be set manually together with textAlign and originX for proper\n     * experience.\n     * some interesting link for the future\n     * https://www.w3.org/International/questions/qa-bidi-unicode-controls\n     * @since 4.5.0\n     * @type {String} 'ltr|rtl'\n     * @default\n     */\n    direction: 'ltr',\n\n    /**\n     * Array of properties that define a style unit (of 'styles').\n     * @type {Array}\n     * @default\n     */\n    _styleProperties: [\n      'stroke',\n      'strokeWidth',\n      'fill',\n      'fontFamily',\n      'fontSize',\n      'fontWeight',\n      'fontStyle',\n      'underline',\n      'overline',\n      'linethrough',\n      'deltaY',\n      'textBackgroundColor',\n    ],\n\n    /**\n     * contains characters bounding boxes\n     */\n    __charBounds: [],\n\n    /**\n     * use this size when measuring text. To avoid IE11 rounding errors\n     * @type {Number}\n     * @default\n     * @readonly\n     * @private\n     */\n    CACHE_FONT_SIZE: 400,\n\n    /**\n     * contains the min text width to avoid getting 0\n     * @type {Number}\n     * @default\n     */\n    MIN_TEXT_WIDTH: 2,\n\n    /**\n     * Constructor\n     * @param {String} text Text string\n     * @param {Object} [options] Options object\n     * @return {fabric.Text} thisArg\n     */\n    initialize: function(text, options) {\n      this.styles = options ? (options.styles || { }) : { };\n      this.text = text;\n      this.__skipDimension = true;\n      this.callSuper('initialize', options);\n      if (this.path) {\n        this.setPathInfo();\n      }\n      this.__skipDimension = false;\n      this.initDimensions();\n      this.setCoords();\n      this.setupState({ propertySet: '_dimensionAffectingProps' });\n    },\n\n    /**\n     * If text has a path, it will add the extra information needed\n     * for path and text calculations\n     * @return {fabric.Text} thisArg\n     */\n    setPathInfo: function() {\n      var path = this.path;\n      if (path) {\n        path.segmentsInfo = fabric.util.getPathSegmentsInfo(path.path);\n      }\n    },\n\n    /**\n     * Return a context for measurement of text string.\n     * if created it gets stored for reuse\n     * this is for internal use, please do not use it\n     * @private\n     * @param {String} text Text string\n     * @param {Object} [options] Options object\n     * @return {fabric.Text} thisArg\n     */\n    getMeasuringContext: function() {\n      // if we did not return we have to measure something.\n      if (!fabric._measuringContext) {\n        fabric._measuringContext = this.canvas && this.canvas.contextCache ||\n          fabric.util.createCanvasElement().getContext('2d');\n      }\n      return fabric._measuringContext;\n    },\n\n    /**\n     * @private\n     * Divides text into lines of text and lines of graphemes.\n     */\n    _splitText: function() {\n      var newLines = this._splitTextIntoLines(this.text);\n      this.textLines = newLines.lines;\n      this._textLines = newLines.graphemeLines;\n      this._unwrappedTextLines = newLines._unwrappedLines;\n      this._text = newLines.graphemeText;\n      return newLines;\n    },\n\n    /**\n     * Initialize or update text dimensions.\n     * Updates this.width and this.height with the proper values.\n     * Does not return dimensions.\n     */\n    initDimensions: function() {\n      if (this.__skipDimension) {\n        return;\n      }\n      this._splitText();\n      this._clearCache();\n      if (this.path) {\n        this.width = this.path.width;\n        this.height = this.path.height;\n      }\n      else {\n        this.width = this.calcTextWidth() || this.cursorWidth || this.MIN_TEXT_WIDTH;\n        this.height = this.calcTextHeight();\n      }\n      if (this.textAlign.indexOf('justify') !== -1) {\n        // once text is measured we need to make space fatter to make justified text.\n        this.enlargeSpaces();\n      }\n      this.saveState({ propertySet: '_dimensionAffectingProps' });\n    },\n\n    /**\n     * Enlarge space boxes and shift the others\n     */\n    enlargeSpaces: function() {\n      var diffSpace, currentLineWidth, numberOfSpaces, accumulatedSpace, line, charBound, spaces;\n      for (var i = 0, len = this._textLines.length; i < len; i++) {\n        if (this.textAlign !== 'justify' && (i === len - 1 || this.isEndOfWrapping(i))) {\n          continue;\n        }\n        accumulatedSpace = 0;\n        line = this._textLines[i];\n        currentLineWidth = this.getLineWidth(i);\n        if (currentLineWidth < this.width && (spaces = this.textLines[i].match(this._reSpacesAndTabs))) {\n          numberOfSpaces = spaces.length;\n          diffSpace = (this.width - currentLineWidth) / numberOfSpaces;\n          for (var j = 0, jlen = line.length; j <= jlen; j++) {\n            charBound = this.__charBounds[i][j];\n            if (this._reSpaceAndTab.test(line[j])) {\n              charBound.width += diffSpace;\n              charBound.kernedWidth += diffSpace;\n              charBound.left += accumulatedSpace;\n              accumulatedSpace += diffSpace;\n            }\n            else {\n              charBound.left += accumulatedSpace;\n            }\n          }\n        }\n      }\n    },\n\n    /**\n     * Detect if the text line is ended with an hard break\n     * text and itext do not have wrapping, return false\n     * @return {Boolean}\n     */\n    isEndOfWrapping: function(lineIndex) {\n      return lineIndex === this._textLines.length - 1;\n    },\n\n    /**\n     * Detect if a line has a linebreak and so we need to account for it when moving\n     * and counting style.\n     * It return always for text and Itext.\n     * @return Number\n     */\n    missingNewlineOffset: function() {\n      return 1;\n    },\n\n    /**\n     * Returns string representation of an instance\n     * @return {String} String representation of text object\n     */\n    toString: function() {\n      return '#<fabric.Text (' + this.complexity() +\n        '): { \"text\": \"' + this.text + '\", \"fontFamily\": \"' + this.fontFamily + '\" }>';\n    },\n\n    /**\n     * Return the dimension and the zoom level needed to create a cache canvas\n     * big enough to host the object to be cached.\n     * @private\n     * @param {Object} dim.x width of object to be cached\n     * @param {Object} dim.y height of object to be cached\n     * @return {Object}.width width of canvas\n     * @return {Object}.height height of canvas\n     * @return {Object}.zoomX zoomX zoom value to unscale the canvas before drawing cache\n     * @return {Object}.zoomY zoomY zoom value to unscale the canvas before drawing cache\n     */\n    _getCacheCanvasDimensions: function() {\n      var dims = this.callSuper('_getCacheCanvasDimensions');\n      var fontSize = this.fontSize;\n      dims.width += fontSize * dims.zoomX;\n      dims.height += fontSize * dims.zoomY;\n      return dims;\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _render: function(ctx) {\n      var path = this.path;\n      path && !path.isNotVisible() && path._render(ctx);\n      this._setTextStyles(ctx);\n      this._renderTextLinesBackground(ctx);\n      this._renderTextDecoration(ctx, 'underline');\n      this._renderText(ctx);\n      this._renderTextDecoration(ctx, 'overline');\n      this._renderTextDecoration(ctx, 'linethrough');\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderText: function(ctx) {\n      if (this.paintFirst === 'stroke') {\n        this._renderTextStroke(ctx);\n        this._renderTextFill(ctx);\n      }\n      else {\n        this._renderTextFill(ctx);\n        this._renderTextStroke(ctx);\n      }\n    },\n\n    /**\n     * Set the font parameter of the context with the object properties or with charStyle\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {Object} [charStyle] object with font style properties\n     * @param {String} [charStyle.fontFamily] Font Family\n     * @param {Number} [charStyle.fontSize] Font size in pixels. ( without px suffix )\n     * @param {String} [charStyle.fontWeight] Font weight\n     * @param {String} [charStyle.fontStyle] Font style (italic|normal)\n     */\n    _setTextStyles: function(ctx, charStyle, forMeasuring) {\n      ctx.textBaseline = 'alphabetic';\n      if (this.path) {\n        switch (this.pathAlign) {\n          case 'center':\n            ctx.textBaseline = 'middle';\n            break;\n          case 'ascender':\n            ctx.textBaseline = 'top';\n            break;\n          case 'descender':\n            ctx.textBaseline = 'bottom';\n            break;\n        }\n      }\n      ctx.font = this._getFontDeclaration(charStyle, forMeasuring);\n    },\n\n    /**\n     * calculate and return the text Width measuring each line.\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @return {Number} Maximum width of fabric.Text object\n     */\n    calcTextWidth: function() {\n      var maxWidth = this.getLineWidth(0);\n\n      for (var i = 1, len = this._textLines.length; i < len; i++) {\n        var currentLineWidth = this.getLineWidth(i);\n        if (currentLineWidth > maxWidth) {\n          maxWidth = currentLineWidth;\n        }\n      }\n      return maxWidth;\n    },\n\n    /**\n     * @private\n     * @param {String} method Method name (\"fillText\" or \"strokeText\")\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {String} line Text to render\n     * @param {Number} left Left position of text\n     * @param {Number} top Top position of text\n     * @param {Number} lineIndex Index of a line in a text\n     */\n    _renderTextLine: function(method, ctx, line, left, top, lineIndex) {\n      this._renderChars(method, ctx, line, left, top, lineIndex);\n    },\n\n    /**\n     * Renders the text background for lines, taking care of style\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderTextLinesBackground: function(ctx) {\n      if (!this.textBackgroundColor && !this.styleHas('textBackgroundColor')) {\n        return;\n      }\n      var heightOfLine,\n          lineLeftOffset, originalFill = ctx.fillStyle,\n          line, lastColor,\n          leftOffset = this._getLeftOffset(),\n          lineTopOffset = this._getTopOffset(),\n          boxStart = 0, boxWidth = 0, charBox, currentColor, path = this.path,\n          drawStart;\n\n      for (var i = 0, len = this._textLines.length; i < len; i++) {\n        heightOfLine = this.getHeightOfLine(i);\n        if (!this.textBackgroundColor && !this.styleHas('textBackgroundColor', i)) {\n          lineTopOffset += heightOfLine;\n          continue;\n        }\n        line = this._textLines[i];\n        lineLeftOffset = this._getLineLeftOffset(i);\n        boxWidth = 0;\n        boxStart = 0;\n        lastColor = this.getValueOfPropertyAt(i, 0, 'textBackgroundColor');\n        for (var j = 0, jlen = line.length; j < jlen; j++) {\n          charBox = this.__charBounds[i][j];\n          currentColor = this.getValueOfPropertyAt(i, j, 'textBackgroundColor');\n          if (path) {\n            ctx.save();\n            ctx.translate(charBox.renderLeft, charBox.renderTop);\n            ctx.rotate(charBox.angle);\n            ctx.fillStyle = currentColor;\n            currentColor && ctx.fillRect(\n              -charBox.width / 2,\n              -heightOfLine / this.lineHeight * (1 - this._fontSizeFraction),\n              charBox.width,\n              heightOfLine / this.lineHeight\n            );\n            ctx.restore();\n          }\n          else if (currentColor !== lastColor) {\n            drawStart = leftOffset + lineLeftOffset + boxStart;\n            if (this.direction === 'rtl') {\n              drawStart = this.width - drawStart - boxWidth;\n            }\n            ctx.fillStyle = lastColor;\n            lastColor && ctx.fillRect(\n              drawStart,\n              lineTopOffset,\n              boxWidth,\n              heightOfLine / this.lineHeight\n            );\n            boxStart = charBox.left;\n            boxWidth = charBox.width;\n            lastColor = currentColor;\n          }\n          else {\n            boxWidth += charBox.kernedWidth;\n          }\n        }\n        if (currentColor && !path) {\n          drawStart = leftOffset + lineLeftOffset + boxStart;\n          if (this.direction === 'rtl') {\n            drawStart = this.width - drawStart - boxWidth;\n          }\n          ctx.fillStyle = currentColor;\n          ctx.fillRect(\n            drawStart,\n            lineTopOffset,\n            boxWidth,\n            heightOfLine / this.lineHeight\n          );\n        }\n        lineTopOffset += heightOfLine;\n      }\n      ctx.fillStyle = originalFill;\n      // if there is text background color no\n      // other shadows should be casted\n      this._removeShadow(ctx);\n    },\n\n    /**\n     * @private\n     * @param {Object} decl style declaration for cache\n     * @param {String} decl.fontFamily fontFamily\n     * @param {String} decl.fontStyle fontStyle\n     * @param {String} decl.fontWeight fontWeight\n     * @return {Object} reference to cache\n     */\n    getFontCache: function(decl) {\n      var fontFamily = decl.fontFamily.toLowerCase();\n      if (!fabric.charWidthsCache[fontFamily]) {\n        fabric.charWidthsCache[fontFamily] = { };\n      }\n      var cache = fabric.charWidthsCache[fontFamily],\n          cacheProp = decl.fontStyle.toLowerCase() + '_' + (decl.fontWeight + '').toLowerCase();\n      if (!cache[cacheProp]) {\n        cache[cacheProp] = { };\n      }\n      return cache[cacheProp];\n    },\n\n    /**\n     * measure and return the width of a single character.\n     * possibly overridden to accommodate different measure logic or\n     * to hook some external lib for character measurement\n     * @private\n     * @param {String} _char, char to be measured\n     * @param {Object} charStyle style of char to be measured\n     * @param {String} [previousChar] previous char\n     * @param {Object} [prevCharStyle] style of previous char\n     */\n    _measureChar: function(_char, charStyle, previousChar, prevCharStyle) {\n      // first i try to return from cache\n      var fontCache = this.getFontCache(charStyle), fontDeclaration = this._getFontDeclaration(charStyle),\n          previousFontDeclaration = this._getFontDeclaration(prevCharStyle), couple = previousChar + _char,\n          stylesAreEqual = fontDeclaration === previousFontDeclaration, width, coupleWidth, previousWidth,\n          fontMultiplier = charStyle.fontSize / this.CACHE_FONT_SIZE, kernedWidth;\n\n      if (previousChar && fontCache[previousChar] !== undefined) {\n        previousWidth = fontCache[previousChar];\n      }\n      if (fontCache[_char] !== undefined) {\n        kernedWidth = width = fontCache[_char];\n      }\n      if (stylesAreEqual && fontCache[couple] !== undefined) {\n        coupleWidth = fontCache[couple];\n        kernedWidth = coupleWidth - previousWidth;\n      }\n      if (width === undefined || previousWidth === undefined || coupleWidth === undefined) {\n        var ctx = this.getMeasuringContext();\n        // send a TRUE to specify measuring font size CACHE_FONT_SIZE\n        this._setTextStyles(ctx, charStyle, true);\n      }\n      if (width === undefined) {\n        kernedWidth = width = ctx.measureText(_char).width;\n        fontCache[_char] = width;\n      }\n      if (previousWidth === undefined && stylesAreEqual && previousChar) {\n        previousWidth = ctx.measureText(previousChar).width;\n        fontCache[previousChar] = previousWidth;\n      }\n      if (stylesAreEqual && coupleWidth === undefined) {\n        // we can measure the kerning couple and subtract the width of the previous character\n        coupleWidth = ctx.measureText(couple).width;\n        fontCache[couple] = coupleWidth;\n        kernedWidth = coupleWidth - previousWidth;\n      }\n      return { width: width * fontMultiplier, kernedWidth: kernedWidth * fontMultiplier };\n    },\n\n    /**\n     * Computes height of character at given position\n     * @param {Number} line the line index number\n     * @param {Number} _char the character index number\n     * @return {Number} fontSize of the character\n     */\n    getHeightOfChar: function(line, _char) {\n      return this.getValueOfPropertyAt(line, _char, 'fontSize');\n    },\n\n    /**\n     * measure a text line measuring all characters.\n     * @param {Number} lineIndex line number\n     * @return {Number} Line width\n     */\n    measureLine: function(lineIndex) {\n      var lineInfo = this._measureLine(lineIndex);\n      if (this.charSpacing !== 0) {\n        lineInfo.width -= this._getWidthOfCharSpacing();\n      }\n      if (lineInfo.width < 0) {\n        lineInfo.width = 0;\n      }\n      return lineInfo;\n    },\n\n    /**\n     * measure every grapheme of a line, populating __charBounds\n     * @param {Number} lineIndex\n     * @return {Object} object.width total width of characters\n     * @return {Object} object.widthOfSpaces length of chars that match this._reSpacesAndTabs\n     */\n    _measureLine: function(lineIndex) {\n      var width = 0, i, grapheme, line = this._textLines[lineIndex], prevGrapheme,\n          graphemeInfo, numOfSpaces = 0, lineBounds = new Array(line.length),\n          positionInPath = 0, startingPoint, totalPathLength, path = this.path,\n          reverse = this.pathSide === 'right';\n\n      this.__charBounds[lineIndex] = lineBounds;\n      for (i = 0; i < line.length; i++) {\n        grapheme = line[i];\n        graphemeInfo = this._getGraphemeBox(grapheme, lineIndex, i, prevGrapheme);\n        lineBounds[i] = graphemeInfo;\n        width += graphemeInfo.kernedWidth;\n        prevGrapheme = grapheme;\n      }\n      // this latest bound box represent the last character of the line\n      // to simplify cursor handling in interactive mode.\n      lineBounds[i] = {\n        left: graphemeInfo ? graphemeInfo.left + graphemeInfo.width : 0,\n        width: 0,\n        kernedWidth: 0,\n        height: this.fontSize\n      };\n      if (path) {\n        totalPathLength = path.segmentsInfo[path.segmentsInfo.length - 1].length;\n        startingPoint = fabric.util.getPointOnPath(path.path, 0, path.segmentsInfo);\n        startingPoint.x += path.pathOffset.x;\n        startingPoint.y += path.pathOffset.y;\n        switch (this.textAlign) {\n          case 'left':\n            positionInPath = reverse ? (totalPathLength - width) : 0;\n            break;\n          case 'center':\n            positionInPath = (totalPathLength - width) / 2;\n            break;\n          case 'right':\n            positionInPath = reverse ? 0 : (totalPathLength - width);\n            break;\n          //todo - add support for justify\n        }\n        positionInPath += this.pathStartOffset * (reverse ? -1 : 1);\n        for (i = reverse ? line.length - 1 : 0;\n          reverse ? i >= 0 : i < line.length;\n          reverse ? i-- : i++) {\n          graphemeInfo = lineBounds[i];\n          if (positionInPath > totalPathLength) {\n            positionInPath %= totalPathLength;\n          }\n          else if (positionInPath < 0) {\n            positionInPath += totalPathLength;\n          }\n          // it would probably much faster to send all the grapheme position for a line\n          // and calculate path position/angle at once.\n          this._setGraphemeOnPath(positionInPath, graphemeInfo, startingPoint);\n          positionInPath += graphemeInfo.kernedWidth;\n        }\n      }\n      return { width: width, numOfSpaces: numOfSpaces };\n    },\n\n    /**\n     * Calculate the angle  and the left,top position of the char that follow a path.\n     * It appends it to graphemeInfo to be reused later at rendering\n     * @private\n     * @param {Number} positionInPath to be measured\n     * @param {Object} graphemeInfo current grapheme box information\n     * @param {Object} startingPoint position of the point\n     */\n    _setGraphemeOnPath: function(positionInPath, graphemeInfo, startingPoint) {\n      var centerPosition = positionInPath + graphemeInfo.kernedWidth / 2,\n          path = this.path;\n\n      // we are at currentPositionOnPath. we want to know what point on the path is.\n      var info = fabric.util.getPointOnPath(path.path, centerPosition, path.segmentsInfo);\n      graphemeInfo.renderLeft = info.x - startingPoint.x;\n      graphemeInfo.renderTop = info.y - startingPoint.y;\n      graphemeInfo.angle = info.angle + (this.pathSide ===  'right' ? Math.PI : 0);\n    },\n\n    /**\n     * Measure and return the info of a single grapheme.\n     * needs the the info of previous graphemes already filled\n     * @private\n     * @param {String} grapheme to be measured\n     * @param {Number} lineIndex index of the line where the char is\n     * @param {Number} charIndex position in the line\n     * @param {String} [prevGrapheme] character preceding the one to be measured\n     */\n    _getGraphemeBox: function(grapheme, lineIndex, charIndex, prevGrapheme, skipLeft) {\n      var style = this.getCompleteStyleDeclaration(lineIndex, charIndex),\n          prevStyle = prevGrapheme ? this.getCompleteStyleDeclaration(lineIndex, charIndex - 1) : { },\n          info = this._measureChar(grapheme, style, prevGrapheme, prevStyle),\n          kernedWidth = info.kernedWidth,\n          width = info.width, charSpacing;\n\n      if (this.charSpacing !== 0) {\n        charSpacing = this._getWidthOfCharSpacing();\n        width += charSpacing;\n        kernedWidth += charSpacing;\n      }\n\n      var box = {\n        width: width,\n        left: 0,\n        height: style.fontSize,\n        kernedWidth: kernedWidth,\n        deltaY: style.deltaY,\n      };\n      if (charIndex > 0 && !skipLeft) {\n        var previousBox = this.__charBounds[lineIndex][charIndex - 1];\n        box.left = previousBox.left + previousBox.width + info.kernedWidth - info.width;\n      }\n      return box;\n    },\n\n    /**\n     * Calculate height of line at 'lineIndex'\n     * @param {Number} lineIndex index of line to calculate\n     * @return {Number}\n     */\n    getHeightOfLine: function(lineIndex) {\n      if (this.__lineHeights[lineIndex]) {\n        return this.__lineHeights[lineIndex];\n      }\n\n      var line = this._textLines[lineIndex],\n          // char 0 is measured before the line cycle because it nneds to char\n          // emptylines\n          maxHeight = this.getHeightOfChar(lineIndex, 0);\n      for (var i = 1, len = line.length; i < len; i++) {\n        maxHeight = Math.max(this.getHeightOfChar(lineIndex, i), maxHeight);\n      }\n\n      return this.__lineHeights[lineIndex] = maxHeight * this.lineHeight * this._fontSizeMult;\n    },\n\n    /**\n     * Calculate text box height\n     */\n    calcTextHeight: function() {\n      var lineHeight, height = 0;\n      for (var i = 0, len = this._textLines.length; i < len; i++) {\n        lineHeight = this.getHeightOfLine(i);\n        height += (i === len - 1 ? lineHeight / this.lineHeight : lineHeight);\n      }\n      return height;\n    },\n\n    /**\n     * @private\n     * @return {Number} Left offset\n     */\n    _getLeftOffset: function() {\n      return this.direction === 'ltr' ? -this.width / 2 : this.width / 2;\n    },\n\n    /**\n     * @private\n     * @return {Number} Top offset\n     */\n    _getTopOffset: function() {\n      return -this.height / 2;\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {String} method Method name (\"fillText\" or \"strokeText\")\n     */\n    _renderTextCommon: function(ctx, method) {\n      ctx.save();\n      var lineHeights = 0, left = this._getLeftOffset(), top = this._getTopOffset();\n      for (var i = 0, len = this._textLines.length; i < len; i++) {\n        var heightOfLine = this.getHeightOfLine(i),\n            maxHeight = heightOfLine / this.lineHeight,\n            leftOffset = this._getLineLeftOffset(i);\n        this._renderTextLine(\n          method,\n          ctx,\n          this._textLines[i],\n          left + leftOffset,\n          top + lineHeights + maxHeight,\n          i\n        );\n        lineHeights += heightOfLine;\n      }\n      ctx.restore();\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderTextFill: function(ctx) {\n      if (!this.fill && !this.styleHas('fill')) {\n        return;\n      }\n\n      this._renderTextCommon(ctx, 'fillText');\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderTextStroke: function(ctx) {\n      if ((!this.stroke || this.strokeWidth === 0) && this.isEmptyStyles()) {\n        return;\n      }\n\n      if (this.shadow && !this.shadow.affectStroke) {\n        this._removeShadow(ctx);\n      }\n\n      ctx.save();\n      this._setLineDash(ctx, this.strokeDashArray);\n      ctx.beginPath();\n      this._renderTextCommon(ctx, 'strokeText');\n      ctx.closePath();\n      ctx.restore();\n    },\n\n    /**\n     * @private\n     * @param {String} method fillText or strokeText.\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {Array} line Content of the line, splitted in an array by grapheme\n     * @param {Number} left\n     * @param {Number} top\n     * @param {Number} lineIndex\n     */\n    _renderChars: function(method, ctx, line, left, top, lineIndex) {\n      // set proper line offset\n      var lineHeight = this.getHeightOfLine(lineIndex),\n          isJustify = this.textAlign.indexOf('justify') !== -1,\n          actualStyle,\n          nextStyle,\n          charsToRender = '',\n          charBox,\n          boxWidth = 0,\n          timeToRender,\n          path = this.path,\n          shortCut = !isJustify && this.charSpacing === 0 && this.isEmptyStyles(lineIndex) && !path,\n          isLtr = this.direction === 'ltr', sign = this.direction === 'ltr' ? 1 : -1,\n          drawingLeft, currentDirection = ctx.canvas.getAttribute('dir');\n      ctx.save();\n      if (currentDirection !== this.direction) {\n        ctx.canvas.setAttribute('dir', isLtr ? 'ltr' : 'rtl');\n        ctx.direction = isLtr ? 'ltr' : 'rtl';\n        ctx.textAlign = isLtr ? 'left' : 'right';\n      }\n      top -= lineHeight * this._fontSizeFraction / this.lineHeight;\n      if (shortCut) {\n        // render all the line in one pass without checking\n        // drawingLeft = isLtr ? left : left - this.getLineWidth(lineIndex);\n        this._renderChar(method, ctx, lineIndex, 0, line.join(''), left, top, lineHeight);\n        ctx.restore();\n        return;\n      }\n      for (var i = 0, len = line.length - 1; i <= len; i++) {\n        timeToRender = i === len || this.charSpacing || path;\n        charsToRender += line[i];\n        charBox = this.__charBounds[lineIndex][i];\n        if (boxWidth === 0) {\n          left += sign * (charBox.kernedWidth - charBox.width);\n          boxWidth += charBox.width;\n        }\n        else {\n          boxWidth += charBox.kernedWidth;\n        }\n        if (isJustify && !timeToRender) {\n          if (this._reSpaceAndTab.test(line[i])) {\n            timeToRender = true;\n          }\n        }\n        if (!timeToRender) {\n          // if we have charSpacing, we render char by char\n          actualStyle = actualStyle || this.getCompleteStyleDeclaration(lineIndex, i);\n          nextStyle = this.getCompleteStyleDeclaration(lineIndex, i + 1);\n          timeToRender = fabric.util.hasStyleChanged(actualStyle, nextStyle, false);\n        }\n        if (timeToRender) {\n          if (path) {\n            ctx.save();\n            ctx.translate(charBox.renderLeft, charBox.renderTop);\n            ctx.rotate(charBox.angle);\n            this._renderChar(method, ctx, lineIndex, i, charsToRender, -boxWidth / 2, 0, lineHeight);\n            ctx.restore();\n          }\n          else {\n            drawingLeft = left;\n            this._renderChar(method, ctx, lineIndex, i, charsToRender, drawingLeft, top, lineHeight);\n          }\n          charsToRender = '';\n          actualStyle = nextStyle;\n          left += sign * boxWidth;\n          boxWidth = 0;\n        }\n      }\n      ctx.restore();\n    },\n\n    /**\n     * This function try to patch the missing gradientTransform on canvas gradients.\n     * transforming a context to transform the gradient, is going to transform the stroke too.\n     * we want to transform the gradient but not the stroke operation, so we create\n     * a transformed gradient on a pattern and then we use the pattern instead of the gradient.\n     * this method has drawbacks: is slow, is in low resolution, needs a patch for when the size\n     * is limited.\n     * @private\n     * @param {fabric.Gradient} filler a fabric gradient instance\n     * @return {CanvasPattern} a pattern to use as fill/stroke style\n     */\n    _applyPatternGradientTransformText: function(filler) {\n      var pCanvas = fabric.util.createCanvasElement(), pCtx,\n          // TODO: verify compatibility with strokeUniform\n          width = this.width + this.strokeWidth, height = this.height + this.strokeWidth;\n      pCanvas.width = width;\n      pCanvas.height = height;\n      pCtx = pCanvas.getContext('2d');\n      pCtx.beginPath(); pCtx.moveTo(0, 0); pCtx.lineTo(width, 0); pCtx.lineTo(width, height);\n      pCtx.lineTo(0, height); pCtx.closePath();\n      pCtx.translate(width / 2, height / 2);\n      pCtx.fillStyle = filler.toLive(pCtx);\n      this._applyPatternGradientTransform(pCtx, filler);\n      pCtx.fill();\n      return pCtx.createPattern(pCanvas, 'no-repeat');\n    },\n\n    handleFiller: function(ctx, property, filler) {\n      var offsetX, offsetY;\n      if (filler.toLive) {\n        if (filler.gradientUnits === 'percentage' || filler.gradientTransform || filler.patternTransform) {\n          // need to transform gradient in a pattern.\n          // this is a slow process. If you are hitting this codepath, and the object\n          // is not using caching, you should consider switching it on.\n          // we need a canvas as big as the current object caching canvas.\n          offsetX = -this.width / 2;\n          offsetY = -this.height / 2;\n          ctx.translate(offsetX, offsetY);\n          ctx[property] = this._applyPatternGradientTransformText(filler);\n          return { offsetX: offsetX, offsetY: offsetY };\n        }\n        else {\n          // is a simple gradient or pattern\n          ctx[property] = filler.toLive(ctx, this);\n          return this._applyPatternGradientTransform(ctx, filler);\n        }\n      }\n      else {\n        // is a color\n        ctx[property] = filler;\n      }\n      return { offsetX: 0, offsetY: 0 };\n    },\n\n    _setStrokeStyles: function(ctx, decl) {\n      ctx.lineWidth = decl.strokeWidth;\n      ctx.lineCap = this.strokeLineCap;\n      ctx.lineDashOffset = this.strokeDashOffset;\n      ctx.lineJoin = this.strokeLineJoin;\n      ctx.miterLimit = this.strokeMiterLimit;\n      return this.handleFiller(ctx, 'strokeStyle', decl.stroke);\n    },\n\n    _setFillStyles: function(ctx, decl) {\n      return this.handleFiller(ctx, 'fillStyle', decl.fill);\n    },\n\n    /**\n     * @private\n     * @param {String} method\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     * @param {Number} lineIndex\n     * @param {Number} charIndex\n     * @param {String} _char\n     * @param {Number} left Left coordinate\n     * @param {Number} top Top coordinate\n     * @param {Number} lineHeight Height of the line\n     */\n    _renderChar: function(method, ctx, lineIndex, charIndex, _char, left, top) {\n      var decl = this._getStyleDeclaration(lineIndex, charIndex),\n          fullDecl = this.getCompleteStyleDeclaration(lineIndex, charIndex),\n          shouldFill = method === 'fillText' && fullDecl.fill,\n          shouldStroke = method === 'strokeText' && fullDecl.stroke && fullDecl.strokeWidth,\n          fillOffsets, strokeOffsets;\n\n      if (!shouldStroke && !shouldFill) {\n        return;\n      }\n      ctx.save();\n\n      shouldFill && (fillOffsets = this._setFillStyles(ctx, fullDecl));\n      shouldStroke && (strokeOffsets = this._setStrokeStyles(ctx, fullDecl));\n\n      ctx.font = this._getFontDeclaration(fullDecl);\n\n\n      if (decl && decl.textBackgroundColor) {\n        this._removeShadow(ctx);\n      }\n      if (decl && decl.deltaY) {\n        top += decl.deltaY;\n      }\n      shouldFill && ctx.fillText(_char, left - fillOffsets.offsetX, top - fillOffsets.offsetY);\n      shouldStroke && ctx.strokeText(_char, left - strokeOffsets.offsetX, top - strokeOffsets.offsetY);\n      ctx.restore();\n    },\n\n    /**\n     * Turns the character into a 'superior figure' (i.e. 'superscript')\n     * @param {Number} start selection start\n     * @param {Number} end selection end\n     * @returns {fabric.Text} thisArg\n     * @chainable\n     */\n    setSuperscript: function(start, end) {\n      return this._setScript(start, end, this.superscript);\n    },\n\n    /**\n     * Turns the character into an 'inferior figure' (i.e. 'subscript')\n     * @param {Number} start selection start\n     * @param {Number} end selection end\n     * @returns {fabric.Text} thisArg\n     * @chainable\n     */\n    setSubscript: function(start, end) {\n      return this._setScript(start, end, this.subscript);\n    },\n\n    /**\n     * Applies 'schema' at given position\n     * @private\n     * @param {Number} start selection start\n     * @param {Number} end selection end\n     * @param {Number} schema\n     * @returns {fabric.Text} thisArg\n     * @chainable\n     */\n    _setScript: function(start, end, schema) {\n      var loc = this.get2DCursorLocation(start, true),\n          fontSize = this.getValueOfPropertyAt(loc.lineIndex, loc.charIndex, 'fontSize'),\n          dy = this.getValueOfPropertyAt(loc.lineIndex, loc.charIndex, 'deltaY'),\n          style = { fontSize: fontSize * schema.size, deltaY: dy + fontSize * schema.baseline };\n      this.setSelectionStyles(style, start, end);\n      return this;\n    },\n\n    /**\n     * @private\n     * @param {Number} lineIndex index text line\n     * @return {Number} Line left offset\n     */\n    _getLineLeftOffset: function(lineIndex) {\n      var lineWidth = this.getLineWidth(lineIndex),\n          lineDiff = this.width - lineWidth, textAlign = this.textAlign, direction = this.direction,\n          isEndOfWrapping, leftOffset = 0, isEndOfWrapping = this.isEndOfWrapping(lineIndex);\n      if (textAlign === 'justify'\n        || (textAlign === 'justify-center' && !isEndOfWrapping)\n        || (textAlign === 'justify-right' && !isEndOfWrapping)\n        || (textAlign === 'justify-left' && !isEndOfWrapping)\n      ) {\n        return 0;\n      }\n      if (textAlign === 'center') {\n        leftOffset = lineDiff / 2;\n      }\n      if (textAlign === 'right') {\n        leftOffset = lineDiff;\n      }\n      if (textAlign === 'justify-center') {\n        leftOffset = lineDiff / 2;\n      }\n      if (textAlign === 'justify-right') {\n        leftOffset = lineDiff;\n      }\n      if (direction === 'rtl') {\n        leftOffset -= lineDiff;\n      }\n      return leftOffset;\n    },\n\n    /**\n     * @private\n     */\n    _clearCache: function() {\n      this.__lineWidths = [];\n      this.__lineHeights = [];\n      this.__charBounds = [];\n    },\n\n    /**\n     * @private\n     */\n    _shouldClearDimensionCache: function() {\n      var shouldClear = this._forceClearCache;\n      shouldClear || (shouldClear = this.hasStateChanged('_dimensionAffectingProps'));\n      if (shouldClear) {\n        this.dirty = true;\n        this._forceClearCache = false;\n      }\n      return shouldClear;\n    },\n\n    /**\n     * Measure a single line given its index. Used to calculate the initial\n     * text bounding box. The values are calculated and stored in __lineWidths cache.\n     * @private\n     * @param {Number} lineIndex line number\n     * @return {Number} Line width\n     */\n    getLineWidth: function(lineIndex) {\n      if (this.__lineWidths[lineIndex] !== undefined) {\n        return this.__lineWidths[lineIndex];\n      }\n\n      var lineInfo = this.measureLine(lineIndex);\n      var width = lineInfo.width;\n      this.__lineWidths[lineIndex] = width;\n      return width;\n    },\n\n    _getWidthOfCharSpacing: function() {\n      if (this.charSpacing !== 0) {\n        return this.fontSize * this.charSpacing / 1000;\n      }\n      return 0;\n    },\n\n    /**\n     * Retrieves the value of property at given character position\n     * @param {Number} lineIndex the line number\n     * @param {Number} charIndex the character number\n     * @param {String} property the property name\n     * @returns the value of 'property'\n     */\n    getValueOfPropertyAt: function(lineIndex, charIndex, property) {\n      var charStyle = this._getStyleDeclaration(lineIndex, charIndex);\n      if (charStyle && typeof charStyle[property] !== 'undefined') {\n        return charStyle[property];\n      }\n      return this[property];\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _renderTextDecoration: function(ctx, type) {\n      if (!this[type] && !this.styleHas(type)) {\n        return;\n      }\n      var heightOfLine, size, _size,\n          lineLeftOffset, dy, _dy,\n          line, lastDecoration,\n          leftOffset = this._getLeftOffset(),\n          topOffset = this._getTopOffset(), top,\n          boxStart, boxWidth, charBox, currentDecoration,\n          maxHeight, currentFill, lastFill, path = this.path,\n          charSpacing = this._getWidthOfCharSpacing(),\n          offsetY = this.offsets[type];\n\n      for (var i = 0, len = this._textLines.length; i < len; i++) {\n        heightOfLine = this.getHeightOfLine(i);\n        if (!this[type] && !this.styleHas(type, i)) {\n          topOffset += heightOfLine;\n          continue;\n        }\n        line = this._textLines[i];\n        maxHeight = heightOfLine / this.lineHeight;\n        lineLeftOffset = this._getLineLeftOffset(i);\n        boxStart = 0;\n        boxWidth = 0;\n        lastDecoration = this.getValueOfPropertyAt(i, 0, type);\n        lastFill = this.getValueOfPropertyAt(i, 0, 'fill');\n        top = topOffset + maxHeight * (1 - this._fontSizeFraction);\n        size = this.getHeightOfChar(i, 0);\n        dy = this.getValueOfPropertyAt(i, 0, 'deltaY');\n        for (var j = 0, jlen = line.length; j < jlen; j++) {\n          charBox = this.__charBounds[i][j];\n          currentDecoration = this.getValueOfPropertyAt(i, j, type);\n          currentFill = this.getValueOfPropertyAt(i, j, 'fill');\n          _size = this.getHeightOfChar(i, j);\n          _dy = this.getValueOfPropertyAt(i, j, 'deltaY');\n          if (path && currentDecoration && currentFill) {\n            ctx.save();\n            ctx.fillStyle = lastFill;\n            ctx.translate(charBox.renderLeft, charBox.renderTop);\n            ctx.rotate(charBox.angle);\n            ctx.fillRect(\n              -charBox.kernedWidth / 2,\n              offsetY * _size + _dy,\n              charBox.kernedWidth,\n              this.fontSize / 15\n            );\n            ctx.restore();\n          }\n          else if (\n            (currentDecoration !== lastDecoration || currentFill !== lastFill || _size !== size || _dy !== dy)\n            && boxWidth > 0\n          ) {\n            var drawStart = leftOffset + lineLeftOffset + boxStart;\n            if (this.direction === 'rtl') {\n              drawStart = this.width - drawStart - boxWidth;\n            }\n            if (lastDecoration && lastFill) {\n              ctx.fillStyle = lastFill;\n              ctx.fillRect(\n                drawStart,\n                top + offsetY * size + dy,\n                boxWidth,\n                this.fontSize / 15\n              );\n            }\n            boxStart = charBox.left;\n            boxWidth = charBox.width;\n            lastDecoration = currentDecoration;\n            lastFill = currentFill;\n            size = _size;\n            dy = _dy;\n          }\n          else {\n            boxWidth += charBox.kernedWidth;\n          }\n        }\n        var drawStart = leftOffset + lineLeftOffset + boxStart;\n        if (this.direction === 'rtl') {\n          drawStart = this.width - drawStart - boxWidth;\n        }\n        ctx.fillStyle = currentFill;\n        currentDecoration && currentFill && ctx.fillRect(\n          drawStart,\n          top + offsetY * size + dy,\n          boxWidth - charSpacing,\n          this.fontSize / 15\n        );\n        topOffset += heightOfLine;\n      }\n      // if there is text background color no\n      // other shadows should be casted\n      this._removeShadow(ctx);\n    },\n\n    /**\n     * return font declaration string for canvas context\n     * @param {Object} [styleObject] object\n     * @returns {String} font declaration formatted for canvas context.\n     */\n    _getFontDeclaration: function(styleObject, forMeasuring) {\n      var style = styleObject || this, family = this.fontFamily,\n          fontIsGeneric = fabric.Text.genericFonts.indexOf(family.toLowerCase()) > -1;\n      var fontFamily = family === undefined ||\n      family.indexOf('\\'') > -1 || family.indexOf(',') > -1 ||\n      family.indexOf('\"') > -1 || fontIsGeneric\n        ? style.fontFamily : '\"' + style.fontFamily + '\"';\n      return [\n        // node-canvas needs \"weight style\", while browsers need \"style weight\"\n        // verify if this can be fixed in JSDOM\n        (fabric.isLikelyNode ? style.fontWeight : style.fontStyle),\n        (fabric.isLikelyNode ? style.fontStyle : style.fontWeight),\n        forMeasuring ? this.CACHE_FONT_SIZE + 'px' : style.fontSize + 'px',\n        fontFamily\n      ].join(' ');\n    },\n\n    /**\n     * Renders text instance on a specified context\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    render: function(ctx) {\n      // do not render if object is not visible\n      if (!this.visible) {\n        return;\n      }\n      if (this.canvas && this.canvas.skipOffscreen && !this.group && !this.isOnScreen()) {\n        return;\n      }\n      if (this._shouldClearDimensionCache()) {\n        this.initDimensions();\n      }\n      this.callSuper('render', ctx);\n    },\n\n    /**\n     * Returns the text as an array of lines.\n     * @param {String} text text to split\n     * @returns {Array} Lines in the text\n     */\n    _splitTextIntoLines: function(text) {\n      var lines = text.split(this._reNewline),\n          newLines = new Array(lines.length),\n          newLine = ['\\n'],\n          newText = [];\n      for (var i = 0; i < lines.length; i++) {\n        newLines[i] = fabric.util.string.graphemeSplit(lines[i]);\n        newText = newText.concat(newLines[i], newLine);\n      }\n      newText.pop();\n      return { _unwrappedLines: newLines, lines: lines, graphemeText: newText, graphemeLines: newLines };\n    },\n\n    /**\n     * Returns object representation of an instance\n     * @param {Array} [propertiesToInclude] Any properties that you might want to additionally include in the output\n     * @return {Object} Object representation of an instance\n     */\n    toObject: function(propertiesToInclude) {\n      var allProperties = additionalProps.concat(propertiesToInclude);\n      var obj = this.callSuper('toObject', allProperties);\n      obj.styles = fabric.util.stylesToArray(this.styles, this.text);\n      if (obj.path) {\n        obj.path = this.path.toObject();\n      }\n      return obj;\n    },\n\n    /**\n     * Sets property to a given value. When changing position/dimension -related properties (left, top, scale, angle, etc.) `set` does not update position of object's borders/controls. If you need to update those, call `setCoords()`.\n     * @param {String|Object} key Property name or object (if object, iterate over the object properties)\n     * @param {Object|Function} value Property value (if function, the value is passed into it and its return value is used as a new one)\n     * @return {fabric.Object} thisArg\n     * @chainable\n     */\n    set: function(key, value) {\n      this.callSuper('set', key, value);\n      var needsDims = false;\n      var isAddingPath = false;\n      if (typeof key === 'object') {\n        for (var _key in key) {\n          if (_key === 'path') {\n            this.setPathInfo();\n          }\n          needsDims = needsDims || this._dimensionAffectingProps.indexOf(_key) !== -1;\n          isAddingPath = isAddingPath || _key === 'path';\n        }\n      }\n      else {\n        needsDims = this._dimensionAffectingProps.indexOf(key) !== -1;\n        isAddingPath = key === 'path';\n      }\n      if (isAddingPath) {\n        this.setPathInfo();\n      }\n      if (needsDims) {\n        this.initDimensions();\n        this.setCoords();\n      }\n      return this;\n    },\n\n    /**\n     * Returns complexity of an instance\n     * @return {Number} complexity\n     */\n    complexity: function() {\n      return 1;\n    }\n  });\n\n  /* _FROM_SVG_START_ */\n  /**\n   * List of attribute names to account for when parsing SVG element (used by {@link fabric.Text.fromElement})\n   * @static\n   * @memberOf fabric.Text\n   * @see: http://www.w3.org/TR/SVG/text.html#TextElement\n   */\n  fabric.Text.ATTRIBUTE_NAMES = fabric.SHARED_ATTRIBUTES.concat(\n    'x y dx dy font-family font-style font-weight font-size letter-spacing text-decoration text-anchor'.split(' '));\n\n  /**\n   * Default SVG font size\n   * @static\n   * @memberOf fabric.Text\n   */\n  fabric.Text.DEFAULT_SVG_FONT_SIZE = 16;\n\n  /**\n   * Returns fabric.Text instance from an SVG element (<b>not yet implemented</b>)\n   * @static\n   * @memberOf fabric.Text\n   * @param {SVGElement} element Element to parse\n   * @param {Function} callback callback function invoked after parsing\n   * @param {Object} [options] Options object\n   */\n  fabric.Text.fromElement = function(element, callback, options) {\n    if (!element) {\n      return callback(null);\n    }\n\n    var parsedAttributes = fabric.parseAttributes(element, fabric.Text.ATTRIBUTE_NAMES),\n        parsedAnchor = parsedAttributes.textAnchor || 'left';\n    options = fabric.util.object.extend((options ? clone(options) : { }), parsedAttributes);\n\n    options.top = options.top || 0;\n    options.left = options.left || 0;\n    if (parsedAttributes.textDecoration) {\n      var textDecoration = parsedAttributes.textDecoration;\n      if (textDecoration.indexOf('underline') !== -1) {\n        options.underline = true;\n      }\n      if (textDecoration.indexOf('overline') !== -1) {\n        options.overline = true;\n      }\n      if (textDecoration.indexOf('line-through') !== -1) {\n        options.linethrough = true;\n      }\n      delete options.textDecoration;\n    }\n    if ('dx' in parsedAttributes) {\n      options.left += parsedAttributes.dx;\n    }\n    if ('dy' in parsedAttributes) {\n      options.top += parsedAttributes.dy;\n    }\n    if (!('fontSize' in options)) {\n      options.fontSize = fabric.Text.DEFAULT_SVG_FONT_SIZE;\n    }\n\n    var textContent = '';\n\n    // The XML is not properly parsed in IE9 so a workaround to get\n    // textContent is through firstChild.data. Another workaround would be\n    // to convert XML loaded from a file to be converted using DOMParser (same way loadSVGFromString() does)\n    if (!('textContent' in element)) {\n      if ('firstChild' in element && element.firstChild !== null) {\n        if ('data' in element.firstChild && element.firstChild.data !== null) {\n          textContent = element.firstChild.data;\n        }\n      }\n    }\n    else {\n      textContent = element.textContent;\n    }\n\n    textContent = textContent.replace(/^\\s+|\\s+$|\\n+/g, '').replace(/\\s+/g, ' ');\n    var originalStrokeWidth = options.strokeWidth;\n    options.strokeWidth = 0;\n\n    var text = new fabric.Text(textContent, options),\n        textHeightScaleFactor = text.getScaledHeight() / text.height,\n        lineHeightDiff = (text.height + text.strokeWidth) * text.lineHeight - text.height,\n        scaledDiff = lineHeightDiff * textHeightScaleFactor,\n        textHeight = text.getScaledHeight() + scaledDiff,\n        offX = 0;\n    /*\n      Adjust positioning:\n        x/y attributes in SVG correspond to the bottom-left corner of text bounding box\n        fabric output by default at top, left.\n    */\n    if (parsedAnchor === 'center') {\n      offX = text.getScaledWidth() / 2;\n    }\n    if (parsedAnchor === 'right') {\n      offX = text.getScaledWidth();\n    }\n    text.set({\n      left: text.left - offX,\n      top: text.top - (textHeight - text.fontSize * (0.07 + text._fontSizeFraction)) / text.lineHeight,\n      strokeWidth: typeof originalStrokeWidth !== 'undefined' ? originalStrokeWidth : 1,\n    });\n    callback(text);\n  };\n  /* _FROM_SVG_END_ */\n\n  /**\n   * Returns fabric.Text instance from an object representation\n   * @static\n   * @memberOf fabric.Text\n   * @param {Object} object plain js Object to create an instance from\n   * @param {Function} [callback] Callback to invoke when an fabric.Text instance is created\n   */\n  fabric.Text.fromObject = function(object, callback) {\n    var objectCopy = clone(object), path = object.path;\n    delete objectCopy.path;\n    return fabric.Object._fromObject('Text', objectCopy, function(textInstance) {\n      textInstance.styles = fabric.util.stylesFromArray(object.styles, object.text);\n      if (path) {\n        fabric.Object._fromObject('Path', path, function(pathInstance) {\n          textInstance.set('path', pathInstance);\n          callback(textInstance);\n        }, 'path');\n      }\n      else {\n        callback(textInstance);\n      }\n    }, 'text');\n  };\n\n  fabric.Text.genericFonts = ['sans-serif', 'serif', 'cursive', 'fantasy', 'monospace'];\n\n  fabric.util.createAccessors && fabric.util.createAccessors(fabric.Text);\n\n})(typeof exports !== 'undefined' ? exports : this);\n\n\n(function() {\n  fabric.util.object.extend(fabric.Text.prototype, /** @lends fabric.Text.prototype */ {\n    /**\n     * Returns true if object has no styling or no styling in a line\n     * @param {Number} lineIndex , lineIndex is on wrapped lines.\n     * @return {Boolean}\n     */\n    isEmptyStyles: function(lineIndex) {\n      if (!this.styles) {\n        return true;\n      }\n      if (typeof lineIndex !== 'undefined' && !this.styles[lineIndex]) {\n        return true;\n      }\n      var obj = typeof lineIndex === 'undefined' ? this.styles : { line: this.styles[lineIndex] };\n      for (var p1 in obj) {\n        for (var p2 in obj[p1]) {\n          // eslint-disable-next-line no-unused-vars\n          for (var p3 in obj[p1][p2]) {\n            return false;\n          }\n        }\n      }\n      return true;\n    },\n\n    /**\n     * Returns true if object has a style property or has it ina specified line\n     * This function is used to detect if a text will use a particular property or not.\n     * @param {String} property to check for\n     * @param {Number} lineIndex to check the style on\n     * @return {Boolean}\n     */\n    styleHas: function(property, lineIndex) {\n      if (!this.styles || !property || property === '') {\n        return false;\n      }\n      if (typeof lineIndex !== 'undefined' && !this.styles[lineIndex]) {\n        return false;\n      }\n      var obj = typeof lineIndex === 'undefined' ? this.styles : { 0: this.styles[lineIndex] };\n      // eslint-disable-next-line\n      for (var p1 in obj) {\n        // eslint-disable-next-line\n        for (var p2 in obj[p1]) {\n          if (typeof obj[p1][p2][property] !== 'undefined') {\n            return true;\n          }\n        }\n      }\n      return false;\n    },\n\n    /**\n     * Check if characters in a text have a value for a property\n     * whose value matches the textbox's value for that property.  If so,\n     * the character-level property is deleted.  If the character\n     * has no other properties, then it is also deleted.  Finally,\n     * if the line containing that character has no other characters\n     * then it also is deleted.\n     *\n     * @param {string} property The property to compare between characters and text.\n     */\n    cleanStyle: function(property) {\n      if (!this.styles || !property || property === '') {\n        return false;\n      }\n      var obj = this.styles, stylesCount = 0, letterCount, stylePropertyValue,\n          allStyleObjectPropertiesMatch = true, graphemeCount = 0, styleObject;\n      // eslint-disable-next-line\n      for (var p1 in obj) {\n        letterCount = 0;\n        // eslint-disable-next-line\n        for (var p2 in obj[p1]) {\n          var styleObject = obj[p1][p2],\n              stylePropertyHasBeenSet = styleObject.hasOwnProperty(property);\n\n          stylesCount++;\n\n          if (stylePropertyHasBeenSet) {\n            if (!stylePropertyValue) {\n              stylePropertyValue = styleObject[property];\n            }\n            else if (styleObject[property] !== stylePropertyValue) {\n              allStyleObjectPropertiesMatch = false;\n            }\n\n            if (styleObject[property] === this[property]) {\n              delete styleObject[property];\n            }\n          }\n          else {\n            allStyleObjectPropertiesMatch = false;\n          }\n\n          if (Object.keys(styleObject).length !== 0) {\n            letterCount++;\n          }\n          else {\n            delete obj[p1][p2];\n          }\n        }\n\n        if (letterCount === 0) {\n          delete obj[p1];\n        }\n      }\n      // if every grapheme has the same style set then\n      // delete those styles and set it on the parent\n      for (var i = 0; i < this._textLines.length; i++) {\n        graphemeCount += this._textLines[i].length;\n      }\n      if (allStyleObjectPropertiesMatch && stylesCount === graphemeCount) {\n        this[property] = stylePropertyValue;\n        this.removeStyle(property);\n      }\n    },\n\n    /**\n     * Remove a style property or properties from all individual character styles\n     * in a text object.  Deletes the character style object if it contains no other style\n     * props.  Deletes a line style object if it contains no other character styles.\n     *\n     * @param {String} props The property to remove from character styles.\n     */\n    removeStyle: function(property) {\n      if (!this.styles || !property || property === '') {\n        return;\n      }\n      var obj = this.styles, line, lineNum, charNum;\n      for (lineNum in obj) {\n        line = obj[lineNum];\n        for (charNum in line) {\n          delete line[charNum][property];\n          if (Object.keys(line[charNum]).length === 0) {\n            delete line[charNum];\n          }\n        }\n        if (Object.keys(line).length === 0) {\n          delete obj[lineNum];\n        }\n      }\n    },\n\n    /**\n     * @private\n     */\n    _extendStyles: function(index, styles) {\n      var loc = this.get2DCursorLocation(index);\n\n      if (!this._getLineStyle(loc.lineIndex)) {\n        this._setLineStyle(loc.lineIndex);\n      }\n\n      if (!this._getStyleDeclaration(loc.lineIndex, loc.charIndex)) {\n        this._setStyleDeclaration(loc.lineIndex, loc.charIndex, {});\n      }\n\n      fabric.util.object.extend(this._getStyleDeclaration(loc.lineIndex, loc.charIndex), styles);\n    },\n\n    /**\n     * Returns 2d representation (lineIndex and charIndex) of cursor (or selection start)\n     * @param {Number} [selectionStart] Optional index. When not given, current selectionStart is used.\n     * @param {Boolean} [skipWrapping] consider the location for unwrapped lines. useful to manage styles.\n     */\n    get2DCursorLocation: function(selectionStart, skipWrapping) {\n      if (typeof selectionStart === 'undefined') {\n        selectionStart = this.selectionStart;\n      }\n      var lines = skipWrapping ? this._unwrappedTextLines : this._textLines,\n          len = lines.length;\n      for (var i = 0; i < len; i++) {\n        if (selectionStart <= lines[i].length) {\n          return {\n            lineIndex: i,\n            charIndex: selectionStart\n          };\n        }\n        selectionStart -= lines[i].length + this.missingNewlineOffset(i);\n      }\n      return {\n        lineIndex: i - 1,\n        charIndex: lines[i - 1].length < selectionStart ? lines[i - 1].length : selectionStart\n      };\n    },\n\n    /**\n     * Gets style of a current selection/cursor (at the start position)\n     * if startIndex or endIndex are not provided, selectionStart or selectionEnd will be used.\n     * @param {Number} [startIndex] Start index to get styles at\n     * @param {Number} [endIndex] End index to get styles at, if not specified selectionEnd or startIndex + 1\n     * @param {Boolean} [complete] get full style or not\n     * @return {Array} styles an array with one, zero or more Style objects\n     */\n    getSelectionStyles: function(startIndex, endIndex, complete) {\n      if (typeof startIndex === 'undefined') {\n        startIndex = this.selectionStart || 0;\n      }\n      if (typeof endIndex === 'undefined') {\n        endIndex = this.selectionEnd || startIndex;\n      }\n      var styles = [];\n      for (var i = startIndex; i < endIndex; i++) {\n        styles.push(this.getStyleAtPosition(i, complete));\n      }\n      return styles;\n    },\n\n    /**\n     * Gets style of a current selection/cursor position\n     * @param {Number} position  to get styles at\n     * @param {Boolean} [complete] full style if true\n     * @return {Object} style Style object at a specified index\n     * @private\n     */\n    getStyleAtPosition: function(position, complete) {\n      var loc = this.get2DCursorLocation(position),\n          style = complete ? this.getCompleteStyleDeclaration(loc.lineIndex, loc.charIndex) :\n            this._getStyleDeclaration(loc.lineIndex, loc.charIndex);\n      return style || {};\n    },\n\n    /**\n     * Sets style of a current selection, if no selection exist, do not set anything.\n     * @param {Object} [styles] Styles object\n     * @param {Number} [startIndex] Start index to get styles at\n     * @param {Number} [endIndex] End index to get styles at, if not specified selectionEnd or startIndex + 1\n     * @return {fabric.IText} thisArg\n     * @chainable\n     */\n    setSelectionStyles: function(styles, startIndex, endIndex) {\n      if (typeof startIndex === 'undefined') {\n        startIndex = this.selectionStart || 0;\n      }\n      if (typeof endIndex === 'undefined') {\n        endIndex = this.selectionEnd || startIndex;\n      }\n      for (var i = startIndex; i < endIndex; i++) {\n        this._extendStyles(i, styles);\n      }\n      /* not included in _extendStyles to avoid clearing cache more than once */\n      this._forceClearCache = true;\n      return this;\n    },\n\n    /**\n     * get the reference, not a clone, of the style object for a given character\n     * @param {Number} lineIndex\n     * @param {Number} charIndex\n     * @return {Object} style object\n     */\n    _getStyleDeclaration: function(lineIndex, charIndex) {\n      var lineStyle = this.styles && this.styles[lineIndex];\n      if (!lineStyle) {\n        return null;\n      }\n      return lineStyle[charIndex];\n    },\n\n    /**\n     * return a new object that contains all the style property for a character\n     * the object returned is newly created\n     * @param {Number} lineIndex of the line where the character is\n     * @param {Number} charIndex position of the character on the line\n     * @return {Object} style object\n     */\n    getCompleteStyleDeclaration: function(lineIndex, charIndex) {\n      var style = this._getStyleDeclaration(lineIndex, charIndex) || { },\n          styleObject = { }, prop;\n      for (var i = 0; i < this._styleProperties.length; i++) {\n        prop = this._styleProperties[i];\n        styleObject[prop] = typeof style[prop] === 'undefined' ? this[prop] : style[prop];\n      }\n      return styleObject;\n    },\n\n    /**\n     * @param {Number} lineIndex\n     * @param {Number} charIndex\n     * @param {Object} style\n     * @private\n     */\n    _setStyleDeclaration: function(lineIndex, charIndex, style) {\n      this.styles[lineIndex][charIndex] = style;\n    },\n\n    /**\n     *\n     * @param {Number} lineIndex\n     * @param {Number} charIndex\n     * @private\n     */\n    _deleteStyleDeclaration: function(lineIndex, charIndex) {\n      delete this.styles[lineIndex][charIndex];\n    },\n\n    /**\n     * @param {Number} lineIndex\n     * @return {Boolean} if the line exists or not\n     * @private\n     */\n    _getLineStyle: function(lineIndex) {\n      return !!this.styles[lineIndex];\n    },\n\n    /**\n     * Set the line style to an empty object so that is initialized\n     * @param {Number} lineIndex\n     * @private\n     */\n    _setLineStyle: function(lineIndex) {\n      this.styles[lineIndex] = {};\n    },\n\n    /**\n     * @param {Number} lineIndex\n     * @private\n     */\n    _deleteLineStyle: function(lineIndex) {\n      delete this.styles[lineIndex];\n    }\n  });\n})();\n\n\n(function() {\n\n  function parseDecoration(object) {\n    if (object.textDecoration) {\n      object.textDecoration.indexOf('underline') > -1 && (object.underline = true);\n      object.textDecoration.indexOf('line-through') > -1 && (object.linethrough = true);\n      object.textDecoration.indexOf('overline') > -1 && (object.overline = true);\n      delete object.textDecoration;\n    }\n  }\n\n  /**\n   * IText class (introduced in <b>v1.4</b>) Events are also fired with \"text:\"\n   * prefix when observing canvas.\n   * @class fabric.IText\n   * @extends fabric.Text\n   * @mixes fabric.Observable\n   *\n   * @fires changed\n   * @fires selection:changed\n   * @fires editing:entered\n   * @fires editing:exited\n   *\n   * @return {fabric.IText} thisArg\n   * @see {@link fabric.IText#initialize} for constructor definition\n   *\n   * <p>Supported key combinations:</p>\n   * <pre>\n   *   Move cursor:                    left, right, up, down\n   *   Select character:               shift + left, shift + right\n   *   Select text vertically:         shift + up, shift + down\n   *   Move cursor by word:            alt + left, alt + right\n   *   Select words:                   shift + alt + left, shift + alt + right\n   *   Move cursor to line start/end:  cmd + left, cmd + right or home, end\n   *   Select till start/end of line:  cmd + shift + left, cmd + shift + right or shift + home, shift + end\n   *   Jump to start/end of text:      cmd + up, cmd + down\n   *   Select till start/end of text:  cmd + shift + up, cmd + shift + down or shift + pgUp, shift + pgDown\n   *   Delete character:               backspace\n   *   Delete word:                    alt + backspace\n   *   Delete line:                    cmd + backspace\n   *   Forward delete:                 delete\n   *   Copy text:                      ctrl/cmd + c\n   *   Paste text:                     ctrl/cmd + v\n   *   Cut text:                       ctrl/cmd + x\n   *   Select entire text:             ctrl/cmd + a\n   *   Quit editing                    tab or esc\n   * </pre>\n   *\n   * <p>Supported mouse/touch combination</p>\n   * <pre>\n   *   Position cursor:                click/touch\n   *   Create selection:               click/touch & drag\n   *   Create selection:               click & shift + click\n   *   Select word:                    double click\n   *   Select line:                    triple click\n   * </pre>\n   */\n  fabric.IText = fabric.util.createClass(fabric.Text, fabric.Observable, /** @lends fabric.IText.prototype */ {\n\n    /**\n     * Type of an object\n     * @type String\n     * @default\n     */\n    type: 'i-text',\n\n    /**\n     * Index where text selection starts (or where cursor is when there is no selection)\n     * @type Number\n     * @default\n     */\n    selectionStart: 0,\n\n    /**\n     * Index where text selection ends\n     * @type Number\n     * @default\n     */\n    selectionEnd: 0,\n\n    /**\n     * Color of text selection\n     * @type String\n     * @default\n     */\n    selectionColor: 'rgba(17,119,255,0.3)',\n\n    /**\n     * Indicates whether text is in editing mode\n     * @type Boolean\n     * @default\n     */\n    isEditing: false,\n\n    /**\n     * Indicates whether a text can be edited\n     * @type Boolean\n     * @default\n     */\n    editable: true,\n\n    /**\n     * Border color of text object while it's in editing mode\n     * @type String\n     * @default\n     */\n    editingBorderColor: 'rgba(102,153,255,0.25)',\n\n    /**\n     * Width of cursor (in px)\n     * @type Number\n     * @default\n     */\n    cursorWidth: 2,\n\n    /**\n     * Color of text cursor color in editing mode.\n     * if not set (default) will take color from the text.\n     * if set to a color value that fabric can understand, it will\n     * be used instead of the color of the text at the current position.\n     * @type String\n     * @default\n     */\n    cursorColor: '',\n\n    /**\n     * Delay between cursor blink (in ms)\n     * @type Number\n     * @default\n     */\n    cursorDelay: 1000,\n\n    /**\n     * Duration of cursor fadein (in ms)\n     * @type Number\n     * @default\n     */\n    cursorDuration: 600,\n\n    /**\n     * Indicates whether internal text char widths can be cached\n     * @type Boolean\n     * @default\n     */\n    caching: true,\n\n    /**\n     * DOM container to append the hiddenTextarea.\n     * An alternative to attaching to the document.body.\n     * Useful to reduce laggish redraw of the full document.body tree and\n     * also with modals event capturing that won't let the textarea take focus.\n     * @type HTMLElement\n     * @default\n     */\n    hiddenTextareaContainer: null,\n\n    /**\n     * @private\n     */\n    _reSpace: /\\s|\\n/,\n\n    /**\n     * @private\n     */\n    _currentCursorOpacity: 0,\n\n    /**\n     * @private\n     */\n    _selectionDirection: null,\n\n    /**\n     * @private\n     */\n    _abortCursorAnimation: false,\n\n    /**\n     * @private\n     */\n    __widthOfSpace: [],\n\n    /**\n     * Helps determining when the text is in composition, so that the cursor\n     * rendering is altered.\n     */\n    inCompositionMode: false,\n\n    /**\n     * Constructor\n     * @param {String} text Text string\n     * @param {Object} [options] Options object\n     * @return {fabric.IText} thisArg\n     */\n    initialize: function(text, options) {\n      this.callSuper('initialize', text, options);\n      this.initBehavior();\n    },\n\n    /**\n     * Sets selection start (left boundary of a selection)\n     * @param {Number} index Index to set selection start to\n     */\n    setSelectionStart: function(index) {\n      index = Math.max(index, 0);\n      this._updateAndFire('selectionStart', index);\n    },\n\n    /**\n     * Sets selection end (right boundary of a selection)\n     * @param {Number} index Index to set selection end to\n     */\n    setSelectionEnd: function(index) {\n      index = Math.min(index, this.text.length);\n      this._updateAndFire('selectionEnd', index);\n    },\n\n    /**\n     * @private\n     * @param {String} property 'selectionStart' or 'selectionEnd'\n     * @param {Number} index new position of property\n     */\n    _updateAndFire: function(property, index) {\n      if (this[property] !== index) {\n        this._fireSelectionChanged();\n        this[property] = index;\n      }\n      this._updateTextarea();\n    },\n\n    /**\n     * Fires the even of selection changed\n     * @private\n     */\n    _fireSelectionChanged: function() {\n      this.fire('selection:changed');\n      this.canvas && this.canvas.fire('text:selection:changed', { target: this });\n    },\n\n    /**\n     * Initialize text dimensions. Render all text on given context\n     * or on a offscreen canvas to get the text width with measureText.\n     * Updates this.width and this.height with the proper values.\n     * Does not return dimensions.\n     * @private\n     */\n    initDimensions: function() {\n      this.isEditing && this.initDelayedCursor();\n      this.clearContextTop();\n      this.callSuper('initDimensions');\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    render: function(ctx) {\n      this.clearContextTop();\n      this.callSuper('render', ctx);\n      // clear the cursorOffsetCache, so we ensure to calculate once per renderCursor\n      // the correct position but not at every cursor animation.\n      this.cursorOffsetCache = { };\n      this.renderCursorOrSelection();\n    },\n\n    /**\n     * @private\n     * @param {CanvasRenderingContext2D} ctx Context to render on\n     */\n    _render: function(ctx) {\n      this.callSuper('_render', ctx);\n    },\n\n    /**\n     * Prepare and clean the contextTop\n     */\n    clearContextTop: function(skipRestore) {\n      if (!this.isEditing || !this.canvas || !this.canvas.contextTop) {\n        return;\n      }\n      var ctx = this.canvas.contextTop, v = this.canvas.viewportTransform;\n      ctx.save();\n      ctx.transform(v[0], v[1], v[2], v[3], v[4], v[5]);\n      this.transform(ctx);\n      this._clearTextArea(ctx);\n      skipRestore || ctx.restore();\n    },\n    /**\n     * Renders cursor or selection (depending on what exists)\n     * it does on the contextTop. If contextTop is not available, do nothing.\n     */\n    renderCursorOrSelection: function() {\n      if (!this.isEditing || !this.canvas || !this.canvas.contextTop) {\n        return;\n      }\n      var boundaries = this._getCursorBoundaries(),\n          ctx = this.canvas.contextTop;\n      this.clearContextTop(true);\n      if (this.selectionStart === this.selectionEnd) {\n        this.renderCursor(boundaries, ctx);\n      }\n      else {\n        this.renderSelection(boundaries, ctx);\n      }\n      ctx.restore();\n    },\n\n    _clearTextArea: function(ctx) {\n      // we add 4 pixel, to be sure to do not leave any pixel out\n      var width = this.width + 4, height = this.height + 4;\n      ctx.clearRect(-width / 2, -height / 2, width, height);\n    },\n\n    /**\n     * Returns cursor boundaries (left, top, leftOffset, topOffset)\n     * @private\n     * @param {Array} chars Array of characters\n     * @param {String} typeOfBoundaries\n     */\n    _getCursorBoundaries: function(position) {\n\n      // left/top are left/top of entire text box\n      // leftOffset/topOffset are offset from that left/top point of a text box\n\n      if (typeof position === 'undefined') {\n        position = this.selectionStart;\n      }\n\n      var left = this._getLeftOffset(),\n          top = this._getTopOffset(),\n          offsets = this._getCursorBoundariesOffsets(position);\n      return {\n        left: left,\n        top: top,\n        leftOffset: offsets.left,\n        topOffset: offsets.top\n      };\n    },\n\n    /**\n     * @private\n     */\n    _getCursorBoundariesOffsets: function(position) {\n      if (this.cursorOffsetCache && 'top' in this.cursorOffsetCache) {\n        return this.cursorOffsetCache;\n      }\n      var lineLeftOffset,\n          lineIndex,\n          charIndex,\n          topOffset = 0,\n          leftOffset = 0,\n          boundaries,\n          cursorPosition = this.get2DCursorLocation(position);\n      charIndex = cursorPosition.charIndex;\n      lineIndex = cursorPosition.lineIndex;\n      for (var i = 0; i < lineIndex; i++) {\n        topOffset += this.getHeightOfLine(i);\n      }\n      lineLeftOffset = this._getLineLeftOffset(lineIndex);\n      var bound = this.__charBounds[lineIndex][charIndex];\n      bound && (leftOffset = bound.left);\n      if (this.charSpacing !== 0 && charIndex === this._textLines[lineIndex].length) {\n        leftOffset -= this._getWidthOfCharSpacing();\n      }\n      boundaries = {\n        top: topOffset,\n        left: lineLeftOffset + (leftOffset > 0 ? leftOffset : 0),\n      };\n      if (this.direction === 'rtl') {\n        boundaries.left *= -1;\n      }\n      this.cursorOffsetCache = boundaries;\n      return this.cursorOffsetCache;\n    },\n\n    /**\n     * Renders cursor\n     * @param {Object} boundaries\n     * @param {CanvasRenderingContext2D} ctx transformed context to draw on\n     */\n    renderCursor: function(boundaries, ctx) {\n      var cursorLocation = this.get2DCursorLocation(),\n          lineIndex = cursorLocation.lineIndex,\n          charIndex = cursorLocation.charIndex > 0 ? cursorLocation.charIndex - 1 : 0,\n          charHeight = this.getValueOfPropertyAt(lineIndex, charIndex, 'fontSize'),\n          multiplier = this.scaleX * this.canvas.getZoom(),\n          cursorWidth = this.cursorWidth / multiplier,\n          topOffset = boundaries.topOffset,\n          dy = this.getValueOfPropertyAt(lineIndex, charIndex, 'deltaY');\n      topOffset += (1 - this._fontSizeFraction) * this.getHeightOfLine(lineIndex) / this.lineHeight\n        - charHeight * (1 - this._fontSizeFraction);\n\n      if (this.inCompositionMode) {\n        this.renderSelection(boundaries, ctx);\n      }\n      ctx.fillStyle = this.cursorColor || this.getValueOfPropertyAt(lineIndex, charIndex, 'fill');\n      ctx.globalAlpha = this.__isMousedown ? 1 : this._currentCursorOpacity;\n      ctx.fillRect(\n        boundaries.left + boundaries.leftOffset - cursorWidth / 2,\n        topOffset + boundaries.top + dy,\n        cursorWidth,\n        charHeight);\n    },\n\n    /**\n     * Renders text selection\n     * @param {Object} boundaries Object with left/top/leftOffset/topOffset\n     * @param {CanvasRenderingContext2D} ctx transformed context to draw on\n     */\n    renderSelection: function(boundaries, ctx) {\n\n      var selectionStart = this.inCompositionMode ? this.hiddenTextarea.selectionStart : this.selectionStart,\n          selectionEnd = this.inCompositionMode ? this.hiddenTextarea.selectionEnd : this.selectionEnd,\n          isJustify = this.textAlign.indexOf('justify') !== -1,\n          start = this.get2DCursorLocation(selectionStart),\n          end = this.get2DCursorLocation(selectionEnd),\n          startLine = start.lineIndex,\n          endLine = end.lineIndex,\n          startChar = start.charIndex < 0 ? 0 : start.charIndex,\n          endChar = end.charIndex < 0 ? 0 : end.charIndex;\n\n      for (var i = startLine; i <= endLine; i++) {\n        var lineOffset = this._getLineLeftOffset(i) || 0,\n            lineHeight = this.getHeightOfLine(i),\n            realLineHeight = 0, boxStart = 0, boxEnd = 0;\n\n        if (i === startLine) {\n          boxStart = this.__charBounds[startLine][startChar].left;\n        }\n        if (i >= startLine && i < endLine) {\n          boxEnd = isJustify && !this.isEndOfWrapping(i) ? this.width : this.getLineWidth(i) || 5; // WTF is this 5?\n        }\n        else if (i === endLine) {\n          if (endChar === 0) {\n            boxEnd = this.__charBounds[endLine][endChar].left;\n          }\n          else {\n            var charSpacing = this._getWidthOfCharSpacing();\n            boxEnd = this.__charBounds[endLine][endChar - 1].left\n              + this.__charBounds[endLine][endChar - 1].width - charSpacing;\n          }\n        }\n        realLineHeight = lineHeight;\n        if (this.lineHeight < 1 || (i === endLine && this.lineHeight > 1)) {\n          lineHeight /= this.lineHeight;\n        }\n        var drawStart = boundaries.left + lineOffset + boxStart,\n            drawWidth = boxEnd - boxStart,\n            drawHeight = lineHeight, extraTop = 0;\n        if (this.inCompositionMode) {\n          ctx.fillStyle = this.compositionColor || 'black';\n          drawHeight = 1;\n          extraTop = lineHeight;\n        }\n        else {\n          ctx.fillStyle = this.selectionColor;\n        }\n        if (this.direction === 'rtl') {\n          drawStart = this.width - drawStart - drawWidth;\n        }\n        ctx.fillRect(\n          drawStart,\n          boundaries.top + boundaries.topOffset + extraTop,\n          drawWidth,\n          drawHeight);\n        boundaries.topOffset += realLineHeight;\n      }\n    },\n\n    /**\n     * High level function to know the height of the cursor.\n     * the currentChar is the one that precedes the cursor\n     * Returns fontSize of char at the current cursor\n     * Unused from the library, is for the end user\n     * @return {Number} Character font size\n     */\n    getCurrentCharFontSize: function() {\n      var cp = this._getCurrentCharIndex();\n      return this.getValueOfPropertyAt(cp.l, cp.c, 'fontSize');\n    },\n\n    /**\n     * High level function to know the color of the cursor.\n     * the currentChar is the one that precedes the cursor\n     * Returns color (fill) of char at the current cursor\n     * if the text object has a pattern or gradient for filler, it will return that.\n     * Unused by the library, is for the end user\n     * @return {String | fabric.Gradient | fabric.Pattern} Character color (fill)\n     */\n    getCurrentCharColor: function() {\n      var cp = this._getCurrentCharIndex();\n      return this.getValueOfPropertyAt(cp.l, cp.c, 'fill');\n    },\n\n    /**\n     * Returns the cursor position for the getCurrent.. functions\n     * @private\n     */\n    _getCurrentCharIndex: function() {\n      var cursorPosition = this.get2DCursorLocation(this.selectionStart, true),\n          charIndex = cursorPosition.charIndex > 0 ? cursorPosition.charIndex - 1 : 0;\n      return { l: cursorPosition.lineIndex, c: charIndex };\n    }\n  });\n\n  /**\n   * Returns fabric.IText instance from an object representation\n   * @static\n   * @memberOf fabric.IText\n   * @param {Object} object Object to create an instance from\n   * @param {function} [callback] invoked with new instance as argument\n   */\n  fabric.IText.fromObject = function(object, callback) {\n    object.styles = fabric.util.stylesFromArray(object.styles, object.text);\n    parseDecoration(object);\n    if (object.styles) {\n      for (var i in object.styles) {\n        for (var j in object.styles[i]) {\n          parseDecoration(object.styles[i][j]);\n        }\n      }\n    }\n    fabric.Object._fromObject('IText', object, callback, 'text');\n  };\n})();\n\n\n(function() {\n\n  var clone = fabric.util.object.clone;\n\n  fabric.util.object.extend(fabric.IText.prototype, /** @lends fabric.IText.prototype */ {\n\n    /**\n     * Initializes all the interactive behavior of IText\n     */\n    initBehavior: function() {\n      this.initAddedHandler();\n      this.initRemovedHandler();\n      this.initCursorSelectionHandlers();\n      this.initDoubleClickSimulation();\n      this.mouseMoveHandler = this.mouseMoveHandler.bind(this);\n    },\n\n    onDeselect: function() {\n      this.isEditing && this.exitEditing();\n      this.selected = false;\n    },\n\n    /**\n     * Initializes \"added\" event handler\n     */\n    initAddedHandler: function() {\n      var _this = this;\n      this.on('added', function() {\n        var canvas = _this.canvas;\n        if (canvas) {\n          if (!canvas._hasITextHandlers) {\n            canvas._hasITextHandlers = true;\n            _this._initCanvasHandlers(canvas);\n          }\n          canvas._iTextInstances = canvas._iTextInstances || [];\n          canvas._iTextInstances.push(_this);\n        }\n      });\n    },\n\n    initRemovedHandler: function() {\n      var _this = this;\n      this.on('removed', function() {\n        var canvas = _this.canvas;\n        if (canvas) {\n          canvas._iTextInstances = canvas._iTextInstances || [];\n          fabric.util.removeFromArray(canvas._iTextInstances, _this);\n          if (canvas._iTextInstances.length === 0) {\n            canvas._hasITextHandlers = false;\n            _this._removeCanvasHandlers(canvas);\n          }\n        }\n      });\n    },\n\n    /**\n     * register canvas event to manage exiting on other instances\n     * @private\n     */\n    _initCanvasHandlers: function(canvas) {\n      canvas._mouseUpITextHandler = function() {\n        if (canvas._iTextInstances) {\n          canvas._iTextInstances.forEach(function(obj) {\n            obj.__isMousedown = false;\n          });\n        }\n      };\n      canvas.on('mouse:up', canvas._mouseUpITextHandler);\n    },\n\n    /**\n     * remove canvas event to manage exiting on other instances\n     * @private\n     */\n    _removeCanvasHandlers: function(canvas) {\n      canvas.off('mouse:up', canvas._mouseUpITextHandler);\n    },\n\n    /**\n     * @private\n     */\n    _tick: function() {\n      this._currentTickState = this._animateCursor(this, 1, this.cursorDuration, '_onTickComplete');\n    },\n\n    /**\n     * @private\n     */\n    _animateCursor: function(obj, targetOpacity, duration, completeMethod) {\n\n      var tickState;\n\n      tickState = {\n        isAborted: false,\n        abort: function() {\n          this.isAborted = true;\n        },\n      };\n\n      obj.animate('_currentCursorOpacity', targetOpacity, {\n        duration: duration,\n        onComplete: function() {\n          if (!tickState.isAborted) {\n            obj[completeMethod]();\n          }\n        },\n        onChange: function() {\n          // we do not want to animate a selection, only cursor\n          if (obj.canvas && obj.selectionStart === obj.selectionEnd) {\n            obj.renderCursorOrSelection();\n          }\n        },\n        abort: function() {\n          return tickState.isAborted;\n        }\n      });\n      return tickState;\n    },\n\n    /**\n     * @private\n     */\n    _onTickComplete: function() {\n\n      var _this = this;\n\n      if (this._cursorTimeout1) {\n        clearTimeout(this._cursorTimeout1);\n      }\n      this._cursorTimeout1 = setTimeout(function() {\n        _this._currentTickCompleteState = _this._animateCursor(_this, 0, this.cursorDuration / 2, '_tick');\n      }, 100);\n    },\n\n    /**\n     * Initializes delayed cursor\n     */\n    initDelayedCursor: function(restart) {\n      var _this = this,\n          delay = restart ? 0 : this.cursorDelay;\n\n      this.abortCursorAnimation();\n      this._currentCursorOpacity = 1;\n      this._cursorTimeout2 = setTimeout(function() {\n        _this._tick();\n      }, delay);\n    },\n\n    /**\n     * Aborts cursor animation and clears all timeouts\n     */\n    abortCursorAnimation: function() {\n      var shouldClear = this._currentTickState || this._currentTickCompleteState,\n          canvas = this.canvas;\n      this._currentTickState && this._currentTickState.abort();\n      this._currentTickCompleteState && this._currentTickCompleteState.abort();\n\n      clearTimeout(this._cursorTimeout1);\n      clearTimeout(this._cursorTimeout2);\n\n      this._currentCursorOpacity = 0;\n      // to clear just itext area we need to transform the context\n      // it may not be worth it\n      if (shouldClear && canvas) {\n        canvas.clearContext(canvas.contextTop || canvas.contextContainer);\n      }\n\n    },\n\n    /**\n     * Selects entire text\n     * @return {fabric.IText} thisArg\n     * @chainable\n     */\n    selectAll: function() {\n      this.selectionStart = 0;\n      this.selectionEnd = this._text.length;\n      this._fireSelectionChanged();\n      this._updateTextarea();\n      return this;\n    },\n\n    /**\n     * Returns selected text\n     * @return {String}\n     */\n    getSelectedText: function() {\n      return this._text.slice(this.selectionStart, this.selectionEnd).join('');\n    },\n\n    /**\n     * Find new selection index representing start of current word according to current selection index\n     * @param {Number} startFrom Current selection index\n     * @return {Number} New selection index\n     */\n    findWordBoundaryLeft: function(startFrom) {\n      var offset = 0, index = startFrom - 1;\n\n      // remove space before cursor first\n      if (this._reSpace.test(this._text[index])) {\n        while (this._reSpace.test(this._text[index])) {\n          offset++;\n          index--;\n        }\n      }\n      while (/\\S/.test(this._text[index]) && index > -1) {\n        offset++;\n        index--;\n      }\n\n      return startFrom - offset;\n    },\n\n    /**\n     * Find new selection index representing end of current word according to current selection index\n     * @param {Number} startFrom Current selection index\n     * @return {Number} New selection index\n     */\n    findWordBoundaryRight: function(startFrom) {\n      var offset = 0, index = startFrom;\n\n      // remove space after cursor first\n      if (this._reSpace.test(this._text[index])) {\n        while (this._reSpace.test(this._text[index])) {\n          offset++;\n          index++;\n        }\n      }\n      while (/\\S/.test(this._text[index]) && index < this._text.length) {\n        offset++;\n        index++;\n      }\n\n      return startFrom + offset;\n    },\n\n    /**\n     * Find new selection index representing start of current line according to current selection index\n     * @param {Number} startFrom Current selection index\n     * @return {Number} New selection index\n     */\n    findLineBoundaryLeft: function(startFrom) {\n      var offset = 0, index = startFrom - 1;\n\n      while (!/\\n/.test(this._text[index]) && index > -1) {\n        offset++;\n        index--;\n      }\n\n      return startFrom - offset;\n    },\n\n    /**\n     * Find new selection index representing end of current line according to current selection index\n     * @param {Number} startFrom Current selection index\n     * @return {Number} New selection index\n     */\n    findLineBoundaryRight: function(startFrom) {\n      var offset = 0, index = startFrom;\n\n      while (!/\\n/.test(this._text[index]) && index < this._text.length) {\n        offset++;\n        index++;\n      }\n\n      return startFrom + offset;\n    },\n\n    /**\n     * Finds index corresponding to beginning or end of a word\n     * @param {Number} selectionStart Index of a character\n     * @param {Number} direction 1 or -1\n     * @return {Number} Index of the beginning or end of a word\n     */\n    searchWordBoundary: function(selectionStart, direction) {\n      var text = this._text,\n          index     = this._reSpace.test(text[selectionStart]) ? selectionStart - 1 : selectionStart,\n          _char     = text[index],\n          // wrong\n          reNonWord = fabric.reNonWord;\n\n      while (!reNonWord.test(_char) && index > 0 && index < text.length) {\n        index += direction;\n        _char = text[index];\n      }\n      if (reNonWord.test(_char)) {\n        index += direction === 1 ? 0 : 1;\n      }\n      return index;\n    },\n\n    /**\n     * Selects a word based on the index\n     * @param {Number} selectionStart Index of a character\n     */\n    selectWord: function(selectionStart) {\n      selectionStart = selectionStart || this.selectionStart;\n      var newSelectionStart = this.searchWordBoundary(selectionStart, -1), /* search backwards */\n          newSelectionEnd = this.searchWordBoundary(selectionStart, 1); /* search forward */\n\n      this.selectionStart = newSelectionStart;\n      this.selectionEnd = newSelectionEnd;\n      this._fireSelectionChanged();\n      this._updateTextarea();\n      this.renderCursorOrSelection();\n    },\n\n    /**\n     * Selects a line based on the index\n     * @param {Number} selectionStart Index of a character\n     * @return {fabric.IText} thisArg\n     * @chainable\n     */\n    selectLine: function(selectionStart) {\n      selectionStart = selectionStart || this.selectionStart;\n      var newSelectionStart = this.findLineBoundaryLeft(selectionStart),\n          newSelectionEnd = this.findLineBoundaryRight(selectionStart);\n\n      this.selectionStart = newSelectionStart;\n      this.selectionEnd = newSelectionEnd;\n      this._fireSelectionChanged();\n      this._updateTextarea();\n      return this;\n    },\n\n    /**\n     * Enters editing state\n     * @return {fabric.IText} thisArg\n     * @chainable\n     */\n    enterEditing: function(e) {\n      if (this.isEditing || !this.editable) {\n        return;\n      }\n\n      if (this.canvas) {\n        this.canvas.calcOffset();\n        this.exitEditingOnOthers(this.canvas);\n      }\n\n      this.isEditing = true;\n\n      this.initHiddenTextarea(e);\n      this.hiddenTextarea.focus();\n      this.hiddenTextarea.value = this.text;\n      this._updateTextarea();\n      this._saveEditingProps();\n      this._setEditingProps();\n      this._textBeforeEdit = this.text;\n\n      this._tick();\n      this.fire('editing:entered');\n      this._fireSelectionChanged();\n      if (!this.canvas) {\n        return this;\n      }\n      this.canvas.fire('text:editing:entered', { target: this });\n      this.initMouseMoveHandler();\n      this.canvas.requestRenderAll();\n      return this;\n    },\n\n    exitEditingOnOthers: function(canvas) {\n      if (canvas._iTextInstances) {\n        canvas._iTextInstances.forEach(function(obj) {\n          obj.selected = false;\n          if (obj.isEditing) {\n            obj.exitEditing();\n          }\n        });\n      }\n    },\n\n    /**\n     * Initializes \"mousemove\" event handler\n     */\n    initMouseMoveHandler: function() {\n      this.canvas.on('mouse:move', this.mouseMoveHandler);\n    },\n\n    /**\n     * @private\n     */\n    mouseMoveHandler: function(options) {\n      if (!this.__isMousedown || !this.isEditing) {\n        return;\n      }\n\n      var newSelectionStart = this.getSelectionStartFromPointer(options.e),\n          currentStart = this.selectionStart,\n          currentEnd = this.selectionEnd;\n      if (\n        (newSelectionStart !== this.__selectionStartOnMouseDown || currentStart === currentEnd)\n        &&\n        (currentStart === newSelectionStart || currentEnd === newSelectionStart)\n      ) {\n        return;\n      }\n      if (newSelectionStart > this.__selectionStartOnMouseDown) {\n        this.selectionStart = this.__selectionStartOnMouseDown;\n        this.selectionEnd = newSelectionStart;\n      }\n      else {\n        this.selectionStart = newSelectionStart;\n        this.selectionEnd = this.__selectionStartOnMouseDown;\n      }\n      if (this.selectionStart !== currentStart || this.selectionEnd !== currentEnd) {\n        this.restartCursorIfNeeded();\n        this._fireSelectionChanged();\n        this._updateTextarea();\n        this.renderCursorOrSelection();\n      }\n    },\n\n    /**\n     * @private\n     */\n    _setEditingProps: function() {\n      this.hoverCursor = 'text';\n\n      if (this.canvas) {\n        this.canvas.defaultCursor = this.canvas.moveCursor = 'text';\n      }\n\n      this.borderColor = this.editingBorderColor;\n      this.hasControls = this.selectable = false;\n      this.lockMovementX = this.lockMovementY = true;\n    },\n\n    /**\n     * convert from textarea to grapheme indexes\n     */\n    fromStringToGraphemeSelection: function(start, end, text) {\n      var smallerTextStart = text.slice(0, start),\n          graphemeStart = fabric.util.string.graphemeSplit(smallerTextStart).length;\n      if (start === end) {\n        return { selectionStart: graphemeStart, selectionEnd: graphemeStart };\n      }\n      var smallerTextEnd = text.slice(start, end),\n          graphemeEnd = fabric.util.string.graphemeSplit(smallerTextEnd).length;\n      return { selectionStart: graphemeStart, selectionEnd: graphemeStart + graphemeEnd };\n    },\n\n    /**\n     * convert from fabric to textarea values\n     */\n    fromGraphemeToStringSelection: function(start, end, _text) {\n      var smallerTextStart = _text.slice(0, start),\n          graphemeStart = smallerTextStart.join('').length;\n      if (start === end) {\n        return { selectionStart: graphemeStart, selectionEnd: graphemeStart };\n      }\n      var smallerTextEnd = _text.slice(start, end),\n          graphemeEnd = smallerTextEnd.join('').length;\n      return { selectionStart: graphemeStart, selectionEnd: graphemeStart + graphemeEnd };\n    },\n\n    /**\n     * @private\n     */\n    _updateTextarea: function() {\n      this.cursorOffsetCache = { };\n      if (!this.hiddenTextarea) {\n        return;\n      }\n      if (!this.inCompositionMode) {\n        var newSelection = this.fromGraphemeToStringSelection(this.selectionStart, this.selectionEnd, this._text);\n        this.hiddenTextarea.selectionStart = newSelection.selectionStart;\n        this.hiddenTextarea.selectionEnd = newSelection.selectionEnd;\n      }\n      this.updateTextareaPosition();\n    },\n\n    /**\n     * @private\n     */\n    updateFromTextArea: function() {\n      if (!this.hiddenTextarea) {\n        return;\n      }\n      this.cursorOffsetCache = { };\n      this.text = this.hiddenTextarea.value;\n      if (this._shouldClearDimensionCache()) {\n        this.initDimensions();\n        this.setCoords();\n      }\n      var newSelection = this.fromStringToGraphemeSelection(\n        this.hiddenTextarea.selectionStart, this.hiddenTextarea.selectionEnd, this.hiddenTextarea.value);\n      this.selectionEnd = this.selectionStart = newSelection.selectionEnd;\n      if (!this.inCompositionMode) {\n        this.selectionStart = newSelection.selectionStart;\n      }\n      this.updateTextareaPosition();\n    },\n\n    /**\n     * @private\n     */\n    updateTextareaPosition: function() {\n      if (this.selectionStart === this.selectionEnd) {\n        var style = this._calcTextareaPosition();\n        this.hiddenTextarea.style.left = style.left;\n        this.hiddenTextarea.style.top = style.top;\n      }\n    },\n\n    /**\n     * @private\n     * @return {Object} style contains style for hiddenTextarea\n     */\n    _calcTextareaPosition: function() {\n      if (!this.canvas) {\n        return { x: 1, y: 1 };\n      }\n      var desiredPosition = this.inCompositionMode ? this.compositionStart : this.selectionStart,\n          boundaries = this._getCursorBoundaries(desiredPosition),\n          cursorLocation = this.get2DCursorLocation(desiredPosition),\n          lineIndex = cursorLocation.lineIndex,\n          charIndex = cursorLocation.charIndex,\n          charHeight = this.getValueOfPropertyAt(lineIndex, charIndex, 'fontSize') * this.lineHeight,\n          leftOffset = boundaries.leftOffset,\n          m = this.calcTransformMatrix(),\n          p = {\n            x: boundaries.left + leftOffset,\n            y: boundaries.top + boundaries.topOffset + charHeight\n          },\n          retinaScaling = this.canvas.getRetinaScaling(),\n          upperCanvas = this.canvas.upperCanvasEl,\n          upperCanvasWidth = upperCanvas.width / retinaScaling,\n          upperCanvasHeight = upperCanvas.height / retinaScaling,\n          maxWidth = upperCanvasWidth - charHeight,\n          maxHeight = upperCanvasHeight - charHeight,\n          scaleX = upperCanvas.clientWidth / upperCanvasWidth,\n          scaleY = upperCanvas.clientHeight / upperCanvasHeight;\n\n      p = fabric.util.transformPoint(p, m);\n      p = fabric.util.transformPoint(p, this.canvas.viewportTransform);\n      p.x *= scaleX;\n      p.y *= scaleY;\n      if (p.x < 0) {\n        p.x = 0;\n      }\n      if (p.x > maxWidth) {\n        p.x = maxWidth;\n      }\n      if (p.y < 0) {\n        p.y = 0;\n      }\n      if (p.y > maxHeight) {\n        p.y = maxHeight;\n      }\n\n      // add canvas offset on document\n      p.x += this.canvas._offset.left;\n      p.y += this.canvas._offset.top;\n\n      return { left: p.x + 'px', top: p.y + 'px', fontSize: charHeight + 'px', charHeight: charHeight };\n    },\n\n    /**\n     * @private\n     */\n    _saveEditingProps: function() {\n      this._savedProps = {\n        hasControls: this.hasControls,\n        borderColor: this.borderColor,\n        lockMovementX: this.lockMovementX,\n        lockMovementY: this.lockMovementY,\n        hoverCursor: this.hoverCursor,\n        selectable: this.selectable,\n        defaultCursor: this.canvas && this.canvas.defaultCursor,\n        moveCursor: this.canvas && this.canvas.moveCursor\n      };\n    },\n\n    /**\n     * @private\n     */\n    _restoreEditingProps: function() {\n      if (!this._savedProps) {\n        return;\n      }\n\n      this.hoverCursor = this._savedProps.hoverCursor;\n      this.hasControls = this._savedProps.hasControls;\n      this.borderColor = this._savedProps.borderColor;\n      this.selectable = this._savedProps.selectable;\n      this.lockMovementX = this._savedProps.lockMovementX;\n      this.lockMovementY = this._savedProps.lockMovementY;\n\n      if (this.canvas) {\n        this.canvas.defaultCursor = this._savedProps.defaultCursor;\n        this.canvas.moveCursor = this._savedProps.moveCursor;\n      }\n    },\n\n    /**\n     * Exits from editing state\n     * @return {fabric.IText} thisArg\n     * @chainable\n     */\n    exitEditing: function() {\n      var isTextChanged = (this._textBeforeEdit !== this.text);\n      var hiddenTextarea = this.hiddenTextarea;\n      this.selected = false;\n      this.isEditing = false;\n\n      this.selectionEnd = this.selectionStart;\n\n      if (hiddenTextarea) {\n        hiddenTextarea.blur && hiddenTextarea.blur();\n        hiddenTextarea.parentNode && hiddenTextarea.parentNode.removeChild(hiddenTextarea);\n      }\n      this.hiddenTextarea = null;\n      this.abortCursorAnimation();\n      this._restoreEditingProps();\n      this._currentCursorOpacity = 0;\n      if (this._shouldClearDimensionCache()) {\n        this.initDimensions();\n        this.setCoords();\n      }\n      this.fire('editing:exited');\n      isTextChanged && this.fire('modified');\n      if (this.canvas) {\n        this.canvas.off('mouse:move', this.mouseMoveHandler);\n        this.canvas.fire('text:editing:exited', { target: this });\n        isTextChanged && this.canvas.fire('object:modified', { target: this });\n      }\n      return this;\n    },\n\n    /**\n     * @private\n     */\n    _removeExtraneousStyles: function() {\n      for (var prop in this.styles) {\n        if (!this._textLines[prop]) {\n          delete this.styles[prop];\n        }\n      }\n    },\n\n    /**\n     * remove and reflow a style block from start to end.\n     * @param {Number} start linear start position for removal (included in removal)\n     * @param {Number} end linear end position for removal ( excluded from removal )\n     */\n    removeStyleFromTo: function(start, end) {\n      var cursorStart = this.get2DCursorLocation(start, true),\n          cursorEnd = this.get2DCursorLocation(end, true),\n          lineStart = cursorStart.lineIndex,\n          charStart = cursorStart.charIndex,\n          lineEnd = cursorEnd.lineIndex,\n          charEnd = cursorEnd.charIndex,\n          i, styleObj;\n      if (lineStart !== lineEnd) {\n        // step1 remove the trailing of lineStart\n        if (this.styles[lineStart]) {\n          for (i = charStart; i < this._unwrappedTextLines[lineStart].length; i++) {\n            delete this.styles[lineStart][i];\n          }\n        }\n        // step2 move the trailing of lineEnd to lineStart if needed\n        if (this.styles[lineEnd]) {\n          for (i = charEnd; i < this._unwrappedTextLines[lineEnd].length; i++) {\n            styleObj = this.styles[lineEnd][i];\n            if (styleObj) {\n              this.styles[lineStart] || (this.styles[lineStart] = { });\n              this.styles[lineStart][charStart + i - charEnd] = styleObj;\n            }\n          }\n        }\n        // step3 detects lines will be completely removed.\n        for (i = lineStart + 1; i <= lineEnd; i++) {\n          delete this.styles[i];\n        }\n        // step4 shift remaining lines.\n        this.shiftLineStyles(lineEnd, lineStart - lineEnd);\n      }\n      else {\n        // remove and shift left on the same line\n        if (this.styles[lineStart]) {\n          styleObj = this.styles[lineStart];\n          var diff = charEnd - charStart, numericChar, _char;\n          for (i = charStart; i < charEnd; i++) {\n            delete styleObj[i];\n          }\n          for (_char in this.styles[lineStart]) {\n            numericChar = parseInt(_char, 10);\n            if (numericChar >= charEnd) {\n              styleObj[numericChar - diff] = styleObj[_char];\n              delete styleObj[_char];\n            }\n          }\n        }\n      }\n    },\n\n    /**\n     * Shifts line styles up or down\n     * @param {Number} lineIndex Index of a line\n     * @param {Number} offset Can any number?\n     */\n    shiftLineStyles: function(lineIndex, offset) {\n      // shift all line styles by offset upward or downward\n      // do not clone deep. we need new array, not new style objects\n      var clonedStyles = clone(this.styles);\n      for (var line in this.styles) {\n        var numericLine = parseInt(line, 10);\n        if (numericLine > lineIndex) {\n          this.styles[numericLine + offset] = clonedStyles[numericLine];\n          if (!clonedStyles[numericLine - offset]) {\n            delete this.styles[numericLine];\n          }\n        }\n      }\n    },\n\n    restartCursorIfNeeded: function() {\n      if (!this._currentTickState || this._currentTickState.isAborted\n        || !this._currentTickCompleteState || this._currentTickCompleteState.isAborted\n      ) {\n        this.initDelayedCursor();\n      }\n    },\n\n    /**\n     * Handle insertion of more consecutive style lines for when one or more\n     * newlines gets added to the text. Since current style needs to be shifted\n     * first we shift the current style of the number lines needed, then we add\n     * new lines from the last to the first.\n     * @param {Number} lineIndex Index of a line\n     * @param {Number} charIndex Index of a char\n     * @param {Number} qty number of lines to add\n     * @param {Array} copiedStyle Array of objects styles\n     */\n    insertNewlineStyleObject: function(lineIndex, charIndex, qty, copiedStyle) {\n      var currentCharStyle,\n          newLineStyles = {},\n          somethingAdded = false,\n          isEndOfLine = this._unwrappedTextLines[lineIndex].length === charIndex;\n\n      qty || (qty = 1);\n      this.shiftLineStyles(lineIndex, qty);\n      if (this.styles[lineIndex]) {\n        currentCharStyle = this.styles[lineIndex][charIndex === 0 ? charIndex : charIndex - 1];\n      }\n      // we clone styles of all chars\n      // after cursor onto the current line\n      for (var index in this.styles[lineIndex]) {\n        var numIndex = parseInt(index, 10);\n        if (numIndex >= charIndex) {\n          somethingAdded = true;\n          newLineStyles[numIndex - charIndex] = this.styles[lineIndex][index];\n          // remove lines from the previous line since they're on a new line now\n          if (!(isEndOfLine && charIndex === 0)) {\n            delete this.styles[lineIndex][index];\n          }\n        }\n      }\n      var styleCarriedOver = false;\n      if (somethingAdded && !isEndOfLine) {\n        // if is end of line, the extra style we copied\n        // is probably not something we want\n        this.styles[lineIndex + qty] = newLineStyles;\n        styleCarriedOver = true;\n      }\n      if (styleCarriedOver) {\n        // skip the last line of since we already prepared it.\n        qty--;\n      }\n      // for the all the lines or all the other lines\n      // we clone current char style onto the next (otherwise empty) line\n      while (qty > 0) {\n        if (copiedStyle && copiedStyle[qty - 1]) {\n          this.styles[lineIndex + qty] = { 0: clone(copiedStyle[qty - 1]) };\n        }\n        else if (currentCharStyle) {\n          this.styles[lineIndex + qty] = { 0: clone(currentCharStyle) };\n        }\n        else {\n          delete this.styles[lineIndex + qty];\n        }\n        qty--;\n      }\n      this._forceClearCache = true;\n    },\n\n    /**\n     * Inserts style object for a given line/char index\n     * @param {Number} lineIndex Index of a line\n     * @param {Number} charIndex Index of a char\n     * @param {Number} quantity number Style object to insert, if given\n     * @param {Array} copiedStyle array of style objects\n     */\n    insertCharStyleObject: function(lineIndex, charIndex, quantity, copiedStyle) {\n      if (!this.styles) {\n        this.styles = {};\n      }\n      var currentLineStyles       = this.styles[lineIndex],\n          currentLineStylesCloned = currentLineStyles ? clone(currentLineStyles) : {};\n\n      quantity || (quantity = 1);\n      // shift all char styles by quantity forward\n      // 0,1,2,3 -> (charIndex=2) -> 0,1,3,4 -> (insert 2) -> 0,1,2,3,4\n      for (var index in currentLineStylesCloned) {\n        var numericIndex = parseInt(index, 10);\n        if (numericIndex >= charIndex) {\n          currentLineStyles[numericIndex + quantity] = currentLineStylesCloned[numericIndex];\n          // only delete the style if there was nothing moved there\n          if (!currentLineStylesCloned[numericIndex - quantity]) {\n            delete currentLineStyles[numericIndex];\n          }\n        }\n      }\n      this._forceClearCache = true;\n      if (copiedStyle) {\n        while (quantity--) {\n          if (!Object.keys(copiedStyle[quantity]).length) {\n            continue;\n          }\n          if (!this.styles[lineIndex]) {\n            this.styles[lineIndex] = {};\n          }\n          this.styles[lineIndex][charIndex + quantity] = clone(copiedStyle[quantity]);\n        }\n        return;\n      }\n      if (!currentLineStyles) {\n        return;\n      }\n      var newStyle = currentLineStyles[charIndex ? charIndex - 1 : 1];\n      while (newStyle && quantity--) {\n        this.styles[lineIndex][charIndex + quantity] = clone(newStyle);\n      }\n    },\n\n    /**\n     * Inserts style object(s)\n     * @param {Array} insertedText Characters at the location where style is inserted\n     * @param {Number} start cursor index for inserting style\n     * @param {Array} [copiedStyle] array of style objects to insert.\n     */\n    insertNewStyleBlock: function(insertedText, start, copiedStyle) {\n      var cursorLoc = this.get2DCursorLocation(start, true),\n          addedLines = [0], linesLength = 0;\n      // get an array of how many char per lines are being added.\n      for (var i = 0; i < insertedText.length; i++) {\n        if (insertedText[i] === '\\n') {\n          linesLength++;\n          addedLines[linesLength] = 0;\n        }\n        else {\n          addedLines[linesLength]++;\n        }\n      }\n      // for the first line copy the style from the current char position.\n      if (addedLines[0] > 0) {\n        this.insertCharStyleObject(cursorLoc.lineIndex, cursorLoc.charIndex, addedLines[0], copiedStyle);\n        copiedStyle = copiedStyle && copiedStyle.slice(addedLines[0] + 1);\n      }\n      linesLength && this.insertNewlineStyleObject(\n        cursorLoc.lineIndex, cursorLoc.charIndex + addedLines[0], linesLength);\n      for (var i = 1; i < linesLength; i++) {\n        if (addedLines[i] > 0) {\n          this.insertCharStyleObject(cursorLoc.lineIndex + i, 0, addedLines[i], copiedStyle);\n        }\n        else if (copiedStyle) {\n          // this test is required in order to close #6841\n          // when a pasted buffer begins with a newline then\n          // this.styles[cursorLoc.lineIndex + i] and copiedStyle[0]\n          // may be undefined for some reason\n          if (this.styles[cursorLoc.lineIndex + i] && copiedStyle[0]) {\n            this.styles[cursorLoc.lineIndex + i][0] = copiedStyle[0];\n          }\n        }\n        copiedStyle = copiedStyle && copiedStyle.slice(addedLines[i] + 1);\n      }\n      // we use i outside the loop to get it like linesLength\n      if (addedLines[i] > 0) {\n        this.insertCharStyleObject(cursorLoc.lineIndex + i, 0, addedLines[i], copiedStyle);\n      }\n    },\n\n    /**\n     * Set the selectionStart and selectionEnd according to the new position of cursor\n     * mimic the key - mouse navigation when shift is pressed.\n     */\n    setSelectionStartEndWithShift: function(start, end, newSelection) {\n      if (newSelection <= start) {\n        if (end === start) {\n          this._selectionDirection = 'left';\n        }\n        else if (this._selectionDirection === 'right') {\n          this._selectionDirection = 'left';\n          this.selectionEnd = start;\n        }\n        this.selectionStart = newSelection;\n      }\n      else if (newSelection > start && newSelection < end) {\n        if (this._selectionDirection === 'right') {\n          this.selectionEnd = newSelection;\n        }\n        else {\n          this.selectionStart = newSelection;\n        }\n      }\n      else {\n        // newSelection is > selection start and end\n        if (end === start) {\n          this._selectionDirection = 'right';\n        }\n        else if (this._selectionDirection === 'left') {\n          this._selectionDirection = 'right';\n          this.selectionStart = end;\n        }\n        this.selectionEnd = newSelection;\n      }\n    },\n\n    setSelectionInBoundaries: function() {\n      var length = this.text.length;\n      if (this.selectionStart > length) {\n        this.selectionStart = length;\n      }\n      else if (this.selectionStart < 0) {\n        this.selectionStart = 0;\n      }\n      if (this.selectionEnd > length) {\n        this.selectionEnd = length;\n      }\n      else if (this.selectionEnd < 0) {\n        this.selectionEnd = 0;\n      }\n    }\n  });\n})();\n\n\nfabric.util.object.extend(fabric.IText.prototype, /** @lends fabric.IText.prototype */ {\n  /**\n   * Initializes \"dbclick\" event handler\n   */\n  initDoubleClickSimulation: function() {\n\n    // for double click\n    this.__lastClickTime = +new Date();\n\n    // for triple click\n    this.__lastLastClickTime = +new Date();\n\n    this.__lastPointer = { };\n\n    this.on('mousedown', this.onMouseDown);\n  },\n\n  /**\n   * Default event handler to simulate triple click\n   * @private\n   */\n  onMouseDown: function(options) {\n    if (!this.canvas) {\n      return;\n    }\n    this.__newClickTime = +new Date();\n    var newPointer = options.pointer;\n    if (this.isTripleClick(newPointer)) {\n      this.fire('tripleclick', options);\n      this._stopEvent(options.e);\n    }\n    this.__lastLastClickTime = this.__lastClickTime;\n    this.__lastClickTime = this.__newClickTime;\n    this.__lastPointer = newPointer;\n    this.__lastIsEditing = this.isEditing;\n    this.__lastSelected = this.selected;\n  },\n\n  isTripleClick: function(newPointer) {\n    return this.__newClickTime - this.__lastClickTime < 500 &&\n        this.__lastClickTime - this.__lastLastClickTime < 500 &&\n        this.__lastPointer.x === newPointer.x &&\n        this.__lastPointer.y === newPointer.y;\n  },\n\n  /**\n   * @private\n   */\n  _stopEvent: function(e) {\n    e.preventDefault && e.preventDefault();\n    e.stopPropagation && e.stopPropagation();\n  },\n\n  /**\n   * Initializes event handlers related to cursor or selection\n   */\n  initCursorSelectionHandlers: function() {\n    this.initMousedownHandler();\n    this.initMouseupHandler();\n    this.initClicks();\n  },\n\n  /**\n   * Default handler for double click, select a word\n   */\n  doubleClickHandler: function(options) {\n    if (!this.isEditing) {\n      return;\n    }\n    this.selectWord(this.getSelectionStartFromPointer(options.e));\n  },\n\n  /**\n   * Default handler for triple click, select a line\n   */\n  tripleClickHandler: function(options) {\n    if (!this.isEditing) {\n      return;\n    }\n    this.selectLine(this.getSelectionStartFromPointer(options.e));\n  },\n\n  /**\n   * Initializes double and triple click event handlers\n   */\n  initClicks: function() {\n    this.on('mousedblclick', this.doubleClickHandler);\n    this.on('tripleclick', this.tripleClickHandler);\n  },\n\n  /**\n   * Default event handler for the basic functionalities needed on _mouseDown\n   * can be overridden to do something different.\n   * Scope of this implementation is: find the click position, set selectionStart\n   * find selectionEnd, initialize the drawing of either cursor or selection area\n   * initializing a mousedDown on a text area will cancel fabricjs knowledge of\n   * current compositionMode. It will be set to false.\n   */\n  _mouseDownHandler: function(options) {\n    if (!this.canvas || !this.editable || (options.e.button && options.e.button !== 1)) {\n      return;\n    }\n\n    this.__isMousedown = true;\n\n    if (this.selected) {\n      this.inCompositionMode = false;\n      this.setCursorByClick(options.e);\n    }\n\n    if (this.isEditing) {\n      this.__selectionStartOnMouseDown = this.selectionStart;\n      if (this.selectionStart === this.selectionEnd) {\n        this.abortCursorAnimation();\n      }\n      this.renderCursorOrSelection();\n    }\n  },\n\n  /**\n   * Default event handler for the basic functionalities needed on mousedown:before\n   * can be overridden to do something different.\n   * Scope of this implementation is: verify the object is already selected when mousing down\n   */\n  _mouseDownHandlerBefore: function(options) {\n    if (!this.canvas || !this.editable || (options.e.button && options.e.button !== 1)) {\n      return;\n    }\n    // we want to avoid that an object that was selected and then becomes unselectable,\n    // may trigger editing mode in some way.\n    this.selected = this === this.canvas._activeObject;\n  },\n\n  /**\n   * Initializes \"mousedown\" event handler\n   */\n  initMousedownHandler: function() {\n    this.on('mousedown', this._mouseDownHandler);\n    this.on('mousedown:before', this._mouseDownHandlerBefore);\n  },\n\n  /**\n   * Initializes \"mouseup\" event handler\n   */\n  initMouseupHandler: function() {\n    this.on('mouseup', this.mouseUpHandler);\n  },\n\n  /**\n   * standard handler for mouse up, overridable\n   * @private\n   */\n  mouseUpHandler: function(options) {\n    this.__isMousedown = false;\n    if (!this.editable || this.group ||\n      (options.transform && options.transform.actionPerformed) ||\n      (options.e.button && options.e.button !== 1)) {\n      return;\n    }\n\n    if (this.canvas) {\n      var currentActive = this.canvas._activeObject;\n      if (currentActive && currentActive !== this) {\n        // avoid running this logic when there is an active object\n        // this because is possible with shift click and fast clicks,\n        // to rapidly deselect and reselect this object and trigger an enterEdit\n        return;\n      }\n    }\n\n    if (this.__lastSelected && !this.__corner) {\n      this.selected = false;\n      this.__lastSelected = false;\n      this.enterEditing(options.e);\n      if (this.selectionStart === this.selectionEnd) {\n        this.initDelayedCursor(true);\n      }\n      else {\n        this.renderCursorOrSelection();\n      }\n    }\n    else {\n      this.selected = true;\n    }\n  },\n\n  /**\n   * Changes cursor location in a text depending on passed pointer (x/y) object\n   * @param {Event} e Event object\n   */\n  setCursorByClick: function(e) {\n    var newSelection = this.getSelectionStartFromPointer(e),\n        start = this.selectionStart, end = this.selectionEnd;\n    if (e.shiftKey) {\n      this.setSelectionStartEndWithShift(start, end, newSelection);\n    }\n    else {\n      this.selectionStart = newSelection;\n      this.selectionEnd = newSelection;\n    }\n    if (this.isEditing) {\n      this._fireSelectionChanged();\n      this._updateTextarea();\n    }\n  },\n\n  /**\n   * Returns index of a character corresponding to where an object was clicked\n   * @param {Event} e Event object\n   * @return {Number} Index of a character\n   */\n  getSelectionStartFromPointer: function(e) {\n    var mouseOffset = this.getLocalPointer(e),\n        prevWidth = 0,\n        width = 0,\n        height = 0,\n        charIndex = 0,\n        lineIndex = 0,\n        lineLeftOffset,\n        line;\n    for (var i = 0, len = this._textLines.length; i < len; i++) {\n      if (height <= mouseOffset.y) {\n        height += this.getHeightOfLine(i) * this.scaleY;\n        lineIndex = i;\n        if (i > 0) {\n          charIndex += this._textLines[i - 1].length + this.missingNewlineOffset(i - 1);\n        }\n      }\n      else {\n        break;\n      }\n    }\n    lineLeftOffset = this._getLineLeftOffset(lineIndex);\n    width = lineLeftOffset * this.scaleX;\n    line = this._textLines[lineIndex];\n    // handling of RTL: in order to get things work correctly,\n    // we assume RTL writing is mirrored compared to LTR writing.\n    // so in position detection we mirror the X offset, and when is time\n    // of rendering it, we mirror it again.\n    if (this.direction === 'rtl') {\n      mouseOffset.x = this.width * this.scaleX - mouseOffset.x + width;\n    }\n    for (var j = 0, jlen = line.length; j < jlen; j++) {\n      prevWidth = width;\n      // i removed something about flipX here, check.\n      width += this.__charBounds[lineIndex][j].kernedWidth * this.scaleX;\n      if (width <= mouseOffset.x) {\n        charIndex++;\n      }\n      else {\n        break;\n      }\n    }\n    return this._getNewSelectionStartFromOffset(mouseOffset, prevWidth, width, charIndex, jlen);\n  },\n\n  /**\n   * @private\n   */\n  _getNewSelectionStartFromOffset: function(mouseOffset, prevWidth, width, index, jlen) {\n    // we need Math.abs because when width is after the last char, the offset is given as 1, while is 0\n    var distanceBtwLastCharAndCursor = mouseOffset.x - prevWidth,\n        distanceBtwNextCharAndCursor = width - mouseOffset.x,\n        offset = distanceBtwNextCharAndCursor > distanceBtwLastCharAndCursor ||\n          distanceBtwNextCharAndCursor < 0 ? 0 : 1,\n        newSelectionStart = index + offset;\n    // if object is horizontally flipped, mirror cursor location from the end\n    if (this.flipX) {\n      newSelectionStart = jlen - newSelectionStart;\n    }\n\n    if (newSelectionStart > this._text.length) {\n      newSelectionStart = this._text.length;\n    }\n\n    return newSelectionStart;\n  }\n});\n\n\nfabric.util.object.extend(fabric.IText.prototype, /** @lends fabric.IText.prototype */ {\n\n  /**\n   * Initializes hidden textarea (needed to bring up keyboard in iOS)\n   */\n  initHiddenTextarea: function() {\n    this.hiddenTextarea = fabric.document.createElement('textarea');\n    this.hiddenTextarea.setAttribute('autocapitalize', 'off');\n    this.hiddenTextarea.setAttribute('autocorrect', 'off');\n    this.hiddenTextarea.setAttribute('autocomplete', 'off');\n    this.hiddenTextarea.setAttribute('spellcheck', 'false');\n    this.hiddenTextarea.setAttribute('data-fabric-hiddentextarea', '');\n    this.hiddenTextarea.setAttribute('wrap', 'off');\n    var style = this._calcTextareaPosition();\n    // line-height: 1px; was removed from the style to fix this:\n    // https://bugs.chromium.org/p/chromium/issues/detail?id=870966\n    this.hiddenTextarea.style.cssText = 'position: absolute; top: ' + style.top +\n    '; left: ' + style.left + '; z-index: -999; opacity: 0; width: 1px; height: 1px; font-size: 1px;' +\n    ' padding\uff70top: ' + style.fontSize + ';';\n\n    if (this.hiddenTextareaContainer) {\n      this.hiddenTextareaContainer.appendChild(this.hiddenTextarea);\n    }\n    else {\n      fabric.document.body.appendChild(this.hiddenTextarea);\n    }\n\n    fabric.util.addListener(this.hiddenTextarea, 'keydown', this.onKeyDown.bind(this));\n    fabric.util.addListener(this.hiddenTextarea, 'keyup', this.onKeyUp.bind(this));\n    fabric.util.addListener(this.hiddenTextarea, 'input', this.onInput.bind(this));\n    fabric.util.addListener(this.hiddenTextarea, 'copy', this.copy.bind(this));\n    fabric.util.addListener(this.hiddenTextarea, 'cut', this.copy.bind(this));\n    fabric.util.addListener(this.hiddenTextarea, 'paste', this.paste.bind(this));\n    fabric.util.addListener(this.hiddenTextarea, 'compositionstart', this.onCompositionStart.bind(this));\n    f