define(["jquery", "flatpickr"], function ($) {
    "use strict";

    return {
        init: function (config) {
            const { trendsUrl, downloadUrl, downloadIcon } = config;

            const noResultsMessage =
                "No se encontraron resultados con los criterios de búsqueda. Pruebe con otros criterios.";

            let allData = [];
            let currentPage = 1;
            let totalPages = 0;
            let totalItems = 0;
            let isInitialLoad = true;

            flatpickr.localize({
                months: {
                    longhand: [
                        "Enero",
                        "Febrero",
                        "Marzo",
                        "Abril",
                        "Mayo",
                        "Junio",
                        "Julio",
                        "Agosto",
                        "Septiembre",
                        "Octubre",
                        "Noviembre",
                        "Diciembre",
                    ],
                    shorthand: [
                        "Ene",
                        "Feb",
                        "Mar",
                        "Abr",
                        "May",
                        "Jun",
                        "Jul",
                        "Ago",
                        "Sep",
                        "Oct",
                        "Nov",
                        "Dic",
                    ],
                },
                weekdays: {
                    longhand: [
                        "Domingo",
                        "Lunes",
                        "Martes",
                        "Miércoles",
                        "Jueves",
                        "Viernes",
                        "Sábado",
                    ],
                    shorthand: [
                        "Dom",
                        "Lun",
                        "Mar",
                        "Mié",
                        "Jue",
                        "Vie",
                        "Sáb",
                    ],
                },
                rangeSeparator: " / ",
                firstDayOfWeek: 1,
            });

            const datePicker = flatpickr("#date-range", {
                mode: "range",
                maxDate: "today",
                dateFormat: "Y-m-d",
                altInput: true,
                altFormat: "d M y",
            });

            function getDefaultDates() {
                const today = new Date();
                const yesterday = new Date();
                yesterday.setDate(today.getDate() - 1);

                return {
                    start: yesterday.toISOString().split("T")[0],
                    end: today.toISOString().split("T")[0],
                };
            }

            function parseDateInput(input) {
                if (!input) return null;

                const parts = input.split(" / ");
                if (parts.length === 2) {
                    return {
                        start: parts[0],
                        end: parts[1],
                    };
                }

                if (parts.length === 1) {
                    const today = new Date().toISOString().split("T")[0];
                    const selectedDate = parts[0];

                    if (selectedDate > today) {
                        return {
                            start: today,
                            end: selectedDate,
                        };
                    } else {
                        return {
                            start: selectedDate,
                            end: today,
                        };
                    }
                }

                return null;
            }

            function getDateParams() {
                const input = $("#date-range").val();
                const parsedDates = parseDateInput(input);

                if (!parsedDates && (isInitialLoad || !input)) {
                    return getDefaultDates();
                }

                return parsedDates || getDefaultDates();
            }

            function makeAjaxRequest(params, successCallback, errorCallback) {
                const defaultErrorHandler = (xhr) => {
                    console.error("Error:", xhr.responseText);
                    alert("Ocurrió un error al procesar la solicitud.");
                };

                $.ajax({
                    url: trendsUrl,
                    data: params,
                    method: "GET",
                    dataType: "json",
                    success: successCallback,
                    error: errorCallback || defaultErrorHandler,
                });
            }

            function getCommonParams() {
                const dateParams = getDateParams();
                const restaurantId = $("#restaurant").val();

                if (!restaurantId) {
                    alert("Selecciona un restaurant");
                    return null;
                }

                return {
                    restaurant_id: restaurantId,
                    start_date: dateParams.start,
                    end_date: dateParams.end,
                };
            }

            function processResponseData(response, isFullExport = false) {
                if (
                    !response?.items?.predictions ||
                    response?.items?.predictions?.length === 0
                ) {
                    $("#trend-table tbody")
                        .empty()
                        .append(
                            `<tr><td colspan='8'>${noResultsMessage}</td></tr>`
                        );
                    if (!isFullExport) {
                        $("#pagination").empty();
                    }
                    return false;
                }

                allData = response?.items?.predictions;

                if (!isFullExport) {
                    currentPage = response.currentPage || 1;
                    totalPages = response.totalPages || 1;
                    totalItems = response.total || 0;
                }

                return true;
            }

            function renderTable() {
                const tbody = $("#trend-table tbody");
                tbody.empty();

                allData.forEach((pred) => {
                    pred.tables.forEach((table) => {
                        const hours = table.by_hour
                            .map((h) => `${h.hour} (${h.total_customers})`)
                            .join(", ");

                        const row = `
                <tr>
                    <td>${pred.date}</td>
                    <td>${pred.total_reservations}</td>
                    <td>${pred.total_customers}</td>
                    <td>${table.table_id}</td>
                    <td>${table.table_label}</td>
                    <td>${table.total_customers}</td>
                    <td>${hours}</td>
                    <td>
                        <a href="#" 
                           class="download-btn" 
                           data-date="${pred.date}"
                           data-reservations="${pred.total_reservations}"
                           data-customers="${pred.total_customers}"
                           data-table-id="${table.table_id}"
                           data-table-label="${table.table_label}"
                           data-table-customers="${table.total_customers}"
                           data-hours="${hours}">
                            <img src="${downloadIcon}" alt="Download" />
                        </a>
                    </td>
                </tr>
            `;
                        tbody.append(row);
                    });
                });
            }

            function setupDownloadButtons() {
                $(document)
                    .off("click", ".download-btn")
                    .on("click", ".download-btn", function (e) {
                        e.preventDefault();
                        const format = $(
                            'input[name="export_format"]:checked'
                        ).val();
                        const $btn = $(this);

                        const data = [
                            {
                                Fecha: $btn.data("date"),
                                "Total de reservas": $btn.data("reservations"),
                                "Total de clientes": $btn.data("customers"),
                                ID: $btn.data("table-id"),
                                Mesa: $btn.data("table-label"),
                                "No. Clientes": $btn.data("table-customers"),
                                Horas: $btn.data("hours"),
                            },
                        ];

                        const downloadUrlWithParams = `${downloadUrl}?data=${encodeURIComponent(
                            JSON.stringify(data)
                        )}&format=${format}&filename=export_${$btn.data(
                            "date"
                        )}_${$btn.data("table-id")}`;

                        const a = document.createElement("a");
                        a.href = downloadUrlWithParams;
                        a.target = "_blank";
                        a.download = `export_${$btn.data("date")}_${$btn.data(
                            "table-id"
                        )}.${format}`;
                        document.body.appendChild(a);
                        a.click();
                        document.body.removeChild(a);
                    });
            }

            function renderPagination() {
                const container = $("#pagination");
                container.empty().addClass("pagination");

                const createPageButton = (pageNum, isActive = false) => {
                    const btn = $("<button>")
                        .addClass("pagination-btn")
                        .text(pageNum);

                    if (isActive) {
                        btn.addClass("active");
                    }

                    btn.on("click", (e) => {
                        e.preventDefault();
                        const pageAsNumber = Number(pageNum);
                        if (pageAsNumber !== currentPage) {
                            currentPage = pageAsNumber;
                            loadPage(currentPage);
                        }
                    });

                    return btn;
                };

                if (currentPage > 1) {
                    container.append(
                        $("<button>")
                            .addClass("pagination-btn pagination-prev")
                            .text("←")
                            .on("click", (e) => {
                                e.preventDefault();
                                currentPage--;
                                loadPage(currentPage);
                            })
                    );
                }

                const range = 2;
                const pages = [];

                if (totalPages > 0) pages.push(1);

                const start = Math.max(2, currentPage - range);
                const end = Math.min(totalPages - 1, currentPage + range);

                if (start > 2) pages.push("...");

                for (let i = start; i <= end; i++) {
                    pages.push(i);
                }

                if (end < totalPages - 1) pages.push("...");

                if (totalPages > 1) pages.push(totalPages);

                pages.forEach((page) => {
                    if (page === "...") {
                        container.append(
                            $("<span>")
                                .addClass("pagination-ellipsis")
                                .text("...")
                        );
                    } else {
                        container.append(
                            createPageButton(page, page == currentPage)
                        );
                    }
                });

                if (currentPage < totalPages) {
                    container.append(
                        $("<button>")
                            .addClass("pagination-btn pagination-next")
                            .text("→")
                            .on("click", (e) => {
                                e.preventDefault();
                                currentPage++;
                                loadPage(currentPage);
                            })
                    );
                }
            }

            function loadPage(page = 1) {
                const commonParams = getCommonParams();
                if (!commonParams) return;

                const params = {
                    ...commonParams,
                    page: page,
                    limit: 20,
                };

                makeAjaxRequest(params, (response) => {
                    if (processResponseData(response)) {
                        renderTable();
                        setupDownloadButtons();
                        renderPagination();
                    }
                });

                isInitialLoad = false;
            }

            function downloadAllData() {
                const commonParams = getCommonParams();
                if (!commonParams) return;

                const format = $('input[name="export_format"]:checked').val();

                makeAjaxRequest(
                    {
                        ...commonParams,
                        page: null,
                        limit: null,
                    },
                    (response) => {
                        if (processResponseData(response, true)) {
                            const allRows = allData.flatMap((pred) =>
                                pred.tables.map((table) => ({
                                    Fecha: pred.date,
                                    "Total de reservas":
                                        pred.total_reservations,
                                    "Total de clientes": pred.total_customers,
                                    ID: table.table_id,
                                    Mesa: table.table_label,
                                    "No. Clientes": table.total_customers,
                                    Horas: table.by_hour
                                        .map(
                                            (h) =>
                                                `${h.hour} (${h.total_customers})`
                                        )
                                        .join(", "),
                                }))
                            );

                            const dataParam = encodeURIComponent(
                                JSON.stringify(allRows)
                            );
                            const filename = `export_completo_${commonParams.restaurant_id}_${commonParams.start_date}_a_${commonParams.end_date}`;
                            const downloadUrlWithParams = `${downloadUrl}?data=${dataParam}&format=${format}&filename=${filename}`;

                            const a = document.createElement("a");
                            a.href = downloadUrlWithParams;
                            a.target = "_blank";
                            a.download = `${filename}.${format}`;
                            document.body.appendChild(a);
                            a.click();
                            document.body.removeChild(a);
                        }
                    }
                );
            }

            function initialize() {
                $(document).ready(function () {
                    const defaultDates = getDefaultDates();
                    datePicker.setDate([defaultDates.start, defaultDates.end]);
                    loadPage();
                });

                $("#load-data").on("click", () => loadPage());
                $("#download-all").on("click", downloadAllData);
                $('input[name="export_format"]').on(
                    "change",
                    setupDownloadButtons
                );

                $(".download-button").on("click", function (event) {
                    event.stopPropagation();
                    $(".download-modal").toggle();
                });

                $(document).on("click", function (e) {
                    if (!$(e.target).closest(".download-container").length) {
                        $(".download-modal").hide();
                    }
                });
            }

            initialize();
        },
    };
});
