define(["jquery", "uiComponent", "ko", "flatpickr", "libChart"], function (
    $,
    Component,
    ko,
) {
    "use strict";

    ko.observableGroup = (observables) => {
        let observableManager = {};
        let throttle = 0;
        let throttleTimeout;

        observableManager.throttle = (duration) => {
            throttle = duration;
            return observableManager;
        };

        observableManager.subscribe = (handler) => {
            const throttledHandler = (val) => {
                if (throttle > 0) {
                    if (!throttleTimeout) {
                        throttleTimeout = setTimeout(() => {
                            throttleTimeout = undefined;
                            handler(val);
                        }, throttle);
                    }
                } else {
                    handler(val);
                }
            };

            for (var i = 0; i < observables.length; i++) {
                observables[i].subscribe(throttledHandler);
            }

            return observableManager;
        };

        return observableManager;
    };

    const getType = (obj) => {
        if (
            obj &&
            typeof obj === "object" &&
            obj.constructor == new Date().constructor
        )
            return "date";
        return typeof obj;
    };

    const scanForObservablesIn = (model, subscribables) => {
        for (const parameter in model) {
            const typeOfData = getType(model[parameter]);
            switch (typeOfData) {
                case "object":
                    scanForObservablesIn(model[parameter], subscribables);
                    break;
                case "array":
                    const underlyingArray = model[parameter]();
                    underlyingArray.forEach((entry, index) => {
                        scanForObservablesIn(
                            underlyingArray[index],
                            subscribables
                        );
                    });
                    break;
                default:
                    if (
                        ko.isComputed(model[parameter]) ||
                        ko.isObservable(model[parameter])
                    ) {
                        subscribables.push(model[parameter]);
                    }
                    break;
            }
        }
    };

    ko.bindingHandlers.chart = {
        init: function (element, valueAccessor, allBindingsAccessor) {
            const chartBinding = valueAccessor();
            const chartType = ko.unwrap(chartBinding.type);
            const data = ko.toJS(chartBinding.data);
            const options = ko.toJS(chartBinding.options) || {};

            element.chart = new Chart(element, {
                type: chartType,
                data: data,
                options: options,
            });
        },
        update: function (element, valueAccessor) {
            const chartBinding = valueAccessor();
            const data = ko.toJS(chartBinding.data);

            if (element.chart) {
                element.chart.data.labels = data.labels;
                element.chart.data.datasets = data.datasets;
                element.chart.update();
            }
        },
    };

    return Component.extend({
        defaults: {
            template: "Threedadv_Customer/view/dashboard",
        },
        baseUrl: window.origin,
        ordersPeriod: ko.observable("24hrs"),
        ordersByPeriodLabels: ko.observableArray([]),
        ordersByPeriodData: ko.observableArray([]),
        totalByGuest: ko.observable(0),
        totalByRegister: ko.observable(0),
        customersByPeriodLabels: ko.observableArray([]),
        customersByPeriodData: ko.observableArray([]),
        customersPeriod: ko.observable("24hrs"),

        trendsData: ko.observable([]),
        restaurantId: ko.observable(""),
        rangeDates: ko.observable(""),
        startDate: ko.observable(""),
        endDate: ko.observable(""),
        restaurantOptions: ko.observableArray([]),

        customersReservationsLabels: ko.observableArray([]),
        totalCustomersByDate: ko.observableArray([]),
        totalReservationsByDate: ko.observableArray([]),

        customersByTableLabels: ko.observableArray([]),
        customersByTableData: ko.observableArray([]),

        customersByTableHoursLabels: ko.observableArray([]),
        customersByTableHoursData: ko.observableArray([]),

        periodOptions: ko.observableArray([
            { value: "24hrs", text: "Ultimas 24hrs" },
            { value: "today", text: "Hoy" },
            { value: "7d", text: "Ultimos 7 Días" },
            { value: "1m", text: "Este Mes" },
            { value: "1y", text: "Año" },
        ]),

        headers: {
            "x-requested-with": "XMLHttpRequest",
        },

        initialize: function (config) {
            const self = this;
            this._super();

            const { restaurantOptions, restaurateur_id } = config;

            this.restaurantOptions = restaurantOptions;

            this.restaurateurId = restaurateur_id;

            this.ordersByPeriod = ko.computed(() =>
                this.getConfigOrdersByPeriod()
            );
            this.ordersByCustomerType = ko.computed(() =>
                this.getConfigOrdersByCustomerType()
            );
            this.customersByPeriod = ko.computed(() =>
                this.getConfigCustomersByPeriod()
            );

            this.getOrdersByPeriod(this.ordersPeriod());
            this.getDataOrdersByCustomerType();
            this.getCustomersByPeriod(this.customersPeriod());

            this.ordersPeriod.subscribe(function (period) {
                self.getOrdersByPeriod(period);
            });

            this.customersPeriod.subscribe(function (period) {
                self.getCustomersByPeriod(period);
            });

            this.rangeDates.subscribe(function (range) {
                const [start, end] = range.split(" / ");
                self.startDate(start);
                self.endDate(end);
            });

            this.customersReservationsByDate = ko.computed(() =>
                this.getConfigCustomersReservationsByDate()
            );

            this.customersByTable = ko.computed(() =>
                this.getConfigCustomersByTable()
            );

            this.customersByTableHours = ko.computed(() =>
                this.getConfigCustomersByTableHours()
            );
        },

        pickerInit: function () {
            flatpickr.localize({
                months: {
                    longhand: [
                        "Enero",
                        "Febrero",
                        "Marzo",
                        "Abril",
                        "Mayo",
                        "Junio",
                        "Julio",
                        "Agosto",
                        "Septiembre",
                        "Octubre",
                        "Noviembre",
                        "Diciembre",
                    ],
                    shorthand: [
                        "Ene",
                        "Feb",
                        "Mar",
                        "Abr",
                        "May",
                        "Jun",
                        "Jul",
                        "Ago",
                        "Sep",
                        "Oct",
                        "Nov",
                        "Dic",
                    ],
                },
                weekdays: {
                    longhand: [
                        "Domingo",
                        "Lunes",
                        "Martes",
                        "Miércoles",
                        "Jueves",
                        "Viernes",
                        "Sábado",
                    ],
                    shorthand: [
                        "Dom",
                        "Lun",
                        "Mar",
                        "Mié",
                        "Jue",
                        "Vie",
                        "Sáb",
                    ],
                },
                rangeSeparator: " / ",
                firstDayOfWeek: 1,
            });

            const datePicker = flatpickr("#date-range", {
                mode: "range",
                maxDate: "today",
                dateFormat: "Y-m-d",
                altInput: true,
                altFormat: "d M y",
            });

            const { start, end } = this.getDefaultDates();
            datePicker.setDate([start, end]);
        },

        getDefaultDates: function () {
            const today = new Date();
            const yesterday = new Date();
            yesterday.setDate(today.getDate() - 1);

            return {
                start: yesterday.toISOString().split("T")[0],
                end: today.toISOString().split("T")[0],
            };
        },

        getConfigOrdersByCustomerType: function () {
            return {
                labels: ["Invitados", "Registrados"],
                datasets: [
                    {
                        label: "Total de reservaciones",
                        data: [this.totalByGuest(), this.totalByRegister()],
                        backgroundColor: ["#002A4B", "#79B8FF"],
                        borderColor: ["#002A4B", "#79B8FF"],
                        borderWidth: 1,
                        hoverOffset: 4,
                        spacing: 5,
                    },
                ],
            };
        },

        getDataOrdersByCustomerType: async function () {
            try {
                const url = `${this.baseUrl}/rest/V1/restaurateurs/${this.restaurateurId}/dashboard/orders-by-customer-type`;
                const response = await fetch(url, {
                    headers: this.headers,
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const orders = await response.json();

                this.totalByGuest(orders.guest);
                this.totalByRegister(orders.register);
            } catch (error) {
                console.error(error);
            }
        },

        getConfigOrdersByPeriod: function () {
            return {
                labels: this.ordersByPeriodLabels(),
                datasets: [
                    {
                        label: "Total de reservaciones",
                        data: this.ordersByPeriodData(),
                        backgroundColor: "#002A4B",
                        borderColor: "#002A4B",
                        borderWidth: 1,
                    },
                ],
            };
        },

        getOrdersByPeriod: async function () {
            try {
                const period = this.ordersPeriod();
                const url = `${this.baseUrl}/rest/V1/restaurateurs/${this.restaurateurId}/dashboard/orders-by-period?period=${period}`;
                const response = await fetch(url, {
                    headers: this.headers,
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const orders = await response.json();
                const labels = [];
                const data = [];

                orders.map((item) => {
                    labels.push(item.x);
                    data.push(item.y);
                });

                this.ordersByPeriodLabels(labels);
                this.ordersByPeriodData(data);
            } catch (error) {
                console.error(error);
            }
        },

        getConfigCustomersByPeriod: function () {
            return {
                labels: this.customersByPeriodLabels(),
                datasets: [
                    {
                        label: "Clientes Creados",
                        data: this.customersByPeriodData(),
                        backgroundColor: "#002A4B",
                        borderColor: "#002A4B",
                        borderWidth: 1,
                    },
                ],
            };
        },

        getCustomersByPeriod: async function () {
            try {
                const period = this.customersPeriod();
                const url = `${this.baseUrl}/rest/V1/restaurateurs/${this.restaurateurId}/dashboard/customers-by-period?period=${period}`;
                const response = await fetch(url, {
                    headers: this.headers,
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const customers = await response.json();
                const labels = [];
                const data = [];

                customers.map((item) => {
                    labels.push(item.x);
                    data.push(item.y);
                });

                this.customersByPeriodLabels(labels);
                this.customersByPeriodData(data);
            } catch (error) {
                console.error(error);
            }
        },

        getTrendsData: async function () {
            try {
                const url = `${
                    this.baseUrl
                }/restaurantTrends/ajax/trends?restaurant_id=${this.restaurantId()}&start_date=${this.startDate()}&end_date=${this.endDate()}`;
                const response = await fetch(url, {
                    headers: this.headers,
                });

                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }

                const { items } = await response.json();

                this.trendsData(items);

                this.getCustomersReservationsByDate();
                this.getCustomersByTable();
                this.getCustomersByTableHours();
            } catch (error) {
                console.error(error);
            }
        },

        getConfigCustomersReservationsByDate: function () {
            return {
                labels: this.customersReservationsLabels(),
                datasets: [
                    {
                        label: "Total de clientes",
                        data: this.totalCustomersByDate(),
                        backgroundColor: "#002A4B",
                        borderColor: "#002A4B",
                        borderWidth: 1,
                    },
                    {
                        label: "Total de reservaciones",
                        data: this.totalReservationsByDate(),
                        backgroundColor: "#79B8FF",
                        borderColor: "#79B8FF",
                        borderWidth: 1,
                    },
                ],
            };
        },

        getCustomersReservationsByDate: async function () {
            if ([this.restaurantId, this.rangeDates].includes("")) {
                alert("El restaurant y la fecha son requeridos.");
                return;
            }

            const labels = [];
            const customersData = [];
            const reservationsData = [];

            this.trendsData().predictions.forEach((item) => {
                labels.push(item.date);
                customersData.push(item.total_customers);
                reservationsData.push(item.total_reservations);
            });

            this.customersReservationsLabels(labels);
            this.totalCustomersByDate(customersData);
            this.totalReservationsByDate(reservationsData);
        },

        getConfigCustomersByTable: function () {
            return {
                labels: this.customersByTableLabels(),
                datasets: [
                    {
                        label: "Total de clientes",
                        data: this.customersByTableData(),
                        backgroundColor: "#002A4B",
                        borderColor: "#002A4B",
                        borderWidth: 1,
                    },
                ],
            };
        },

        getCustomersByTable: async function () {
            const tableTotals = {};

            this.trendsData().predictions.forEach((prediction) => {
                prediction.tables.forEach((table) => {
                    const label = table.table_label || `Mesa ${table.table_id}`;
                    tableTotals[label] =
                        (tableTotals[label] || 0) +
                        parseInt(table.total_customers);
                });
            });

            const labels = Object.keys(tableTotals);
            const customersData = Object.values(tableTotals);

            this.customersByTableLabels(labels);
            this.customersByTableData(customersData);
        },

        getConfigCustomersByTableHours: function () {
            return {
                labels: this.customersByTableHoursLabels(),
                datasets: this.customersByTableHoursData(),
            };
        },

        getCustomersByTableHours: async function () {
            const tableHourMap = {};
            const labelsSet = new Set();

            this.trendsData().predictions.forEach((prediction) => {
                prediction.tables.forEach((table) => {
                    const label = table.table_label || `Mesa ${table.table_id}`;
                    if (!tableHourMap[label]) {
                        tableHourMap[label] = {};
                    }

                    (table.by_hour || []).forEach((entry) => {
                        const hour = entry.hour.slice(0, 2);
                        labelsSet.add(hour);
                        const total = parseInt(entry.total_customers, 10);
                        tableHourMap[label][hour] =
                            (tableHourMap[label][hour] || 0) +
                            (Number.isNaN(total) ? 0 : total);
                    });
                });
            });

            const labels = Array.from(labelsSet).sort(
                (a, b) => parseInt(a) - parseInt(b)
            );

            const generateColors = (count) => {
                const colors = [];
                const hueStep = 360 / count;

                for (let i = 0; i < count; i++) {
                    const hue = Math.floor(hueStep * i);
                    colors.push({
                        background: `hsla(${hue}, 70%, 50%, 0.2)`,
                        border: `hsla(${hue}, 70%, 40%, 1)`,
                    });
                }
                return colors;
            };

            const tableLabels = Object.keys(tableHourMap);
            const colorPalette = generateColors(tableLabels.length);

            const datasets = tableLabels.map((label, index) => ({
                label: label,
                data: labels.map(
                    (h) => parseInt(tableHourMap[label][h], 10) || 0
                ),
                fill: false,
                tension: 0.3,
                borderWidth: 2,
                backgroundColor: colorPalette[index].background,
                borderColor: colorPalette[index].border,
                pointBackgroundColor: colorPalette[index].border,
                pointBorderColor: "#fff",
                pointHoverBackgroundColor: "#fff",
                pointHoverBorderColor: colorPalette[index].border,
            }));

            this.customersByTableHoursLabels(labels);
            this.customersByTableHoursData(datasets);
        },
    });
});
